"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const eui_1 = require("@elastic/eui");
const eui_2 = require("@elastic/eui");
const moment_1 = tslib_1.__importDefault(require("moment"));
const react_1 = tslib_1.__importDefault(require("react"));
const styled_components_1 = tslib_1.__importDefault(require("styled-components"));
const i18n_1 = require("x-pack/plugins/apm/common/i18n");
const variables_1 = require("../../../style/variables");
const TooltipFieldName = styled_components_1.default.span `
  font-family: ${variables_1.fontFamilyCode};
`;
const PropertyLabel = styled_components_1.default.div `
  margin-bottom: ${variables_1.px(variables_1.units.half)};
  font-size: ${variables_1.fontSizes.small};
  color: ${variables_1.colors.gray3};

  span {
    cursor: help;
  }
`;
PropertyLabel.displayName = 'PropertyLabel';
const PropertyValueDimmed = styled_components_1.default.span `
  color: ${variables_1.colors.gray3};
`;
const PropertyValue = styled_components_1.default.div `
  display: inline-block;
  line-height: ${variables_1.px(variables_1.unit)};
`;
PropertyValue.displayName = 'PropertyValue';
const PropertyValueTruncated = styled_components_1.default.span `
  display: inline-block;
  line-height: ${variables_1.px(variables_1.unit)};
  ${variables_1.truncate('100%')};
`;
function TimestampValue({ timestamp }) {
    const time = moment_1.default(timestamp);
    const timeAgo = timestamp ? time.fromNow() : i18n_1.NOT_AVAILABLE_LABEL;
    const timestampFull = timestamp
        ? time.format('MMMM Do YYYY, HH:mm:ss.SSS')
        : i18n_1.NOT_AVAILABLE_LABEL;
    return (react_1.default.createElement(PropertyValue, null,
        timeAgo,
        " ",
        react_1.default.createElement(PropertyValueDimmed, null,
            "(",
            timestampFull,
            ")")));
}
function getPropertyLabel({ fieldName, label }) {
    if (fieldName) {
        return (react_1.default.createElement(PropertyLabel, null,
            react_1.default.createElement(eui_2.EuiToolTip, { content: react_1.default.createElement(TooltipFieldName, null, fieldName) },
                react_1.default.createElement("span", null, label))));
    }
    return react_1.default.createElement(PropertyLabel, null, label);
}
function getPropertyValue({ val, fieldName, truncated = false }) {
    if (fieldName === '@timestamp') {
        return react_1.default.createElement(TimestampValue, { timestamp: val });
    }
    if (truncated) {
        return (react_1.default.createElement(eui_2.EuiToolTip, { content: String(val) },
            react_1.default.createElement(PropertyValueTruncated, null, String(val))));
    }
    return react_1.default.createElement(PropertyValue, null, val);
}
function StickyProperties({ stickyProperties }) {
    /**
     * Note: the padding and margin styles here are strange because
     * EUI flex groups and items have a default "gutter" applied that
     * won't allow percentage widths to line up correctly, so we have
     * to turn the gutter off with gutterSize: none. When we do that,
     * the top/bottom spacing *also* collapses, so we have to add
     * padding between each item without adding it to the outside of
     * the flex group itself.
     *
     * Hopefully we can make EUI handle this better and remove all this.
     */
    const itemStyles = {
        padding: '1em 1em 1em 0'
    };
    const groupStyles = {
        marginTop: '-1em',
        marginBottom: '-1em'
    };
    return (react_1.default.createElement(eui_1.EuiFlexGroup, { wrap: true, gutterSize: "none", style: groupStyles }, stickyProperties &&
        stickyProperties.map(({ width = 0, ...prop }, i) => {
            return (react_1.default.createElement(eui_1.EuiFlexItem, { key: i, style: {
                    minWidth: width,
                    ...itemStyles
                } },
                getPropertyLabel(prop),
                getPropertyValue(prop)));
        })));
}
exports.StickyProperties = StickyProperties;
