"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const eui_1 = require("@elastic/eui");
const eui_2 = require("@elastic/eui");
const i18n_1 = require("@kbn/i18n");
const lodash_1 = require("lodash");
const react_1 = tslib_1.__importDefault(require("react"));
const styled_components_1 = tslib_1.__importDefault(require("styled-components"));
const variables_1 = require("../../../style/variables");
const agents_1 = require("../../../utils/documentation/agents");
const NestedKeyValueTable_1 = require("./NestedKeyValueTable");
const propertyConfig_1 = require("./propertyConfig");
const TableContainer = styled_components_1.default.div `
  padding-bottom: ${variables_1.px(variables_1.units.double)};
`;
const TableInfo = styled_components_1.default.div `
  padding: ${variables_1.px(variables_1.unit)} 0 0;
  text-align: center;
  font-size: ${variables_1.fontSize};
  color: ${variables_1.colors.gray2};
  line-height: 1.5;
`;
const TableInfoHeader = styled_components_1.default(TableInfo) `
  font-size: ${variables_1.fontSizes.large};
  color: ${variables_1.colors.black2};
`;
const EuiIconWithSpace = styled_components_1.default(eui_1.EuiIcon) `
  margin-right: ${variables_1.px(variables_1.units.half)};
`;
function getPropertyTabNames(selected) {
    return propertyConfig_1.PROPERTY_CONFIG.filter(({ key, required }) => required || selected.includes(key)).map(({ key, label }) => ({ key, label }));
}
exports.getPropertyTabNames = getPropertyTabNames;
function getAgentFeatureText(featureName) {
    switch (featureName) {
        case 'user':
            return i18n_1.i18n.translate('xpack.apm.propertiesTable.userTab.agentFeatureText', {
                defaultMessage: 'You can configure your agent to add contextual information about your users.'
            });
        case 'tags':
            return i18n_1.i18n.translate('xpack.apm.propertiesTable.tagsTab.agentFeatureText', {
                defaultMessage: 'You can configure your agent to add filterable tags on transactions.'
            });
        case 'custom':
            return i18n_1.i18n.translate('xpack.apm.propertiesTable.customTab.agentFeatureText', {
                defaultMessage: 'You can configure your agent to add custom contextual information on transactions.'
            });
    }
}
function AgentFeatureTipMessage({ featureName, agentName }) {
    const docsUrl = agents_1.getAgentFeatureDocsUrl(featureName, agentName);
    if (!docsUrl) {
        return null;
    }
    return (react_1.default.createElement(TableInfo, null,
        react_1.default.createElement(EuiIconWithSpace, { type: "iInCircle" }),
        getAgentFeatureText(featureName),
        ' ',
        react_1.default.createElement(eui_2.EuiLink, { target: "_blank", rel: "noopener", href: docsUrl }, i18n_1.i18n.translate('xpack.apm.propertiesTable.agentFeature.learnMoreLinkLabel', { defaultMessage: 'Learn more in the documentation.' }))));
}
exports.AgentFeatureTipMessage = AgentFeatureTipMessage;
exports.sortKeysByConfig = (object, currentKey) => {
    const indexedPropertyConfig = lodash_1.indexBy(propertyConfig_1.PROPERTY_CONFIG, 'key');
    const presorted = lodash_1.get(indexedPropertyConfig, `${currentKey}.presortedKeys`, []);
    return lodash_1.uniq([...presorted, ...Object.keys(object).sort()]);
};
function PropertiesTable({ propData, propKey, agentName }) {
    return (react_1.default.createElement(TableContainer, null,
        propData ? (react_1.default.createElement(NestedKeyValueTable_1.NestedKeyValueTable, { data: propData, parentKey: propKey, keySorter: exports.sortKeysByConfig, depth: 1 })) : (react_1.default.createElement(TableInfoHeader, null, i18n_1.i18n.translate('xpack.apm.propertiesTable.agentFeature.noDataAvailableLabel', { defaultMessage: 'No data available' }))),
        react_1.default.createElement(AgentFeatureTipMessage, { featureName: propKey, agentName: agentName })));
}
exports.PropertiesTable = PropertiesTable;
