"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const createHashHistory_1 = tslib_1.__importDefault(require("history/createHashHistory"));
const lodash_1 = require("lodash");
const querystring_1 = tslib_1.__importDefault(require("querystring"));
const rison_node_1 = tslib_1.__importDefault(require("rison-node"));
const chrome_1 = tslib_1.__importDefault(require("ui/chrome"));
const url_1 = tslib_1.__importDefault(require("url"));
function toQuery(search) {
    return search ? querystring_1.default.parse(search.slice(1)) : {};
}
exports.toQuery = toQuery;
function fromQuery(query) {
    const encodedQuery = encodeQuery(query, ['_g', '_a']);
    return stringifyWithoutEncoding(encodedQuery);
}
exports.fromQuery = fromQuery;
function encodeQuery(query, exclude = []) {
    return lodash_1.mapValues(query, (value, key) => {
        if (exclude.includes(key)) {
            return encodeURI(value);
        }
        return querystring_1.default.escape(value);
    });
}
exports.encodeQuery = encodeQuery;
function stringifyWithoutEncoding(query) {
    return querystring_1.default.stringify(query, undefined, undefined, {
        encodeURIComponent: (v) => v
    });
}
function risonSafeDecode(value) {
    if (!value) {
        return {};
    }
    try {
        const decoded = rison_node_1.default.decode(value);
        return lodash_1.isPlainObject(decoded) ? decoded : {};
    }
    catch (e) {
        return {};
    }
}
// Kibana default set in: https://github.com/elastic/kibana/blob/e13e47fc4eb6112f2a5401408e9f765eae90f55d/x-pack/plugins/apm/public/utils/timepicker/index.js#L31-L35
// TODO: store this in config or a shared constant?
const DEFAULT_KIBANA_TIME_RANGE = {
    time: {
        from: 'now-24h',
        mode: 'quick',
        to: 'now'
    }
};
function getQueryWithRisonParams(location, query = {}) {
    // Preserve current _g and _a
    const currentQuery = toQuery(location.search);
    const decodedG = risonSafeDecode(currentQuery._g);
    const combinedG = { ...DEFAULT_KIBANA_TIME_RANGE, ...decodedG, ...query._g };
    const encodedG = rison_node_1.default.encode(combinedG);
    const encodedA = query._a ? rison_node_1.default.encode(query._a) : '';
    const queryWithRisonParams = {
        ...query,
        _g: encodedG,
        _a: encodedA
    };
    if (currentQuery.kuery) {
        queryWithRisonParams.kuery = currentQuery.kuery;
    }
    return queryWithRisonParams;
}
function getKibanaHref({ location, pathname = '', hash, query = {} }) {
    const queryWithRisonParams = getQueryWithRisonParams(location, query);
    const search = stringifyWithoutEncoding(queryWithRisonParams);
    const href = url_1.default.format({
        pathname: chrome_1.default.addBasePath(pathname),
        hash: `${hash}?${search}`
    });
    return href;
}
exports.getKibanaHref = getKibanaHref;
// This is downright horrible 😭 💔
// Angular decodes encoded url tokens like "%2F" to "/" which causes problems when path params contains forward slashes
// This was originally fixed in Angular, but roled back to avoid breaking backwards compatability: https://github.com/angular/angular.js/commit/2bdf7126878c87474bb7588ce093d0a3c57b0026
function legacyEncodeURIComponent(rawUrl) {
    return (rawUrl &&
        encodeURIComponent(rawUrl)
            .replace(/~/g, '%7E')
            .replace(/%/g, '~'));
}
exports.legacyEncodeURIComponent = legacyEncodeURIComponent;
function legacyDecodeURIComponent(encodedUrl) {
    return encodedUrl && decodeURIComponent(encodedUrl.replace(/~/g, '%'));
}
exports.legacyDecodeURIComponent = legacyDecodeURIComponent;
// Make history singleton available across APM project.
// This is not great. Other options are to use context or withRouter helper
// React Context API is unstable and will change soon-ish (probably 16.3)
// withRouter helper from react-router overrides several props (eg. `location`) which makes it less desireable
exports.history = createHashHistory_1.default();
