"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const i18n_1 = require("@kbn/i18n");
const lodash_1 = require("lodash");
const chrome_1 = tslib_1.__importDefault(require("ui/chrome"));
const url_1 = tslib_1.__importDefault(require("url"));
const uuid_1 = tslib_1.__importDefault(require("uuid"));
const constants_1 = require("../../../../../common/constants");
// @ts-ignore
const watcher_1 = require("../../../../services/rest/watcher");
function getSlackPathUrl(slackUrl) {
    if (slackUrl) {
        const { path } = url_1.default.parse(slackUrl);
        return path;
    }
}
async function createErrorGroupWatch({ emails = [], schedule, serviceName, slackUrl, threshold, timeRange }) {
    const id = `apm-${uuid_1.default.v4()}`;
    const apmIndexPatternTitle = chrome_1.default.getInjected('apmIndexPatternTitle');
    const slackUrlPath = getSlackPathUrl(slackUrl);
    const emailTemplate = i18n_1.i18n.translate('xpack.apm.serviceDetails.enableErrorReportsPanel.emailTemplateText', {
        defaultMessage: 'Your service {serviceName} has error groups which exceeds {threshold} occurrences within {timeRange}{br}' +
            '{br}' +
            '{errorGroupsBuckets}{br}' +
            '{errorLogMessage}{br}' +
            '{errorCulprit}N/A{slashErrorCulprit}{br}' +
            '{docCountParam} occurrences{br}' +
            '{slashErrorGroupsBucket}',
        values: {
            serviceName: '"{{ctx.metadata.serviceName}}"',
            threshold: '{{ctx.metadata.threshold}}',
            timeRange: '"{{ctx.metadata.timeRangeValue}}{{ctx.metadata.timeRangeUnit}}"',
            errorGroupsBuckets: '{{#ctx.payload.aggregations.error_groups.buckets}}',
            errorLogMessage: '<strong>{{sample.hits.hits.0._source.error.log.message}}{{^sample.hits.hits.0._source.error.log.message}}{{sample.hits.hits.0._source.error.exception.message}}{{/sample.hits.hits.0._source.error.log.message}}</strong>',
            errorCulprit: '{{sample.hits.hits.0._source.error.culprit}}{{^sample.hits.hits.0._source.error.culprit}}',
            slashErrorCulprit: '{{/sample.hits.hits.0._source.error.culprit}}',
            docCountParam: '{{doc_count}}',
            slashErrorGroupsBucket: '{{/ctx.payload.aggregations.error_groups.buckets}}',
            br: '<br/>'
        }
    });
    const slackTemplate = i18n_1.i18n.translate('xpack.apm.serviceDetails.enableErrorReportsPanel.slackTemplateText', {
        defaultMessage: `Your service {serviceName} has error groups which exceeds {threshold} occurrences within {timeRange}
{errorGroupsBuckets}
{errorLogMessage}
{errorCulprit}N/A{slashErrorCulprit}
{docCountParam} occurrences
{slashErrorGroupsBucket}`,
        values: {
            serviceName: '"{{ctx.metadata.serviceName}}"',
            threshold: '{{ctx.metadata.threshold}}',
            timeRange: '"{{ctx.metadata.timeRangeValue}}{{ctx.metadata.timeRangeUnit}}"',
            errorGroupsBuckets: '{{#ctx.payload.aggregations.error_groups.buckets}}',
            errorLogMessage: '>*{{sample.hits.hits.0._source.error.log.message}}{{^sample.hits.hits.0._source.error.log.message}}{{sample.hits.hits.0._source.error.exception.message}}{{/sample.hits.hits.0._source.error.log.message}}*',
            errorCulprit: '>{{#sample.hits.hits.0._source.error.culprit}}`{{sample.hits.hits.0._source.error.culprit}}`{{/sample.hits.hits.0._source.error.culprit}}{{^sample.hits.hits.0._source.error.culprit}}',
            slashErrorCulprit: '{{/sample.hits.hits.0._source.error.culprit}}',
            docCountParam: '>{{doc_count}}',
            slashErrorGroupsBucket: '{{/ctx.payload.aggregations.error_groups.buckets}}'
        }
    });
    const actions = {
        log_error: { logging: { text: emailTemplate } }
    };
    const body = {
        metadata: {
            emails,
            trigger: i18n_1.i18n.translate('xpack.apm.serviceDetails.enableErrorReportsPanel.triggerText', {
                defaultMessage: 'This value must be changed in trigger section'
            }),
            serviceName,
            threshold,
            timeRangeValue: timeRange.value,
            timeRangeUnit: timeRange.unit,
            slackUrlPath
        },
        trigger: {
            schedule
        },
        input: {
            search: {
                request: {
                    indices: [apmIndexPatternTitle],
                    body: {
                        size: 0,
                        query: {
                            bool: {
                                filter: [
                                    { term: { [constants_1.SERVICE_NAME]: '{{ctx.metadata.serviceName}}' } },
                                    { term: { [constants_1.PROCESSOR_EVENT]: 'error' } },
                                    {
                                        range: {
                                            '@timestamp': {
                                                gte: 'now-{{ctx.metadata.timeRangeValue}}{{ctx.metadata.timeRangeUnit}}'
                                            }
                                        }
                                    }
                                ]
                            }
                        },
                        aggs: {
                            error_groups: {
                                terms: {
                                    min_doc_count: '{{ctx.metadata.threshold}}',
                                    field: constants_1.ERROR_GROUP_ID,
                                    size: 10,
                                    order: {
                                        _count: 'desc'
                                    }
                                },
                                aggs: {
                                    sample: {
                                        top_hits: {
                                            _source: [
                                                constants_1.ERROR_LOG_MESSAGE,
                                                constants_1.ERROR_EXC_MESSAGE,
                                                constants_1.ERROR_EXC_HANDLED,
                                                constants_1.ERROR_CULPRIT,
                                                constants_1.ERROR_GROUP_ID,
                                                '@timestamp'
                                            ],
                                            sort: [
                                                {
                                                    '@timestamp': 'desc'
                                                }
                                            ],
                                            size: 1
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        },
        condition: {
            script: {
                source: 'return ctx.payload.aggregations.error_groups.buckets.length > 0'
            }
        },
        actions
    };
    if (slackUrlPath) {
        body.actions.slack_webhook = {
            webhook: {
                scheme: 'https',
                host: 'hooks.slack.com',
                port: 443,
                method: 'POST',
                path: '{{ctx.metadata.slackUrlPath}}',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: `__json__::${JSON.stringify({
                    text: slackTemplate
                })}`
            }
        };
    }
    if (!lodash_1.isEmpty(emails)) {
        body.actions.email = {
            email: {
                to: '{{#join}}ctx.metadata.emails{{/join}}',
                subject: i18n_1.i18n.translate('xpack.apm.serviceDetails.enableErrorReportsPanel.emailSubjectText', {
                    defaultMessage: '{serviceName} has error groups which exceeds the threshold',
                    values: { serviceName: '"{{ctx.metadata.serviceName}}"' }
                }),
                body: {
                    html: emailTemplate
                }
            }
        };
    }
    await watcher_1.createWatch(id, body);
    return id;
}
exports.createErrorGroupWatch = createErrorGroupWatch;
