"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const eui_1 = require("@elastic/eui");
const i18n_1 = require("@kbn/i18n");
const react_1 = require("@kbn/i18n/react");
const react_2 = tslib_1.__importStar(require("react"));
const notify_1 = require("ui/notify");
const KibanaLink_1 = require("x-pack/plugins/apm/public/components/shared/Links/KibanaLink");
const MLJobLink_1 = require("x-pack/plugins/apm/public/components/shared/Links/MLJobLink");
const ml_1 = require("x-pack/plugins/apm/public/services/rest/ml");
const savedObjects_1 = require("x-pack/plugins/apm/public/services/rest/savedObjects");
const machineLearningJobs_1 = require("x-pack/plugins/apm/public/store/reactReduxRequest/machineLearningJobs");
const TransactionSelect_1 = require("./TransactionSelect");
class MachineLearningFlyout extends react_2.Component {
    constructor() {
        super(...arguments);
        this.state = {
            isLoading: false,
            hasIndexPattern: false,
            hasMLJob: false,
            selectedTransactionType: this.props.urlParams.transactionType
        };
        this.createJob = async () => {
            this.setState({ isLoading: true });
            try {
                const { serviceName, transactionType } = this.props.urlParams;
                if (serviceName) {
                    const res = await ml_1.startMLJob({ serviceName, transactionType });
                    const didSucceed = res.datafeeds[0].success && res.jobs[0].success;
                    if (!didSucceed) {
                        throw new Error('Creating ML job failed');
                    }
                    this.addSuccessToast();
                }
            }
            catch (e) {
                this.addErrorToast();
            }
            this.setState({ isLoading: false });
            this.props.onClose();
        };
        this.addErrorToast = () => {
            const { urlParams } = this.props;
            const { serviceName = 'unknown' } = urlParams;
            if (!serviceName) {
                return;
            }
            notify_1.toastNotifications.addWarning({
                title: i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.jobCreationFailedNotificationTitle', {
                    defaultMessage: 'Job creation failed'
                }),
                text: (react_2.default.createElement("p", null, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.jobCreationFailedNotificationText', {
                    defaultMessage: 'Your current license may not allow for creating machine learning jobs, or this job may already exist.'
                })))
            });
        };
        this.addSuccessToast = () => {
            const { location, urlParams } = this.props;
            const { serviceName = 'unknown', transactionType } = urlParams;
            notify_1.toastNotifications.addSuccess({
                title: i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.jobCreatedNotificationTitle', {
                    defaultMessage: 'Job successfully created'
                }),
                text: (react_2.default.createElement("p", null,
                    i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.jobCreatedNotificationText', {
                        defaultMessage: 'The analysis is now running for {serviceName} ({transactionType}). It might take a while before results are added to the response times graph.',
                        values: {
                            serviceName,
                            transactionType
                        }
                    }),
                    ' ',
                    react_2.default.createElement(MLJobLink_1.MLJobLink, { serviceName: serviceName, transactionType: transactionType, location: location }, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.jobCreatedNotificationText.viewJobLinkText', {
                        defaultMessage: 'View job'
                    }))))
            });
        };
    }
    async componentDidMount() {
        const indexPattern = await savedObjects_1.getAPMIndexPattern();
        this.setState({ hasIndexPattern: !!indexPattern });
    }
    componentDidUpdate(prevProps) {
        if (prevProps.urlParams.transactionType !==
            this.props.urlParams.transactionType) {
            this.setState({
                selectedTransactionType: this.props.urlParams.transactionType
            });
        }
    }
    render() {
        const { isOpen, onClose, urlParams, location } = this.props;
        const { serviceName, transactionType } = urlParams;
        const { isLoading, hasIndexPattern, selectedTransactionType } = this.state;
        if (!isOpen || !serviceName) {
            return null;
        }
        return (react_2.default.createElement(machineLearningJobs_1.MLJobsRequest, { serviceName: serviceName, render: ({ data, status }) => {
                if (status === 'LOADING') {
                    return null;
                }
                const hasMLJob = data.jobs.some(job => job.jobId &&
                    job.jobId.startsWith(ml_1.getMlPrefix(serviceName, selectedTransactionType)));
                return (react_2.default.createElement(eui_1.EuiFlyout, { onClose: onClose, size: "s" },
                    react_2.default.createElement(eui_1.EuiFlyoutHeader, null,
                        react_2.default.createElement(eui_1.EuiTitle, null,
                            react_2.default.createElement("h2", null, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.enableAnomalyDetectionTitle', {
                                defaultMessage: 'Enable anomaly detection'
                            }))),
                        react_2.default.createElement(eui_1.EuiSpacer, { size: "s" })),
                    react_2.default.createElement(eui_1.EuiFlyoutBody, null,
                        hasMLJob && (react_2.default.createElement("div", null,
                            react_2.default.createElement(eui_1.EuiCallOut, { title: i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.callout.jobExistsTitle', {
                                    defaultMessage: 'Job already exists'
                                }), color: "success", iconType: "check" },
                                react_2.default.createElement("p", null,
                                    i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.callout.jobExistsDescription', {
                                        defaultMessage: 'There is currently a job running for {serviceName} ({transactionType}).',
                                        values: {
                                            serviceName,
                                            transactionType
                                        }
                                    }),
                                    ' ',
                                    react_2.default.createElement(MLJobLink_1.MLJobLink, { serviceName: serviceName, transactionType: transactionType, location: location }, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.callout.jobExistsDescription.viewJobLinkText', {
                                        defaultMessage: 'View existing job'
                                    })))),
                            react_2.default.createElement(eui_1.EuiSpacer, { size: "m" }))),
                        !hasIndexPattern && (react_2.default.createElement("div", null,
                            react_2.default.createElement(eui_1.EuiCallOut, { title: react_2.default.createElement("span", null,
                                    react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.apm.serviceDetails.enableAnomalyDetectionPanel.callout.noPatternTitle", defaultMessage: "No APM index pattern available. To create a job, please import the APM index pattern via the {setupInstructionLink}", values: {
                                            setupInstructionLink: (react_2.default.createElement(KibanaLink_1.KibanaLink, { pathname: '/app/kibana', hash: `/home/tutorial/apm` }, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.callout.noPatternTitle.setupInstructionLinkText', {
                                                defaultMessage: 'Setup Instructions'
                                            })))
                                        } })), color: "warning", iconType: "alert" }),
                            react_2.default.createElement(eui_1.EuiSpacer, { size: "m" }))),
                        react_2.default.createElement(eui_1.EuiText, null,
                            react_2.default.createElement("p", null,
                                react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.apm.serviceDetails.enableAnomalyDetectionPanel.createMLJobDescription", defaultMessage: "Here you can create a machine learning job to calculate anomaly scores on durations for APM transactions\n                        within the {serviceName} service. Once enabled, {transactionDurationGraphText} will show the expected bounds and annotate\n                        the graph once the anomaly score is >=75.", values: {
                                        serviceName,
                                        transactionDurationGraphText: (react_2.default.createElement("b", null, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.createMLJobDescription.transactionDurationGraphText', {
                                            defaultMessage: 'the transaction duration graph'
                                        })))
                                    } })),
                            react_2.default.createElement("p", null,
                                react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.apm.serviceDetails.enableAnomalyDetectionPanel.manageMLJobDescription", defaultMessage: "Jobs can be created for each service + transaction type combination.\n                        Once a job is created, you can manage it and see more details in the {mlJobsPageLink}.", values: {
                                        mlJobsPageLink: (react_2.default.createElement(KibanaLink_1.KibanaLink, { pathname: '/app/ml' }, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.manageMLJobDescription.mlJobsPageLinkText', {
                                            defaultMessage: 'Machine Learning jobs management page'
                                        })))
                                    } }),
                                ' ',
                                react_2.default.createElement("em", null, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.manageMLJobDescription.noteText', {
                                    defaultMessage: 'Note: It might take a few minutes for the job to begin calculating results.'
                                })))),
                        react_2.default.createElement(eui_1.EuiSpacer, null)),
                    react_2.default.createElement(eui_1.EuiFlyoutFooter, null,
                        react_2.default.createElement(eui_1.EuiFlexGroup, { justifyContent: "spaceBetween", alignItems: "flexEnd" },
                            react_2.default.createElement(eui_1.EuiFlexItem, null, this.props.serviceTransactionTypes.length > 1 ? (react_2.default.createElement(TransactionSelect_1.TransactionSelect, { types: this.props.serviceTransactionTypes, selected: this.state.selectedTransactionType, existingJobs: data.jobs, onChange: value => this.setState({
                                    selectedTransactionType: value
                                }) })) : null),
                            react_2.default.createElement(eui_1.EuiFlexItem, { grow: false },
                                react_2.default.createElement(eui_1.EuiFormRow, null,
                                    react_2.default.createElement(eui_1.EuiButton, { onClick: this.createJob, fill: true, disabled: isLoading || hasMLJob || !hasIndexPattern }, i18n_1.i18n.translate('xpack.apm.serviceDetails.enableAnomalyDetectionPanel.createNewJobButtonLabel', {
                                        defaultMessage: 'Create new job'
                                    }))))))));
            } }));
    }
}
exports.MachineLearningFlyout = MachineLearningFlyout;
