"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const eui_1 = require("@elastic/eui");
const i18n_1 = require("@kbn/i18n");
const idx_1 = tslib_1.__importDefault(require("idx"));
const lodash_1 = require("lodash");
const react_1 = tslib_1.__importDefault(require("react"));
const styled_components_1 = tslib_1.__importDefault(require("styled-components"));
const constants_1 = require("x-pack/plugins/apm/common/constants");
const i18n_2 = require("x-pack/plugins/apm/common/i18n");
const KibanaLink_1 = require("x-pack/plugins/apm/public/components/shared/Links/KibanaLink");
const url_helpers_1 = require("x-pack/plugins/apm/public/components/shared/Links/url_helpers");
const url_helpers_2 = require("x-pack/plugins/apm/public/components/shared/Links/url_helpers");
const constants_2 = require("x-pack/plugins/apm/public/constants");
const constants_3 = require("../../../../../common/constants");
const variables_1 = require("../../../../style/variables");
const DiscoverErrorLink_1 = require("../../../shared/Links/DiscoverLinks/DiscoverErrorLink");
const PropertiesTable_1 = require("../../../shared/PropertiesTable");
const Stacktrace_1 = require("../../../shared/Stacktrace");
const StickyProperties_1 = require("../../../shared/StickyProperties");
const Container = styled_components_1.default.div `
  position: relative;
  border: 1px solid ${variables_1.colors.gray4};
  border-radius: ${variables_1.borderRadius};
  margin-top: ${variables_1.px(variables_1.units.plus)};
`;
const HeaderContainer = styled_components_1.default.div `
  display: flex;
  justify-content: space-between;
  align-items: flex-start;
  padding: ${variables_1.px(variables_1.units.plus)} ${variables_1.px(variables_1.units.plus)} 0;
  margin-bottom: ${variables_1.px(variables_1.unit)};
`;
const PaddedContainer = styled_components_1.default.div `
  padding: ${variables_1.px(variables_1.units.plus)} ${variables_1.px(variables_1.units.plus)} 0;
`;
const logStacktraceTab = {
    key: 'log_stacktrace',
    label: i18n_1.i18n.translate('xpack.apm.propertiesTable.tabs.logStacktraceLabel', {
        defaultMessage: 'Log stacktrace'
    })
};
const exceptionStacktraceTab = {
    key: 'exception_stacktrace',
    label: i18n_1.i18n.translate('xpack.apm.propertiesTable.tabs.exceptionStacktraceLabel', {
        defaultMessage: 'Exception stacktrace'
    })
};
function DetailView({ errorGroup, urlParams, location }) {
    if (errorGroup.status !== constants_2.STATUS.SUCCESS) {
        return null;
    }
    const { transaction, error, occurrencesCount } = errorGroup.data;
    if (!error) {
        return null;
    }
    const transactionLink = getTransactionLink(error, transaction);
    const stickyProperties = [
        {
            fieldName: '@timestamp',
            label: i18n_1.i18n.translate('xpack.apm.errorGroupDetails.timestampLabel', {
                defaultMessage: 'Timestamp'
            }),
            val: error['@timestamp'],
            width: '50%'
        },
        {
            fieldName: constants_3.REQUEST_URL_FULL,
            label: 'URL',
            val: idx_1.default(error, _ => _.context.page.url) ||
                idx_1.default(transaction, _ => _.context.request.url.full) ||
                i18n_2.NOT_AVAILABLE_LABEL,
            truncated: true,
            width: '50%'
        },
        {
            fieldName: constants_3.REQUEST_METHOD,
            label: i18n_1.i18n.translate('xpack.apm.errorGroupDetails.requestMethodLabel', {
                defaultMessage: 'Request method'
            }),
            val: lodash_1.get(error, constants_3.REQUEST_METHOD, i18n_2.NOT_AVAILABLE_LABEL),
            width: '25%'
        },
        {
            fieldName: constants_3.ERROR_EXC_HANDLED,
            label: i18n_1.i18n.translate('xpack.apm.errorGroupDetails.handledLabel', {
                defaultMessage: 'Handled'
            }),
            val: String(lodash_1.get(error, constants_3.ERROR_EXC_HANDLED, i18n_2.NOT_AVAILABLE_LABEL)),
            width: '25%'
        },
        {
            fieldName: constants_3.TRANSACTION_ID,
            label: i18n_1.i18n.translate('xpack.apm.errorGroupDetails.transactionSampleIdLabel', {
                defaultMessage: 'Transaction sample ID'
            }),
            val: transactionLink || i18n_2.NOT_AVAILABLE_LABEL,
            width: '25%'
        },
        {
            fieldName: constants_3.USER_ID,
            label: i18n_1.i18n.translate('xpack.apm.errorGroupDetails.userIdLabel', {
                defaultMessage: 'User ID'
            }),
            val: lodash_1.get(error, constants_3.USER_ID, i18n_2.NOT_AVAILABLE_LABEL),
            width: '25%'
        }
    ];
    const tabs = getTabs(error);
    const currentTab = getCurrentTab(tabs, urlParams.detailTab);
    return (react_1.default.createElement(Container, null,
        react_1.default.createElement(HeaderContainer, null,
            react_1.default.createElement(eui_1.EuiTitle, { size: "s" },
                react_1.default.createElement("h3", null, i18n_1.i18n.translate('xpack.apm.errorGroupDetails.errorOccurrenceTitle', {
                    defaultMessage: 'Error occurrence'
                }))),
            react_1.default.createElement(DiscoverErrorLink_1.DiscoverErrorLink, { error: error, kuery: urlParams.kuery },
                react_1.default.createElement(eui_1.EuiButtonEmpty, { iconType: "discoverApp" }, i18n_1.i18n.translate('xpack.apm.errorGroupDetails.viewOccurrencesInDiscoverButtonLabel', {
                    defaultMessage: 'View {occurrencesCount} occurrences in Discover',
                    values: { occurrencesCount }
                })))),
        react_1.default.createElement(PaddedContainer, null,
            react_1.default.createElement(StickyProperties_1.StickyProperties, { stickyProperties: stickyProperties })),
        react_1.default.createElement(eui_1.EuiSpacer, null),
        react_1.default.createElement(eui_1.EuiTabs, null, tabs.map(({ key, label }) => {
            return (react_1.default.createElement(eui_1.EuiTab, { onClick: () => {
                    url_helpers_2.history.replace({
                        ...location,
                        search: url_helpers_2.fromQuery({
                            ...url_helpers_2.toQuery(location.search),
                            detailTab: key
                        })
                    });
                }, isSelected: currentTab.key === key, key: key }, label));
        })),
        react_1.default.createElement(PaddedContainer, null,
            react_1.default.createElement(TabContent, { error: error, currentTab: currentTab }))));
}
exports.DetailView = DetailView;
function getTransactionLink(error, transaction) {
    if (!transaction || !lodash_1.get(error, 'transaction.sampled')) {
        return;
    }
    const path = `/${transaction.context.service.name}/transactions/${url_helpers_1.legacyEncodeURIComponent(transaction.transaction.type)}/${url_helpers_1.legacyEncodeURIComponent(transaction.transaction.name)}`;
    return (react_1.default.createElement(KibanaLink_1.KibanaLink, { hash: path, query: {
            transactionId: transaction.transaction.id,
            traceId: lodash_1.get(transaction, constants_3.TRACE_ID)
        } }, transaction.transaction.id));
}
function TabContent({ error, currentTab }) {
    const codeLanguage = error.context.service.name;
    const agentName = error.context.service.agent.name;
    const excStackframes = lodash_1.get(error, constants_1.ERROR_EXC_STACKTRACE);
    const logStackframes = lodash_1.get(error, constants_1.ERROR_LOG_STACKTRACE);
    switch (currentTab.key) {
        case logStacktraceTab.key:
            return (react_1.default.createElement(Stacktrace_1.Stacktrace, { stackframes: logStackframes, codeLanguage: codeLanguage }));
        case exceptionStacktraceTab.key:
            return (react_1.default.createElement(Stacktrace_1.Stacktrace, { stackframes: excStackframes, codeLanguage: codeLanguage }));
        default:
            const propData = error.context[currentTab.key];
            return (react_1.default.createElement(PropertiesTable_1.PropertiesTable, { propData: propData, propKey: currentTab.key, agentName: agentName }));
    }
}
exports.TabContent = TabContent;
// Ensure the selected tab exists or use the first
function getCurrentTab(tabs = [], selectedTabKey) {
    const selectedTab = tabs.find(({ key }) => key === selectedTabKey);
    return selectedTab ? selectedTab : lodash_1.first(tabs) || {};
}
exports.getCurrentTab = getCurrentTab;
function getTabs(error) {
    const hasLogStacktrace = lodash_1.get(error, constants_1.ERROR_LOG_STACKTRACE, []).length > 0;
    const contextKeys = Object.keys(error.context);
    return [
        ...(hasLogStacktrace ? [logStacktraceTab] : []),
        exceptionStacktraceTab,
        ...PropertiesTable_1.getPropertyTabNames(contextKeys)
    ];
}
exports.getTabs = getTabs;
