"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const constants_1 = require("../nodes/constants");
class ElasticsearchMetadataAdapter {
    constructor(framework) {
        this.framework = framework;
    }
    async getMetricMetadata(req, sourceConfiguration, nodeId, nodeType) {
        const idFieldName = getIdFieldName(sourceConfiguration, nodeType);
        const metricQuery = {
            index: sourceConfiguration.metricAlias,
            body: {
                query: {
                    bool: {
                        filter: {
                            term: { [idFieldName]: nodeId },
                        },
                    },
                },
                size: 1,
                _source: [constants_1.NAME_FIELDS[nodeType]],
                aggs: {
                    metrics: {
                        terms: {
                            field: 'metricset.module',
                            size: 1000,
                        },
                        aggs: {
                            names: {
                                terms: {
                                    field: 'metricset.name',
                                    size: 1000,
                                },
                            },
                        },
                    },
                },
            },
        };
        const response = await this.framework.callWithRequest(req, 'search', metricQuery);
        const buckets = response.aggregations && response.aggregations.metrics
            ? response.aggregations.metrics.buckets
            : [];
        const sampleDoc = lodash_1.first(response.hits.hits);
        return {
            id: nodeId,
            name: lodash_1.get(sampleDoc, `_source.${constants_1.NAME_FIELDS[nodeType]}`),
            buckets,
        };
    }
    async getLogMetadata(req, sourceConfiguration, nodeId, nodeType) {
        const idFieldName = getIdFieldName(sourceConfiguration, nodeType);
        const logQuery = {
            index: sourceConfiguration.logAlias,
            body: {
                query: {
                    bool: {
                        filter: {
                            term: { [idFieldName]: nodeId },
                        },
                    },
                },
                size: 1,
                _source: [constants_1.NAME_FIELDS[nodeType]],
                aggs: {
                    metrics: {
                        terms: {
                            field: 'fileset.module',
                            size: 1000,
                        },
                        aggs: {
                            names: {
                                terms: {
                                    field: 'fileset.name',
                                    size: 1000,
                                },
                            },
                        },
                    },
                },
            },
        };
        const response = await this.framework.callWithRequest(req, 'search', logQuery);
        const buckets = response.aggregations && response.aggregations.metrics
            ? response.aggregations.metrics.buckets
            : [];
        const sampleDoc = lodash_1.first(response.hits.hits);
        return {
            id: nodeId,
            name: lodash_1.get(sampleDoc, `_source.${constants_1.NAME_FIELDS[nodeType]}`),
            buckets,
        };
    }
}
exports.ElasticsearchMetadataAdapter = ElasticsearchMetadataAdapter;
const getIdFieldName = (sourceConfiguration, nodeType) => {
    switch (nodeType) {
        case 'host':
            return sourceConfiguration.fields.host;
        case 'container':
            return sourceConfiguration.fields.container;
        default:
            return sourceConfiguration.fields.pod;
    }
};
