"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const typescript_fsa_1 = require("typescript-fsa");
const dist_1 = require("typescript-fsa-reducers/dist");
const reselect_1 = require("reselect");
const loading_state_1 = require("../loading_state");
exports.createGraphqlInitialState = (initialData) => ({
    current: {
        progress: 'idle',
    },
    last: {
        result: 'uninitialized',
    },
    data: initialData,
});
exports.createGraphqlOperationActionCreators = (stateKey, operationKey) => {
    const actionCreator = typescript_fsa_1.actionCreatorFactory(`x-pack/infra/remote/${stateKey}/${operationKey}`);
    const resolve = actionCreator('RESOLVE');
    const resolveEffect = actionCreator.async('RESOLVE');
    return {
        resolve,
        resolveStarted: resolveEffect.started,
        resolveDone: resolveEffect.done,
        resolveFailed: resolveEffect.failed,
    };
};
exports.createGraphqlOperationReducer = (operationKey, initialState, actionCreators, reduceSuccess = state => state) => dist_1.reducerWithInitialState(initialState)
    .caseWithAction(actionCreators.resolveStarted, (state, action) => ({
    ...state,
    current: {
        progress: 'running',
        time: Date.now(),
        parameters: {
            operationKey,
            variables: action.payload,
        },
    },
}))
    .caseWithAction(actionCreators.resolveDone, (state, action) => ({
    ...state,
    current: {
        progress: 'idle',
    },
    last: {
        result: 'success',
        parameters: {
            operationKey,
            variables: action.payload.params,
        },
        time: Date.now(),
        isExhausted: false,
    },
    data: reduceSuccess(state.data, action),
}))
    .caseWithAction(actionCreators.resolveFailed, (state, action) => ({
    ...state,
    current: {
        progress: 'idle',
    },
    last: {
        result: 'failure',
        reason: `${action.payload}`,
        time: Date.now(),
        parameters: {
            operationKey,
            variables: action.payload.params,
        },
    },
}))
    .build();
exports.createGraphqlQueryEpic = (graphqlQuery, actionCreators) => (action$, state$, { apolloClient$ }) => action$.pipe(operators_1.filter(actionCreators.resolve.match), operators_1.withLatestFrom(apolloClient$), operators_1.switchMap(([{ payload: variables }, apolloClient]) => rxjs_1.from(apolloClient.query({
    query: graphqlQuery,
    variables,
    fetchPolicy: 'no-cache',
})).pipe(operators_1.map(result => actionCreators.resolveDone({ params: variables, result })), operators_1.catchError(error => [actionCreators.resolveFailed({ params: variables, error })]), operators_1.startWith(actionCreators.resolveStarted(variables)))));
exports.createGraphqlStateSelectors = (selectState = parentState => parentState) => {
    const selectData = reselect_1.createSelector(selectState, state => state.data);
    const selectLoadingProgress = reselect_1.createSelector(selectState, state => state.current);
    const selectLoadingProgressOperationInfo = reselect_1.createSelector(selectLoadingProgress, progress => (loading_state_1.isRunningLoadingProgress(progress) ? progress.parameters : null));
    const selectIsLoading = reselect_1.createSelector(selectLoadingProgress, loading_state_1.isRunningLoadingProgress);
    const selectIsIdle = reselect_1.createSelector(selectLoadingProgress, loading_state_1.isIdleLoadingProgress);
    const selectLoadingResult = reselect_1.createSelector(selectState, state => state.last);
    const selectLoadingResultOperationInfo = reselect_1.createSelector(selectLoadingResult, result => (!loading_state_1.isUninitializedLoadingResult(result) ? result.parameters : null));
    const selectLoadingResultTime = reselect_1.createSelector(selectLoadingResult, result => (!loading_state_1.isUninitializedLoadingResult(result) ? result.time : null));
    const selectIsUninitialized = reselect_1.createSelector(selectLoadingResult, loading_state_1.isUninitializedLoadingResult);
    const selectIsSuccess = reselect_1.createSelector(selectLoadingResult, loading_state_1.isSuccessLoadingResult);
    const selectIsFailure = reselect_1.createSelector(selectLoadingResult, loading_state_1.isFailureLoadingResult);
    const selectLoadingState = reselect_1.createSelector(selectLoadingProgress, selectLoadingResult, (loadingProgress, loadingResult) => ({
        current: loadingProgress,
        last: loadingResult,
        policy: {
            policy: 'manual',
        },
    }));
    return {
        selectData,
        selectIsFailure,
        selectIsIdle,
        selectIsLoading,
        selectIsSuccess,
        selectIsUninitialized,
        selectLoadingProgress,
        selectLoadingProgressOperationInfo,
        selectLoadingResult,
        selectLoadingResultOperationInfo,
        selectLoadingResultTime,
        selectLoadingState,
    };
};
