"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const constate_1 = require("constate");
const mergeAll_1 = tslib_1.__importDefault(require("lodash/fp/mergeAll"));
const react_1 = tslib_1.__importDefault(require("react"));
const react_2 = require("@kbn/i18n/react");
const memoize_1 = require("ui/utils/memoize");
const source_configuration_1 = require("../../../common/source_configuration");
const createContainerProps = memoize_1.memoizeLast((initialFormState) => {
    const actions = {
        resetForm: () => state => ({
            ...state,
            updates: [],
        }),
        updateName: name => state => ({
            ...state,
            updates: addOrCombineLastUpdate(state.updates, { setName: { name } }),
        }),
        updateLogAlias: logAlias => state => ({
            ...state,
            updates: addOrCombineLastUpdate(state.updates, { setAliases: { logAlias } }),
        }),
        updateMetricAlias: metricAlias => state => ({
            ...state,
            updates: addOrCombineLastUpdate(state.updates, { setAliases: { metricAlias } }),
        }),
        updateField: (field, value) => state => ({
            ...state,
            updates: addOrCombineLastUpdate(state.updates, { setFields: { [field]: value } }),
        }),
    };
    const getCurrentFormState = memoize_1.memoizeLast((previousFormState, updates) => updates
        .map(source_configuration_1.convertChangeToUpdater)
        .reduce((state, updater) => updater(state), previousFormState));
    const selectors = {
        getCurrentFormState: () => ({ updates }) => getCurrentFormState(initialFormState, updates),
        getNameFieldValidationErrors: () => state => validateInputFieldNotEmpty(selectors.getCurrentFormState()(state).name),
        getLogAliasFieldValidationErrors: () => state => validateInputFieldNotEmpty(selectors.getCurrentFormState()(state).logAlias),
        getMetricAliasFieldValidationErrors: () => state => validateInputFieldNotEmpty(selectors.getCurrentFormState()(state).metricAlias),
        getFieldFieldValidationErrors: field => state => validateInputFieldNotEmpty(selectors.getCurrentFormState()(state).fields[field]),
        isFormValid: () => state => [
            selectors.getNameFieldValidationErrors()(state),
            selectors.getLogAliasFieldValidationErrors()(state),
            selectors.getMetricAliasFieldValidationErrors()(state),
            selectors.getFieldFieldValidationErrors('container')(state),
            selectors.getFieldFieldValidationErrors('host')(state),
            selectors.getFieldFieldValidationErrors('pod')(state),
            selectors.getFieldFieldValidationErrors('tiebreaker')(state),
            selectors.getFieldFieldValidationErrors('timestamp')(state),
        ].every(errors => errors.length === 0),
    };
    return {
        actions,
        initialState: { updates: [] },
        selectors,
    };
});
exports.WithSourceConfigurationFormState = ({ children, initialFormState, onMount, }) => (react_1.default.createElement(constate_1.Container, Object.assign({}, createContainerProps(initialFormState), { context: "source-configuration-form", onMount: onMount }), args => {
    const currentFormState = args.getCurrentFormState();
    return children({
        ...args,
        getNameFieldProps: () => createInputFieldProps({
            errors: args.getNameFieldValidationErrors(),
            name: 'name',
            onChange: args.updateName,
            value: currentFormState.name,
        }),
        getLogAliasFieldProps: () => createInputFieldProps({
            errors: args.getLogAliasFieldValidationErrors(),
            name: 'logAlias',
            onChange: args.updateLogAlias,
            value: currentFormState.logAlias,
        }),
        getMetricAliasFieldProps: () => createInputFieldProps({
            errors: args.getMetricAliasFieldValidationErrors(),
            name: 'metricAlias',
            onChange: args.updateMetricAlias,
            value: currentFormState.metricAlias,
        }),
        getFieldFieldProps: field => createInputFieldProps({
            errors: args.getFieldFieldValidationErrors(field),
            name: `${field}Field`,
            onChange: newValue => args.updateField(field, newValue),
            value: currentFormState.fields[field],
        }),
    });
}));
const addOrCombineLastUpdate = (updates, newUpdate) => [
    ...updates.slice(0, -1),
    ...maybeCombineUpdates(updates[updates.length - 1], newUpdate),
];
const createInputFieldProps = ({ errors, name, onChange, value, }) => ({
    error: errors,
    isInvalid: errors.length > 0,
    name,
    onChange: (evt) => onChange(evt.currentTarget.value),
    value,
});
const validateInputFieldNotEmpty = (value) => value === ''
    ? [
        react_1.default.createElement(react_2.FormattedMessage, { id: "xpack.infra.sourceConfiguration.fieldEmptyErrorMessage", defaultMessage: "The field must not be empty" }),
    ]
    : [];
/**
 * Tries to combine the given updates by naively checking whether they can be
 * merged into one update.
 *
 * This is only judged to be the case when all of the following conditions are
 * met:
 *
 * 1. The update only contains one operation.
 * 2. The operation is the same on in both updates.
 * 3. The operation is known to be safe to combine.
 */
const maybeCombineUpdates = (firstUpdate, secondUpdate) => {
    if (!firstUpdate) {
        return [secondUpdate];
    }
    const firstKeys = Object.keys(firstUpdate);
    const secondKeys = Object.keys(secondUpdate);
    const isSingleOperation = firstKeys.length === secondKeys.length && firstKeys.length === 1;
    const isSameOperation = firstKeys[0] === secondKeys[0];
    // to guard against future operations, which might not be safe to merge naively
    const isMergeableOperation = mergeableOperations.indexOf(firstKeys[0]) > -1;
    if (isSingleOperation && isSameOperation && isMergeableOperation) {
        return [mergeAll_1.default([firstUpdate, secondUpdate])];
    }
    return [firstUpdate, secondUpdate];
};
const mergeableOperations = ['setName', 'setDescription', 'setAliases', 'setFields'];
