"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var graphql_1 = require("graphql");
var resolveFromParentTypename_1 = require("./resolveFromParentTypename");
var backcompatOptions = { commentDescriptions: true };
function typeFromAST(node, getType) {
    switch (node.kind) {
        case graphql_1.Kind.OBJECT_TYPE_DEFINITION:
            return makeObjectType(node, getType);
        case graphql_1.Kind.INTERFACE_TYPE_DEFINITION:
            return makeInterfaceType(node, getType);
        case graphql_1.Kind.ENUM_TYPE_DEFINITION:
            return makeEnumType(node, getType);
        case graphql_1.Kind.UNION_TYPE_DEFINITION:
            return makeUnionType(node, getType);
        case graphql_1.Kind.SCALAR_TYPE_DEFINITION:
            return makeScalarType(node, getType);
        case graphql_1.Kind.INPUT_OBJECT_TYPE_DEFINITION:
            return makeInputObjectType(node, getType);
        default:
            return null;
    }
}
exports.default = typeFromAST;
function makeObjectType(node, getType) {
    return new graphql_1.GraphQLObjectType({
        name: node.name.value,
        fields: function () { return makeFields(node.fields, getType); },
        interfaces: function () {
            return node.interfaces.map(function (iface) { return getType(iface.name.value, 'interface'); });
        },
        description: graphql_1.getDescription(node, backcompatOptions),
    });
}
function makeInterfaceType(node, getType) {
    return new graphql_1.GraphQLInterfaceType({
        name: node.name.value,
        fields: function () { return makeFields(node.fields, getType); },
        description: graphql_1.getDescription(node, backcompatOptions),
        resolveType: function (parent, context, info) {
            return resolveFromParentTypename_1.default(parent, info.schema);
        },
    });
}
function makeEnumType(node, getType) {
    var values = {};
    node.values.forEach(function (value) {
        values[value.name.value] = {
            description: graphql_1.getDescription(value, backcompatOptions),
        };
    });
    return new graphql_1.GraphQLEnumType({
        name: node.name.value,
        values: values,
        description: graphql_1.getDescription(node, backcompatOptions),
    });
}
function makeUnionType(node, getType) {
    return new graphql_1.GraphQLUnionType({
        name: node.name.value,
        types: function () {
            return node.types.map(function (type) { return resolveType(type, getType, 'object'); });
        },
        description: graphql_1.getDescription(node, backcompatOptions),
        resolveType: function (parent, context, info) {
            return resolveFromParentTypename_1.default(parent, info.schema);
        },
    });
}
function makeScalarType(node, getType) {
    return new graphql_1.GraphQLScalarType({
        name: node.name.value,
        description: graphql_1.getDescription(node, backcompatOptions),
        serialize: function () { return null; },
        // Note: validation calls the parse functions to determine if a
        // literal value is correct. Returning null would cause use of custom
        // scalars to always fail validation. Returning false causes them to
        // always pass validation.
        parseValue: function () { return false; },
        parseLiteral: function () { return false; },
    });
}
function makeInputObjectType(node, getType) {
    return new graphql_1.GraphQLInputObjectType({
        name: node.name.value,
        fields: function () { return makeValues(node.fields, getType); },
        description: graphql_1.getDescription(node, backcompatOptions),
    });
}
function makeFields(nodes, getType) {
    var result = {};
    nodes.forEach(function (node) {
        result[node.name.value] = {
            type: resolveType(node.type, getType, 'object'),
            args: makeValues(node.arguments, getType),
            description: graphql_1.getDescription(node, backcompatOptions),
        };
    });
    return result;
}
function makeValues(nodes, getType) {
    var result = {};
    nodes.forEach(function (node) {
        var type = resolveType(node.type, getType, 'input');
        result[node.name.value] = {
            type: type,
            defaultValue: graphql_1.valueFromAST(node.defaultValue, type),
            description: graphql_1.getDescription(node, backcompatOptions),
        };
    });
    return result;
}
function resolveType(node, getType, type) {
    switch (node.kind) {
        case graphql_1.Kind.LIST_TYPE:
            return new graphql_1.GraphQLList(resolveType(node.type, getType, type));
        case graphql_1.Kind.NON_NULL_TYPE:
            return new graphql_1.GraphQLNonNull(resolveType(node.type, getType, type));
        default:
            return getType(node.name.value, type);
    }
}
//# sourceMappingURL=typeFromAST.js.map