/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common;

import java.io.IOException;
import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.IsoFields;
import java.time.temporal.TemporalField;
import java.time.zone.ZoneOffsetTransition;
import java.util.List;
import java.util.Objects;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.unit.TimeValue;

public abstract class Rounding
implements Writeable {
    public static String format(long epochMillis) {
        return Instant.ofEpochMilli(epochMillis) + "/" + epochMillis;
    }

    public abstract void innerWriteTo(StreamOutput var1) throws IOException;

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        out.writeByte(this.id());
        this.innerWriteTo(out);
    }

    public abstract byte id();

    public abstract long round(long var1);

    public abstract long nextRoundingValue(long var1);

    public abstract boolean equals(Object var1);

    public abstract int hashCode();

    public static Builder builder(DateTimeUnit unit) {
        return new Builder(unit);
    }

    public static Builder builder(TimeValue interval) {
        return new Builder(interval);
    }

    public static Rounding read(StreamInput in) throws IOException {
        Rounding rounding;
        byte id = in.readByte();
        switch (id) {
            case 1: {
                rounding = new TimeUnitRounding(in);
                break;
            }
            case 2: {
                rounding = new TimeIntervalRounding(in);
                break;
            }
            default: {
                throw new ElasticsearchException("unknown rounding id [" + id + "]", new Object[0]);
            }
        }
        return rounding;
    }

    static class TimeIntervalRounding
    extends Rounding {
        static final byte ID = 2;
        private final long interval;
        private final ZoneId timeZone;

        public String toString() {
            return "TimeIntervalRounding{interval=" + this.interval + ", timeZone=" + this.timeZone + '}';
        }

        TimeIntervalRounding(long interval, ZoneId timeZone) {
            if (interval < 1L) {
                throw new IllegalArgumentException("Zero or negative time interval not supported");
            }
            this.interval = interval;
            this.timeZone = timeZone;
        }

        TimeIntervalRounding(StreamInput in) throws IOException {
            this.interval = in.readVLong();
            this.timeZone = ZoneId.of(in.readString());
        }

        @Override
        public byte id() {
            return 2;
        }

        @Override
        public long round(long utcMillis) {
            Instant utcInstant = Instant.ofEpochMilli(utcMillis);
            LocalDateTime rawLocalDateTime = Instant.ofEpochMilli(utcMillis).atZone(this.timeZone).toLocalDateTime();
            long localMillis = utcMillis + (long)(this.timeZone.getRules().getOffset(utcInstant).getTotalSeconds() * 1000);
            assert (localMillis == rawLocalDateTime.toInstant(ZoneOffset.UTC).toEpochMilli());
            long roundedMillis = TimeIntervalRounding.roundKey(localMillis, this.interval) * this.interval;
            LocalDateTime roundedLocalDateTime = Instant.ofEpochMilli(roundedMillis).atZone(ZoneOffset.UTC).toLocalDateTime();
            List<ZoneOffset> currentOffsets = this.timeZone.getRules().getValidOffsets(roundedLocalDateTime);
            if (!currentOffsets.isEmpty()) {
                ZoneOffsetTransition previousTransition = this.timeZone.getRules().previousTransition(utcInstant.plusMillis(1L));
                for (int offsetIndex = currentOffsets.size() - 1; 0 <= offsetIndex; --offsetIndex) {
                    OffsetDateTime offsetTime = roundedLocalDateTime.atOffset(currentOffsets.get(offsetIndex));
                    Instant offsetInstant = offsetTime.toInstant();
                    if (previousTransition != null && offsetInstant.isBefore(previousTransition.getInstant())) {
                        return this.round(previousTransition.getInstant().toEpochMilli() - 1L);
                    }
                    if (utcInstant.isBefore(offsetTime.toInstant())) continue;
                    return offsetInstant.toEpochMilli();
                }
                OffsetDateTime offsetTime = roundedLocalDateTime.atOffset(currentOffsets.get(0));
                Instant offsetInstant = offsetTime.toInstant();
                assert (false) : this + " failed to round " + utcMillis + " down: " + offsetInstant + " is the earliest possible";
                return offsetInstant.toEpochMilli();
            }
            ZoneOffsetTransition zoneOffsetTransition = this.timeZone.getRules().getTransition(roundedLocalDateTime);
            return zoneOffsetTransition.getInstant().toEpochMilli();
        }

        private static long roundKey(long value, long interval) {
            if (value < 0L) {
                return (value - interval + 1L) / interval;
            }
            return value / interval;
        }

        @Override
        public long nextRoundingValue(long time) {
            int offsetSeconds = this.timeZone.getRules().getOffset(Instant.ofEpochMilli(time)).getTotalSeconds();
            return ZonedDateTime.ofInstant(Instant.ofEpochMilli(time), ZoneOffset.UTC).plusSeconds(offsetSeconds).plusNanos(this.interval * 1000000L).withZoneSameLocal(this.timeZone).toInstant().toEpochMilli();
        }

        @Override
        public void innerWriteTo(StreamOutput out) throws IOException {
            out.writeVLong(this.interval);
            String tz = ZoneOffset.UTC.equals(this.timeZone) ? "UTC" : this.timeZone.getId();
            out.writeString(tz);
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.interval, this.timeZone);
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            TimeIntervalRounding other = (TimeIntervalRounding)obj;
            return Objects.equals(this.interval, other.interval) && Objects.equals(this.timeZone, other.timeZone);
        }
    }

    static class TimeUnitRounding
    extends Rounding {
        static final byte ID = 1;
        private final DateTimeUnit unit;
        private final ZoneId timeZone;
        private final boolean unitRoundsToMidnight;

        TimeUnitRounding(DateTimeUnit unit, ZoneId timeZone) {
            this.unit = unit;
            this.timeZone = timeZone;
            this.unitRoundsToMidnight = this.unit.field.getBaseUnit().getDuration().toMillis() > 3600000L;
        }

        TimeUnitRounding(StreamInput in) throws IOException {
            this.unit = DateTimeUnit.resolve(in.readByte());
            this.timeZone = ZoneId.of(in.readString());
            this.unitRoundsToMidnight = this.unit.getField().getBaseUnit().getDuration().toMillis() > 3600000L;
        }

        @Override
        public byte id() {
            return 1;
        }

        private LocalDateTime truncateLocalDateTime(LocalDateTime localDateTime) {
            localDateTime = localDateTime.withNano(0);
            assert (localDateTime.getNano() == 0);
            if (this.unit.equals((Object)DateTimeUnit.SECOND_OF_MINUTE)) {
                return localDateTime;
            }
            localDateTime = localDateTime.withSecond(0);
            assert (localDateTime.getSecond() == 0);
            if (this.unit.equals((Object)DateTimeUnit.MINUTES_OF_HOUR)) {
                return localDateTime;
            }
            localDateTime = localDateTime.withMinute(0);
            assert (localDateTime.getMinute() == 0);
            if (this.unit.equals((Object)DateTimeUnit.HOUR_OF_DAY)) {
                return localDateTime;
            }
            localDateTime = localDateTime.withHour(0);
            assert (localDateTime.getHour() == 0);
            if (this.unit.equals((Object)DateTimeUnit.DAY_OF_MONTH)) {
                return localDateTime;
            }
            if (this.unit.equals((Object)DateTimeUnit.WEEK_OF_WEEKYEAR)) {
                localDateTime = localDateTime.with(ChronoField.DAY_OF_WEEK, 1L);
                assert (localDateTime.getDayOfWeek() == DayOfWeek.MONDAY);
                return localDateTime;
            }
            localDateTime = localDateTime.withDayOfMonth(1);
            assert (localDateTime.getDayOfMonth() == 1);
            if (this.unit.equals((Object)DateTimeUnit.MONTH_OF_YEAR)) {
                return localDateTime;
            }
            if (this.unit.equals((Object)DateTimeUnit.QUARTER_OF_YEAR)) {
                int quarter = (int)IsoFields.QUARTER_OF_YEAR.getFrom(localDateTime);
                int month = (quarter - 1) * 3 + 1;
                localDateTime = localDateTime.withMonth(month);
                assert (localDateTime.getMonthValue() % 3 == 1);
                return localDateTime;
            }
            if (this.unit.equals((Object)DateTimeUnit.YEAR_OF_CENTURY)) {
                localDateTime = localDateTime.withMonth(1);
                assert (localDateTime.getMonthValue() == 1);
                return localDateTime;
            }
            throw new IllegalArgumentException("NOT YET IMPLEMENTED for unit " + (Object)((Object)this.unit));
        }

        @Override
        public long round(long utcMillis) {
            if (this.unitRoundsToMidnight) {
                ZonedDateTime zonedDateTime = Instant.ofEpochMilli(utcMillis).atZone(this.timeZone);
                LocalDateTime localDateTime = zonedDateTime.toLocalDateTime();
                LocalDateTime localMidnight = this.truncateLocalDateTime(localDateTime);
                return this.firstTimeOnDay(localMidnight);
            }
            while (true) {
                Instant truncatedTime = this.truncateAsLocalTime(utcMillis);
                ZoneOffsetTransition previousTransition = this.timeZone.getRules().previousTransition(Instant.ofEpochMilli(utcMillis));
                if (previousTransition == null) {
                    return truncatedTime.toEpochMilli();
                }
                long previousTransitionMillis = previousTransition.getInstant().toEpochMilli();
                if (truncatedTime != null && previousTransitionMillis <= truncatedTime.toEpochMilli()) {
                    return truncatedTime.toEpochMilli();
                }
                utcMillis = previousTransitionMillis - 1L;
            }
        }

        private long firstTimeOnDay(LocalDateTime localMidnight) {
            assert (localMidnight.toLocalTime().equals(LocalTime.of(0, 0, 0))) : "firstTimeOnDay should only be called at midnight";
            assert (this.unitRoundsToMidnight) : "firstTimeOnDay should only be called if unitRoundsToMidnight";
            List<ZoneOffset> currentOffsets = this.timeZone.getRules().getValidOffsets(localMidnight);
            if (currentOffsets.size() >= 1) {
                ZoneOffset firstOffset = currentOffsets.get(0);
                OffsetDateTime offsetMidnight = localMidnight.atOffset(firstOffset);
                return offsetMidnight.toInstant().toEpochMilli();
            }
            ZoneOffsetTransition zoneOffsetTransition = this.timeZone.getRules().getTransition(localMidnight);
            return zoneOffsetTransition.getInstant().toEpochMilli();
        }

        private Instant truncateAsLocalTime(long utcMillis) {
            assert (!this.unitRoundsToMidnight) : "truncateAsLocalTime should not be called if unitRoundsToMidnight";
            LocalDateTime truncatedLocalDateTime = this.truncateLocalDateTime(Instant.ofEpochMilli(utcMillis).atZone(this.timeZone).toLocalDateTime());
            List<ZoneOffset> currentOffsets = this.timeZone.getRules().getValidOffsets(truncatedLocalDateTime);
            if (currentOffsets.size() >= 1) {
                for (int offsetIndex = currentOffsets.size() - 1; offsetIndex >= 0; --offsetIndex) {
                    Instant result = truncatedLocalDateTime.atOffset(currentOffsets.get(offsetIndex)).toInstant();
                    if (result.toEpochMilli() > utcMillis) continue;
                    return result;
                }
                assert (false) : "rounded time not found for " + utcMillis + " with " + this;
                return null;
            }
            return null;
        }

        private LocalDateTime nextRelevantMidnight(LocalDateTime localMidnight) {
            assert (localMidnight.toLocalTime().equals(LocalTime.of(0, 0, 0))) : "nextRelevantMidnight should only be called at midnight";
            assert (this.unitRoundsToMidnight) : "firstTimeOnDay should only be called if unitRoundsToMidnight";
            switch (this.unit) {
                case DAY_OF_MONTH: {
                    return localMidnight.plus(1L, ChronoUnit.DAYS);
                }
                case WEEK_OF_WEEKYEAR: {
                    return localMidnight.plus(7L, ChronoUnit.DAYS);
                }
                case MONTH_OF_YEAR: {
                    return localMidnight.plus(1L, ChronoUnit.MONTHS);
                }
                case QUARTER_OF_YEAR: {
                    return localMidnight.plus(3L, ChronoUnit.MONTHS);
                }
                case YEAR_OF_CENTURY: {
                    return localMidnight.plus(1L, ChronoUnit.YEARS);
                }
            }
            throw new IllegalArgumentException("Unknown round-to-midnight unit: " + (Object)((Object)this.unit));
        }

        @Override
        public long nextRoundingValue(long utcMillis) {
            if (this.unitRoundsToMidnight) {
                ZonedDateTime zonedDateTime = Instant.ofEpochMilli(utcMillis).atZone(this.timeZone);
                LocalDateTime localDateTime = zonedDateTime.toLocalDateTime();
                LocalDateTime earlierLocalMidnight = this.truncateLocalDateTime(localDateTime);
                LocalDateTime localMidnight = this.nextRelevantMidnight(earlierLocalMidnight);
                return this.firstTimeOnDay(localMidnight);
            }
            long unitSize = this.unit.field.getBaseUnit().getDuration().toMillis();
            long roundedAfterOneIncrement = this.round(utcMillis + unitSize);
            if (utcMillis < roundedAfterOneIncrement) {
                return roundedAfterOneIncrement;
            }
            return this.round(utcMillis + 2L * unitSize);
        }

        @Override
        public void innerWriteTo(StreamOutput out) throws IOException {
            out.writeByte(this.unit.getId());
            String tz = ZoneOffset.UTC.equals(this.timeZone) ? "UTC" : this.timeZone.getId();
            out.writeString(tz);
        }

        @Override
        public int hashCode() {
            return Objects.hash(new Object[]{this.unit, this.timeZone});
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            TimeUnitRounding other = (TimeUnitRounding)obj;
            return Objects.equals((Object)this.unit, (Object)other.unit) && Objects.equals(this.timeZone, other.timeZone);
        }

        public String toString() {
            return "[" + this.timeZone + "][" + (Object)((Object)this.unit) + "]";
        }
    }

    public static class Builder {
        private final DateTimeUnit unit;
        private final long interval;
        private ZoneId timeZone = ZoneOffset.UTC;

        public Builder(DateTimeUnit unit) {
            this.unit = unit;
            this.interval = -1L;
        }

        public Builder(TimeValue interval) {
            this.unit = null;
            if (interval.millis() < 1L) {
                throw new IllegalArgumentException("Zero or negative time interval not supported");
            }
            this.interval = interval.millis();
        }

        public Builder timeZone(ZoneId timeZone) {
            if (timeZone == null) {
                throw new IllegalArgumentException("Setting null as timezone is not supported");
            }
            this.timeZone = timeZone;
            return this;
        }

        public Rounding build() {
            Rounding timeZoneRounding = this.unit != null ? new TimeUnitRounding(this.unit, this.timeZone) : new TimeIntervalRounding(this.interval, this.timeZone);
            return timeZoneRounding;
        }
    }

    public static enum DateTimeUnit {
        WEEK_OF_WEEKYEAR(1, IsoFields.WEEK_OF_WEEK_BASED_YEAR),
        YEAR_OF_CENTURY(2, ChronoField.YEAR_OF_ERA),
        QUARTER_OF_YEAR(3, IsoFields.QUARTER_OF_YEAR),
        MONTH_OF_YEAR(4, ChronoField.MONTH_OF_YEAR),
        DAY_OF_MONTH(5, ChronoField.DAY_OF_MONTH),
        HOUR_OF_DAY(6, ChronoField.HOUR_OF_DAY),
        MINUTES_OF_HOUR(7, ChronoField.MINUTE_OF_HOUR),
        SECOND_OF_MINUTE(8, ChronoField.SECOND_OF_MINUTE);

        private final byte id;
        private final TemporalField field;

        private DateTimeUnit(byte id, TemporalField field) {
            this.id = id;
            this.field = field;
        }

        public byte getId() {
            return this.id;
        }

        public TemporalField getField() {
            return this.field;
        }

        public static DateTimeUnit resolve(byte id) {
            switch (id) {
                case 1: {
                    return WEEK_OF_WEEKYEAR;
                }
                case 2: {
                    return YEAR_OF_CENTURY;
                }
                case 3: {
                    return QUARTER_OF_YEAR;
                }
                case 4: {
                    return MONTH_OF_YEAR;
                }
                case 5: {
                    return DAY_OF_MONTH;
                }
                case 6: {
                    return HOUR_OF_DAY;
                }
                case 7: {
                    return MINUTES_OF_HOUR;
                }
                case 8: {
                    return SECOND_OF_MINUTE;
                }
            }
            throw new ElasticsearchException("Unknown date time unit id [" + id + "]", new Object[0]);
        }
    }
}

