/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.geometry.simplify;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.PriorityQueue;
import org.elasticsearch.geometry.Geometry;
import org.elasticsearch.geometry.Line;
import org.elasticsearch.geometry.LinearRing;
import org.elasticsearch.geometry.Polygon;
import org.elasticsearch.geometry.simplify.SimplificationErrorCalculator;

public abstract class StreamingGeometrySimplifier<T extends Geometry> {
    protected final int maxPoints;
    protected final SimplificationErrorCalculator calculator;
    protected final PointError[] points;
    protected PointError lastRemoved;
    protected final Monitor monitor;
    protected int length;
    protected int objCount = 0;
    protected String description;
    protected PointConstructor pointConstructor = new PointConstructor(this){};
    protected PointResetter pointResetter = new PointResetter(this){};
    protected final PriorityQueue<PointError> queue = new PriorityQueue();

    protected StreamingGeometrySimplifier(String description, int maxPoints, SimplificationErrorCalculator calculator, Monitor monitor) {
        this.description = description;
        this.maxPoints = maxPoints;
        this.calculator = calculator;
        this.monitor = monitor;
        this.points = new PointError[maxPoints];
        this.length = 0;
    }

    public void reset() {
        this.length = 0;
        this.queue.clear();
    }

    public void consume(double x, double y) {
        this.consume(this.makePointErrorFor(this.length, x, y));
    }

    public void consume(PointError pointError) {
        if (this.length > 1) {
            this.points[this.length - 1].error = this.calculator.calculateError(this.points[this.length - 2], this.points[this.length - 1], pointError);
            this.queue.add(this.points[this.length - 1]);
        }
        if (this.length == this.maxPoints) {
            PointError toRemove = (PointError)this.queue.remove();
            this.removeAndAdd(toRemove.index, pointError);
            this.notifyMonitorPointRemoved(toRemove);
        } else {
            this.points[this.length] = pointError;
            ++this.length;
            this.notifyMonitorPointAdded();
        }
    }

    public abstract T produce();

    private PointError makePointErrorFor(int index, double x, double y) {
        if (index == this.maxPoints) {
            if (this.lastRemoved == null) {
                ++this.objCount;
                return this.pointConstructor.newPoint(index, x, y);
            }
            return this.pointResetter.resetPoint(this.lastRemoved, index, x, y);
        }
        if (this.points[index] == null) {
            ++this.objCount;
            return this.pointConstructor.newPoint(index, x, y);
        }
        this.points[index] = this.pointResetter.resetPoint(this.points[index], index, x, y);
        return this.points[index];
    }

    private void removeAndAdd(int toRemove, PointError pointError) {
        assert (toRemove > 0);
        this.lastRemoved = this.points[toRemove];
        System.arraycopy(this.points, toRemove + 1, this.points, toRemove, this.maxPoints - toRemove - 1);
        this.points[this.length - 1] = pointError;
        for (int i = toRemove; i < this.length; ++i) {
            this.points[i].index = i;
        }
        this.updateErrorAt(toRemove - 1);
        this.updateErrorAt(toRemove);
        if (toRemove < this.maxPoints - 1) {
            this.updateErrorAt(this.maxPoints - 2);
        }
    }

    private void updateErrorAt(int index) {
        if (index > 0 && index < this.length - 1) {
            double error = this.calculator.calculateError(this.points[index - 1], this.points[index], this.points[index + 1]);
            double delta = Math.abs(error - this.points[index].error);
            this.points[index].error = error;
            if (delta > 1.0E-10 && this.queue.remove(this.points[index])) {
                this.queue.add(this.points[index]);
            }
        }
    }

    protected void notifyMonitorPointRemoved(PointError removed) {
        if (this.monitor != null) {
            PointError previous = this.points[removed.index - 1];
            PointError next = this.points[removed.index];
            this.monitor.pointRemoved(this.description + ".addAndRemovePoint()", this.getCurrentPoints(), removed, removed.error, previous, next);
        }
    }

    protected void notifyMonitorPointAdded() {
        if (this.monitor != null) {
            this.monitor.pointAdded(this.description + ".addPoint()", this.getCurrentPoints());
        }
    }

    private List<SimplificationErrorCalculator.PointLike> getCurrentPoints() {
        return new ArrayList<SimplificationErrorCalculator.PointLike>(Arrays.asList(this.points).subList(0, this.length));
    }

    static LinearRing produceLinearRing(StreamingGeometrySimplifier<?> simplifier) {
        if (simplifier.length < 1) {
            throw new IllegalArgumentException("No points have been consumed");
        }
        if (simplifier.length < 4) {
            throw new IllegalArgumentException("LinearRing cannot have less than 4 points");
        }
        double[] x = new double[simplifier.length];
        double[] y = new double[simplifier.length];
        for (int i = 0; i < simplifier.length; ++i) {
            x[i] = simplifier.points[i].x;
            y[i] = simplifier.points[i].y;
        }
        if (x[simplifier.length - 1] != x[0] || y[simplifier.length - 1] != y[0]) {
            String inequality = "(" + x[0] + " " + y[0] + ") != (" + x[simplifier.length - 1] + " " + y[simplifier.length - 1] + ")";
            throw new IllegalArgumentException("LinearRing cannot have first and last points differ: " + inequality);
        }
        return new LinearRing(x, y);
    }

    public static interface PointConstructor {
        default public PointError newPoint(int index, double x, double y) {
            return new PointError(index, x, y);
        }
    }

    public static interface PointResetter {
        default public PointError resetPoint(PointError point, int index, double x, double y) {
            return point.reset(index, x, y);
        }
    }

    public static interface Monitor {
        public void pointAdded(String var1, List<SimplificationErrorCalculator.PointLike> var2);

        public void pointRemoved(String var1, List<SimplificationErrorCalculator.PointLike> var2, SimplificationErrorCalculator.PointLike var3, double var4, SimplificationErrorCalculator.PointLike var6, SimplificationErrorCalculator.PointLike var7);

        public void startSimplification(String var1, int var2);

        public void endSimplification(String var1, List<SimplificationErrorCalculator.PointLike> var2);
    }

    public static class PointError
    implements SimplificationErrorCalculator.PointLike,
    Comparable<PointError> {
        private int index;
        private double x;
        private double y;
        double error = 0.0;

        protected PointError(int index, double x, double y) {
            this.index = index;
            this.x = x;
            this.y = y;
        }

        @Override
        public int compareTo(PointError o) {
            return (int)Math.signum(this.error - o.error);
        }

        public String toString() {
            return "[" + this.index + "] POINT( " + this.x + " " + this.y + " ) [error:" + this.error + "]";
        }

        @Override
        public double x() {
            return this.x;
        }

        @Override
        public double y() {
            return this.y;
        }

        public PointError reset(int index, double x, double y) {
            this.index = index;
            this.x = x;
            this.y = y;
            return this;
        }
    }

    public static class PolygonSimplifier
    extends StreamingGeometrySimplifier<Polygon> {
        public PolygonSimplifier(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public PolygonSimplifier(int maxPoints, SimplificationErrorCalculator calculator, Monitor monitor) {
            super("Polygon", maxPoints, calculator, monitor);
        }

        @Override
        public Polygon produce() {
            return new Polygon(StreamingGeometrySimplifier.produceLinearRing(this));
        }
    }

    public static class LinearRingSimplifier
    extends StreamingGeometrySimplifier<LinearRing> {
        public LinearRingSimplifier(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public LinearRingSimplifier(int maxPoints, SimplificationErrorCalculator calculator, Monitor monitor) {
            super("LinearRing", maxPoints, calculator, monitor);
            assert (maxPoints >= 4);
        }

        @Override
        public LinearRing produce() {
            return StreamingGeometrySimplifier.produceLinearRing(this);
        }
    }

    public static class LineSimplifier
    extends StreamingGeometrySimplifier<Line> {
        public LineSimplifier(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public LineSimplifier(int maxPoints, SimplificationErrorCalculator calculator, Monitor monitor) {
            super("Line", maxPoints, calculator, monitor);
        }

        @Override
        public Line produce() {
            if (this.length < 1) {
                throw new IllegalArgumentException("No points have been consumed");
            }
            double[] x = new double[this.length];
            double[] y = new double[this.length];
            for (int i = 0; i < this.length; ++i) {
                x[i] = this.points[i].x;
                y[i] = this.points[i].y;
            }
            return new Line(x, y);
        }
    }
}

