/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.ingest.geoip;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.admin.indices.flush.FlushRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.CheckedSupplier;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.MatchQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.index.reindex.DeleteByQueryAction;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.ingest.geoip.EnterpriseGeoIpTaskState;
import org.elasticsearch.ingest.geoip.GeoIpTaskState;
import org.elasticsearch.ingest.geoip.HttpClient;
import org.elasticsearch.ingest.geoip.IngestGeoIpMetadata;
import org.elasticsearch.ingest.geoip.direct.DatabaseConfiguration;
import org.elasticsearch.ingest.geoip.direct.DatabaseConfigurationMetadata;
import org.elasticsearch.persistent.AllocatedPersistentTask;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.tasks.TaskId;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentParserConfiguration;
import org.elasticsearch.xcontent.XContentType;

public class EnterpriseGeoIpDownloader
extends AllocatedPersistentTask {
    private static final Logger logger = LogManager.getLogger(EnterpriseGeoIpDownloader.class);
    private static final Pattern SHA256_CHECKSUM_PATTERN = Pattern.compile("(\\w{64})\\s\\s(.*)");
    private static final Pattern MD5_CHECKSUM_PATTERN = Pattern.compile("(\\w{32})");
    static String DEFAULT_MAXMIND_ENDPOINT = System.getProperty("ingest.geoip.downloader.maxmind.endpoint.default", "https://download.maxmind.com/geoip/databases");
    static String DEFAULT_IPINFO_ENDPOINT = System.getProperty("ingest.ip_location.downloader.ipinfo.endpoint.default", "https://ipinfo.io/data");
    static final String DATABASES_INDEX = ".geoip_databases";
    static final int MAX_CHUNK_SIZE = 0x100000;
    private final Client client;
    private final HttpClient httpClient;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    protected volatile EnterpriseGeoIpTaskState state;
    private volatile Scheduler.ScheduledCancellable scheduledPeriodicRun;
    private final AtomicInteger queuedRuns = new AtomicInteger(0);
    private final Supplier<TimeValue> pollIntervalSupplier;
    private final Function<String, char[]> tokenProvider;

    EnterpriseGeoIpDownloader(Client client, HttpClient httpClient, ClusterService clusterService, ThreadPool threadPool, long id, String type, String action, String description, TaskId parentTask, Map<String, String> headers, Supplier<TimeValue> pollIntervalSupplier, Function<String, char[]> tokenProvider) {
        super(id, type, action, description, parentTask, headers);
        this.client = client;
        this.httpClient = httpClient;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.pollIntervalSupplier = pollIntervalSupplier;
        this.tokenProvider = tokenProvider;
    }

    void setState(EnterpriseGeoIpTaskState state) {
        assert (this.state == null) : "setState() cannot be called when state is already non-null. This most likely happened because setState() was called twice";
        assert (state != null) : "Should never call setState with a null state. Pass an EnterpriseGeoIpTaskState.EMPTY instead.";
        this.state = state;
    }

    void updateDatabases() throws IOException {
        ProjectId projectId = ProjectId.DEFAULT;
        ClusterState clusterState = this.clusterService.state();
        IndexAbstraction geoipIndex = (IndexAbstraction)clusterState.getMetadata().getProject(projectId).getIndicesLookup().get(DATABASES_INDEX);
        if (geoipIndex != null) {
            logger.trace("the geoip index [{}] exists", (Object)DATABASES_INDEX);
            if (!clusterState.routingTable(projectId).index(geoipIndex.getWriteIndex()).allPrimaryShardsActive()) {
                logger.debug("not updating databases because not all primary shards of [{}] index are active yet", (Object)DATABASES_INDEX);
                return;
            }
            ClusterBlockException blockException = clusterState.blocks().indexBlockedException(projectId, ClusterBlockLevel.WRITE, geoipIndex.getWriteIndex().getName());
            if (blockException != null) {
                throw blockException;
            }
        }
        logger.trace("Updating databases");
        IngestGeoIpMetadata geoIpMeta = (IngestGeoIpMetadata)clusterState.metadata().getProject(projectId).custom("ingest_geoip", (Metadata.ProjectCustom)IngestGeoIpMetadata.EMPTY);
        boolean addedSomething = false;
        Set<String> existingDatabaseNames = this.state.getDatabases().keySet();
        for (Map.Entry<String, DatabaseConfigurationMetadata> entry2 : geoIpMeta.getDatabases().entrySet()) {
            String id = entry2.getKey();
            DatabaseConfiguration database = entry2.getValue().database();
            if (existingDatabaseNames.contains(database.name() + ".mmdb")) continue;
            logger.debug("A new database appeared [{}]", (Object)database.name());
            if (!this.processDatabase(id, database)) continue;
            addedSomething = true;
        }
        boolean droppedSomething = false;
        Set databases = geoIpMeta.getDatabases().values().stream().map(c -> c.database().name() + ".mmdb").collect(Collectors.toSet());
        EnterpriseGeoIpTaskState _state = this.state;
        List<Tuple> metas = _state.getDatabases().entrySet().stream().map(entry -> Tuple.tuple((Object)((String)entry.getKey()), (Object)((GeoIpTaskState.Metadata)entry.getValue()))).toList();
        for (Tuple metaTuple : metas) {
            String name = (String)metaTuple.v1();
            GeoIpTaskState.Metadata meta = (GeoIpTaskState.Metadata)metaTuple.v2();
            if (databases.contains(name)) continue;
            logger.debug("Dropping [{}], databases was {}", (Object)name, databases);
            _state = _state.remove(name);
            this.deleteOldChunks(name, meta.lastChunk() + 1);
            droppedSomething = true;
        }
        if (droppedSomething) {
            this.state = _state;
            this.updateTaskState();
        }
        if (!addedSomething && !droppedSomething) {
            RuntimeException accumulator = null;
            for (Map.Entry<String, DatabaseConfigurationMetadata> entry3 : geoIpMeta.getDatabases().entrySet()) {
                String id = entry3.getKey();
                DatabaseConfiguration database = entry3.getValue().database();
                try {
                    this.processDatabase(id, database);
                }
                catch (Exception e) {
                    accumulator = (RuntimeException)ExceptionsHelper.useOrSuppress(accumulator, (Throwable)ExceptionsHelper.convertToRuntime((Exception)e));
                }
            }
            if (accumulator != null) {
                throw accumulator;
            }
        }
    }

    boolean processDatabase(String id, DatabaseConfiguration database) throws IOException {
        String name = database.name();
        logger.debug("Processing database [{}] for configuration [{}]", (Object)name, (Object)database.id());
        try (ProviderDownload downloader = this.downloaderFor(database);){
            if (downloader != null && downloader.validCredentials()) {
                String fileName = name + ".mmdb";
                this.processDatabase(fileName, downloader.checksum(), downloader.download());
                boolean bl = true;
                return bl;
            }
            logger.warn("No credentials found to download database [{}], skipping download...", (Object)id);
            boolean bl = false;
            return bl;
        }
    }

    private void processDatabase(String name, Checksum checksum, CheckedSupplier<InputStream, IOException> source) {
        GeoIpTaskState.Metadata metadata = this.state.getDatabases().getOrDefault(name, GeoIpTaskState.Metadata.EMPTY);
        if (checksum.matches(metadata)) {
            this.updateTimestamp(name, metadata);
            return;
        }
        logger.debug("downloading database [{}]", (Object)name);
        long start = System.currentTimeMillis();
        try (InputStream is = (InputStream)source.get();){
            int firstChunk = metadata.lastChunk() + 1;
            Tuple<Integer, String> tuple = this.indexChunks(name, is, firstChunk, checksum, start);
            int lastChunk = (Integer)tuple.v1();
            String md5 = (String)tuple.v2();
            if (lastChunk > firstChunk) {
                String sha256 = checksum.type == Checksum.Type.SHA256 ? checksum.checksum : null;
                this.state = this.state.put(name, new GeoIpTaskState.Metadata(start, firstChunk, lastChunk - 1, md5, start, sha256));
                this.updateTaskState();
                logger.info("successfully downloaded database [{}]", (Object)name);
                this.deleteOldChunks(name, firstChunk);
            }
        }
        catch (Exception e) {
            logger.error(() -> "error downloading database [" + name + "]", (Throwable)e);
        }
    }

    void deleteOldChunks(String name, int firstChunk) {
        BoolQueryBuilder queryBuilder = new BoolQueryBuilder().filter((QueryBuilder)new MatchQueryBuilder("name", (Object)name)).filter((QueryBuilder)new RangeQueryBuilder("chunk").to((Object)firstChunk, false));
        DeleteByQueryRequest request = new DeleteByQueryRequest();
        request.indices(new String[]{DATABASES_INDEX});
        request.setQuery((QueryBuilder)queryBuilder);
        this.client.execute((ActionType)DeleteByQueryAction.INSTANCE, (ActionRequest)request, ActionListener.wrap(r -> {}, e -> logger.warn("could not delete old chunks for database [" + name + "]", (Throwable)e)));
    }

    protected void updateTimestamp(String name, GeoIpTaskState.Metadata old) {
        logger.debug("database [{}] is up to date, updated timestamp", (Object)name);
        this.state = this.state.put(name, new GeoIpTaskState.Metadata(old.lastUpdate(), old.firstChunk(), old.lastChunk(), old.md5(), System.currentTimeMillis(), old.sha256()));
        this.updateTaskState();
    }

    void updateTaskState() {
        PlainActionFuture future = new PlainActionFuture();
        this.updatePersistentTaskState(this.state, (ActionListener)future);
        this.state = (EnterpriseGeoIpTaskState)((PersistentTasksCustomMetadata.PersistentTask)future.actionGet()).getState();
    }

    Tuple<Integer, String> indexChunks(String name, InputStream is, int chunk, Checksum checksum, long timestamp) throws IOException {
        MessageDigest md5 = MessageDigests.md5();
        MessageDigest digest = checksum.digest();
        byte[] buf = EnterpriseGeoIpDownloader.getChunk(is);
        while (buf.length != 0) {
            md5.update(buf);
            if (digest != null) {
                digest.update(buf);
            }
            IndexRequest indexRequest = new IndexRequest(DATABASES_INDEX).id(name + "_" + chunk + "_" + timestamp).create(true).source(XContentType.SMILE, new Object[]{"name", name, "chunk", chunk, "data", buf});
            this.client.index(indexRequest).actionGet();
            ++chunk;
            buf = EnterpriseGeoIpDownloader.getChunk(is);
        }
        FlushRequest flushRequest = new FlushRequest(new String[]{DATABASES_INDEX});
        this.client.admin().indices().flush(flushRequest).actionGet();
        RefreshRequest refreshRequest = new RefreshRequest(new String[]{DATABASES_INDEX});
        this.client.admin().indices().refresh(refreshRequest).actionGet();
        String actualMd5 = MessageDigests.toHexString((byte[])md5.digest());
        String actualChecksum = digest == null ? actualMd5 : MessageDigests.toHexString((byte[])digest.digest());
        String expectedChecksum = checksum.checksum;
        if (!Objects.equals(expectedChecksum, actualChecksum)) {
            throw new IOException("checksum mismatch, expected [" + expectedChecksum + "], actual [" + actualChecksum + "]");
        }
        return Tuple.tuple((Object)chunk, (Object)actualMd5);
    }

    static byte[] getChunk(InputStream is) throws IOException {
        int chunkSize;
        int read;
        byte[] buf = new byte[0x100000];
        for (chunkSize = 0; chunkSize < 0x100000 && (read = is.read(buf, chunkSize, 0x100000 - chunkSize)) != -1; chunkSize += read) {
        }
        if (chunkSize < 0x100000) {
            buf = Arrays.copyOf(buf, chunkSize);
        }
        return buf;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void restartPeriodicRun() {
        if (this.isCancelled() || this.isCompleted() || this.threadPool.scheduler().isShutdown()) {
            logger.debug("Not restarting periodic run because task is cancelled, completed, or shutting down");
            return;
        }
        logger.debug("Restarting periodic run");
        EnterpriseGeoIpDownloader enterpriseGeoIpDownloader = this;
        synchronized (enterpriseGeoIpDownloader) {
            if (this.scheduledPeriodicRun != null) {
                boolean cancelSuccessful = this.scheduledPeriodicRun.cancel();
                logger.debug("Cancelled scheduled run: [{}]", (Object)cancelSuccessful);
            }
            this.scheduledPeriodicRun = this.threadPool.schedule(this::runPeriodic, this.pollIntervalSupplier.get(), (Executor)this.threadPool.generic());
        }
        this.requestRunOnDemand();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runPeriodic() {
        if (this.isCancelled() || this.isCompleted() || this.threadPool.scheduler().isShutdown()) {
            logger.debug("Not running periodic downloader because task is cancelled, completed, or shutting down");
            return;
        }
        logger.trace("Running periodic downloader");
        this.requestRunOnDemand();
        EnterpriseGeoIpDownloader enterpriseGeoIpDownloader = this;
        synchronized (enterpriseGeoIpDownloader) {
            this.scheduledPeriodicRun = this.threadPool.schedule(this::runPeriodic, this.pollIntervalSupplier.get(), (Executor)this.threadPool.generic());
        }
    }

    public void requestRunOnDemand() {
        if (this.isCancelled() || this.isCompleted()) {
            logger.debug("Not requesting downloader to run on demand because task is cancelled or completed");
            return;
        }
        logger.trace("Requesting downloader run on demand");
        if (this.queuedRuns.getAndIncrement() == 0) {
            logger.trace("Scheduling downloader run on demand");
            this.threadPool.generic().submit(this::runOnDemand);
        }
    }

    private void runOnDemand() {
        if (this.isCancelled() || this.isCompleted()) {
            logger.debug("Not running downloader on demand because task is cancelled or completed");
            return;
        }
        int currentQueueSize = this.queuedRuns.get();
        logger.trace("Running downloader on demand");
        try {
            this.runDownloader();
            logger.trace("Downloader completed successfully");
        }
        finally {
            if (this.queuedRuns.addAndGet(-currentQueueSize) > 0) {
                logger.debug("Downloader on demand requested again while running, scheduling another run");
                this.threadPool.generic().submit(this::runOnDemand);
            }
        }
    }

    void runDownloader() {
        if (this.isCancelled() || this.isCompleted()) {
            logger.debug("Not running downloader because task is cancelled or completed");
            return;
        }
        assert (this.state != null) : "this.setState() is null. You need to call setState() before calling runDownloader()";
        try {
            this.updateDatabases();
        }
        catch (Exception e) {
            logger.error("exception during databases update", (Throwable)e);
        }
        try {
            this.cleanDatabases();
        }
        catch (Exception e) {
            logger.error("exception during databases cleanup", (Throwable)e);
        }
    }

    private void cleanDatabases() {
        List<Tuple> expiredDatabases = this.state.getDatabases().entrySet().stream().filter(e -> !((GeoIpTaskState.Metadata)e.getValue()).isNewEnough(this.clusterService.state().metadata().settings())).map(entry -> Tuple.tuple((Object)((String)entry.getKey()), (Object)((GeoIpTaskState.Metadata)entry.getValue()))).toList();
        expiredDatabases.forEach(e -> {
            String name = (String)e.v1();
            GeoIpTaskState.Metadata meta = (GeoIpTaskState.Metadata)e.v2();
            this.deleteOldChunks(name, meta.lastChunk() + 1);
            this.state = this.state.put(name, new GeoIpTaskState.Metadata(meta.lastUpdate(), meta.firstChunk(), meta.lastChunk(), meta.md5(), meta.lastCheck() - 1L));
            this.updateTaskState();
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void onCancelled() {
        EnterpriseGeoIpDownloader enterpriseGeoIpDownloader = this;
        synchronized (enterpriseGeoIpDownloader) {
            if (this.scheduledPeriodicRun != null) {
                this.scheduledPeriodicRun.cancel();
            }
        }
        this.markAsCompleted();
    }

    private ProviderDownload downloaderFor(DatabaseConfiguration database) {
        DatabaseConfiguration.Provider provider = database.provider();
        if (provider instanceof DatabaseConfiguration.Maxmind) {
            DatabaseConfiguration.Maxmind maxmind = (DatabaseConfiguration.Maxmind)provider;
            return new MaxmindDownload(database.name(), maxmind);
        }
        provider = database.provider();
        if (provider instanceof DatabaseConfiguration.Ipinfo) {
            DatabaseConfiguration.Ipinfo ipinfo = (DatabaseConfiguration.Ipinfo)provider;
            return new IpinfoDownload(database.name(), ipinfo);
        }
        throw new IllegalArgumentException(Strings.format((String)"Unexpected provider [%s] for configuration [%s]", (Object[])new Object[]{database.provider().getClass(), database.id()}));
    }

    static interface ProviderDownload
    extends Closeable {
        public HttpClient.PasswordAuthenticationHolder buildCredentials();

        public String url(String var1);

        public boolean validCredentials();

        public Checksum checksum() throws IOException;

        public CheckedSupplier<InputStream, IOException> download();

        @Override
        public void close() throws IOException;
    }

    record Checksum(Type type, String checksum) {
        public Checksum {
            Objects.requireNonNull(type);
            Objects.requireNonNull(checksum);
        }

        static Checksum md5(String checksum) {
            return new Checksum(Type.MD5, checksum);
        }

        static Checksum sha256(String checksum) {
            return new Checksum(Type.SHA256, checksum);
        }

        MessageDigest digest() {
            return switch (this.type.ordinal()) {
                default -> throw new MatchException(null, null);
                case 0 -> null;
                case 1 -> MessageDigests.sha256();
            };
        }

        boolean matches(GeoIpTaskState.Metadata metadata) {
            return switch (this.type.ordinal()) {
                default -> throw new MatchException(null, null);
                case 0 -> this.checksum.equals(metadata.md5());
                case 1 -> this.checksum.equals(metadata.sha256());
            };
        }

        static enum Type {
            MD5,
            SHA256;

        }
    }

    class MaxmindDownload
    implements ProviderDownload {
        final String name;
        final DatabaseConfiguration.Maxmind maxmind;
        HttpClient.PasswordAuthenticationHolder auth;

        MaxmindDownload(String name, DatabaseConfiguration.Maxmind maxmind) {
            this.name = name;
            this.maxmind = maxmind;
            this.auth = this.buildCredentials();
        }

        @Override
        public HttpClient.PasswordAuthenticationHolder buildCredentials() {
            String username = this.maxmind.accountId();
            if (username == null || username.isEmpty() || username.isBlank()) {
                return null;
            }
            char[] passwordChars = EnterpriseGeoIpDownloader.this.tokenProvider.apply("maxmind");
            if (passwordChars == null || passwordChars.length == 0) {
                return null;
            }
            return new HttpClient.PasswordAuthenticationHolder(username, passwordChars);
        }

        @Override
        public boolean validCredentials() {
            return this.auth != null && this.auth.get() != null;
        }

        @Override
        public String url(String suffix) {
            Object endpointPattern = DEFAULT_MAXMIND_ENDPOINT;
            if (((String)endpointPattern).contains("%")) {
                throw new IllegalArgumentException("Invalid endpoint [" + (String)endpointPattern + "]");
            }
            if (!((String)endpointPattern).endsWith("/")) {
                endpointPattern = (String)endpointPattern + "/";
            }
            endpointPattern = (String)endpointPattern + "%s/download?suffix=%s";
            return Strings.format((String)endpointPattern, (Object[])new Object[]{this.name, suffix});
        }

        @Override
        public Checksum checksum() throws IOException {
            String sha256Url = this.url("tar.gz.sha256");
            String result = new String(EnterpriseGeoIpDownloader.this.httpClient.getBytes(this.auth.get(), sha256Url), StandardCharsets.UTF_8).trim();
            Matcher matcher = SHA256_CHECKSUM_PATTERN.matcher(result);
            boolean match = matcher.matches();
            if (!match) {
                throw new RuntimeException("Unexpected sha256 response from [" + sha256Url + "]");
            }
            String sha256 = matcher.group(1);
            return Checksum.sha256(sha256);
        }

        @Override
        public CheckedSupplier<InputStream, IOException> download() {
            String tgzUrl = this.url("tar.gz");
            return () -> EnterpriseGeoIpDownloader.this.httpClient.get(this.auth.get(), tgzUrl);
        }

        @Override
        public void close() throws IOException {
            if (this.auth != null) {
                this.auth.close();
            }
        }
    }

    class IpinfoDownload
    implements ProviderDownload {
        final String name;
        final DatabaseConfiguration.Ipinfo ipinfo;
        HttpClient.PasswordAuthenticationHolder auth;
        private static final Set<String> FREE_DATABASES = Set.of("asn", "country", "country_asn");

        IpinfoDownload(String name, DatabaseConfiguration.Ipinfo ipinfo) {
            this.name = name;
            this.ipinfo = ipinfo;
            this.auth = this.buildCredentials();
        }

        @Override
        public HttpClient.PasswordAuthenticationHolder buildCredentials() {
            char[] tokenChars = EnterpriseGeoIpDownloader.this.tokenProvider.apply("ipinfo");
            if (tokenChars == null || tokenChars.length == 0) {
                return null;
            }
            return new HttpClient.PasswordAuthenticationHolder(new String(tokenChars), new char[0]);
        }

        @Override
        public boolean validCredentials() {
            return this.auth != null && this.auth.get() != null;
        }

        @Override
        public String url(String suffix) {
            Object internalName = FREE_DATABASES.contains(this.name) ? "free/" + this.name : this.name;
            Object endpointPattern = DEFAULT_IPINFO_ENDPOINT;
            if (((String)endpointPattern).contains("%")) {
                throw new IllegalArgumentException("Invalid endpoint [" + (String)endpointPattern + "]");
            }
            if (!((String)endpointPattern).endsWith("/")) {
                endpointPattern = (String)endpointPattern + "/";
            }
            endpointPattern = (String)endpointPattern + "%s.%s";
            return Strings.format((String)endpointPattern, (Object[])new Object[]{internalName, suffix});
        }

        @Override
        public Checksum checksum() throws IOException {
            Map checksums;
            String checksumJsonUrl = this.url("mmdb/checksums");
            byte[] data = EnterpriseGeoIpDownloader.this.httpClient.getBytes(this.auth.get(), checksumJsonUrl);
            try (XContentParser parser = XContentType.JSON.xContent().createParser(XContentParserConfiguration.EMPTY, data);){
                checksums = parser.map();
            }
            String md5 = (String)((Map)checksums.get("checksums")).get("md5");
            logger.trace("checksum was [{}]", (Object)md5);
            Matcher matcher = MD5_CHECKSUM_PATTERN.matcher(md5);
            boolean match = matcher.matches();
            if (!match) {
                throw new RuntimeException("Unexpected md5 response from [" + checksumJsonUrl + "]");
            }
            return Checksum.md5(md5);
        }

        @Override
        public CheckedSupplier<InputStream, IOException> download() {
            String mmdbUrl = this.url("mmdb");
            return () -> EnterpriseGeoIpDownloader.this.httpClient.get(this.auth.get(), mmdbUrl);
        }

        @Override
        public void close() throws IOException {
            if (this.auth != null) {
                this.auth.close();
            }
        }
    }
}

