/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.blobstore.fs;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.OverlappingFileLockException;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.AccessDeniedException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefIterator;
import org.apache.lucene.util.Constants;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.common.blobstore.OptionalBytesReference;
import org.elasticsearch.common.blobstore.fs.FsBlobStore;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.BlobContainerUtils;
import org.elasticsearch.common.blobstore.support.BlobMetadata;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Iterators;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.util.concurrent.KeyedLock;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.repositories.blobstore.RequestedRangeNotSatisfiedException;

public class FsBlobContainer
extends AbstractBlobContainer {
    private static final Logger logger = LogManager.getLogger(FsBlobContainer.class);
    private static final String TEMP_FILE_PREFIX = "pending-";
    protected final FsBlobStore blobStore;
    protected final Path path;
    private static final KeyedLock<Path> writeMutexes = new KeyedLock();

    public FsBlobContainer(FsBlobStore blobStore, BlobPath blobPath, Path path) {
        super(blobPath);
        this.blobStore = blobStore;
        this.path = path;
    }

    public Path getPath() {
        return this.path;
    }

    @Override
    public Map<String, BlobMetadata> listBlobs(OperationPurpose purpose) throws IOException {
        return this.listBlobsByPrefix(purpose, null);
    }

    @Override
    public Map<String, BlobContainer> children(OperationPurpose purpose) throws IOException {
        HashMap<String, FsBlobContainer> builder = new HashMap<String, FsBlobContainer>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.path);){
            for (Path file : stream) {
                if (!Files.isDirectory(file, new LinkOption[0])) continue;
                String name = file.getFileName().toString();
                builder.put(name, new FsBlobContainer(this.blobStore, this.path().add(name), file));
            }
        }
        return Collections.unmodifiableMap(builder);
    }

    @Override
    public Map<String, BlobMetadata> listBlobsByPrefix(OperationPurpose purpose, String blobNamePrefix) throws IOException {
        HashMap<String, BlobMetadata> builder = new HashMap<String, BlobMetadata>();
        blobNamePrefix = blobNamePrefix == null ? "" : blobNamePrefix;
        try (DirectoryStream<Path> stream = this.newDirectoryStreamIfFound(blobNamePrefix);){
            for (Path file : stream) {
                BasicFileAttributes attrs;
                try {
                    attrs = Files.readAttributes(file, BasicFileAttributes.class, new LinkOption[0]);
                }
                catch (FileNotFoundException | NoSuchFileException e) {
                    continue;
                }
                catch (AccessDeniedException e) {
                    logger.warn(Strings.format((String)"file [%s] became inaccessible while listing [%s/%s]", (Object[])new Object[]{file, this.path, blobNamePrefix}), (Throwable)e);
                    assert (Constants.WINDOWS) : e;
                    continue;
                }
                if (!attrs.isRegularFile()) continue;
                builder.put(file.getFileName().toString(), new BlobMetadata(file.getFileName().toString(), attrs.size()));
            }
        }
        return Collections.unmodifiableMap(builder);
    }

    private DirectoryStream<Path> newDirectoryStreamIfFound(String blobNamePrefix) throws IOException {
        try {
            return Files.newDirectoryStream(this.path, blobNamePrefix + "*");
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            return new DirectoryStream<Path>(this){

                @Override
                public Iterator<Path> iterator() {
                    return Collections.emptyIterator();
                }

                @Override
                public void close() {
                }
            };
        }
    }

    @Override
    public DeleteResult delete(OperationPurpose purpose) throws IOException {
        final AtomicLong filesDeleted = new AtomicLong(0L);
        final AtomicLong bytesDeleted = new AtomicLong(0L);
        Files.walkFileTree(this.path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(this){

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException impossible) throws IOException {
                assert (impossible == null);
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                filesDeleted.incrementAndGet();
                bytesDeleted.addAndGet(attrs.size());
                return FileVisitResult.CONTINUE;
            }
        });
        return new DeleteResult(filesDeleted.get(), bytesDeleted.get());
    }

    @Override
    public void deleteBlobsIgnoringIfNotExists(OperationPurpose purpose, Iterator<String> blobNames) throws IOException {
        this.blobStore.deleteBlobs(Iterators.map(blobNames, blobName -> this.path.resolve((String)blobName).toString()));
    }

    @Override
    public boolean blobExists(OperationPurpose purpose, String blobName) {
        return Files.exists(this.path.resolve(blobName), new LinkOption[0]);
    }

    @Override
    public InputStream readBlob(OperationPurpose purpose, String name) throws IOException {
        assert (BlobContainer.assertPurposeConsistency(purpose, name));
        Path resolvedPath = this.path.resolve(name);
        try {
            return Files.newInputStream(resolvedPath, new OpenOption[0]);
        }
        catch (FileNotFoundException fnfe) {
            throw new NoSuchFileException("[" + name + "] blob not found");
        }
    }

    @Override
    public InputStream readBlob(OperationPurpose purpose, String blobName, long position, long length) throws IOException {
        assert (BlobContainer.assertPurposeConsistency(purpose, blobName));
        SeekableByteChannel channel = Files.newByteChannel(this.path.resolve(blobName), new OpenOption[0]);
        if (position > 0L) {
            if (channel.size() <= position) {
                SeekableByteChannel seekableByteChannel = channel;
                try {
                    throw new RequestedRangeNotSatisfiedException(blobName, position, length);
                }
                catch (Throwable throwable) {
                    if (seekableByteChannel != null) {
                        try {
                            seekableByteChannel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
            }
            channel.position(position);
        }
        assert (channel.position() == position);
        return Streams.limitStream(Channels.newInputStream(channel), length);
    }

    @Override
    public long readBlobPreferredLength() {
        return Long.MAX_VALUE;
    }

    @Override
    public void writeBlob(OperationPurpose purpose, String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (BlobContainer.assertPurposeConsistency(purpose, blobName));
        Path file = this.path.resolve(blobName);
        try {
            this.writeToPath(inputStream, file, blobSize);
        }
        catch (FileAlreadyExistsException faee) {
            if (failIfAlreadyExists) {
                throw faee;
            }
            this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(blobName));
            this.writeToPath(inputStream, file, blobSize);
        }
        IOUtils.fsync((Path)this.path, (boolean)true);
    }

    @Override
    public void writeBlob(OperationPurpose purpose, String blobName, BytesReference bytes, boolean failIfAlreadyExists) throws IOException {
        assert (BlobContainer.assertPurposeConsistency(purpose, blobName));
        Path file = this.path.resolve(blobName);
        try {
            FsBlobContainer.writeToPath(bytes, file);
        }
        catch (FileAlreadyExistsException faee) {
            if (failIfAlreadyExists) {
                throw faee;
            }
            this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(blobName));
            FsBlobContainer.writeToPath(bytes, file);
        }
        IOUtils.fsync((Path)this.path, (boolean)true);
    }

    @Override
    public void writeMetadataBlob(OperationPurpose purpose, String blobName, boolean failIfAlreadyExists, boolean atomic, CheckedConsumer<OutputStream, IOException> writer) throws IOException {
        assert (purpose != OperationPurpose.SNAPSHOT_DATA && BlobContainer.assertPurposeConsistency(purpose, blobName)) : purpose;
        if (atomic) {
            String tempBlob = FsBlobContainer.tempBlobName(blobName);
            try {
                this.writeToPath(purpose, tempBlob, true, writer);
                this.moveBlobAtomic(purpose, tempBlob, blobName, failIfAlreadyExists);
            }
            catch (IOException ex) {
                try {
                    this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(tempBlob));
                }
                catch (IOException e) {
                    ex.addSuppressed(e);
                }
                throw ex;
            }
        }
        this.writeToPath(purpose, blobName, failIfAlreadyExists, writer);
        IOUtils.fsync((Path)this.path, (boolean)true);
    }

    private void writeToPath(OperationPurpose purpose, String blobName, boolean failIfAlreadyExists, CheckedConsumer<OutputStream, IOException> writer) throws IOException {
        Path file = this.path.resolve(blobName);
        try (OutputStream out = FsBlobContainer.blobOutputStream(file);){
            writer.accept((Object)out);
        }
        catch (FileAlreadyExistsException faee) {
            if (failIfAlreadyExists) {
                throw faee;
            }
            this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(blobName));
            try (OutputStream out2 = FsBlobContainer.blobOutputStream(file);){
                writer.accept((Object)out2);
            }
        }
        IOUtils.fsync((Path)file, (boolean)false);
    }

    @Override
    public void writeBlobAtomic(OperationPurpose purpose, String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (purpose != OperationPurpose.SNAPSHOT_DATA && BlobContainer.assertPurposeConsistency(purpose, blobName)) : purpose;
        String tempBlob = FsBlobContainer.tempBlobName(blobName);
        Path tempBlobPath = this.path.resolve(tempBlob);
        try {
            this.writeToPath(inputStream, tempBlobPath, blobSize);
            this.moveBlobAtomic(purpose, tempBlob, blobName, failIfAlreadyExists);
        }
        catch (IOException ex) {
            try {
                this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(tempBlob));
            }
            catch (IOException e) {
                ex.addSuppressed(e);
            }
            throw ex;
        }
        finally {
            IOUtils.fsync((Path)this.path, (boolean)true);
        }
    }

    @Override
    public void writeBlobAtomic(OperationPurpose purpose, String blobName, BytesReference bytes, boolean failIfAlreadyExists) throws IOException {
        assert (purpose != OperationPurpose.SNAPSHOT_DATA && BlobContainer.assertPurposeConsistency(purpose, blobName)) : purpose;
        String tempBlob = FsBlobContainer.tempBlobName(blobName);
        Path tempBlobPath = this.path.resolve(tempBlob);
        try {
            FsBlobContainer.writeToPath(bytes, tempBlobPath);
            this.moveBlobAtomic(purpose, tempBlob, blobName, failIfAlreadyExists);
        }
        catch (IOException ex) {
            try {
                this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(tempBlob));
            }
            catch (IOException e) {
                ex.addSuppressed(e);
            }
            throw ex;
        }
        finally {
            IOUtils.fsync((Path)this.path, (boolean)true);
        }
    }

    @Override
    public void copyBlob(OperationPurpose purpose, BlobContainer sourceBlobContainer, String sourceBlobName, String blobName, long blobSize) throws IOException {
        if (!(sourceBlobContainer instanceof FsBlobContainer)) {
            throw new IllegalArgumentException("source blob container must be a FsBlobContainer");
        }
        FsBlobContainer sourceContainer = (FsBlobContainer)sourceBlobContainer;
        Path sourceBlobPath = sourceContainer.path.resolve(sourceBlobName);
        String tempBlob = FsBlobContainer.tempBlobName(blobName);
        Path tempBlobPath = this.path.resolve(tempBlob);
        Files.copy(sourceBlobPath, tempBlobPath, StandardCopyOption.REPLACE_EXISTING);
        try {
            this.moveBlobAtomic(purpose, tempBlob, blobName, false);
        }
        catch (IOException ex) {
            try {
                this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(tempBlob));
            }
            catch (IOException e) {
                ex.addSuppressed(e);
            }
            throw ex;
        }
    }

    private static void writeToPath(BytesReference bytes, Path tempBlobPath) throws IOException {
        try (OutputStream outputStream = Files.newOutputStream(tempBlobPath, StandardOpenOption.CREATE_NEW);){
            bytes.writeTo(outputStream);
        }
        IOUtils.fsync((Path)tempBlobPath, (boolean)false);
    }

    private void writeToPath(InputStream inputStream, Path tempBlobPath, long blobSize) throws IOException {
        try (OutputStream outputStream = Files.newOutputStream(tempBlobPath, StandardOpenOption.CREATE_NEW);){
            int bufferSize = this.blobStore.bufferSizeInBytes();
            long bytesWritten = org.elasticsearch.core.Streams.copy((InputStream)inputStream, (OutputStream)outputStream, (byte[])new byte[blobSize < (long)bufferSize ? Math.toIntExact(blobSize) : bufferSize]);
            assert (bytesWritten == blobSize) : "expected [" + blobSize + "] bytes but wrote [" + bytesWritten + "]";
        }
        IOUtils.fsync((Path)tempBlobPath, (boolean)false);
    }

    public void moveBlobAtomic(OperationPurpose purpose, String sourceBlobName, String targetBlobName, boolean failIfAlreadyExists) throws IOException {
        Path sourceBlobPath = this.path.resolve(sourceBlobName);
        Path targetBlobPath = this.path.resolve(targetBlobName);
        try {
            if (failIfAlreadyExists && Files.exists(targetBlobPath, new LinkOption[0])) {
                throw new FileAlreadyExistsException("blob [" + String.valueOf(targetBlobPath) + "] already exists, cannot overwrite");
            }
            Files.move(sourceBlobPath, targetBlobPath, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            if (failIfAlreadyExists) {
                throw e;
            }
            this.moveBlobNonAtomic(purpose, targetBlobName, sourceBlobPath, targetBlobPath, e);
        }
    }

    private void moveBlobNonAtomic(OperationPurpose purpose, String targetBlobName, Path sourceBlobPath, Path targetBlobPath, IOException e) throws IOException {
        try {
            this.deleteBlobsIgnoringIfNotExists(purpose, Iterators.single(targetBlobName));
            Files.move(sourceBlobPath, targetBlobPath, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException ex) {
            ex.addSuppressed(e);
            throw e;
        }
    }

    public static String tempBlobName(String blobName) {
        return TEMP_FILE_PREFIX + blobName + "-" + UUIDs.randomBase64UUID();
    }

    public static boolean isTempBlobName(String blobName) {
        return blobName.startsWith(TEMP_FILE_PREFIX);
    }

    private static OutputStream blobOutputStream(Path file) throws IOException {
        return Files.newOutputStream(file, StandardOpenOption.CREATE_NEW);
    }

    @Override
    public void getRegister(OperationPurpose purpose, String key, ActionListener<OptionalBytesReference> listener) {
        ActionListener.completeWith(listener, () -> FsBlobContainer.doUncontendedCompareAndExchangeRegister(this.path.resolve(key), BytesArray.EMPTY, BytesArray.EMPTY));
    }

    @Override
    public void compareAndExchangeRegister(OperationPurpose purpose, String key, BytesReference expected, BytesReference updated, ActionListener<OptionalBytesReference> listener) {
        ActionListener.completeWith(listener, () -> FsBlobContainer.doCompareAndExchangeRegister(this.path.resolve(key), expected, updated));
    }

    private static OptionalBytesReference doCompareAndExchangeRegister(Path registerPath, BytesReference expected, BytesReference updated) throws IOException {
        try (Releasable mutex = writeMutexes.tryAcquire(registerPath);){
            OptionalBytesReference optionalBytesReference = mutex == null ? OptionalBytesReference.MISSING : FsBlobContainer.doUncontendedCompareAndExchangeRegister(registerPath, expected, updated);
            return optionalBytesReference;
        }
    }

    @SuppressForbidden(reason="write to channel that we have open for locking purposes already directly")
    private static OptionalBytesReference doUncontendedCompareAndExchangeRegister(Path registerPath, BytesReference expected, BytesReference updated) throws IOException {
        OptionalBytesReference optionalBytesReference;
        block14: {
            BlobContainerUtils.ensureValidRegisterContent(updated);
            LockedFileChannel lockedFileChannel = LockedFileChannel.open(registerPath);
            try {
                FileChannel fileChannel = lockedFileChannel.fileChannel();
                ByteBuffer readBuf = ByteBuffer.allocate(32);
                while (readBuf.remaining() > 0 && fileChannel.read(readBuf) != -1) {
                }
                BytesArray found = new BytesArray(readBuf.array(), readBuf.arrayOffset(), readBuf.position());
                readBuf.clear();
                if (fileChannel.read(readBuf) != -1) {
                    throw new IllegalStateException("register contains more than [32] bytes");
                }
                if (expected.equals(found)) {
                    BytesRef page;
                    long pageStart = 0L;
                    BytesRefIterator iterator = updated.iterator();
                    while ((page = iterator.next()) != null) {
                        ByteBuffer writeBuf = ByteBuffer.wrap(page.bytes, page.offset, page.length);
                        while (writeBuf.remaining() > 0) {
                            fileChannel.write(writeBuf, pageStart + (long)writeBuf.position());
                        }
                        pageStart += (long)page.length;
                    }
                    fileChannel.force(true);
                }
                optionalBytesReference = OptionalBytesReference.of(found);
                if (lockedFileChannel == null) break block14;
            }
            catch (Throwable throwable) {
                try {
                    if (lockedFileChannel != null) {
                        try {
                            lockedFileChannel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (OverlappingFileLockException e) {
                    assert (false) : e;
                    return OptionalBytesReference.MISSING;
                }
            }
            lockedFileChannel.close();
        }
        return optionalBytesReference;
    }

    private record LockedFileChannel(FileChannel fileChannel, Closeable fileLock) implements Closeable
    {
        private static final ReentrantLock mutex = new ReentrantLock();

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        static LockedFileChannel open(Path path) throws IOException {
            ArrayList<Closeable> resources = new ArrayList<Closeable>(3);
            try {
                mutex.lock();
                resources.add(mutex::unlock);
                FileChannel fileChannel = LockedFileChannel.openOrCreateAtomic(path);
                resources.add(fileChannel);
                Closeable fileLock = fileChannel.lock()::close;
                resources.add(fileLock);
                LockedFileChannel result = new LockedFileChannel(fileChannel, fileLock);
                resources.clear();
                LockedFileChannel lockedFileChannel = result;
                return lockedFileChannel;
            }
            finally {
                IOUtils.closeWhileHandlingException(resources);
            }
        }

        private static FileChannel openOrCreateAtomic(Path path) throws IOException {
            try {
                if (!Files.exists(path, new LinkOption[0])) {
                    return FileChannel.open(path, StandardOpenOption.READ, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
                }
            }
            catch (FileAlreadyExistsException fileAlreadyExistsException) {
                // empty catch block
            }
            return FileChannel.open(path, StandardOpenOption.READ, StandardOpenOption.WRITE);
        }

        @Override
        public void close() throws IOException {
            Closeable[] closeableArray = new Closeable[3];
            closeableArray[0] = this.fileLock;
            closeableArray[1] = this.fileChannel;
            closeableArray[2] = mutex::unlock;
            IOUtils.close((Closeable[])closeableArray);
        }
    }
}

