/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.io.IOException;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import javax.net.ssl.X509ExtendedTrustManager;
import org.elasticsearch.common.ssl.KeyStoreUtil;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.common.ssl.SslFileUtil;
import org.elasticsearch.common.ssl.SslTrustConfig;
import org.elasticsearch.common.ssl.StoredCertificate;

public final class StoreTrustConfig
implements SslTrustConfig {
    private final String truststorePath;
    private final char[] password;
    private final String type;
    private final String algorithm;
    private final boolean requireTrustAnchors;
    private final Path configBasePath;

    public StoreTrustConfig(String path, char[] password, String type, String algorithm, boolean requireTrustAnchors, Path configBasePath) {
        this.truststorePath = Objects.requireNonNull(path, "Truststore path cannot be null");
        this.type = Objects.requireNonNull(type, "Truststore type cannot be null");
        this.algorithm = Objects.requireNonNull(algorithm, "Truststore algorithm cannot be null");
        this.password = Objects.requireNonNull(password, "Truststore password cannot be null (but may be empty)");
        this.requireTrustAnchors = requireTrustAnchors;
        this.configBasePath = configBasePath;
    }

    @Override
    public Collection<Path> getDependentFiles() {
        return List.of(this.resolvePath());
    }

    private Path resolvePath() {
        return this.configBasePath.resolve(this.truststorePath);
    }

    @Override
    public Collection<? extends StoredCertificate> getConfiguredCertificates() {
        Path path = this.resolvePath();
        KeyStore trustStore = this.readKeyStore(path);
        return KeyStoreUtil.stream(trustStore, ex -> this.keystoreException(path, (GeneralSecurityException)ex)).map(entry -> {
            X509Certificate certificate = entry.getX509Certificate();
            if (certificate != null) {
                boolean hasKey = entry.isKeyEntry();
                return new StoredCertificate(certificate, this.truststorePath, this.type, entry.getAlias(), hasKey);
            }
            return null;
        }).filter(Objects::nonNull).toList();
    }

    @Override
    public X509ExtendedTrustManager createTrustManager() {
        Path path = this.resolvePath();
        try {
            KeyStore store = this.readKeyStore(path);
            if (this.requireTrustAnchors) {
                StoreTrustConfig.checkTrustStore(store, path);
            }
            return KeyStoreUtil.createTrustManager(store, this.algorithm);
        }
        catch (GeneralSecurityException e) {
            throw this.keystoreException(path, e);
        }
    }

    private KeyStore readKeyStore(Path path) {
        try {
            return KeyStoreUtil.readKeyStore(path, this.type, this.password);
        }
        catch (SecurityException e) {
            throw SslFileUtil.accessControlFailure(this.fileTypeForException(), List.of(path), e, this.configBasePath);
        }
        catch (IOException e) {
            throw SslFileUtil.ioException(this.fileTypeForException(), List.of(path), e, this.getAdditionalErrorDetails());
        }
        catch (GeneralSecurityException e) {
            throw this.keystoreException(path, e);
        }
    }

    private SslConfigException keystoreException(Path path, GeneralSecurityException e) {
        String extra = this.getAdditionalErrorDetails();
        return SslFileUtil.securityException(this.fileTypeForException(), List.of(path), e, extra);
    }

    private String getAdditionalErrorDetails() {
        String extra = this.password.length == 0 ? "(no password was provided)" : "(a keystore password was provided)";
        return extra;
    }

    private String fileTypeForException() {
        return "[" + this.type + "] keystore (as a truststore)";
    }

    private static void checkTrustStore(KeyStore store, Path path) throws GeneralSecurityException {
        Enumeration<String> aliases = store.aliases();
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            if (!store.isCertificateEntry(alias)) continue;
            return;
        }
        throw new SslConfigException("the truststore [" + String.valueOf(path) + "] does not contain any trusted certificate entries");
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        StoreTrustConfig that = (StoreTrustConfig)o;
        return this.truststorePath.equals(that.truststorePath) && Arrays.equals(this.password, that.password) && this.type.equals(that.type) && this.algorithm.equals(that.algorithm);
    }

    public int hashCode() {
        int result = Objects.hash(this.truststorePath, this.type, this.algorithm);
        result = 31 * result + Arrays.hashCode(this.password);
        return result;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("StoreTrustConfig{");
        sb.append("path=").append(this.truststorePath);
        sb.append(", password=").append(this.password.length == 0 ? "<empty>" : "<non-empty>");
        sb.append(", type=").append(this.type);
        sb.append(", algorithm=").append(this.algorithm);
        sb.append('}');
        return sb.toString();
    }

    @Override
    public boolean hasExplicitConfig() {
        return true;
    }
}

