/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.datafeed;

import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.xpack.core.ml.datafeed.DatafeedTimingStats;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.DataCounts;

public class DatafeedTimingStatsReporter {
    private static final Logger logger = LogManager.getLogger(DatafeedTimingStatsReporter.class);
    private volatile DatafeedTimingStats persistedTimingStats;
    private final DatafeedTimingStats currentTimingStats;
    private final DatafeedTimingStatsPersister persister;
    private volatile boolean allowedPersisting;
    private CountDownLatch persistInProgressLatch;
    private static final double MIN_VALID_RATIO = 0.9;
    private static final double MAX_VALID_ABS_DIFFERENCE_MS = 10000.0;

    public DatafeedTimingStatsReporter(DatafeedTimingStats timingStats, DatafeedTimingStatsPersister persister) {
        Objects.requireNonNull(timingStats);
        this.persistedTimingStats = new DatafeedTimingStats(timingStats);
        this.currentTimingStats = new DatafeedTimingStats(timingStats);
        this.persister = Objects.requireNonNull(persister);
        this.allowedPersisting = true;
    }

    public DatafeedTimingStats getCurrentTimingStats() {
        return new DatafeedTimingStats(this.currentTimingStats);
    }

    public void reportSearchDuration(TimeValue searchDuration) {
        if (searchDuration == null) {
            return;
        }
        this.currentTimingStats.incrementSearchTimeMs((double)searchDuration.millis());
        this.flushIfDifferSignificantly();
    }

    public void reportDataCounts(DataCounts dataCounts) {
        if (dataCounts == null) {
            return;
        }
        this.currentTimingStats.incrementBucketCount(dataCounts.getBucketCount());
        if (dataCounts.getLatestRecordTimeStamp() != null) {
            this.currentTimingStats.setLatestRecordTimestamp(dataCounts.getLatestRecordTimeStamp().toInstant());
        }
        this.flushIfDifferSignificantly();
    }

    public void finishReporting() {
        try {
            this.flush(WriteRequest.RefreshPolicy.IMMEDIATE, true);
        }
        catch (InterruptedException e) {
            logger.warn("[{}] interrupted while finishing reporting of datafeed timing stats", (Object)this.currentTimingStats.getJobId());
            Thread.currentThread().interrupt();
        }
    }

    public void disallowPersisting() {
        this.allowedPersisting = false;
    }

    private void flushIfDifferSignificantly() {
        if (DatafeedTimingStatsReporter.differSignificantly(this.currentTimingStats, this.persistedTimingStats)) {
            try {
                this.flush(WriteRequest.RefreshPolicy.NONE, false);
            }
            catch (InterruptedException e) {
                assert (false) : "This should never happen when flush is called with mustWait set to false";
                Thread.currentThread().interrupt();
            }
        }
    }

    private synchronized void flush(WriteRequest.RefreshPolicy refreshPolicy, boolean mustWait) throws InterruptedException {
        String jobId = this.currentTimingStats.getJobId();
        if (this.allowedPersisting && mustWait && this.persistInProgressLatch != null) {
            this.persistInProgressLatch.await();
            this.persistInProgressLatch = null;
        }
        if (!this.allowedPersisting) {
            logger.trace("[{}] not persisting datafeed timing stats as persistence is disallowed", (Object)jobId);
            return;
        }
        if (this.persistInProgressLatch != null && this.persistInProgressLatch.getCount() > 0L) {
            logger.trace("[{}] not persisting datafeed timing stats as the previous persist is still in progress", (Object)jobId);
            return;
        }
        if (this.currentTimingStats.equals((Object)this.persistedTimingStats)) {
            logger.trace("[{}] not persisting datafeed timing stats as they are identical to latest already persisted", (Object)jobId);
            return;
        }
        CountDownLatch thisPersistLatch = new CountDownLatch(1);
        DatafeedTimingStats thisPersistTimingStats = new DatafeedTimingStats(this.currentTimingStats);
        this.persistInProgressLatch = thisPersistLatch;
        this.persister.persistDatafeedTimingStats(thisPersistTimingStats, refreshPolicy, (ActionListener<BulkResponse>)ActionListener.wrap(r -> {
            this.persistedTimingStats = thisPersistTimingStats;
            thisPersistLatch.countDown();
        }, e -> {
            thisPersistLatch.countDown();
            logger.warn(() -> "[" + jobId + "] failed to report datafeed timing stats", (Throwable)e);
        }));
        if (mustWait) {
            thisPersistLatch.await();
        }
    }

    public static boolean differSignificantly(DatafeedTimingStats stats1, DatafeedTimingStats stats2) {
        return DatafeedTimingStatsReporter.countsDifferSignificantly(stats1.getSearchCount(), stats2.getSearchCount()) || DatafeedTimingStatsReporter.differSignificantly(stats1.getTotalSearchTimeMs(), stats2.getTotalSearchTimeMs()) || DatafeedTimingStatsReporter.differSignificantly(stats1.getAvgSearchTimePerBucketMs(), stats2.getAvgSearchTimePerBucketMs()) || DatafeedTimingStatsReporter.differSignificantly(stats1.getExponentialAvgSearchTimePerHourMs(), stats2.getExponentialAvgSearchTimePerHourMs());
    }

    private static boolean countsDifferSignificantly(long value1, long value2) {
        return (double)value2 / (double)value1 < 0.9 || (double)value1 / (double)value2 < 0.9;
    }

    private static boolean differSignificantly(Double value1, Double value2) {
        if (value1 != null && value2 != null) {
            return value2 / value1 < 0.9 || value1 / value2 < 0.9 || Math.abs(value1 - value2) > 10000.0;
        }
        return value1 != null || value2 != null;
    }

    @FunctionalInterface
    public static interface DatafeedTimingStatsPersister {
        public void persistDatafeedTimingStats(DatafeedTimingStats var1, WriteRequest.RefreshPolicy var2, ActionListener<BulkResponse> var3);
    }
}

