/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.ingest.geoip;

import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.admin.indices.flush.FlushRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.client.internal.ProjectClient;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.MatchQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.index.reindex.DeleteByQueryAction;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.ingest.geoip.GeoIpTaskState;
import org.elasticsearch.ingest.geoip.HttpClient;
import org.elasticsearch.ingest.geoip.stats.GeoIpDownloaderStats;
import org.elasticsearch.persistent.AllocatedPersistentTask;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.tasks.TaskId;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentParserConfiguration;
import org.elasticsearch.xcontent.XContentType;

public class GeoIpDownloader
extends AllocatedPersistentTask {
    private static final Logger logger = LogManager.getLogger(GeoIpDownloader.class);
    private static final String DEFAULT_ENDPOINT = System.getProperty("ingest.geoip.downloader.endpoint.default", "https://geoip.elastic.co/v1/database");
    public static final Setting<String> ENDPOINT_SETTING = Setting.simpleString((String)"ingest.geoip.downloader.endpoint", (String)DEFAULT_ENDPOINT, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final String GEOIP_DOWNLOADER = "geoip-downloader";
    static final String DATABASES_INDEX = ".geoip_databases";
    static final String DATABASES_INDEX_PATTERN = ".geoip_databases*";
    static final int MAX_CHUNK_SIZE = 0x100000;
    private final ProjectClient client;
    private final HttpClient httpClient;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final String endpoint;
    protected volatile GeoIpTaskState state;
    private volatile Scheduler.ScheduledCancellable scheduled;
    private volatile GeoIpDownloaderStats stats = GeoIpDownloaderStats.EMPTY;
    private final Supplier<TimeValue> pollIntervalSupplier;
    private final Supplier<Boolean> eagerDownloadSupplier;
    private final Supplier<Boolean> atLeastOneGeoipProcessorSupplier;
    private final ProjectId projectId;

    GeoIpDownloader(ProjectClient client, HttpClient httpClient, ClusterService clusterService, ThreadPool threadPool, Settings settings, long id, String type, String action, String description, TaskId parentTask, Map<String, String> headers, Supplier<TimeValue> pollIntervalSupplier, Supplier<Boolean> eagerDownloadSupplier, Supplier<Boolean> atLeastOneGeoipProcessorSupplier, ProjectId projectId) {
        super(id, type, action, description, parentTask, headers);
        this.client = client;
        this.httpClient = httpClient;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.endpoint = (String)ENDPOINT_SETTING.get(settings);
        this.pollIntervalSupplier = pollIntervalSupplier;
        this.eagerDownloadSupplier = eagerDownloadSupplier;
        this.atLeastOneGeoipProcessorSupplier = atLeastOneGeoipProcessorSupplier;
        this.projectId = projectId;
    }

    void setState(GeoIpTaskState state) {
        assert (this.state == null);
        assert (state != null);
        this.state = state;
    }

    void updateDatabases() throws IOException {
        ClusterState clusterState = this.clusterService.state();
        IndexAbstraction geoipIndex = (IndexAbstraction)clusterState.getMetadata().getProject(this.projectId).getIndicesLookup().get(DATABASES_INDEX);
        if (geoipIndex != null) {
            logger.trace("The {} index is not null", (Object)DATABASES_INDEX);
            if (!clusterState.routingTable(this.projectId).index(geoipIndex.getWriteIndex()).allPrimaryShardsActive()) {
                logger.debug("Not updating geoip database because not all primary shards of the [.geoip_databases] index are active.");
                return;
            }
            ClusterBlockException blockException = clusterState.blocks().indexBlockedException(this.projectId, ClusterBlockLevel.WRITE, geoipIndex.getWriteIndex().getName());
            if (blockException != null) {
                logger.debug("Not updating geoip database because there is a write block on the " + geoipIndex.getWriteIndex().getName() + " index", (Throwable)blockException);
                return;
            }
        }
        if (this.eagerDownloadSupplier.get().booleanValue() || this.atLeastOneGeoipProcessorSupplier.get().booleanValue()) {
            logger.trace("Updating geoip databases");
            List response = this.fetchDatabasesOverview();
            for (Map res : response) {
                if (!res.get("name").toString().endsWith(".tgz")) continue;
                this.processDatabase(res);
            }
        } else {
            logger.trace("Not updating geoip databases because no geoip processors exist in the cluster and eager downloading is not configured");
        }
    }

    private <T> List<T> fetchDatabasesOverview() throws IOException {
        String url = this.endpoint + "?elastic_geoip_service_tos=agree";
        logger.debug("fetching geoip databases overview from [{}]", (Object)url);
        byte[] data = this.httpClient.getBytes(url);
        try (XContentParser parser = XContentType.JSON.xContent().createParser(XContentParserConfiguration.EMPTY, data);){
            List list = parser.list();
            return list;
        }
    }

    void processDatabase(Map<String, Object> databaseInfo) {
        String name = databaseInfo.get("name").toString().replace(".tgz", "") + ".mmdb";
        String md5 = (String)databaseInfo.get("md5_hash");
        Object url = databaseInfo.get("url").toString();
        if (!((String)url).startsWith("http")) {
            int lastSlash = this.endpoint.substring(8).lastIndexOf(47);
            url = (lastSlash != -1 ? this.endpoint.substring(0, lastSlash + 8) : this.endpoint) + "/" + (String)url;
        }
        this.processDatabase(name, md5, (String)url);
    }

    private void processDatabase(String name, String md5, String url) {
        GeoIpTaskState.Metadata metadata = this.state.getDatabases().getOrDefault(name, GeoIpTaskState.Metadata.EMPTY);
        if (Objects.equals(metadata.md5(), md5)) {
            this.updateTimestamp(name, metadata);
            return;
        }
        logger.debug("downloading geoip database [{}] for project [{}]", (Object)name, (Object)this.projectId);
        long start = System.currentTimeMillis();
        try (InputStream is = this.httpClient.get(url);){
            int firstChunk = metadata.lastChunk() + 1;
            int lastChunk = this.indexChunks(name, is, firstChunk, md5, start);
            if (lastChunk > firstChunk) {
                this.state = this.state.put(name, new GeoIpTaskState.Metadata(start, firstChunk, lastChunk - 1, md5, start));
                this.updateTaskState();
                this.stats = this.stats.successfulDownload(System.currentTimeMillis() - start).databasesCount(this.state.getDatabases().size());
                logger.info("successfully downloaded geoip database [{}] for project [{}]", (Object)name, (Object)this.projectId);
                this.deleteOldChunks(name, firstChunk);
            }
        }
        catch (Exception e) {
            this.stats = this.stats.failedDownload();
            logger.error(() -> "error downloading geoip database [" + name + "] for project [" + String.valueOf(this.projectId) + "]", (Throwable)e);
        }
    }

    void deleteOldChunks(String name, int firstChunk) {
        BoolQueryBuilder queryBuilder = new BoolQueryBuilder().filter((QueryBuilder)new MatchQueryBuilder("name", (Object)name)).filter((QueryBuilder)new RangeQueryBuilder("chunk").to((Object)firstChunk, false));
        DeleteByQueryRequest request = new DeleteByQueryRequest();
        request.indices(new String[]{DATABASES_INDEX});
        request.setQuery((QueryBuilder)queryBuilder);
        this.client.execute((ActionType)DeleteByQueryAction.INSTANCE, (ActionRequest)request, ActionListener.wrap(r -> {}, e -> logger.warn("could not delete old chunks for geoip database [" + name + "]", (Throwable)e)));
    }

    protected void updateTimestamp(String name, GeoIpTaskState.Metadata old) {
        logger.debug("geoip database [{}] is up to date for project [{}], updated timestamp", (Object)name, (Object)this.projectId);
        this.state = this.state.put(name, new GeoIpTaskState.Metadata(old.lastUpdate(), old.firstChunk(), old.lastChunk(), old.md5(), System.currentTimeMillis()));
        this.stats = this.stats.skippedDownload();
        this.updateTaskState();
    }

    void updateTaskState() {
        PlainActionFuture future = new PlainActionFuture();
        this.updateProjectPersistentTaskState(this.projectId, this.state, (ActionListener)future);
        this.state = (GeoIpTaskState)((PersistentTasksCustomMetadata.PersistentTask)future.actionGet()).getState();
    }

    int indexChunks(String name, InputStream is, int chunk, String expectedMd5, long timestamp) throws IOException {
        MessageDigest md = MessageDigests.md5();
        byte[] buf = GeoIpDownloader.getChunk(is);
        while (buf.length != 0) {
            md.update(buf);
            IndexRequest indexRequest = new IndexRequest(DATABASES_INDEX).id(name + "_" + chunk + "_" + timestamp).create(true).source(XContentType.SMILE, new Object[]{"name", name, "chunk", chunk, "data", buf});
            this.client.index(indexRequest).actionGet();
            ++chunk;
            buf = GeoIpDownloader.getChunk(is);
        }
        FlushRequest flushRequest = new FlushRequest(new String[]{DATABASES_INDEX});
        this.client.admin().indices().flush(flushRequest).actionGet();
        RefreshRequest refreshRequest = new RefreshRequest(new String[]{DATABASES_INDEX});
        this.client.admin().indices().refresh(refreshRequest).actionGet();
        String actualMd5 = MessageDigests.toHexString((byte[])md.digest());
        if (!Objects.equals(expectedMd5, actualMd5)) {
            throw new IOException("md5 checksum mismatch, expected [" + expectedMd5 + "], actual [" + actualMd5 + "]");
        }
        return chunk;
    }

    static byte[] getChunk(InputStream is) throws IOException {
        int chunkSize;
        int read;
        byte[] buf = new byte[0x100000];
        for (chunkSize = 0; chunkSize < 0x100000 && (read = is.read(buf, chunkSize, 0x100000 - chunkSize)) != -1; chunkSize += read) {
        }
        if (chunkSize < 0x100000) {
            buf = Arrays.copyOf(buf, chunkSize);
        }
        return buf;
    }

    void runDownloader() {
        assert (this.state != null);
        if (this.isCancelled() || this.isCompleted()) {
            return;
        }
        try {
            this.updateDatabases();
        }
        catch (Exception e) {
            this.stats = this.stats.failedDownload();
            logger.error("exception during geoip databases update", (Throwable)e);
        }
        try {
            this.cleanDatabases();
        }
        catch (Exception e) {
            logger.error("exception during geoip databases cleanup", (Throwable)e);
        }
        this.scheduleNextRun(this.pollIntervalSupplier.get());
    }

    public void requestReschedule() {
        if (this.isCancelled() || this.isCompleted()) {
            return;
        }
        if (this.scheduled != null && this.scheduled.cancel()) {
            this.scheduleNextRun(TimeValue.ZERO);
        }
    }

    private void cleanDatabases() {
        List<Tuple> expiredDatabases = this.state.getDatabases().entrySet().stream().filter(e -> !((GeoIpTaskState.Metadata)e.getValue()).isNewEnough(this.clusterService.state().metadata().settings())).map(entry -> Tuple.tuple((Object)((String)entry.getKey()), (Object)((GeoIpTaskState.Metadata)entry.getValue()))).toList();
        expiredDatabases.forEach(e -> {
            String name = (String)e.v1();
            GeoIpTaskState.Metadata meta = (GeoIpTaskState.Metadata)e.v2();
            this.deleteOldChunks(name, meta.lastChunk() + 1);
            this.state = this.state.put(name, new GeoIpTaskState.Metadata(meta.lastUpdate(), meta.firstChunk(), meta.lastChunk(), meta.md5(), meta.lastCheck() - 1L));
            this.updateTaskState();
        });
        this.stats = this.stats.expiredDatabases(expiredDatabases.size());
    }

    protected void onCancelled() {
        if (this.scheduled != null) {
            this.scheduled.cancel();
        }
        this.markAsCompleted();
    }

    public GeoIpDownloaderStats getStatus() {
        return this.isCancelled() || this.isCompleted() ? null : this.stats;
    }

    private void scheduleNextRun(TimeValue time) {
        if (!this.threadPool.scheduler().isShutdown()) {
            this.scheduled = this.threadPool.schedule(this::runDownloader, time, (Executor)this.threadPool.generic());
        }
    }
}

