/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.ProjectSecrets;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3ClientSettings;
import org.elasticsearch.repositories.s3.S3Repository;

public class S3ClientsManager
implements ClusterStateApplier {
    private static final Logger logger = LogManager.getLogger(S3ClientsManager.class);
    private static final String S3_SETTING_PREFIX = "s3.";
    private final Settings nodeS3Settings;
    private final Function<S3ClientSettings, AmazonS3Reference> clientBuilder;
    private final Executor executor;
    private final AtomicBoolean managerClosed = new AtomicBoolean(false);
    private final Map<ProjectId, PerProjectClientsHolder> perProjectClientsHolders;
    private final ClusterClientsHolder clusterClientsHolder;

    S3ClientsManager(Settings nodeSettings, Function<S3ClientSettings, AmazonS3Reference> clientBuilder, Executor executor, boolean supportsMultipleProjects) {
        this.nodeS3Settings = Settings.builder().put(nodeSettings.getByPrefix(S3_SETTING_PREFIX), false).normalizePrefix(S3_SETTING_PREFIX).build();
        this.clientBuilder = clientBuilder;
        this.executor = executor;
        this.clusterClientsHolder = new ClusterClientsHolder(this);
        this.perProjectClientsHolders = supportsMultipleProjects ? new ConcurrentHashMap() : null;
    }

    public void applyClusterState(ClusterChangedEvent event) {
        assert (this.perProjectClientsHolders != null) : "expect per-project clients holders to be non-null";
        Map currentProjects = event.state().metadata().projects();
        HashMap<ProjectId, PerProjectClientsHolder> updatedPerProjectClients = new HashMap<ProjectId, PerProjectClientsHolder>();
        ArrayList<PerProjectClientsHolder> clientsHoldersToClose = new ArrayList<PerProjectClientsHolder>();
        for (ProjectMetadata project : currentProjects.values()) {
            if (ProjectId.DEFAULT.equals((Object)project.id())) continue;
            ProjectSecrets projectSecrets = (ProjectSecrets)project.custom("project_state_secrets");
            if (projectSecrets == null || projectSecrets.getSettingNames().stream().noneMatch(key -> key.startsWith(S3_SETTING_PREFIX))) {
                PerProjectClientsHolder removed = this.perProjectClientsHolders.remove(project.id());
                if (removed == null) continue;
                clientsHoldersToClose.add(removed);
                continue;
            }
            Settings currentSettings = Settings.builder().put(this.nodeS3Settings).setSecureSettings(projectSecrets.getSettings()).build();
            Map<String, S3ClientSettings> clientSettings = S3ClientSettings.load(currentSettings).entrySet().stream().filter(entry -> ((S3ClientSettings)entry.getValue()).credentials != null).collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, Map.Entry::getValue));
            if (clientSettings.isEmpty()) {
                logger.warn("Skipping project [{}] with no client settings", new Object[]{project.id()});
                continue;
            }
            if (!this.newOrUpdated(project.id(), clientSettings)) continue;
            updatedPerProjectClients.put(project.id(), new PerProjectClientsHolder(this, project.id(), clientSettings));
        }
        for (ProjectId projectId : updatedPerProjectClients.keySet()) {
            assert (!ProjectId.DEFAULT.equals((Object)projectId));
            PerProjectClientsHolder old = this.perProjectClientsHolders.put(projectId, (PerProjectClientsHolder)updatedPerProjectClients.get(projectId));
            if (old == null) continue;
            clientsHoldersToClose.add(old);
        }
        for (ProjectId projectId : this.perProjectClientsHolders.keySet()) {
            assert (!ProjectId.DEFAULT.equals((Object)projectId));
            if (currentProjects.containsKey(projectId)) continue;
            PerProjectClientsHolder removed = this.perProjectClientsHolders.remove(projectId);
            clientsHoldersToClose.add(removed);
        }
        if (!clientsHoldersToClose.isEmpty()) {
            this.closePerProjectClientsAsync(clientsHoldersToClose);
        }
    }

    private void closePerProjectClientsAsync(final List<PerProjectClientsHolder> clientsHoldersToClose) {
        this.executor.execute((Runnable)new AbstractRunnable(this){

            protected void doRun() throws Exception {
                IOUtils.closeWhileHandlingException((Iterable)clientsHoldersToClose);
            }

            public void onFailure(Exception e) {
                logger.warn("Failed to close s3 clients", (Throwable)e);
            }
        });
    }

    ClusterClientsHolder getClusterClientsHolder() {
        return this.clusterClientsHolder;
    }

    Map<ProjectId, PerProjectClientsHolder> getPerProjectClientsHolders() {
        return this.perProjectClientsHolders == null ? null : Map.copyOf(this.perProjectClientsHolders);
    }

    boolean isManagerClosed() {
        return this.managerClosed.get();
    }

    void refreshAndClearCacheForClusterClients(Map<String, S3ClientSettings> clientsSettings) {
        this.clusterClientsHolder.refreshAndClearCache(clientsSettings);
    }

    S3ClientSettings settingsForClient(ProjectId projectId, RepositoryMetadata repositoryMetadata) {
        return this.getClientsHolderSafe(projectId).singleClientSettings(repositoryMetadata);
    }

    AmazonS3Reference client(ProjectId projectId, RepositoryMetadata repositoryMetadata) {
        return this.getClientsHolderSafe(projectId).client(repositoryMetadata);
    }

    void releaseCachedClients(ProjectId projectId) {
        ClientsHolder<?> old = this.getClientsHolder(projectId);
        if (old != null) {
            old.clearCache();
        }
    }

    private ClientsHolder<?> getClientsHolderSafe(ProjectId projectId) {
        ClientsHolder<?> clientsHolder = this.getClientsHolder(projectId);
        if (clientsHolder == null) {
            throw new IllegalArgumentException("no s3 client is configured for project [" + String.valueOf(projectId) + "]");
        }
        return clientsHolder;
    }

    @Nullable
    private ClientsHolder<?> getClientsHolder(ProjectId projectId) {
        if (ProjectId.DEFAULT.equals((Object)Objects.requireNonNull(projectId))) {
            return this.clusterClientsHolder;
        }
        assert (this.perProjectClientsHolders != null) : "expect per-project clients holders to be non-null";
        return this.perProjectClientsHolders.get(projectId);
    }

    void close() {
        if (this.managerClosed.compareAndSet(false, true)) {
            if (this.perProjectClientsHolders != null) {
                IOUtils.closeWhileHandlingException(this.perProjectClientsHolders.values());
            }
            IOUtils.closeWhileHandlingException((Closeable)this.clusterClientsHolder);
        } else assert (false) : "attempting to close s3 clients manager multiple times";
    }

    private boolean newOrUpdated(ProjectId projectId, Map<String, S3ClientSettings> currentClientSettings) {
        PerProjectClientsHolder old = this.perProjectClientsHolders.get(projectId);
        if (old == null) {
            return true;
        }
        return !currentClientSettings.equals(old.allClientSettings());
    }

    final class ClusterClientsHolder
    extends ClientsHolder<S3ClientSettings> {
        private volatile Map<String, S3ClientSettings> staticClientSettings = Map.of("default", S3ClientSettings.getClientSettings(Settings.EMPTY, "default"));
        private volatile Map<Settings, S3ClientSettings> derivedClientSettings = Collections.emptyMap();

        ClusterClientsHolder(S3ClientsManager this$0) {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        S3ClientSettings clientKey(RepositoryMetadata repositoryMetadata) {
            Settings settings = repositoryMetadata.settings();
            S3ClientSettings existing = this.derivedClientSettings.get(settings);
            if (existing != null) {
                return existing;
            }
            String clientName = (String)S3Repository.CLIENT_NAME.get(settings);
            S3ClientSettings staticSettings = this.staticClientSettings.get(clientName);
            if (staticSettings != null) {
                ClusterClientsHolder clusterClientsHolder = this;
                synchronized (clusterClientsHolder) {
                    S3ClientSettings existing2 = this.derivedClientSettings.get(settings);
                    if (existing2 != null) {
                        return existing2;
                    }
                    S3ClientSettings newSettings = staticSettings.refine(settings);
                    this.derivedClientSettings = Maps.copyMapWithAddedOrReplacedEntry(this.derivedClientSettings, (Object)settings, (Object)newSettings);
                    return newSettings;
                }
            }
            throw new IllegalArgumentException("Unknown s3 client name [" + clientName + "]. Existing client configs: " + Strings.collectionToDelimitedString(this.staticClientSettings.keySet(), (String)","));
        }

        @Override
        S3ClientSettings singleClientSettings(S3ClientSettings clientKey) {
            return clientKey;
        }

        @Override
        Map<String, S3ClientSettings> allClientSettings() {
            return this.staticClientSettings;
        }

        @Override
        void doClearCache() {
            this.derivedClientSettings = Collections.emptyMap();
        }

        @Override
        ProjectId projectId() {
            return ProjectId.DEFAULT;
        }

        synchronized void refreshAndClearCache(Map<String, S3ClientSettings> clientsSettings) {
            this.clearCache();
            this.staticClientSettings = Maps.ofEntries(clientsSettings.entrySet());
            assert (this.staticClientSettings.containsKey("default")) : "always at least have 'default'";
        }
    }

    final class PerProjectClientsHolder
    extends ClientsHolder<String> {
        private final ProjectId projectId;
        private final Map<String, S3ClientSettings> clientSettings;

        PerProjectClientsHolder(S3ClientsManager this$0, ProjectId projectId, Map<String, S3ClientSettings> clientSettings) {
            assert (!ProjectId.DEFAULT.equals((Object)projectId));
            this.projectId = projectId;
            this.clientSettings = clientSettings;
        }

        @Override
        Map<String, S3ClientSettings> allClientSettings() {
            return this.clientSettings;
        }

        @Override
        String clientKey(RepositoryMetadata repositoryMetadata) {
            return (String)S3Repository.CLIENT_NAME.get(repositoryMetadata.settings());
        }

        @Override
        S3ClientSettings singleClientSettings(String clientKey) {
            S3ClientSettings settings = this.clientSettings.get(clientKey);
            if (settings == null) {
                throw new IllegalArgumentException("s3 client [" + clientKey + "] does not exist for project [" + String.valueOf(this.projectId) + "]");
            }
            return settings;
        }

        @Override
        ProjectId projectId() {
            return this.projectId;
        }
    }

    abstract class ClientsHolder<K>
    implements Closeable {
        protected volatile Map<K, AmazonS3Reference> clientsCache = Collections.emptyMap();
        private final AtomicBoolean closed = new AtomicBoolean(false);

        ClientsHolder() {
        }

        abstract K clientKey(RepositoryMetadata var1);

        abstract S3ClientSettings singleClientSettings(K var1);

        abstract Map<String, S3ClientSettings> allClientSettings();

        abstract ProjectId projectId();

        S3ClientSettings singleClientSettings(RepositoryMetadata repositoryMetadata) {
            return this.singleClientSettings(this.clientKey(repositoryMetadata));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        final AmazonS3Reference client(RepositoryMetadata repositoryMetadata) {
            K clientKey = this.clientKey(repositoryMetadata);
            AmazonS3Reference clientReference = this.clientsCache.get(clientKey);
            if (clientReference != null && clientReference.tryIncRef()) {
                return clientReference;
            }
            S3ClientSettings settings = this.singleClientSettings(clientKey);
            ClientsHolder clientsHolder = this;
            synchronized (clientsHolder) {
                AmazonS3Reference existing = this.clientsCache.get(clientKey);
                if (existing != null && existing.tryIncRef()) {
                    return existing;
                }
                if (this.closed.get()) {
                    throw new AlreadyClosedException("Project [" + String.valueOf(this.projectId()) + "] clients holder is closed");
                }
                if (S3ClientsManager.this.managerClosed.get()) {
                    assert (this.clientsCache.isEmpty()) : "expect empty cache, but got " + String.valueOf(this.clientsCache);
                    throw new AlreadyClosedException("s3 clients manager is closed");
                }
                AmazonS3Reference newClientReference = S3ClientsManager.this.clientBuilder.apply(settings);
                this.clientsCache = Maps.copyMapWithAddedEntry(this.clientsCache, clientKey, (Object)((Object)newClientReference));
                return newClientReference;
            }
        }

        final synchronized void clearCache() {
            IOUtils.closeWhileHandlingException(this.clientsCache.values());
            this.clientsCache = Collections.emptyMap();
            this.doClearCache();
        }

        void doClearCache() {
        }

        @Override
        public final void close() {
            if (this.closed.compareAndSet(false, true)) {
                this.clearCache();
            }
        }

        final boolean isClosed() {
            return this.closed.get();
        }
    }
}

