/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.compute.aggregation;

import java.util.Arrays;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.RamUsageEstimator;
import org.elasticsearch.common.breaker.CircuitBreaker;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.common.util.ObjectArray;
import org.elasticsearch.compute.aggregation.GroupingAggregatorEvaluationContext;
import org.elasticsearch.compute.aggregation.GroupingAggregatorState;
import org.elasticsearch.compute.aggregation.SeenGroupIds;
import org.elasticsearch.compute.aggregation.TimeSeriesGroupingAggregatorEvaluationContext;
import org.elasticsearch.compute.data.Block;
import org.elasticsearch.compute.data.BlockFactory;
import org.elasticsearch.compute.data.DoubleBlock;
import org.elasticsearch.compute.data.DoubleVector;
import org.elasticsearch.compute.data.IntVector;
import org.elasticsearch.compute.data.LongBlock;
import org.elasticsearch.compute.operator.DriverContext;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;

public class RateLongAggregator {
    public static LongRateGroupingState initGrouping(DriverContext driverContext) {
        return new LongRateGroupingState(driverContext.bigArrays(), driverContext.breaker());
    }

    public static void combine(LongRateGroupingState current, int groupId, long timestamp, long value) {
        current.append(groupId, timestamp, value);
    }

    public static void combineIntermediate(LongRateGroupingState current, int groupId, LongBlock timestamps, LongBlock values, int sampleCount, double reset, int otherPosition) {
        current.combine(groupId, timestamps, values, sampleCount, reset, otherPosition);
    }

    public static void combineStates(LongRateGroupingState current, int currentGroupId, LongRateGroupingState otherState, int otherGroupId) {
        current.combineState(currentGroupId, otherState, otherGroupId);
    }

    public static Block evaluateFinal(LongRateGroupingState state, IntVector selected, GroupingAggregatorEvaluationContext evalContext) {
        return state.evaluateFinal(selected, evalContext);
    }

    public static final class LongRateGroupingState
    implements Releasable,
    Accountable,
    GroupingAggregatorState {
        private ObjectArray<LongRateState> states;
        private final BigArrays bigArrays;
        private final CircuitBreaker breaker;
        private long stateBytes;

        LongRateGroupingState(BigArrays bigArrays, CircuitBreaker breaker) {
            this.bigArrays = bigArrays;
            this.breaker = breaker;
            this.states = bigArrays.newObjectArray(1L);
        }

        void ensureCapacity(int groupId) {
            this.states = this.bigArrays.grow(this.states, (long)(groupId + 1));
        }

        void adjustBreaker(long bytes) {
            this.breaker.addEstimateBytesAndMaybeBreak(bytes, "<<rate aggregation>>");
            this.stateBytes += bytes;
            assert (this.stateBytes >= 0L) : this.stateBytes;
        }

        void append(int groupId, long timestamp, long value) {
            this.ensureCapacity(groupId);
            LongRateState state = (LongRateState)this.states.get((long)groupId);
            if (state == null) {
                this.adjustBreaker(LongRateState.bytesUsed(1));
                state = new LongRateState(new long[]{timestamp}, new long[]{value});
                this.states.set((long)groupId, (Object)state);
            } else if (state.entries() == 1) {
                this.adjustBreaker(LongRateState.bytesUsed(2));
                state = new LongRateState(new long[]{state.timestamps[0], timestamp}, new long[]{state.values[0], value});
                this.states.set((long)groupId, (Object)state);
                this.adjustBreaker(-LongRateState.bytesUsed(1));
            } else {
                state.append(timestamp, value);
            }
        }

        void combine(int groupId, LongBlock timestamps, LongBlock values, int sampleCount, double reset, int otherPosition) {
            int valueCount = timestamps.getValueCount(otherPosition);
            if (valueCount == 0) {
                return;
            }
            int firstIndex = timestamps.getFirstValueIndex(otherPosition);
            this.ensureCapacity(groupId);
            LongRateState state = (LongRateState)this.states.get((long)groupId);
            if (state == null) {
                this.adjustBreaker(LongRateState.bytesUsed(valueCount));
                state = new LongRateState(valueCount);
                state.reset = reset;
                state.sampleCount = sampleCount;
                this.states.set((long)groupId, (Object)state);
                for (int i = 0; i < valueCount; ++i) {
                    state.timestamps[i] = timestamps.getLong(firstIndex + i);
                    state.values[i] = values.getLong(firstIndex + i);
                }
            } else {
                this.adjustBreaker(LongRateState.bytesUsed(state.entries() + valueCount));
                LongRateState newState = new LongRateState(state.entries() + valueCount);
                newState.reset = state.reset + reset;
                newState.sampleCount = state.sampleCount + sampleCount;
                this.states.set((long)groupId, (Object)newState);
                this.merge(state, newState, firstIndex, valueCount, timestamps, values);
                this.adjustBreaker(-LongRateState.bytesUsed(state.entries()));
            }
        }

        void merge(LongRateState curr, LongRateState dst, int firstIndex, int rightCount, LongBlock timestamps, LongBlock values) {
            int i = 0;
            int j = 0;
            int k = 0;
            int leftCount = curr.entries();
            while (i < leftCount && j < rightCount) {
                long t1 = curr.timestamps[i];
                long t2 = timestamps.getLong(firstIndex + j);
                if (t1 > t2) {
                    dst.timestamps[k] = t1;
                    dst.values[k] = curr.values[i];
                    ++i;
                } else {
                    dst.timestamps[k] = t2;
                    dst.values[k] = values.getLong(firstIndex + j);
                    ++j;
                }
                ++k;
            }
            if (i < leftCount) {
                System.arraycopy(curr.timestamps, i, dst.timestamps, k, leftCount - i);
                System.arraycopy(curr.values, i, dst.values, k, leftCount - i);
            }
            while (j < rightCount) {
                dst.timestamps[k] = timestamps.getLong(firstIndex + j);
                dst.values[k] = values.getLong(firstIndex + j);
                ++k;
                ++j;
            }
        }

        void combineState(int groupId, LongRateGroupingState otherState, int otherGroupId) {
            LongRateState other;
            LongRateState longRateState = other = (long)otherGroupId < otherState.states.size() ? (LongRateState)otherState.states.get((long)otherGroupId) : null;
            if (other == null) {
                return;
            }
            this.ensureCapacity(groupId);
            LongRateState curr = (LongRateState)this.states.get((long)groupId);
            if (curr == null) {
                int len = other.entries();
                this.adjustBreaker(LongRateState.bytesUsed(len));
                curr = new LongRateState(Arrays.copyOf(other.timestamps, len), Arrays.copyOf(other.values, len));
                curr.reset = other.reset;
                curr.sampleCount = other.sampleCount;
                this.states.set((long)groupId, (Object)curr);
            } else {
                this.states.set((long)groupId, (Object)this.mergeState(curr, other));
            }
        }

        LongRateState mergeState(LongRateState s1, LongRateState s2) {
            int newLen = s1.entries() + s2.entries();
            this.adjustBreaker(LongRateState.bytesUsed(newLen));
            LongRateState dst = new LongRateState(newLen);
            dst.reset = s1.reset + s2.reset;
            dst.sampleCount = s1.sampleCount + s2.sampleCount;
            int i = 0;
            int j = 0;
            int k = 0;
            while (i < s1.entries() && j < s2.entries()) {
                if (s1.timestamps[i] > s2.timestamps[j]) {
                    dst.timestamps[k] = s1.timestamps[i];
                    dst.values[k] = s1.values[i];
                    ++i;
                } else {
                    dst.timestamps[k] = s2.timestamps[j];
                    dst.values[k] = s2.values[j];
                    ++j;
                }
                ++k;
            }
            System.arraycopy(s1.timestamps, i, dst.timestamps, k, s1.entries() - i);
            System.arraycopy(s1.values, i, dst.values, k, s1.entries() - i);
            System.arraycopy(s2.timestamps, j, dst.timestamps, k, s2.entries() - j);
            System.arraycopy(s2.values, j, dst.values, k, s2.entries() - j);
            return dst;
        }

        public long ramBytesUsed() {
            return this.states.ramBytesUsed() + this.stateBytes;
        }

        public void close() {
            Releasables.close((Releasable[])new Releasable[]{this.states, () -> this.adjustBreaker(-this.stateBytes)});
        }

        @Override
        public void toIntermediate(Block[] blocks, int offset, IntVector selected, DriverContext driverContext) {
            assert (blocks.length >= offset + 3) : "blocks=" + blocks.length + ",offset=" + offset;
            BlockFactory blockFactory = driverContext.blockFactory();
            int positionCount = selected.getPositionCount();
            try (LongBlock.Builder timestamps = blockFactory.newLongBlockBuilder(positionCount * 2);
                 LongBlock.Builder values = blockFactory.newLongBlockBuilder(positionCount * 2);
                 IntVector.FixedBuilder sampleCounts = blockFactory.newIntVectorFixedBuilder(positionCount);
                 DoubleVector.FixedBuilder resets = blockFactory.newDoubleVectorFixedBuilder(positionCount);){
                for (int i = 0; i < positionCount; ++i) {
                    LongRateState state;
                    int groupId = selected.getInt(i);
                    LongRateState longRateState = state = (long)groupId < this.states.size() ? (LongRateState)this.states.get((long)groupId) : null;
                    if (state != null) {
                        timestamps.beginPositionEntry();
                        for (long t : state.timestamps) {
                            timestamps.appendLong(t);
                        }
                        timestamps.endPositionEntry();
                        values.beginPositionEntry();
                        for (long v : state.values) {
                            values.appendLong(v);
                        }
                        values.endPositionEntry();
                        sampleCounts.appendInt(i, state.sampleCount);
                        resets.appendDouble(i, state.reset);
                        continue;
                    }
                    timestamps.appendNull();
                    values.appendNull();
                    sampleCounts.appendInt(i, 0);
                    resets.appendDouble(i, 0.0);
                }
                blocks[offset] = timestamps.build();
                blocks[offset + 1] = values.build();
                blocks[offset + 2] = sampleCounts.build().asBlock();
                blocks[offset + 3] = resets.build().asBlock();
            }
        }

        private static double computeRateWithoutExtrapolate(LongRateState state) {
            int len = state.entries();
            assert (len >= 2) : "rate requires at least two samples; got " + len;
            long firstTS = state.timestamps[state.timestamps.length - 1];
            long lastTS = state.timestamps[0];
            double reset = state.reset;
            for (int i = 1; i < len; ++i) {
                if (state.values[i - 1] >= state.values[i]) continue;
                reset += (double)state.values[i];
            }
            double firstValue = state.values[len - 1];
            double lastValue = (double)state.values[0] + reset;
            return (lastValue - firstValue) * 1000.0 / (double)(lastTS - firstTS);
        }

        private static double extrapolateRate(LongRateState state, long rangeStart, long rangeEnd) {
            double endGap;
            int len = state.entries();
            assert (len >= 2) : "rate requires at least two samples; got " + len;
            long firstTS = state.timestamps[state.timestamps.length - 1];
            long lastTS = state.timestamps[0];
            double reset = state.reset;
            for (int i = 1; i < len; ++i) {
                if (state.values[i - 1] >= state.values[i]) continue;
                reset += (double)state.values[i];
            }
            double firstValue = state.values[len - 1];
            double lastValue = (double)state.values[0] + reset;
            double sampleTS = lastTS - firstTS;
            double averageSampleInterval = sampleTS / (double)state.sampleCount;
            double slope = (lastValue - firstValue) / sampleTS;
            double startGap = firstTS - rangeStart;
            if (startGap > 0.0) {
                if (startGap > averageSampleInterval * 1.1) {
                    startGap = averageSampleInterval / 2.0;
                }
                firstValue = Math.max(0.0, firstValue - startGap * slope);
            }
            if ((endGap = (double)(rangeEnd - lastTS)) > 0.0) {
                if (endGap > averageSampleInterval * 1.1) {
                    endGap = averageSampleInterval / 2.0;
                }
                lastValue += endGap * slope;
            }
            return (lastValue - firstValue) * 1000.0 / (double)(rangeEnd - rangeStart);
        }

        Block evaluateFinal(IntVector selected, GroupingAggregatorEvaluationContext evalContext) {
            int positionCount = selected.getPositionCount();
            try (DoubleBlock.Builder rates = evalContext.blockFactory().newDoubleBlockBuilder(positionCount);){
                for (int p = 0; p < positionCount; ++p) {
                    double rate;
                    LongRateState state;
                    int groupId = selected.getInt(p);
                    LongRateState longRateState = state = (long)groupId < this.states.size() ? (LongRateState)this.states.get((long)groupId) : null;
                    if (state == null || state.sampleCount < 2) {
                        rates.appendNull();
                        continue;
                    }
                    int len = state.entries();
                    if (evalContext instanceof TimeSeriesGroupingAggregatorEvaluationContext) {
                        TimeSeriesGroupingAggregatorEvaluationContext tsContext = (TimeSeriesGroupingAggregatorEvaluationContext)evalContext;
                        rate = LongRateGroupingState.extrapolateRate(state, tsContext.rangeStartInMillis(groupId), tsContext.rangeEndInMillis(groupId));
                    } else {
                        rate = LongRateGroupingState.computeRateWithoutExtrapolate(state);
                    }
                    rates.appendDouble(rate);
                }
                DoubleBlock doubleBlock = rates.build();
                return doubleBlock;
            }
        }

        @Override
        public void enableGroupIdTracking(SeenGroupIds seenGroupIds) {
        }
    }

    private static class LongRateState {
        static final long BASE_RAM_USAGE = RamUsageEstimator.sizeOfObject(LongRateState.class);
        final long[] timestamps;
        final long[] values;
        int sampleCount = 0;
        double reset = 0.0;

        LongRateState(int initialSize) {
            this.timestamps = new long[initialSize];
            this.values = new long[initialSize];
        }

        LongRateState(long[] ts, long[] vs) {
            this.timestamps = ts;
            this.values = vs;
            this.sampleCount = this.values.length;
        }

        private long dv(long v0, long v1) {
            return v0 > v1 ? v1 : v1 - v0;
        }

        void append(long t, long v) {
            assert (this.timestamps.length == 2) : "expected two timestamps; got " + this.timestamps.length;
            assert (t < this.timestamps[1]) : "@timestamp goes backward: " + t + " >= " + this.timestamps[1];
            this.reset += (double)(this.dv(v, this.values[1]) + this.dv(this.values[1], this.values[0]) - this.dv(v, this.values[0]));
            this.timestamps[1] = t;
            this.values[1] = v;
            ++this.sampleCount;
        }

        int entries() {
            return this.timestamps.length;
        }

        static long bytesUsed(int entries) {
            long ts = RamUsageEstimator.alignObjectSize((long)((long)RamUsageEstimator.NUM_BYTES_ARRAY_HEADER + 8L * (long)entries));
            long vs = RamUsageEstimator.alignObjectSize((long)((long)RamUsageEstimator.NUM_BYTES_ARRAY_HEADER + 8L * (long)entries));
            return BASE_RAM_USAGE + ts + vs;
        }
    }
}

