/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.blobstore.testkit.integrity;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.BooleanSupplier;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.RateLimiter;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.support.RefCountingListener;
import org.elasticsearch.action.support.SubscribableListener;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.common.blobstore.support.BlobMetadata;
import org.elasticsearch.common.collect.Iterators;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.CancellableThreads;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.ThrottledIterator;
import org.elasticsearch.common.util.concurrent.ThrottledTaskRunner;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.core.Strings;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshots;
import org.elasticsearch.index.snapshots.blobstore.RateLimitingInputStream;
import org.elasticsearch.index.snapshots.blobstore.SlicedInputStream;
import org.elasticsearch.index.snapshots.blobstore.SnapshotFiles;
import org.elasticsearch.repositories.IndexId;
import org.elasticsearch.repositories.RepositoryData;
import org.elasticsearch.repositories.RepositoryVerificationException;
import org.elasticsearch.repositories.ShardGeneration;
import org.elasticsearch.repositories.blobstore.BlobStoreRepository;
import org.elasticsearch.repositories.blobstore.testkit.integrity.IndexDescription;
import org.elasticsearch.repositories.blobstore.testkit.integrity.RepositoryVerifyIntegrityParams;
import org.elasticsearch.repositories.blobstore.testkit.integrity.RepositoryVerifyIntegrityResponse;
import org.elasticsearch.repositories.blobstore.testkit.integrity.RepositoryVerifyIntegrityResponseChunk;
import org.elasticsearch.repositories.blobstore.testkit.integrity.RepositoryVerifyIntegrityTask;
import org.elasticsearch.snapshots.SnapshotId;
import org.elasticsearch.snapshots.SnapshotInfo;
import org.elasticsearch.snapshots.SnapshotShardFailure;
import org.elasticsearch.tasks.TaskCancelledException;

class RepositoryIntegrityVerifier {
    private static final Logger logger = LogManager.getLogger(RepositoryIntegrityVerifier.class);
    private final LongSupplier currentTimeMillisSupplier;
    private final BlobStoreRepository blobStoreRepository;
    private final RepositoryVerifyIntegrityResponseChunk.Writer responseChunkWriter;
    private final String repositoryName;
    private final RepositoryVerifyIntegrityParams requestParams;
    private final RepositoryData repositoryData;
    private final BooleanSupplier isCancelledSupplier;
    private final CancellableRunner metadataTaskRunner;
    private final CancellableRunner snapshotTaskRunner;
    private final RateLimiter rateLimiter;
    private final Set<String> unreadableSnapshotInfoUuids = ConcurrentCollections.newConcurrentSet();
    private final long snapshotCount;
    private final AtomicLong snapshotProgress = new AtomicLong();
    private final long indexCount;
    private final AtomicLong indexProgress = new AtomicLong();
    private final long indexSnapshotCount;
    private final AtomicLong indexSnapshotProgress = new AtomicLong();
    private final AtomicLong blobsVerified = new AtomicLong();
    private final AtomicLong blobBytesVerified = new AtomicLong();
    private final AtomicLong throttledNanos;
    private final AtomicLong failedShardSnapshotsCount = new AtomicLong();
    private final Set<String> failedShardSnapshotDescriptions = ConcurrentCollections.newConcurrentSet();

    RepositoryIntegrityVerifier(LongSupplier currentTimeMillisSupplier, BlobStoreRepository blobStoreRepository, RepositoryVerifyIntegrityResponseChunk.Writer responseChunkWriter, RepositoryVerifyIntegrityParams requestParams, RepositoryData repositoryData, CancellableThreads cancellableThreads) {
        this.currentTimeMillisSupplier = currentTimeMillisSupplier;
        this.blobStoreRepository = blobStoreRepository;
        this.repositoryName = blobStoreRepository.getMetadata().name();
        this.responseChunkWriter = responseChunkWriter;
        this.requestParams = requestParams;
        this.repositoryData = repositoryData;
        this.isCancelledSupplier = () -> ((CancellableThreads)cancellableThreads).isCancelled();
        this.snapshotTaskRunner = new CancellableRunner(new ThrottledTaskRunner("verify-blob", requestParams.blobThreadPoolConcurrency(), (Executor)blobStoreRepository.threadPool().executor("snapshot")), cancellableThreads);
        this.metadataTaskRunner = new CancellableRunner(new ThrottledTaskRunner("verify-metadata", requestParams.metaThreadPoolConcurrency(), (Executor)blobStoreRepository.threadPool().executor("snapshot_meta")), cancellableThreads);
        this.snapshotCount = repositoryData.getSnapshotIds().size();
        this.indexCount = repositoryData.getIndices().size();
        this.indexSnapshotCount = repositoryData.getIndexSnapshotCount();
        this.rateLimiter = new RateLimiter.SimpleRateLimiter(requestParams.maxBytesPerSec().getMbFrac());
        this.throttledNanos = new AtomicLong(requestParams.verifyBlobContents() ? 1L : 0L);
    }

    RepositoryVerifyIntegrityTask.Status getStatus() {
        return new RepositoryVerifyIntegrityTask.Status(this.repositoryName, this.repositoryData.getGenId(), this.repositoryData.getUuid(), this.snapshotCount, this.snapshotProgress.get(), this.indexCount, this.indexProgress.get(), this.indexSnapshotCount, this.indexSnapshotProgress.get(), this.blobsVerified.get(), this.blobBytesVerified.get(), this.throttledNanos.get());
    }

    void start(final ActionListener<RepositoryVerifyIntegrityResponse> listener) {
        logger.info("[{}] verifying metadata integrity for index generation [{}]: repo UUID [{}], cluster UUID [{}], snapshots [{}], indices [{}], index snapshots [{}]", (Object)this.repositoryName, (Object)this.repositoryData.getGenId(), (Object)this.repositoryData.getUuid(), (Object)this.repositoryData.getClusterUUID(), (Object)this.getSnapshotCount(), (Object)this.getIndexCount(), (Object)this.getIndexSnapshotCount());
        SubscribableListener.newForked(this::verifySnapshots).andThen(this::checkFailedShardSnapshotCount).andThen(this::verifyIndices).andThenAccept(v -> this.ensureNotCancelled()).andThen(l -> this.blobStoreRepository.getRepositoryData((Executor)this.blobStoreRepository.threadPool().executor("management"), l)).addListener((ActionListener)new ActionListener<RepositoryData>(){

            public void onResponse(RepositoryData finalRepositoryData) {
                logger.info("[{}] completed verifying metadata integrity for index generation [{}]: repo UUID [{}], cluster UUID [{}]", (Object)RepositoryIntegrityVerifier.this.repositoryName, (Object)RepositoryIntegrityVerifier.this.repositoryData.getGenId(), (Object)RepositoryIntegrityVerifier.this.repositoryData.getUuid(), (Object)RepositoryIntegrityVerifier.this.repositoryData.getClusterUUID());
                listener.onResponse((Object)new RepositoryVerifyIntegrityResponse(RepositoryIntegrityVerifier.this.getStatus(), finalRepositoryData.getGenId()));
            }

            public void onFailure(Exception e) {
                logger.warn(() -> Strings.format((String)"[%s] failed verifying metadata integrity for index generation [%d]: repo UUID [%s], cluster UUID [%s]", (Object[])new Object[]{RepositoryIntegrityVerifier.this.repositoryName, RepositoryIntegrityVerifier.this.repositoryData.getGenId(), RepositoryIntegrityVerifier.this.repositoryData.getUuid(), RepositoryIntegrityVerifier.this.repositoryData.getClusterUUID()}), (Throwable)e);
                listener.onFailure(e);
            }
        });
    }

    private void ensureNotCancelled() {
        if (this.isCancelledSupplier.getAsBoolean()) {
            throw new TaskCancelledException("task cancelled");
        }
    }

    private void verifySnapshots(ActionListener<Void> listener) {
        new SnapshotsVerifier().run(listener);
    }

    private void checkFailedShardSnapshotCount(ActionListener<Void> listener) {
        if ((long)this.failedShardSnapshotDescriptions.size() < this.failedShardSnapshotsCount.get()) {
            listener.onFailure((Exception)new RepositoryVerificationException(this.repositoryName, Strings.format((String)"Cannot verify the integrity of all index snapshots because this repository contains too many shard snapshot failures: there are [%d] shard snapshot failures but [?%s] is set to [%d]. Please increase this limit if it is safe to do so.", (Object[])new Object[]{this.failedShardSnapshotsCount.get(), "max_failed_shard_snapshots", this.requestParams.maxFailedShardSnapshots()})));
        } else {
            listener.onResponse(null);
        }
    }

    private void verifyIndices(ActionListener<Void> listener) {
        RefCountingListener listeners = new RefCountingListener(listener);
        RepositoryIntegrityVerifier.runThrottled(Iterators.failFast(this.repositoryData.getIndices().values().iterator(), () -> this.isCancelledSupplier.getAsBoolean() || listeners.isFailing()), (releasable, indexId) -> new IndexVerifier((IndexId)indexId).run((ActionListener<Void>)ActionListener.releaseAfter((ActionListener)listeners.acquire(), (Releasable)releasable)), this.requestParams.indexVerificationConcurrency(), this.indexProgress, (Releasable)listeners);
    }

    private static String getShardSnapshotDescription(SnapshotId snapshotId, String index, int shardId) {
        return snapshotId.getUUID() + "/" + index + "/" + shardId;
    }

    private static Map<String, BlobStoreIndexShardSnapshot.FileInfo> getFilesByPhysicalName(List<BlobStoreIndexShardSnapshot.FileInfo> fileInfos) {
        Map filesByPhysicalName = Maps.newHashMapWithExpectedSize((int)fileInfos.size());
        for (BlobStoreIndexShardSnapshot.FileInfo fileInfo : fileInfos) {
            filesByPhysicalName.put(fileInfo.physicalName(), fileInfo);
        }
        return filesByPhysicalName;
    }

    private static <T> void runThrottled(Iterator<T> iterator, BiConsumer<Releasable, T> itemConsumer, int maxConcurrency, AtomicLong progressCounter, Releasable onCompletion) {
        ThrottledIterator.run(iterator, (ref, item) -> {
            Releasable[] releasableArray = new Releasable[2];
            releasableArray[0] = progressCounter::incrementAndGet;
            releasableArray[1] = ref;
            itemConsumer.accept(Releasables.wrap((Releasable[])releasableArray), item);
        }, (int)maxConcurrency, () -> ((Releasable)onCompletion).close());
    }

    private RepositoryVerifyIntegrityResponseChunk.Builder anomaly(String anomaly) {
        return new RepositoryVerifyIntegrityResponseChunk.Builder(this.responseChunkWriter, RepositoryVerifyIntegrityResponseChunk.Type.ANOMALY, this.currentTimeMillisSupplier.getAsLong()).anomaly(anomaly);
    }

    public long getSnapshotCount() {
        return this.snapshotCount;
    }

    public long getIndexCount() {
        return this.indexCount;
    }

    public long getIndexSnapshotCount() {
        return this.indexSnapshotCount;
    }

    private static class CancellableRunner {
        private final ThrottledTaskRunner delegate;
        private final CancellableThreads cancellableThreads;

        CancellableRunner(ThrottledTaskRunner delegate, CancellableThreads cancellableThreads) {
            this.delegate = delegate;
            this.cancellableThreads = cancellableThreads;
        }

        void run(final AbstractRunnable runnable) {
            this.delegate.enqueueTask((ActionListener)new ActionListener<Releasable>(){

                public void onResponse(Releasable releasable) {
                    try (Releasable releasable2 = releasable;){
                        if (cancellableThreads.isCancelled()) {
                            runnable.onFailure((Exception)new TaskCancelledException("task cancelled"));
                        } else {
                            try {
                                cancellableThreads.execute(() -> ((AbstractRunnable)runnable).run());
                            }
                            catch (RuntimeException e) {
                                runnable.onFailure((Exception)e);
                            }
                        }
                    }
                }

                public void onFailure(Exception e) {
                    runnable.onFailure(e);
                }
            });
        }
    }

    private class SnapshotsVerifier {
        final Map<String, Set<String>> indexNamesBySnapshotName;

        SnapshotsVerifier() {
            this.indexNamesBySnapshotName = Maps.newHashMapWithExpectedSize((int)RepositoryIntegrityVerifier.this.repositoryData.getIndices().size());
            for (IndexId indexId : RepositoryIntegrityVerifier.this.repositoryData.getIndices().values()) {
                for (SnapshotId snapshotId : RepositoryIntegrityVerifier.this.repositoryData.getSnapshots(indexId)) {
                    this.indexNamesBySnapshotName.computeIfAbsent(snapshotId.getName(), ignored -> new HashSet()).add(indexId.getName());
                }
            }
        }

        void run(ActionListener<Void> listener) {
            RefCountingListener listeners = new RefCountingListener(listener);
            RepositoryIntegrityVerifier.runThrottled(Iterators.failFast(RepositoryIntegrityVerifier.this.repositoryData.getSnapshotIds().iterator(), () -> RepositoryIntegrityVerifier.this.isCancelledSupplier.getAsBoolean() || listeners.isFailing()), (releasable, snapshotId) -> new SnapshotVerifier((SnapshotId)snapshotId).run((ActionListener<Void>)ActionListener.assertOnce((ActionListener)ActionListener.releaseAfter((ActionListener)listeners.acquire(), (Releasable)releasable))), RepositoryIntegrityVerifier.this.requestParams.snapshotVerificationConcurrency(), RepositoryIntegrityVerifier.this.snapshotProgress, (Releasable)listeners);
        }

        private class SnapshotVerifier {
            private final SnapshotId snapshotId;

            SnapshotVerifier(SnapshotId snapshotId) {
                this.snapshotId = snapshotId;
            }

            void run(final ActionListener<Void> listener) {
                if (RepositoryIntegrityVerifier.this.isCancelledSupplier.getAsBoolean()) {
                    listener.onResponse(null);
                    return;
                }
                RepositoryIntegrityVerifier.this.blobStoreRepository.getSnapshotInfo(this.snapshotId, (ActionListener)new ActionListener<SnapshotInfo>(){

                    public void onResponse(SnapshotInfo snapshotInfo) {
                        SnapshotVerifier.this.verifySnapshotInfo(snapshotInfo, (ActionListener<Void>)listener);
                    }

                    public void onFailure(Exception e) {
                        RepositoryIntegrityVerifier.this.unreadableSnapshotInfoUuids.add(SnapshotVerifier.this.snapshotId.getUUID());
                        RepositoryIntegrityVerifier.this.anomaly("failed to load snapshot info").snapshotId(SnapshotVerifier.this.snapshotId).exception(e).write((ActionListener<Void>)listener);
                    }
                });
            }

            void verifySnapshotInfo(SnapshotInfo snapshotInfo, ActionListener<Void> listener) {
                RepositoryVerifyIntegrityResponseChunk.Builder chunkBuilder = new RepositoryVerifyIntegrityResponseChunk.Builder(RepositoryIntegrityVerifier.this.responseChunkWriter, RepositoryVerifyIntegrityResponseChunk.Type.SNAPSHOT_INFO, RepositoryIntegrityVerifier.this.currentTimeMillisSupplier.getAsLong()).snapshotInfo(snapshotInfo);
                SubscribableListener chunkWrittenStep = SubscribableListener.newForked(chunkBuilder::write);
                if (RepositoryIntegrityVerifier.this.failedShardSnapshotsCount.get() < (long)RepositoryIntegrityVerifier.this.requestParams.maxFailedShardSnapshots()) {
                    for (SnapshotShardFailure shardFailure : snapshotInfo.shardFailures()) {
                        if (RepositoryIntegrityVerifier.this.failedShardSnapshotsCount.getAndIncrement() >= (long)RepositoryIntegrityVerifier.this.requestParams.maxFailedShardSnapshots()) continue;
                        RepositoryIntegrityVerifier.this.failedShardSnapshotDescriptions.add(RepositoryIntegrityVerifier.getShardSnapshotDescription(this.snapshotId, shardFailure.index(), shardFailure.shardId()));
                    }
                } else {
                    RepositoryIntegrityVerifier.this.failedShardSnapshotsCount.addAndGet(snapshotInfo.shardFailures().size());
                }
                SubscribableListener snapshotContentsOkStep = chunkWrittenStep.andThen(l -> {
                    if (Set.copyOf(snapshotInfo.indices()).equals(SnapshotsVerifier.this.indexNamesBySnapshotName.get(this.snapshotId.getName()))) {
                        l.onResponse(null);
                    } else {
                        RepositoryIntegrityVerifier.this.anomaly("snapshot contents mismatch").snapshotId(this.snapshotId).write((ActionListener<Void>)l);
                    }
                });
                SubscribableListener globalMetadataOkStep = Boolean.TRUE.equals(snapshotInfo.includeGlobalState()) ? snapshotContentsOkStep.andThen(this::verifySnapshotGlobalMetadata) : snapshotContentsOkStep;
                globalMetadataOkStep.addListener(listener);
            }

            private void verifySnapshotGlobalMetadata(ActionListener<Void> listener) {
                RepositoryIntegrityVerifier.this.metadataTaskRunner.run((AbstractRunnable)ActionRunnable.wrap(listener, l -> {
                    try {
                        RepositoryIntegrityVerifier.this.blobStoreRepository.getSnapshotGlobalMetadata(this.snapshotId);
                        l.onResponse(null);
                    }
                    catch (Exception e) {
                        RepositoryIntegrityVerifier.this.anomaly("failed to load global metadata").snapshotId(this.snapshotId).exception(e).write((ActionListener<Void>)l);
                    }
                }));
            }
        }
    }

    private class IndexVerifier {
        private final IndexId indexId;
        private final ShardContainerContentsDeduplicator shardContainerContentsDeduplicator = new ShardContainerContentsDeduplicator();
        private final IndexDescriptionsDeduplicator indexDescriptionsDeduplicator = new IndexDescriptionsDeduplicator();
        private final AtomicInteger totalSnapshotCounter = new AtomicInteger();
        private final AtomicInteger restorableSnapshotCounter = new AtomicInteger();

        IndexVerifier(IndexId indexId) {
            this.indexId = indexId;
        }

        void run(ActionListener<Void> listener) {
            SubscribableListener.newForked(l -> {
                RefCountingListener listeners = new RefCountingListener(1, l);
                RepositoryIntegrityVerifier.runThrottled(Iterators.failFast(RepositoryIntegrityVerifier.this.repositoryData.getSnapshots(this.indexId).iterator(), () -> RepositoryIntegrityVerifier.this.isCancelledSupplier.getAsBoolean() || listeners.isFailing()), (releasable, snapshotId) -> this.verifyIndexSnapshot((SnapshotId)snapshotId, (ActionListener<Void>)ActionListener.releaseAfter((ActionListener)listeners.acquire(), (Releasable)releasable)), RepositoryIntegrityVerifier.this.requestParams.indexSnapshotVerificationConcurrency(), RepositoryIntegrityVerifier.this.indexSnapshotProgress, (Releasable)listeners);
            }).andThen(l -> {
                RepositoryIntegrityVerifier.this.ensureNotCancelled();
                new RepositoryVerifyIntegrityResponseChunk.Builder(RepositoryIntegrityVerifier.this.responseChunkWriter, RepositoryVerifyIntegrityResponseChunk.Type.INDEX_RESTORABILITY, RepositoryIntegrityVerifier.this.currentTimeMillisSupplier.getAsLong()).indexRestorability(this.indexId, this.totalSnapshotCounter.get(), this.restorableSnapshotCounter.get()).write((ActionListener<Void>)l);
            }).addListener(listener);
        }

        private void verifyIndexSnapshot(SnapshotId snapshotId, ActionListener<Void> listener) {
            this.totalSnapshotCounter.incrementAndGet();
            this.indexDescriptionsDeduplicator.get(snapshotId).andThen((l, indexDescription) -> {
                if (indexDescription == null) {
                    l.onResponse(null);
                } else {
                    new ShardSnapshotsVerifier(snapshotId, (IndexDescription)indexDescription).run((ActionListener<Void>)l);
                }
            }).addListener(listener);
        }

        private SubscribableListener<Void> blobContentsListeners(final IndexDescription indexDescription, final ShardContainerContents shardContainerContents, final BlobStoreIndexShardSnapshot.FileInfo fileInfo) {
            return shardContainerContents.blobContentsListeners().computeIfAbsent(fileInfo.name(), ignored -> {
                if (RepositoryIntegrityVerifier.this.requestParams.verifyBlobContents()) {
                    return SubscribableListener.newForked(listener -> RepositoryIntegrityVerifier.this.snapshotTaskRunner.run((AbstractRunnable)ActionRunnable.run((ActionListener)listener, () -> {
                        try (SlicedInputStream slicedStream = new SlicedInputStream(fileInfo.numberOfParts()){

                            protected InputStream openSlice(int slice) throws IOException {
                                return RepositoryIntegrityVerifier.this.blobStoreRepository.shardContainer(indexDescription.indexId(), shardContainerContents.shardId()).readBlob(OperationPurpose.REPOSITORY_ANALYSIS, fileInfo.partName(slice));
                            }
                        };
                             RateLimitingInputStream rateLimitedStream = new RateLimitingInputStream((InputStream)slicedStream, () -> RepositoryIntegrityVerifier.this.rateLimiter, RepositoryIntegrityVerifier.this.throttledNanos::addAndGet);
                             IndexInputWrapper indexInput = new IndexInputWrapper((InputStream)rateLimitedStream, fileInfo.length());){
                            CodecUtil.checksumEntireFile((IndexInput)indexInput);
                        }
                    })));
                }
                RepositoryIntegrityVerifier.this.blobBytesVerified.addAndGet(fileInfo.length());
                return SubscribableListener.nullSuccess();
            });
        }

        private class ShardContainerContentsDeduplicator {
            private final Map<Integer, SubscribableListener<ShardContainerContents>> listenersByShardId = ConcurrentCollections.newConcurrentMap();

            private ShardContainerContentsDeduplicator() {
            }

            SubscribableListener<ShardContainerContents> get(int shardId) {
                return this.listenersByShardId.computeIfAbsent(shardId, ignored -> SubscribableListener.newForked(shardContainerContentsListener -> RepositoryIntegrityVerifier.this.metadataTaskRunner.run((AbstractRunnable)ActionRunnable.wrap((ActionListener)shardContainerContentsListener, l -> this.load(shardId, (ActionListener<ShardContainerContents>)l)))));
            }

            private void load(int shardId, ActionListener<ShardContainerContents> listener) {
                Map blobsByName;
                IndexDescription indexDescription = new IndexDescription(IndexVerifier.this.indexId, null, 0);
                try {
                    blobsByName = RepositoryIntegrityVerifier.this.blobStoreRepository.shardContainer(IndexVerifier.this.indexId, shardId).listBlobs(OperationPurpose.REPOSITORY_ANALYSIS);
                }
                catch (Exception e) {
                    RepositoryIntegrityVerifier.this.anomaly("failed to list shard container contents").shardDescription(new IndexDescription(IndexVerifier.this.indexId, null, 0), shardId).exception(e).write((ActionListener<Void>)listener.map(v -> null));
                    return;
                }
                ShardGeneration shardGen = RepositoryIntegrityVerifier.this.repositoryData.shardGenerations().getShardGen(IndexVerifier.this.indexId, shardId);
                if (shardGen == null) {
                    RepositoryIntegrityVerifier.this.anomaly("shard generation not defined").shardDescription(indexDescription, shardId).write((ActionListener<Void>)listener.map(v -> new ShardContainerContents(shardId, blobsByName, null, null, ConcurrentCollections.newConcurrentMap())));
                    return;
                }
                SubscribableListener.newForked(l -> {
                    try {
                        l.onResponse((Object)RepositoryIntegrityVerifier.this.blobStoreRepository.getBlobStoreIndexShardSnapshots(IndexVerifier.this.indexId, shardId, shardGen));
                    }
                    catch (Exception e) {
                        RepositoryIntegrityVerifier.this.anomaly("failed to load shard generation").shardDescription(indexDescription, shardId).shardGeneration(shardGen).exception(e).write((ActionListener<Void>)l.map(v -> null));
                    }
                }).andThenApply(blobStoreIndexShardSnapshots -> new ShardContainerContents(shardId, blobsByName, shardGen, ShardContainerContentsDeduplicator.getFilesByPhysicalNameBySnapshotName(blobStoreIndexShardSnapshots), ConcurrentCollections.newConcurrentMap())).addListener(listener);
            }

            private static Map<String, Map<String, BlobStoreIndexShardSnapshot.FileInfo>> getFilesByPhysicalNameBySnapshotName(BlobStoreIndexShardSnapshots blobStoreIndexShardSnapshots) {
                if (blobStoreIndexShardSnapshots == null) {
                    return null;
                }
                Map filesByPhysicalNameBySnapshotName = Maps.newHashMapWithExpectedSize((int)blobStoreIndexShardSnapshots.snapshots().size());
                for (SnapshotFiles snapshotFiles : blobStoreIndexShardSnapshots.snapshots()) {
                    filesByPhysicalNameBySnapshotName.put(snapshotFiles.snapshot(), RepositoryIntegrityVerifier.getFilesByPhysicalName(snapshotFiles.indexFiles()));
                }
                return filesByPhysicalNameBySnapshotName;
            }
        }

        private class IndexDescriptionsDeduplicator {
            private final Map<String, SubscribableListener<IndexDescription>> listenersByBlobId = ConcurrentCollections.newConcurrentMap();

            private IndexDescriptionsDeduplicator() {
            }

            SubscribableListener<IndexDescription> get(SnapshotId snapshotId) {
                String indexMetaBlobId = RepositoryIntegrityVerifier.this.repositoryData.indexMetaDataGenerations().indexMetaBlobId(snapshotId, IndexVerifier.this.indexId);
                return this.listenersByBlobId.computeIfAbsent(indexMetaBlobId, ignored -> SubscribableListener.newForked(indexDescriptionListener -> RepositoryIntegrityVerifier.this.metadataTaskRunner.run((AbstractRunnable)ActionRunnable.wrap((ActionListener)indexDescriptionListener, l -> this.load(snapshotId, indexMetaBlobId, (ActionListener<IndexDescription>)l)))));
            }

            private void load(SnapshotId snapshotId, String indexMetaBlobId, ActionListener<IndexDescription> listener) {
                try {
                    listener.onResponse((Object)new IndexDescription(IndexVerifier.this.indexId, indexMetaBlobId, RepositoryIntegrityVerifier.this.blobStoreRepository.getSnapshotIndexMetaData(RepositoryIntegrityVerifier.this.repositoryData, snapshotId, IndexVerifier.this.indexId).getNumberOfShards()));
                }
                catch (Exception e) {
                    RepositoryIntegrityVerifier.this.anomaly("failed to load index metadata").indexDescription(new IndexDescription(IndexVerifier.this.indexId, indexMetaBlobId, 0)).exception(e).write((ActionListener<Void>)listener.map(v -> null));
                }
            }
        }

        private record ShardContainerContents(int shardId, Map<String, BlobMetadata> blobsByName, @Nullable ShardGeneration shardGeneration, @Nullable Map<String, Map<String, BlobStoreIndexShardSnapshot.FileInfo>> filesByPhysicalNameBySnapshotName, Map<String, SubscribableListener<Void>> blobContentsListeners) {
        }

        private class ShardSnapshotsVerifier {
            private final SnapshotId snapshotId;
            private final IndexDescription indexDescription;
            private final AtomicInteger restorableShardCount = new AtomicInteger();

            ShardSnapshotsVerifier(SnapshotId snapshotId, IndexDescription indexDescription) {
                this.snapshotId = snapshotId;
                this.indexDescription = indexDescription;
            }

            void run(ActionListener<Void> listener) {
                try (RefCountingListener listeners = new RefCountingListener(1, listener.map(v -> {
                    if (!RepositoryIntegrityVerifier.this.unreadableSnapshotInfoUuids.contains(this.snapshotId.getUUID()) && this.indexDescription.shardCount() == this.restorableShardCount.get()) {
                        IndexVerifier.this.restorableSnapshotCounter.incrementAndGet();
                    }
                    return v;
                }));){
                    for (int shardId = 0; shardId < this.indexDescription.shardCount(); ++shardId) {
                        if (RepositoryIntegrityVerifier.this.failedShardSnapshotDescriptions.contains(RepositoryIntegrityVerifier.getShardSnapshotDescription(this.snapshotId, IndexVerifier.this.indexId.getName(), shardId))) continue;
                        IndexVerifier.this.shardContainerContentsDeduplicator.get(shardId).andThen((l, shardContainerContents) -> {
                            if (shardContainerContents == null) {
                                l.onResponse(null);
                            } else {
                                new ShardSnapshotVerifier((ShardContainerContents)shardContainerContents).run((ActionListener<Void>)l);
                            }
                        }).addListener(listeners.acquire());
                    }
                }
            }

            private class ShardSnapshotVerifier {
                private final ShardContainerContents shardContainerContents;
                private volatile boolean isRestorable = true;

                ShardSnapshotVerifier(ShardContainerContents shardContainerContents) {
                    this.shardContainerContents = shardContainerContents;
                }

                void run(ActionListener<Void> listener) {
                    RepositoryIntegrityVerifier.this.metadataTaskRunner.run((AbstractRunnable)ActionRunnable.wrap(listener, this::verifyShardSnapshot));
                }

                private void verifyShardSnapshot(ActionListener<Void> listener) {
                    BlobStoreIndexShardSnapshot blobStoreIndexShardSnapshot;
                    int shardId = this.shardContainerContents.shardId();
                    try {
                        blobStoreIndexShardSnapshot = RepositoryIntegrityVerifier.this.blobStoreRepository.loadShardSnapshot(RepositoryIntegrityVerifier.this.blobStoreRepository.shardContainer(IndexVerifier.this.indexId, shardId), ShardSnapshotsVerifier.this.snapshotId);
                    }
                    catch (Exception e) {
                        RepositoryIntegrityVerifier.this.anomaly("failed to load shard snapshot").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, shardId).exception(e).write(listener);
                        return;
                    }
                    RefCountingListener listeners = new RefCountingListener(1, listener.map(v -> {
                        if (this.isRestorable) {
                            ShardSnapshotsVerifier.this.restorableShardCount.incrementAndGet();
                        }
                        return v;
                    }));
                    ActionListener shardGenerationConsistencyListener = listeners.acquire();
                    RepositoryIntegrityVerifier.runThrottled(Iterators.failFast(blobStoreIndexShardSnapshot.indexFiles().iterator(), () -> RepositoryIntegrityVerifier.this.isCancelledSupplier.getAsBoolean() || listeners.isFailing()), (releasable, fileInfo) -> this.verifyFileInfo((BlobStoreIndexShardSnapshot.FileInfo)fileInfo, (ActionListener<Void>)ActionListener.releaseAfter((ActionListener)listeners.acquire(), (Releasable)releasable)), 1, RepositoryIntegrityVerifier.this.blobsVerified, (Releasable)listeners);
                    this.verifyShardGenerationConsistency(blobStoreIndexShardSnapshot, (ActionListener<Void>)shardGenerationConsistencyListener);
                }

                private void verifyFileInfo(BlobStoreIndexShardSnapshot.FileInfo fileInfo, ActionListener<Void> listener) {
                    if (fileInfo.metadata().hashEqualsContents()) {
                        listener.onResponse(null);
                        return;
                    }
                    for (int partIndex = 0; partIndex < fileInfo.numberOfParts(); ++partIndex) {
                        String blobName = fileInfo.partName(partIndex);
                        BlobMetadata blobInfo = this.shardContainerContents.blobsByName().get(blobName);
                        if (blobInfo == null) {
                            this.isRestorable = false;
                            String physicalFileName = fileInfo.physicalName();
                            RepositoryIntegrityVerifier.this.anomaly("missing blob").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, this.shardContainerContents.shardId()).blobName(blobName, physicalFileName).part(partIndex, fileInfo.numberOfParts()).fileLength(ByteSizeValue.ofBytes((long)fileInfo.length())).partLength(ByteSizeValue.ofBytes((long)fileInfo.partBytes(partIndex))).write(listener);
                            return;
                        }
                        if (blobInfo.length() == fileInfo.partBytes(partIndex)) continue;
                        this.isRestorable = false;
                        String physicalFileName = fileInfo.physicalName();
                        ByteSizeValue blobLength = ByteSizeValue.ofBytes((long)blobInfo.length());
                        RepositoryIntegrityVerifier.this.anomaly("mismatched blob length").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, this.shardContainerContents.shardId()).blobName(blobName, physicalFileName).part(partIndex, fileInfo.numberOfParts()).fileLength(ByteSizeValue.ofBytes((long)fileInfo.length())).partLength(ByteSizeValue.ofBytes((long)fileInfo.partBytes(partIndex))).blobLength(blobLength).write(listener);
                        return;
                    }
                    IndexVerifier.this.blobContentsListeners(ShardSnapshotsVerifier.this.indexDescription, this.shardContainerContents, fileInfo).addListener(listener.delegateResponse((l, e) -> {
                        this.isRestorable = false;
                        String physicalFileName = fileInfo.physicalName();
                        RepositoryIntegrityVerifier.this.anomaly("corrupt data blob").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, this.shardContainerContents.shardId()).blobName(fileInfo.name(), physicalFileName).part(-1, fileInfo.numberOfParts()).fileLength(ByteSizeValue.ofBytes((long)fileInfo.length())).exception((Exception)e).write((ActionListener<Void>)l);
                    }));
                }

                private void verifyShardGenerationConsistency(BlobStoreIndexShardSnapshot blobStoreIndexShardSnapshot, ActionListener<Void> listener) {
                    Map<String, Map<String, BlobStoreIndexShardSnapshot.FileInfo>> summaryFilesByPhysicalNameBySnapshotName = this.shardContainerContents.filesByPhysicalNameBySnapshotName();
                    if (summaryFilesByPhysicalNameBySnapshotName == null) {
                        listener.onResponse(null);
                        return;
                    }
                    int shardId = this.shardContainerContents.shardId();
                    Map<String, BlobStoreIndexShardSnapshot.FileInfo> summaryFilesByPhysicalName = summaryFilesByPhysicalNameBySnapshotName.get(ShardSnapshotsVerifier.this.snapshotId.getName());
                    if (summaryFilesByPhysicalName == null) {
                        RepositoryIntegrityVerifier.this.anomaly("snapshot not in shard generation").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, shardId).shardGeneration(this.shardContainerContents.shardGeneration()).write(listener);
                        return;
                    }
                    Map<String, BlobStoreIndexShardSnapshot.FileInfo> snapshotFiles = RepositoryIntegrityVerifier.getFilesByPhysicalName(blobStoreIndexShardSnapshot.indexFiles());
                    for (BlobStoreIndexShardSnapshot.FileInfo summaryFile : summaryFilesByPhysicalName.values()) {
                        BlobStoreIndexShardSnapshot.FileInfo snapshotFile = snapshotFiles.get(summaryFile.physicalName());
                        if (snapshotFile == null) {
                            RepositoryIntegrityVerifier.this.anomaly("blob in shard generation but not snapshot").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, shardId).shardGeneration(this.shardContainerContents.shardGeneration()).physicalFileName(summaryFile.physicalName()).write(listener);
                            return;
                        }
                        if (summaryFile.isSame(snapshotFile)) continue;
                        RepositoryIntegrityVerifier.this.anomaly("snapshot shard generation mismatch").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, shardId).shardGeneration(this.shardContainerContents.shardGeneration()).physicalFileName(summaryFile.physicalName()).write(listener);
                        return;
                    }
                    for (BlobStoreIndexShardSnapshot.FileInfo snapshotFile : blobStoreIndexShardSnapshot.indexFiles()) {
                        if (summaryFilesByPhysicalName.get(snapshotFile.physicalName()) != null) continue;
                        RepositoryIntegrityVerifier.this.anomaly("blob in snapshot but not shard generation").snapshotId(ShardSnapshotsVerifier.this.snapshotId).shardDescription(ShardSnapshotsVerifier.this.indexDescription, shardId).shardGeneration(this.shardContainerContents.shardGeneration()).physicalFileName(snapshotFile.physicalName()).write(listener);
                        return;
                    }
                    listener.onResponse(null);
                }
            }
        }
    }

    private class IndexInputWrapper
    extends IndexInput {
        private final InputStream inputStream;
        private final long length;
        long filePointer;

        IndexInputWrapper(InputStream inputStream, long length) {
            super("");
            this.filePointer = 0L;
            this.inputStream = inputStream;
            this.length = length;
        }

        public byte readByte() throws IOException {
            if (RepositoryIntegrityVerifier.this.isCancelledSupplier.getAsBoolean()) {
                throw new TaskCancelledException("task cancelled");
            }
            int read = this.inputStream.read();
            if (read == -1) {
                throw new EOFException();
            }
            ++this.filePointer;
            RepositoryIntegrityVerifier.this.blobBytesVerified.incrementAndGet();
            return (byte)read;
        }

        public void readBytes(byte[] b, int offset, int len) throws IOException {
            while (len > 0) {
                if (RepositoryIntegrityVerifier.this.isCancelledSupplier.getAsBoolean()) {
                    throw new TaskCancelledException("task cancelled");
                }
                int read = this.inputStream.read(b, offset, len);
                if (read == -1) {
                    throw new EOFException();
                }
                this.filePointer += (long)read;
                RepositoryIntegrityVerifier.this.blobBytesVerified.addAndGet(read);
                len -= read;
                offset += read;
            }
        }

        public void close() {
        }

        public long getFilePointer() {
            return this.filePointer;
        }

        public void seek(long pos) {
            if (this.filePointer != pos) {
                assert (false) : "cannot seek";
                throw new UnsupportedOperationException("seek");
            }
        }

        public long length() {
            return this.length;
        }

        public IndexInput slice(String sliceDescription, long offset, long length) {
            assert (false);
            throw new UnsupportedOperationException("slice");
        }
    }
}

