/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.inference.registry;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.action.support.SubscribableListener;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.support.master.AcknowledgedRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.ElasticsearchClient;
import org.elasticsearch.client.internal.OriginSettingClient;
import org.elasticsearch.cluster.AckedBatchedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateAckListener;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateTaskExecutor;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.SimpleBatchedAckListenerTaskExecutor;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.cluster.service.MasterServiceTaskQueue;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.index.query.ConstantScoreQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.reindex.BulkByScrollResponse;
import org.elasticsearch.index.reindex.DeleteByQueryAction;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.inference.InferenceService;
import org.elasticsearch.inference.MinimalServiceSettings;
import org.elasticsearch.inference.Model;
import org.elasticsearch.inference.ModelConfigurations;
import org.elasticsearch.inference.TaskType;
import org.elasticsearch.inference.UnparsedModel;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.sort.SortOrder;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentObject;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentFactory;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xpack.core.inference.action.GetInferenceModelAction;
import org.elasticsearch.xpack.inference.registry.ModelRegistryMetadata;
import org.elasticsearch.xpack.inference.services.ServiceUtils;

public class ModelRegistry
implements ClusterStateListener {
    private static final String TASK_TYPE_FIELD = "task_type";
    private static final String MODEL_ID_FIELD = "model_id";
    private static final Logger logger = LogManager.getLogger(ModelRegistry.class);
    private final OriginSettingClient client;
    private final Map<String, InferenceService.DefaultConfigId> defaultConfigIds;
    private final MasterServiceTaskQueue<MetadataTask> metadataTaskQueue;
    private final AtomicBoolean upgradeMetadataInProgress = new AtomicBoolean(false);
    private final Set<String> preventDeletionLock = Collections.newSetFromMap(new ConcurrentHashMap());
    private volatile Metadata lastMetadata;

    public static UnparsedModel unparsedModelFromMap(ModelConfigMap modelConfigMap) {
        if (modelConfigMap.config() == null) {
            throw new ElasticsearchStatusException("Missing config map", RestStatus.BAD_REQUEST, new Object[0]);
        }
        String inferenceEntityId = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), MODEL_ID_FIELD);
        String service = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), "service");
        String taskTypeStr = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), TaskType.NAME);
        TaskType taskType = TaskType.fromString((String)taskTypeStr);
        return new UnparsedModel(inferenceEntityId, taskType, service, modelConfigMap.config(), modelConfigMap.secrets());
    }

    public ModelRegistry(ClusterService clusterService, Client client) {
        this.client = new OriginSettingClient(client, "inference");
        this.defaultConfigIds = new ConcurrentHashMap<String, InferenceService.DefaultConfigId>();
        SimpleBatchedAckListenerTaskExecutor<MetadataTask> executor = new SimpleBatchedAckListenerTaskExecutor<MetadataTask>(){

            public Tuple<ClusterState, ClusterStateAckListener> executeTask(MetadataTask task, ClusterState clusterState) throws Exception {
                ModelRegistryMetadata updated = task.executeTask(ModelRegistryMetadata.fromState(clusterState.metadata()));
                Metadata.Builder newMetadata = Metadata.builder((Metadata)clusterState.metadata()).putCustom("model_registry", (Metadata.Custom)updated);
                return new Tuple((Object)ClusterState.builder((ClusterState)clusterState).metadata(newMetadata).build(), (Object)task);
            }
        };
        this.metadataTaskQueue = clusterService.createTaskQueue("model_registry", Priority.NORMAL, (ClusterStateTaskExecutor)executor);
    }

    public boolean containsDefaultConfigId(String inferenceEntityId) {
        return this.defaultConfigIds.containsKey(inferenceEntityId);
    }

    public synchronized void putDefaultIdIfAbsent(InferenceService.DefaultConfigId defaultConfigId) {
        this.defaultConfigIds.putIfAbsent(defaultConfigId.inferenceId(), defaultConfigId);
    }

    public synchronized void addDefaultIds(InferenceService.DefaultConfigId defaultConfigId) throws IllegalStateException {
        InferenceService.DefaultConfigId config = this.defaultConfigIds.get(defaultConfigId.inferenceId());
        if (config != null) {
            throw new IllegalStateException("Cannot add default endpoint to the inference endpoint registry with duplicate inference id [" + defaultConfigId.inferenceId() + "] declared by service [" + defaultConfigId.service().name() + "]. The inference Id is already use by [" + config.service().name() + "] service.");
        }
        this.defaultConfigIds.put(defaultConfigId.inferenceId(), defaultConfigId);
    }

    public void clearDefaultIds() {
        this.defaultConfigIds.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MinimalServiceSettings getMinimalServiceSettings(String inferenceEntityId) throws ResourceNotFoundException {
        ModelRegistry modelRegistry = this;
        synchronized (modelRegistry) {
            if (this.lastMetadata == null) {
                throw new IllegalStateException("initial cluster state not set yet");
            }
        }
        InferenceService.DefaultConfigId config = this.defaultConfigIds.get(inferenceEntityId);
        if (config != null) {
            return config.settings();
        }
        ModelRegistryMetadata state = ModelRegistryMetadata.fromState(this.lastMetadata);
        MinimalServiceSettings existing = state.getMinimalServiceSettings(inferenceEntityId);
        if (state.isUpgraded() && existing == null) {
            throw new ResourceNotFoundException(inferenceEntityId + " does not exist in this cluster.", new Object[0]);
        }
        return existing;
    }

    public void getModelWithSecrets(String inferenceEntityId, ActionListener<UnparsedModel> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            if (searchResponse.getHits().getHits().length == 0) {
                InferenceService.DefaultConfigId maybeDefault = this.defaultConfigIds.get(inferenceEntityId);
                if (maybeDefault != null) {
                    this.getDefaultConfig(true, maybeDefault, listener);
                } else {
                    delegate.onFailure((Exception)((Object)this.inferenceNotFoundException(inferenceEntityId)));
                }
                return;
            }
            delegate.onResponse((Object)ModelRegistry.unparsedModelFromMap(this.createModelConfigMap(searchResponse.getHits(), inferenceEntityId)));
        });
        QueryBuilder queryBuilder = ModelRegistry.documentIdQuery(inferenceEntityId);
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*", ".secrets-inference*"}).setQuery(queryBuilder).setSize(2).request();
        this.client.search(modelSearch, searchListener);
    }

    public void getModel(String inferenceEntityId, ActionListener<UnparsedModel> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            if (searchResponse.getHits().getHits().length == 0) {
                InferenceService.DefaultConfigId maybeDefault = this.defaultConfigIds.get(inferenceEntityId);
                if (maybeDefault != null) {
                    this.getDefaultConfig(true, maybeDefault, listener);
                } else {
                    delegate.onFailure((Exception)((Object)this.inferenceNotFoundException(inferenceEntityId)));
                }
                return;
            }
            List<UnparsedModel> modelConfigs = this.parseHitsAsModels(searchResponse.getHits()).stream().map(ModelRegistry::unparsedModelFromMap).toList();
            assert (modelConfigs.size() == 1);
            delegate.onResponse((Object)modelConfigs.get(0));
        });
        QueryBuilder queryBuilder = ModelRegistry.documentIdQuery(inferenceEntityId);
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery(queryBuilder).setSize(1).setTrackTotalHits(false).request();
        this.client.search(modelSearch, searchListener);
    }

    private ResourceNotFoundException inferenceNotFoundException(String inferenceEntityId) {
        return new ResourceNotFoundException("Inference endpoint not found [{}]", new Object[]{inferenceEntityId});
    }

    public void getModelsByTaskType(TaskType taskType, ActionListener<List<UnparsedModel>> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            List<UnparsedModel> modelConfigs = this.parseHitsAsModels(searchResponse.getHits()).stream().map(ModelRegistry::unparsedModelFromMap).toList();
            List<InferenceService.DefaultConfigId> defaultConfigsForTaskType = ModelRegistry.taskTypeMatchedDefaults(taskType, this.defaultConfigIds.values());
            this.addAllDefaultConfigsIfMissing(true, modelConfigs, defaultConfigsForTaskType, (ActionListener<List<UnparsedModel>>)delegate);
        });
        ConstantScoreQueryBuilder queryBuilder = QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.termsQuery((String)TASK_TYPE_FIELD, (String[])new String[]{taskType.toString()}));
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery((QueryBuilder)queryBuilder).setSize(10000).setTrackTotalHits(false).addSort(MODEL_ID_FIELD, SortOrder.ASC).request();
        this.client.search(modelSearch, searchListener);
    }

    public void getAllModels(boolean persistDefaultEndpoints, ActionListener<List<UnparsedModel>> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            List<UnparsedModel> foundConfigs = this.parseHitsAsModels(searchResponse.getHits()).stream().map(ModelRegistry::unparsedModelFromMap).toList();
            this.addAllDefaultConfigsIfMissing(persistDefaultEndpoints, foundConfigs, this.defaultConfigIds.values(), (ActionListener<List<UnparsedModel>>)delegate);
        });
        ConstantScoreQueryBuilder queryBuilder = QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.existsQuery((String)TASK_TYPE_FIELD));
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery((QueryBuilder)queryBuilder).setSize(10000).setTrackTotalHits(false).addSort(MODEL_ID_FIELD, SortOrder.ASC).request();
        this.client.search(modelSearch, searchListener);
    }

    private void addAllDefaultConfigsIfMissing(boolean persistDefaultEndpoints, List<UnparsedModel> foundConfigs, Collection<InferenceService.DefaultConfigId> matchedDefaults, ActionListener<List<UnparsedModel>> listener) {
        Set foundIds = foundConfigs.stream().map(UnparsedModel::inferenceEntityId).collect(Collectors.toSet());
        List<InferenceService.DefaultConfigId> missing = matchedDefaults.stream().filter(d -> !foundIds.contains(d.inferenceId())).toList();
        if (missing.isEmpty()) {
            listener.onResponse(foundConfigs);
        } else {
            GroupedActionListener groupedListener = new GroupedActionListener(missing.size(), listener.delegateFailure((delegate, listOfModels) -> {
                ArrayList<UnparsedModel> allConfigs = new ArrayList<UnparsedModel>();
                allConfigs.addAll(foundConfigs);
                allConfigs.addAll((Collection<UnparsedModel>)listOfModels);
                allConfigs.sort(Comparator.comparing(UnparsedModel::inferenceEntityId));
                delegate.onResponse(allConfigs);
            }));
            for (InferenceService.DefaultConfigId required : missing) {
                this.getDefaultConfig(persistDefaultEndpoints, required, (ActionListener<UnparsedModel>)groupedListener);
            }
        }
    }

    private void getDefaultConfig(boolean persistDefaultEndpoints, InferenceService.DefaultConfigId defaultConfig, ActionListener<UnparsedModel> listener) {
        defaultConfig.service().defaultConfigs(listener.delegateFailureAndWrap((delegate, models) -> {
            boolean foundModel = false;
            for (Model m : models) {
                if (!m.getInferenceEntityId().equals(defaultConfig.inferenceId())) continue;
                foundModel = true;
                if (persistDefaultEndpoints) {
                    this.storeDefaultEndpoint(m, () -> listener.onResponse((Object)ModelRegistry.modelToUnparsedModel(m)));
                    break;
                }
                listener.onResponse((Object)ModelRegistry.modelToUnparsedModel(m));
                break;
            }
            if (!foundModel) {
                listener.onFailure((Exception)new IllegalStateException("Configuration not found for default inference id [" + defaultConfig.inferenceId() + "]"));
            }
        }));
    }

    private void storeDefaultEndpoint(Model preconfigured, Runnable runAfter) {
        ActionListener responseListener = ActionListener.wrap(success -> logger.debug("Added default inference endpoint [{}]", (Object)preconfigured.getInferenceEntityId()), exception -> {
            if (exception instanceof ResourceAlreadyExistsException) {
                logger.debug("Default inference id [{}] already exists", (Object)preconfigured.getInferenceEntityId());
            } else {
                logger.error("Failed to store default inference id [" + preconfigured.getInferenceEntityId() + "]", (Throwable)exception);
            }
        });
        this.storeModel(preconfigured, false, (ActionListener<Boolean>)ActionListener.runAfter((ActionListener)responseListener, (Runnable)runAfter), AcknowledgedRequest.DEFAULT_ACK_TIMEOUT);
    }

    private ArrayList<ModelConfigMap> parseHitsAsModels(SearchHits hits) {
        ArrayList<ModelConfigMap> modelConfigs = new ArrayList<ModelConfigMap>();
        for (SearchHit hit : hits) {
            modelConfigs.add(new ModelConfigMap(hit.getSourceAsMap(), Map.of()));
        }
        return modelConfigs;
    }

    private ModelConfigMap createModelConfigMap(SearchHits hits, String inferenceEntityId) {
        Map mappedHits = Arrays.stream(hits.getHits()).collect(Collectors.toMap(hit -> {
            if (hit.getIndex().startsWith(".inference")) {
                return ".inference";
            }
            if (hit.getIndex().startsWith(".secrets-inference")) {
                return ".secrets-inference";
            }
            logger.warn(Strings.format((String)"Found invalid index for inference endpoint [%s] at index [%s]", (Object[])new Object[]{inferenceEntityId, hit.getIndex()}));
            throw new IllegalArgumentException(Strings.format((String)"Invalid result while loading inference endpoint [%s] index: [%s]. Try deleting and reinitializing the service", (Object[])new Object[]{inferenceEntityId, hit.getIndex()}));
        }, Function.identity()));
        if (!mappedHits.containsKey(".inference") || !mappedHits.containsKey(".secrets-inference") || mappedHits.size() > 2) {
            logger.warn(Strings.format((String)"Failed to load inference endpoint [%s], found endpoint parts from index prefixes: [%s]", (Object[])new Object[]{inferenceEntityId, mappedHits.keySet()}));
            throw new IllegalStateException(Strings.format((String)"Failed to load inference endpoint [%s]. Endpoint is in an invalid state, try deleting and reinitializing the service", (Object[])new Object[]{inferenceEntityId}));
        }
        return new ModelConfigMap(((SearchHit)mappedHits.get(".inference")).getSourceAsMap(), ((SearchHit)mappedHits.get(".secrets-inference")).getSourceAsMap());
    }

    public void updateModelTransaction(Model newModel, Model existingModel, ActionListener<Boolean> finalListener) {
        String inferenceEntityId = newModel.getConfigurations().getInferenceEntityId();
        logger.info("Attempting to store update to inference endpoint [{}]", (Object)inferenceEntityId);
        if (this.preventDeletionLock.contains(inferenceEntityId)) {
            logger.warn(Strings.format((String)"Attempted to update endpoint [{}] that is already being updated", (Object[])new Object[]{inferenceEntityId}));
            finalListener.onFailure((Exception)new ElasticsearchStatusException("Endpoint [{}] is currently being updated. Try again once the update completes", RestStatus.CONFLICT, new Object[]{inferenceEntityId}));
            return;
        }
        this.preventDeletionLock.add(inferenceEntityId);
        SubscribableListener.newForked(subListener -> {
            IndexRequestBuilder configRequestBuilder = ModelRegistry.createIndexRequestBuilder(inferenceEntityId, ".inference", (ToXContentObject)newModel.getConfigurations(), true, (Client)this.client);
            ActionListener storeConfigListener = subListener.delegateResponse((l, e) -> {
                this.preventDeletionLock.remove(inferenceEntityId);
                l.onFailure(e);
            });
            this.client.prepareBulk().add(configRequestBuilder).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(storeConfigListener);
        }).andThen((subListener, configResponse) -> {
            if (configResponse.hasFailures()) {
                logger.error(Strings.format((String)"Failed to update inference endpoint [%s] due to [%s]", (Object[])new Object[]{inferenceEntityId, configResponse.buildFailureMessage()}));
                this.preventDeletionLock.remove(inferenceEntityId);
                finalListener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to update inference endpoint [%s] due to [%s]", (Object[])new Object[]{inferenceEntityId, configResponse.buildFailureMessage()}), RestStatus.INTERNAL_SERVER_ERROR, new Object[]{configResponse.buildFailureMessage()}));
            } else {
                IndexRequestBuilder secretsRequestBuilder = ModelRegistry.createIndexRequestBuilder(inferenceEntityId, ".secrets-inference", (ToXContentObject)newModel.getSecrets(), true, (Client)this.client);
                ActionListener storeSecretsListener = subListener.delegateResponse((l, e) -> {
                    this.preventDeletionLock.remove(inferenceEntityId);
                    l.onFailure(e);
                });
                this.client.prepareBulk().add(secretsRequestBuilder).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(storeSecretsListener);
            }
        }).andThen((subListener, secretsResponse) -> {
            if (secretsResponse.hasFailures()) {
                IndexRequestBuilder configRequestBuilder = ModelRegistry.createIndexRequestBuilder(inferenceEntityId, ".inference", (ToXContentObject)existingModel.getConfigurations(), true, (Client)this.client);
                logger.error("Failed to update inference endpoint secrets [{}], attempting rolling back to previous state", (Object)inferenceEntityId);
                ActionListener rollbackConfigListener = subListener.delegateResponse((l, e) -> {
                    this.preventDeletionLock.remove(inferenceEntityId);
                    l.onFailure(e);
                });
                this.client.prepareBulk().add(configRequestBuilder).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(rollbackConfigListener);
            } else {
                this.preventDeletionLock.remove(inferenceEntityId);
                finalListener.onResponse((Object)true);
            }
        }).andThen((subListener, configResponse) -> {
            this.preventDeletionLock.remove(inferenceEntityId);
            if (configResponse.hasFailures()) {
                logger.error(Strings.format((String)"Failed to update inference endpoint [%s] due to [%s]", (Object[])new Object[]{inferenceEntityId, configResponse.buildFailureMessage()}));
                finalListener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to rollback while handling failure to update inference endpoint [%s]. Endpoint may be in an inconsistent state due to [%s]", (Object[])new Object[]{inferenceEntityId, configResponse.buildFailureMessage()}), RestStatus.INTERNAL_SERVER_ERROR, new Object[0]));
            } else {
                logger.warn("Failed to update inference endpoint [{}], successfully rolled back to previous state", (Object)inferenceEntityId);
                finalListener.onResponse((Object)false);
            }
        });
    }

    public void storeModel(Model model, ActionListener<Boolean> listener, TimeValue timeout) {
        this.storeModel(model, true, listener, timeout);
    }

    private void storeModel(Model model, boolean updateClusterState, ActionListener<Boolean> listener, TimeValue timeout) {
        ActionListener<BulkResponse> bulkResponseActionListener = this.getStoreIndexListener(model, updateClusterState, listener, timeout);
        String inferenceEntityId = model.getConfigurations().getInferenceEntityId();
        IndexRequestBuilder configRequestBuilder = ModelRegistry.createIndexRequestBuilder(inferenceEntityId, ".inference", (ToXContentObject)model.getConfigurations(), false, (Client)this.client);
        IndexRequestBuilder secretsRequestBuilder = ModelRegistry.createIndexRequestBuilder(inferenceEntityId, ".secrets-inference", (ToXContentObject)model.getSecrets(), false, (Client)this.client);
        this.client.prepareBulk().add(configRequestBuilder).add(secretsRequestBuilder).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(bulkResponseActionListener);
    }

    private ActionListener<BulkResponse> getStoreIndexListener(Model model, boolean updateClusterState, ActionListener<Boolean> listener, TimeValue timeout) {
        AtomicBoolean partialFailure = new AtomicBoolean(false);
        ActionListener cleanupListener = listener.delegateResponse((delegate, ex) -> {
            if (partialFailure.get()) {
                this.deleteModel(model.getInferenceEntityId(), (ActionListener<Boolean>)ActionListener.running(() -> delegate.onFailure(ex)));
            } else {
                delegate.onFailure(ex);
            }
        });
        return ActionListener.wrap(bulkItemResponses -> {
            String inferenceEntityId = model.getInferenceEntityId();
            if (bulkItemResponses.getItems().length == 0) {
                logger.warn(Strings.format((String)"Storing inference endpoint [%s] failed, no items were received from the bulk response", (Object[])new Object[]{inferenceEntityId}));
                cleanupListener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to store inference endpoint [%s], invalid bulk response received. Try reinitializing the service", (Object[])new Object[]{inferenceEntityId}), RestStatus.INTERNAL_SERVER_ERROR, new Object[0]));
                return;
            }
            BulkItemResponse.Failure failure = ModelRegistry.getFirstBulkFailure(bulkItemResponses);
            if (failure == null) {
                if (updateClusterState) {
                    ActionListener<AcknowledgedResponse> storeListener = this.getStoreMetadataListener(inferenceEntityId, (ActionListener<Boolean>)cleanupListener);
                    try {
                        this.metadataTaskQueue.submitTask("add model [" + inferenceEntityId + "]", (ClusterStateTaskListener)new AddModelMetadataTask(inferenceEntityId, new MinimalServiceSettings(model), storeListener), timeout);
                    }
                    catch (Exception exc) {
                        storeListener.onFailure(exc);
                    }
                } else {
                    cleanupListener.onResponse((Object)Boolean.TRUE);
                }
                return;
            }
            for (BulkItemResponse aResponse : bulkItemResponses.getItems()) {
                ModelRegistry.logBulkFailure(inferenceEntityId, aResponse);
                partialFailure.compareAndSet(false, !aResponse.isFailed());
            }
            if (ExceptionsHelper.unwrapCause((Throwable)failure.getCause()) instanceof VersionConflictEngineException) {
                cleanupListener.onFailure((Exception)new ResourceAlreadyExistsException("Inference endpoint [{}] already exists", new Object[]{inferenceEntityId}));
                return;
            }
            cleanupListener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to store inference endpoint [%s]", (Object[])new Object[]{inferenceEntityId}), RestStatus.INTERNAL_SERVER_ERROR, (Throwable)failure.getCause(), new Object[0]));
        }, e -> {
            String errorMessage = Strings.format((String)"Failed to store inference endpoint [%s]", (Object[])new Object[]{model.getInferenceEntityId()});
            logger.warn(errorMessage, (Throwable)e);
            cleanupListener.onFailure((Exception)new ElasticsearchStatusException(errorMessage, RestStatus.INTERNAL_SERVER_ERROR, (Throwable)e, new Object[0]));
        });
    }

    private ActionListener<AcknowledgedResponse> getStoreMetadataListener(final String inferenceEntityId, final ActionListener<Boolean> listener) {
        return new ActionListener<AcknowledgedResponse>(){

            public void onResponse(AcknowledgedResponse resp) {
                listener.onResponse((Object)true);
            }

            public void onFailure(Exception exc) {
                logger.warn(Strings.format((String)"Failed to add inference endpoint [%s] minimal service settings to cluster state", (Object[])new Object[]{inferenceEntityId}), (Throwable)exc);
                ModelRegistry.this.deleteModel(inferenceEntityId, (ActionListener<Boolean>)ActionListener.running(() -> listener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to add the inference endpoint [%s]. The service may be in an inconsistent state. Please try deleting and re-adding the endpoint.", (Object[])new Object[]{inferenceEntityId}), RestStatus.INTERNAL_SERVER_ERROR, (Throwable)exc, new Object[0]))));
            }
        };
    }

    private static void logBulkFailure(String inferenceEntityId, BulkItemResponse item) {
        if (item.isFailed()) {
            logger.warn(Strings.format((String)"Failed to store inference endpoint [%s] index: [%s]", (Object[])new Object[]{inferenceEntityId, item.getIndex()}), (Throwable)item.getFailure().getCause());
        }
    }

    private static BulkItemResponse.Failure getFirstBulkFailure(BulkResponse bulkResponse) {
        for (BulkItemResponse item : bulkResponse.getItems()) {
            if (!item.isFailed()) continue;
            return item.getFailure();
        }
        return null;
    }

    public synchronized void removeDefaultConfigs(Set<String> inferenceEntityIds, ActionListener<Boolean> listener) {
        if (inferenceEntityIds.isEmpty()) {
            listener.onResponse((Object)true);
            return;
        }
        this.defaultConfigIds.keySet().removeAll(inferenceEntityIds);
        this.deleteModels(inferenceEntityIds, false, listener);
    }

    public void deleteModel(String inferenceEntityId, ActionListener<Boolean> listener) {
        this.deleteModels(Set.of(inferenceEntityId), listener);
    }

    public void deleteModels(Set<String> inferenceEntityIds, ActionListener<Boolean> listener) {
        this.deleteModels(inferenceEntityIds, true, listener);
    }

    private void deleteModels(Set<String> inferenceEntityIds, boolean updateClusterState, ActionListener<Boolean> listener) {
        HashSet<String> lockedInferenceIds = new HashSet<String>(inferenceEntityIds);
        lockedInferenceIds.retainAll(this.preventDeletionLock);
        if (!lockedInferenceIds.isEmpty()) {
            listener.onFailure((Exception)new ElasticsearchStatusException(org.elasticsearch.common.Strings.format((String)"The inference endpoint(s) %s are currently being updated, please wait until after they are finished updating to delete.", (Object[])new Object[]{lockedInferenceIds}), RestStatus.CONFLICT, new Object[0]));
            return;
        }
        DeleteByQueryRequest request = ModelRegistry.createDeleteRequest(inferenceEntityIds);
        this.client.execute((ActionType)DeleteByQueryAction.INSTANCE, (ActionRequest)request, this.getDeleteModelClusterStateListener(inferenceEntityIds, updateClusterState, listener));
    }

    private ActionListener<BulkByScrollResponse> getDeleteModelClusterStateListener(final Set<String> inferenceEntityIds, final boolean updateClusterState, final ActionListener<Boolean> listener) {
        return new ActionListener<BulkByScrollResponse>(){

            public void onResponse(BulkByScrollResponse bulkByScrollResponse) {
                if (!updateClusterState) {
                    listener.onResponse((Object)Boolean.TRUE);
                    return;
                }
                var clusterStateListener = new ActionListener<AcknowledgedResponse>(){

                    public void onResponse(AcknowledgedResponse acknowledgedResponse) {
                        listener.onResponse((Object)acknowledgedResponse.isAcknowledged());
                    }

                    public void onFailure(Exception exc) {
                        listener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to delete the inference endpoint [%s]. The service may be in an inconsistent state. Please try deleting the endpoint again.", (Object[])new Object[]{inferenceEntityIds}), RestStatus.INTERNAL_SERVER_ERROR, (Throwable)exc, new Object[0]));
                    }
                };
                try {
                    ModelRegistry.this.metadataTaskQueue.submitTask("delete models [" + inferenceEntityIds + "]", (ClusterStateTaskListener)new DeleteModelMetadataTask(inferenceEntityIds, clusterStateListener), null);
                }
                catch (Exception exc) {
                    clusterStateListener.onFailure(exc);
                }
            }

            public void onFailure(Exception exc) {
                listener.onFailure(exc);
            }
        };
    }

    private static DeleteByQueryRequest createDeleteRequest(Set<String> inferenceEntityIds) {
        DeleteByQueryRequest request = (DeleteByQueryRequest)new DeleteByQueryRequest().setAbortOnVersionConflict(false);
        request.indices(new String[]{".inference*", ".secrets-inference*"});
        request.setQuery(ModelRegistry.documentIdsQuery(inferenceEntityIds));
        request.setRefresh(true);
        return request;
    }

    private static IndexRequest createIndexRequest(String docId, String indexName, ToXContentObject body, boolean allowOverwriting) {
        IndexRequest indexRequest;
        block8: {
            XContentBuilder builder = XContentFactory.jsonBuilder();
            try {
                IndexRequest request = new IndexRequest(indexName);
                XContentBuilder source = body.toXContent(builder, (ToXContent.Params)new ToXContent.MapParams(Map.of("for_index", Boolean.TRUE.toString())));
                DocWriteRequest.OpType operation = allowOverwriting ? DocWriteRequest.OpType.INDEX : DocWriteRequest.OpType.CREATE;
                indexRequest = request.opType(operation).id(docId).source(source);
                if (builder == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (builder != null) {
                        try {
                            builder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new ElasticsearchException(Strings.format((String)"Unexpected serialization exception for index [%s] doc [%s]", (Object[])new Object[]{indexName, docId}), (Throwable)ex, new Object[0]);
                }
            }
            builder.close();
        }
        return indexRequest;
    }

    static IndexRequestBuilder createIndexRequestBuilder(String inferenceId, String indexName, ToXContentObject body, boolean allowOverwriting, Client client) {
        IndexRequestBuilder indexRequestBuilder;
        block8: {
            XContentBuilder xContentBuilder = XContentFactory.jsonBuilder();
            try {
                XContentBuilder source = body.toXContent(xContentBuilder, (ToXContent.Params)new ToXContent.MapParams(Map.of("for_index", Boolean.TRUE.toString())));
                indexRequestBuilder = ((IndexRequestBuilder)new IndexRequestBuilder((ElasticsearchClient)client).setIndex(indexName)).setCreate(!allowOverwriting).setId(Model.documentId((String)inferenceId)).setSource(source);
                if (xContentBuilder == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (xContentBuilder != null) {
                        try {
                            xContentBuilder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new ElasticsearchException("Unexpected serialization exception for index [{}] inference ID [{}]", (Throwable)ex, new Object[]{indexName, inferenceId});
                }
            }
            xContentBuilder.close();
        }
        return indexRequestBuilder;
    }

    private static UnparsedModel modelToUnparsedModel(Model model) {
        UnparsedModel unparsedModel;
        block8: {
            XContentBuilder builder = XContentFactory.jsonBuilder();
            try {
                model.getConfigurations().toXContent(builder, (ToXContent.Params)new ToXContent.MapParams(Map.of("for_index", Boolean.TRUE.toString())));
                Map modelConfigMap = (Map)XContentHelper.convertToMap((BytesReference)BytesReference.bytes((XContentBuilder)builder), (boolean)false, (XContentType)builder.contentType()).v2();
                unparsedModel = ModelRegistry.unparsedModelFromMap(new ModelConfigMap(modelConfigMap, new HashMap<String, Object>()));
                if (builder == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (builder != null) {
                        try {
                            builder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new ElasticsearchException("[{}] Error serializing inference endpoint configuration", new Object[]{model.getInferenceEntityId(), ex});
                }
            }
            builder.close();
        }
        return unparsedModel;
    }

    private static QueryBuilder documentIdQuery(String inferenceEntityId) {
        return QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.idsQuery().addIds(new String[]{Model.documentId((String)inferenceEntityId)}));
    }

    private static QueryBuilder documentIdsQuery(Set<String> inferenceEntityIds) {
        String[] documentIdsArray = (String[])inferenceEntityIds.stream().map(Model::documentId).toArray(String[]::new);
        return QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.idsQuery().addIds(documentIdsArray));
    }

    static Optional<InferenceService.DefaultConfigId> idMatchedDefault(String inferenceId, List<InferenceService.DefaultConfigId> defaultConfigIds) {
        return defaultConfigIds.stream().filter(defaultConfigId -> defaultConfigId.inferenceId().equals(inferenceId)).findFirst();
    }

    static List<InferenceService.DefaultConfigId> taskTypeMatchedDefaults(TaskType taskType, Collection<InferenceService.DefaultConfigId> defaultConfigIds) {
        return defaultConfigIds.stream().filter(defaultConfigId -> defaultConfigId.settings().taskType().equals((Object)taskType)).collect(Collectors.toList());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clusterChanged(ClusterChangedEvent event) {
        if (this.lastMetadata == null || event.metadataChanged()) {
            ModelRegistry modelRegistry = this;
            synchronized (modelRegistry) {
                this.lastMetadata = event.state().metadata();
            }
        }
        if (!event.localNodeMaster()) {
            return;
        }
        ModelRegistryMetadata state = ModelRegistryMetadata.fromState(event.state().metadata());
        if (event.state().blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        if (state.isUpgraded()) {
            return;
        }
        if (!this.upgradeMetadataInProgress.compareAndSet(false, true)) {
            return;
        }
        this.client.execute((ActionType)GetInferenceModelAction.INSTANCE, (ActionRequest)new GetInferenceModelAction.Request("*", TaskType.ANY, false), (ActionListener)new ActionListener<GetInferenceModelAction.Response>(){

            public void onResponse(GetInferenceModelAction.Response response) {
                HashMap<String, MinimalServiceSettings> map = new HashMap<String, MinimalServiceSettings>();
                for (ModelConfigurations model : response.getEndpoints()) {
                    if (ModelRegistry.this.defaultConfigIds.containsKey(model.getInferenceEntityId())) continue;
                    map.put(model.getInferenceEntityId(), new MinimalServiceSettings(model.getService(), model.getTaskType(), model.getServiceSettings().dimensions(), model.getServiceSettings().similarity(), model.getServiceSettings().elementType()));
                }
                ModelRegistry.this.metadataTaskQueue.submitTask("model registry auto upgrade", (ClusterStateTaskListener)new UpgradeModelsMetadataTask(map, (ActionListener<AcknowledgedResponse>)ActionListener.running(() -> ModelRegistry.this.upgradeMetadataInProgress.set(false))), null);
            }

            public void onFailure(Exception e) {
                ModelRegistry.this.upgradeMetadataInProgress.set(false);
            }
        });
    }

    public record ModelConfigMap(Map<String, Object> config, Map<String, Object> secrets) {
    }

    private static class AddModelMetadataTask
    extends MetadataTask {
        private final String inferenceEntityId;
        private final MinimalServiceSettings settings;

        AddModelMetadataTask(String inferenceEntityId, MinimalServiceSettings settings, ActionListener<AcknowledgedResponse> listener) {
            super(listener);
            this.inferenceEntityId = inferenceEntityId;
            this.settings = settings;
        }

        @Override
        ModelRegistryMetadata executeTask(ModelRegistryMetadata current) {
            return current.withAddedModel(this.inferenceEntityId, this.settings);
        }
    }

    private static class DeleteModelMetadataTask
    extends MetadataTask {
        private final Set<String> inferenceEntityIds;

        DeleteModelMetadataTask(Set<String> inferenceEntityId, ActionListener<AcknowledgedResponse> listener) {
            super(listener);
            this.inferenceEntityIds = inferenceEntityId;
        }

        @Override
        ModelRegistryMetadata executeTask(ModelRegistryMetadata current) {
            return current.withRemovedModel(this.inferenceEntityIds);
        }
    }

    private static class UpgradeModelsMetadataTask
    extends MetadataTask {
        private final Map<String, MinimalServiceSettings> fromIndex;

        UpgradeModelsMetadataTask(Map<String, MinimalServiceSettings> fromIndex, ActionListener<AcknowledgedResponse> listener) {
            super(listener);
            this.fromIndex = fromIndex;
        }

        @Override
        ModelRegistryMetadata executeTask(ModelRegistryMetadata current) {
            return current.withUpgradedModels(this.fromIndex);
        }
    }

    private static abstract class MetadataTask
    extends AckedBatchedClusterStateUpdateTask {
        MetadataTask(ActionListener<AcknowledgedResponse> listener) {
            super(TimeValue.THIRTY_SECONDS, listener);
        }

        abstract ModelRegistryMetadata executeTask(ModelRegistryMetadata var1);
    }
}

