/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc.saml;

import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.xpack.security.authc.saml.IdpConfiguration;
import org.elasticsearch.xpack.security.authc.saml.SamlAttributes;
import org.elasticsearch.xpack.security.authc.saml.SamlNameId;
import org.elasticsearch.xpack.security.authc.saml.SamlResponseHandler;
import org.elasticsearch.xpack.security.authc.saml.SamlToken;
import org.elasticsearch.xpack.security.authc.saml.SamlUtils;
import org.elasticsearch.xpack.security.authc.saml.SpConfiguration;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.schema.XSURI;
import org.opensaml.saml.common.SAMLObject;
import org.opensaml.saml.saml2.core.Assertion;
import org.opensaml.saml.saml2.core.Attribute;
import org.opensaml.saml.saml2.core.AttributeStatement;
import org.opensaml.saml.saml2.core.AudienceRestriction;
import org.opensaml.saml.saml2.core.AuthnStatement;
import org.opensaml.saml.saml2.core.Conditions;
import org.opensaml.saml.saml2.core.EncryptedAssertion;
import org.opensaml.saml.saml2.core.EncryptedAttribute;
import org.opensaml.saml.saml2.core.Response;
import org.opensaml.saml.saml2.core.StatusResponseType;
import org.opensaml.saml.saml2.core.Subject;
import org.opensaml.saml.saml2.core.SubjectConfirmation;
import org.opensaml.saml.saml2.core.SubjectConfirmationData;
import org.opensaml.xmlsec.encryption.support.DecryptionException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

class SamlAuthenticator
extends SamlResponseHandler {
    private static final String RESPONSE_TAG_NAME = "Response";
    private static final Set<String> SPECIAL_ATTRIBUTE_NAMES = Set.of("nameid", "nameid:persistent");

    SamlAuthenticator(Clock clock, IdpConfiguration idp, SpConfiguration sp, TimeValue maxSkew) {
        super(clock, idp, sp, maxSkew);
    }

    SamlAttributes authenticate(SamlToken token) {
        Element root = this.parseSamlMessage(token.getContent());
        if (RESPONSE_TAG_NAME.equals(root.getLocalName()) && "urn:oasis:names:tc:SAML:2.0:protocol".equals(root.getNamespaceURI())) {
            try {
                return this.authenticateResponse(root, token.getAllowedSamlRequestIds());
            }
            catch (ElasticsearchSecurityException e) {
                this.logger.trace("Rejecting SAML response [{}...] because {}", (Object)org.elasticsearch.common.Strings.cleanTruncate((String)SamlUtils.toString(root), (int)512), (Object)e.getMessage());
                throw e;
            }
        }
        throw SamlUtils.samlException("SAML content [{}] should have a root element of Namespace=[{}] Tag=[{}]", root, "urn:oasis:names:tc:SAML:2.0:protocol", RESPONSE_TAG_NAME);
    }

    private SamlAttributes authenticateResponse(Element element, Collection<String> allowedSamlRequestIds) {
        boolean requireSignedAssertions;
        Response response = this.buildXmlObject(element, Response.class);
        if (response == null) {
            throw SamlUtils.samlException("Cannot convert element {} into Response object", element);
        }
        if (this.logger.isTraceEnabled()) {
            this.logger.trace(SamlUtils.describeSamlObject((SAMLObject)response));
        }
        if (response.isSigned()) {
            this.validateSignature(response.getSignature(), response.getIssuer());
            requireSignedAssertions = false;
        } else {
            requireSignedAssertions = true;
        }
        this.checkInResponseTo((StatusResponseType)response, allowedSamlRequestIds);
        SamlAuthenticator.checkStatus(response.getStatus());
        this.checkIssuer(response.getIssuer(), (XMLObject)response);
        this.checkResponseDestination(response);
        Tuple<Assertion, List<Attribute>> details = this.extractDetails(response, allowedSamlRequestIds, requireSignedAssertions);
        Assertion assertion = (Assertion)details.v1();
        SamlNameId nameId = SamlNameId.forSubject(assertion.getSubject());
        String session = SamlAuthenticator.getSessionIndex(assertion);
        List<SamlAttributes.SamlAttribute> attributes = ((List)details.v2()).stream().map(SamlAttributes.SamlAttribute::new).toList();
        if (this.logger.isTraceEnabled()) {
            StringBuilder sb = new StringBuilder();
            sb.append("The SAML Assertion contained the following attributes: \n");
            for (SamlAttributes.SamlAttribute attr : attributes) {
                sb.append(attr).append("\n");
            }
            this.logger.trace(sb.toString());
        }
        if (attributes.isEmpty() && nameId == null) {
            this.logger.debug("The Attribute Statements of SAML Response with ID [{}] contained no attributes and the SAML Assertion Subject did not contain a SAML NameID. Please verify that the Identity Provider configuration with regards to attribute release is correct.", (Object)response.getID());
            throw SamlUtils.samlException("Could not process any SAML attributes in {}", response.getElementQName());
        }
        return new SamlAttributes(nameId, session, attributes);
    }

    private static String getSessionIndex(Assertion assertion) {
        return assertion.getAuthnStatements().stream().map(as -> as.getSessionIndex()).filter(Objects::nonNull).findFirst().orElse(null);
    }

    private void checkResponseDestination(Response response) {
        String asc = this.getSpConfiguration().getAscUrl();
        if (!asc.equals(response.getDestination()) && (response.isSigned() || org.elasticsearch.common.Strings.hasText((String)response.getDestination()))) {
            throw SamlUtils.samlException("SAML response " + response.getID() + " is for destination " + response.getDestination() + " but this realm uses " + asc, new Object[0]);
        }
    }

    private Tuple<Assertion, List<Attribute>> extractDetails(Response response, Collection<String> allowedSamlRequestIds, boolean requireSignedAssertions) {
        int assertionCount = response.getAssertions().size() + response.getEncryptedAssertions().size();
        if (assertionCount > 1) {
            throw SamlUtils.samlException("Expecting only 1 assertion, but response contains multiple (" + assertionCount + ")", new Object[0]);
        }
        Iterator iterator = response.getAssertions().iterator();
        if (iterator.hasNext()) {
            Assertion assertion = (Assertion)iterator.next();
            return new Tuple((Object)assertion, this.processAssertion(assertion, requireSignedAssertions, allowedSamlRequestIds));
        }
        iterator = response.getEncryptedAssertions().iterator();
        if (iterator.hasNext()) {
            EncryptedAssertion encrypted = (EncryptedAssertion)iterator.next();
            Assertion assertion = this.decrypt(encrypted);
            SamlAuthenticator.moveToNewDocument((XMLObject)assertion);
            assertion.getDOM().setIdAttribute("ID", true);
            return new Tuple((Object)assertion, this.processAssertion(assertion, requireSignedAssertions, allowedSamlRequestIds));
        }
        throw SamlUtils.samlException("No assertions found in SAML response", new Object[0]);
    }

    private static void moveToNewDocument(XMLObject xmlObject) {
        Element element = xmlObject.getDOM();
        Document doc = element.getOwnerDocument().getImplementation().createDocument(null, null, null);
        doc.adoptNode(element);
        doc.appendChild(element);
    }

    private Assertion decrypt(EncryptedAssertion encrypted) {
        if (this.decrypter == null) {
            throw SamlUtils.samlException("SAML assertion [" + SamlAuthenticator.text((XMLObject)encrypted, 32) + "] is encrypted, but no decryption key is available", new Object[0]);
        }
        try {
            return this.decrypter.decrypt(encrypted);
        }
        catch (DecryptionException e) {
            this.logger.debug(() -> Strings.format((String)"Failed to decrypt SAML assertion [%s] with [%s]", (Object[])new Object[]{SamlAuthenticator.text((XMLObject)encrypted, 512), SamlAuthenticator.describe(this.getSpConfiguration().getEncryptionCredentials())}), (Throwable)e);
            throw SamlUtils.samlException("Failed to decrypt SAML assertion " + SamlAuthenticator.text((XMLObject)encrypted, 32), (Exception)((Object)e), new Object[0]);
        }
    }

    private List<Attribute> processAssertion(Assertion assertion, boolean requireSignature, Collection<String> allowedSamlRequestIds) {
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("(Possibly decrypted) Assertion: {}", (Object)SamlUtils.getXmlContent((SAMLObject)assertion, true));
            this.logger.trace(SamlUtils.describeSamlObject((SAMLObject)assertion));
        }
        if (assertion.isSigned()) {
            this.validateSignature(assertion.getSignature(), assertion.getIssuer());
        } else if (requireSignature) {
            throw SamlUtils.samlException("Assertion [{}] is not signed, but a signature is required", assertion.getElementQName());
        }
        this.checkConditions(assertion.getConditions());
        this.checkIssuer(assertion.getIssuer(), (XMLObject)assertion);
        this.checkSubject(assertion.getSubject(), (XMLObject)assertion, allowedSamlRequestIds);
        this.checkAuthnStatement(assertion.getAuthnStatements());
        ArrayList<Attribute> attributes = new ArrayList<Attribute>();
        for (AttributeStatement statement : assertion.getAttributeStatements()) {
            this.logger.trace("SAML AttributeStatement has [{}] attributes and [{}] encrypted attributes", (Object)statement.getAttributes().size(), (Object)statement.getEncryptedAttributes().size());
            attributes.addAll(statement.getAttributes());
            for (EncryptedAttribute enc : statement.getEncryptedAttributes()) {
                Attribute attribute = this.decrypt(enc);
                if (attribute == null) continue;
                this.logger.trace("Successfully decrypted attribute: {}" + SamlUtils.getXmlContent((SAMLObject)attribute, true));
                attributes.add(attribute);
            }
        }
        this.warnOnSpecialAttributeNames(assertion, attributes);
        return attributes;
    }

    private void checkAuthnStatement(List<AuthnStatement> authnStatements) {
        if (authnStatements.size() != 1) {
            throw SamlUtils.samlException("SAML Assertion subject contains [{}] Authn Statements while exactly one was expected.", authnStatements.size());
        }
        AuthnStatement authnStatement = authnStatements.get(0);
        Instant now = this.now();
        Instant pastNow = now.minusMillis(this.maxSkewInMillis());
        if (authnStatement.getSessionNotOnOrAfter() != null && !pastNow.isBefore(authnStatement.getSessionNotOnOrAfter())) {
            throw SamlUtils.samlException("Rejecting SAML assertion's Authentication Statement because [{}] is on/after [{}]", pastNow, authnStatement.getSessionNotOnOrAfter());
        }
        List<String> reqAuthnCtxClassRef = this.getSpConfiguration().getReqAuthnCtxClassRef();
        if (!reqAuthnCtxClassRef.isEmpty()) {
            String authnCtxClassRefValue = null;
            if (authnStatement.getAuthnContext() != null && authnStatement.getAuthnContext().getAuthnContextClassRef() != null) {
                authnCtxClassRefValue = authnStatement.getAuthnContext().getAuthnContextClassRef().getURI();
            }
            if (org.elasticsearch.common.Strings.isNullOrEmpty(authnCtxClassRefValue) || !reqAuthnCtxClassRef.contains(authnCtxClassRefValue)) {
                throw SamlUtils.samlException("Rejecting SAML assertion as the AuthnContextClassRef [{}] is not one of the ({}) that were requested in the corresponding AuthnRequest", authnCtxClassRefValue, reqAuthnCtxClassRef);
            }
        }
    }

    private Attribute decrypt(EncryptedAttribute encrypted) {
        if (this.decrypter == null) {
            this.logger.info("SAML message has encrypted attribute [" + SamlAuthenticator.text((XMLObject)encrypted, 32) + "], but no encryption key has been configured");
            return null;
        }
        try {
            return this.decrypter.decrypt(encrypted);
        }
        catch (DecryptionException e) {
            this.logger.info("Failed to decrypt SAML attribute " + SamlAuthenticator.text((XMLObject)encrypted, 32), (Throwable)e);
            return null;
        }
    }

    private void checkConditions(Conditions conditions) {
        if (conditions != null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("SAML Assertion was intended for the following Service providers: {}", (Object)conditions.getAudienceRestrictions().stream().map(r -> SamlAuthenticator.text((XMLObject)r, 32)).collect(Collectors.joining(" | ")));
                this.logger.trace("SAML Assertion is only valid between: " + conditions.getNotBefore() + " and " + conditions.getNotOnOrAfter());
            }
            this.checkAudienceRestrictions(conditions.getAudienceRestrictions());
            this.checkLifetimeRestrictions(conditions);
        }
    }

    private void checkSubject(Subject assertionSubject, XMLObject parent, Collection<String> allowedSamlRequestIds) {
        if (assertionSubject == null) {
            throw SamlUtils.samlException("SAML Assertion ({}) has no Subject", SamlAuthenticator.text(parent, 16));
        }
        List<SubjectConfirmationData> confirmationData = assertionSubject.getSubjectConfirmations().stream().filter(data -> data.getMethod().equals("urn:oasis:names:tc:SAML:2.0:cm:bearer")).map(SubjectConfirmation::getSubjectConfirmationData).filter(Objects::nonNull).toList();
        if (confirmationData.size() != 1) {
            throw SamlUtils.samlException("SAML Assertion subject contains [{}] bearer SubjectConfirmation, while exactly one was expected.", confirmationData.size());
        }
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("SAML Assertion Subject Confirmation intended recipient is: " + confirmationData.get(0).getRecipient());
            this.logger.trace("SAML Assertion Subject Confirmation is only valid before: " + confirmationData.get(0).getNotOnOrAfter());
            this.logger.trace("SAML Assertion Subject Confirmation is in response to: " + confirmationData.get(0).getInResponseTo());
        }
        this.checkRecipient(confirmationData.get(0));
        this.checkLifetimeRestrictions(confirmationData.get(0));
        SamlAuthenticator.checkSubjectInResponseTo(confirmationData.get(0), allowedSamlRequestIds);
    }

    private static void checkSubjectInResponseTo(SubjectConfirmationData subjectConfirmationData, Collection<String> allowedSamlRequestIds) {
        if (org.elasticsearch.common.Strings.hasText((String)subjectConfirmationData.getInResponseTo()) && !allowedSamlRequestIds.contains(subjectConfirmationData.getInResponseTo())) {
            throw SamlUtils.samlException("SAML Assertion SubjectConfirmationData is in-response-to [{}] but expected one of [{}]", subjectConfirmationData.getInResponseTo(), allowedSamlRequestIds);
        }
    }

    private void checkRecipient(SubjectConfirmationData subjectConfirmationData) {
        SpConfiguration sp = this.getSpConfiguration();
        if (!sp.getAscUrl().equals(subjectConfirmationData.getRecipient())) {
            throw SamlUtils.samlException("SAML Assertion SubjectConfirmationData Recipient [{}] does not match expected value [{}]", subjectConfirmationData.getRecipient(), sp.getAscUrl());
        }
    }

    private void checkAudienceRestrictions(List<AudienceRestriction> restrictions) {
        if (!restrictions.stream().allMatch(this::checkAudienceRestriction)) {
            throw SamlUtils.samlException("Conditions [{}] do not match required audience [{}]", restrictions.stream().map(r -> SamlAuthenticator.text((XMLObject)r, 56, 8)).collect(Collectors.joining(" | ")), this.getSpConfiguration().getEntityId());
        }
    }

    private boolean checkAudienceRestriction(AudienceRestriction restriction) {
        String spEntityId = this.getSpConfiguration().getEntityId();
        if (!restriction.getAudiences().stream().map(XSURI::getURI).anyMatch(spEntityId::equals)) {
            restriction.getAudiences().stream().map(XSURI::getURI).forEach(uri -> {
                int diffChar;
                for (diffChar = 0; diffChar < uri.length() && diffChar < spEntityId.length() && uri.charAt(diffChar) == spEntityId.charAt(diffChar); ++diffChar) {
                }
                if (diffChar >= spEntityId.length() / 2) {
                    this.logger.info("Audience restriction [{}] does not match required audience [{}] (difference starts at character [#{}] [{}] vs [{}])", uri, (Object)spEntityId, (Object)diffChar, (Object)uri.substring(diffChar), (Object)spEntityId.substring(diffChar));
                } else {
                    this.logger.info("Audience restriction [{}] does not match required audience [{}]", uri, (Object)spEntityId);
                }
            });
            return false;
        }
        return true;
    }

    private void checkLifetimeRestrictions(Conditions conditions) {
        Instant now = this.now();
        Instant futureNow = now.plusMillis(this.maxSkewInMillis());
        Instant pastNow = now.minusMillis(this.maxSkewInMillis());
        if (conditions.getNotBefore() != null && futureNow.isBefore(conditions.getNotBefore())) {
            throw SamlUtils.samlException("Rejecting SAML assertion because [{}] is before [{}]", futureNow, conditions.getNotBefore());
        }
        if (conditions.getNotOnOrAfter() != null && !pastNow.isBefore(conditions.getNotOnOrAfter())) {
            throw SamlUtils.samlException("Rejecting SAML assertion because [{}] is on/after [{}]", pastNow, conditions.getNotOnOrAfter());
        }
    }

    private void checkLifetimeRestrictions(SubjectConfirmationData subjectConfirmationData) {
        this.validateNotOnOrAfter(subjectConfirmationData.getNotOnOrAfter());
    }

    private void warnOnSpecialAttributeNames(Assertion assertion, List<Attribute> attributes) {
        attributes.forEach(attribute -> {
            this.warnOnSpecialAttributeName(assertion, attribute.getName(), "name");
            String attributeFriendlyName = attribute.getFriendlyName();
            if (attributeFriendlyName != null) {
                this.warnOnSpecialAttributeName(assertion, attributeFriendlyName, "friendly name");
            }
        });
    }

    private void warnOnSpecialAttributeName(Assertion assertion, String attributeName, String fieldNameForLogMessage) {
        if (SPECIAL_ATTRIBUTE_NAMES.contains(attributeName)) {
            this.logger.warn("SAML assertion [{}] has attribute with {} [{}] which clashes with a special attribute name. Attributes with a name clash may prevent authentication or interfere will role mapping. Change your IdP configuration to use a different attribute {} that will not clash with any of [{}]", (Object)assertion.getElementQName(), (Object)fieldNameForLogMessage, (Object)attributeName, (Object)fieldNameForLogMessage, (Object)String.join((CharSequence)",", SPECIAL_ATTRIBUTE_NAMES));
        }
    }
}

