/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.plugins.cli;

import java.io.IOException;
import java.net.Proxy;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.elasticsearch.Build;
import org.elasticsearch.cli.Terminal;
import org.elasticsearch.cli.UserException;
import org.elasticsearch.env.Environment;
import org.elasticsearch.plugins.PluginDescriptor;
import org.elasticsearch.plugins.cli.InstallPluginAction;
import org.elasticsearch.plugins.cli.InstallablePlugin;
import org.elasticsearch.plugins.cli.PluginSyncException;
import org.elasticsearch.plugins.cli.PluginsConfig;
import org.elasticsearch.plugins.cli.ProxyUtils;
import org.elasticsearch.plugins.cli.RemovePluginAction;
import org.elasticsearch.xcontent.cbor.CborXContent;
import org.elasticsearch.xcontent.yaml.YamlXContent;

public class SyncPluginsAction {
    public static final String ELASTICSEARCH_PLUGINS_YML = "elasticsearch-plugins.yml";
    public static final String ELASTICSEARCH_PLUGINS_YML_CACHE = ".elasticsearch-plugins.yml.cache";
    private final Terminal terminal;
    private final Environment env;

    public SyncPluginsAction(Terminal terminal, Environment env) {
        this.terminal = terminal;
        this.env = env;
    }

    public static void ensureNoConfigFile(Environment env) throws UserException {
        Path pluginsConfig = env.configDir().resolve(ELASTICSEARCH_PLUGINS_YML);
        if (Files.exists(pluginsConfig, new LinkOption[0])) {
            throw new UserException(64, "Plugins config [" + pluginsConfig + "] exists, which is used by Elasticsearch on startup to ensure the correct plugins are installed. Instead of using this tool, you need to update this config file and restart Elasticsearch.");
        }
    }

    public void execute() throws Exception {
        Path configPath = this.env.configDir().resolve(ELASTICSEARCH_PLUGINS_YML);
        Path previousConfigPath = this.env.pluginsDir().resolve(ELASTICSEARCH_PLUGINS_YML_CACHE);
        if (!Files.exists(configPath, new LinkOption[0])) {
            throw new PluginSyncException("Plugins config does not exist: " + configPath.toAbsolutePath());
        }
        if (!Files.exists(this.env.pluginsDir(), new LinkOption[0])) {
            throw new PluginSyncException("Plugins directory missing: " + this.env.pluginsDir());
        }
        PluginsConfig pluginsConfig = PluginsConfig.parseConfig(configPath, YamlXContent.yamlXContent);
        pluginsConfig.validate(InstallPluginAction.OFFICIAL_PLUGINS, InstallPluginAction.PLUGINS_CONVERTED_TO_MODULES);
        Optional<PluginsConfig> cachedPluginsConfig = Files.exists(previousConfigPath, new LinkOption[0]) ? Optional.of(PluginsConfig.parseConfig(previousConfigPath, CborXContent.cborXContent)) : Optional.empty();
        PluginChanges changes = this.getPluginChanges(pluginsConfig, cachedPluginsConfig);
        if (changes.isEmpty()) {
            this.terminal.println((CharSequence)"No plugins to install, remove or upgrade");
            return;
        }
        this.performSync(pluginsConfig, changes);
        PluginsConfig.writeConfig(CborXContent.cborXContent, pluginsConfig, previousConfigPath);
    }

    PluginChanges getPluginChanges(PluginsConfig pluginsConfig, Optional<PluginsConfig> cachedPluginsConfig) throws PluginSyncException {
        List<PluginDescriptor> existingPlugins = this.getExistingPlugins();
        List<InstallablePlugin> pluginsThatShouldExist = this.getPluginsThatShouldExist(pluginsConfig);
        List<InstallablePlugin> pluginsThatActuallyExist = existingPlugins.stream().map(info -> new InstallablePlugin(info.getName())).collect(Collectors.toList());
        Set existingPluginIds = pluginsThatActuallyExist.stream().map(InstallablePlugin::getId).collect(Collectors.toSet());
        List<InstallablePlugin> pluginsToInstall = SyncPluginsAction.difference(pluginsThatShouldExist, pluginsThatActuallyExist);
        List<InstallablePlugin> pluginsToRemove = SyncPluginsAction.difference(pluginsThatActuallyExist, pluginsThatShouldExist);
        List<InstallablePlugin> pluginsToMaybeUpgrade = SyncPluginsAction.difference(pluginsThatShouldExist, pluginsToRemove).stream().filter(each -> existingPluginIds.contains(each.getId())).collect(Collectors.toList());
        List<InstallablePlugin> pluginsToUpgrade = this.getPluginsToUpgrade(pluginsToMaybeUpgrade, cachedPluginsConfig, existingPlugins);
        pluginsToRemove.sort(Comparator.comparing(InstallablePlugin::getId));
        pluginsToInstall.sort(Comparator.comparing(InstallablePlugin::getId));
        pluginsToMaybeUpgrade.sort(Comparator.comparing(InstallablePlugin::getId));
        return new PluginChanges(pluginsToRemove, pluginsToInstall, pluginsToUpgrade);
    }

    private List<InstallablePlugin> getPluginsThatShouldExist(PluginsConfig pluginsConfig) {
        ArrayList<InstallablePlugin> pluginsThatShouldExist = new ArrayList<InstallablePlugin>(pluginsConfig.getPlugins());
        Iterator shouldExistIterator = pluginsThatShouldExist.iterator();
        while (shouldExistIterator.hasNext()) {
            InstallablePlugin each = (InstallablePlugin)shouldExistIterator.next();
            if (!InstallPluginAction.PLUGINS_CONVERTED_TO_MODULES.contains(each.getId())) continue;
            this.terminal.errorPrintln("[" + each.getId() + "] is no longer a plugin but instead a module packaged with this distribution of Elasticsearch");
            shouldExistIterator.remove();
        }
        return pluginsThatShouldExist;
    }

    private void performSync(PluginsConfig pluginsConfig, PluginChanges changes) throws Exception {
        Proxy proxy = ProxyUtils.buildProxy(pluginsConfig.getProxy());
        RemovePluginAction removePluginAction = new RemovePluginAction(this.terminal, this.env, true);
        InstallPluginAction installPluginAction = new InstallPluginAction(this.terminal, this.env, true);
        installPluginAction.setProxy(proxy);
        this.performSync(installPluginAction, removePluginAction, changes);
    }

    void performSync(InstallPluginAction installAction, RemovePluginAction removeAction, PluginChanges changes) throws Exception {
        this.logRequiredChanges(changes);
        if (!changes.remove.isEmpty()) {
            removeAction.setPurge(true);
            removeAction.execute(changes.remove);
        }
        if (!changes.install.isEmpty()) {
            installAction.execute(changes.install);
        }
        if (!changes.upgrade.isEmpty()) {
            removeAction.setPurge(false);
            removeAction.execute(changes.upgrade);
            installAction.execute(changes.upgrade);
        }
    }

    private List<InstallablePlugin> getPluginsToUpgrade(List<InstallablePlugin> pluginsToMaybeUpgrade, Optional<PluginsConfig> cachedPluginsConfig, List<PluginDescriptor> existingPlugins) {
        HashMap cachedPluginIdToLocation = new HashMap();
        cachedPluginsConfig.ifPresent(config -> config.getPlugins().forEach(p -> cachedPluginIdToLocation.put(p.getId(), p.getLocation())));
        return pluginsToMaybeUpgrade.stream().filter(eachPlugin -> {
            String eachPluginId = eachPlugin.getId();
            if (!Objects.equals(eachPlugin.getLocation(), cachedPluginIdToLocation.get(eachPluginId))) {
                this.terminal.println(Terminal.Verbosity.VERBOSE, (CharSequence)String.format(Locale.ROOT, "Location for plugin [%s] has changed from [%s] to [%s], reinstalling", eachPluginId, cachedPluginIdToLocation.get(eachPluginId), eachPlugin.getLocation()));
                return true;
            }
            if (InstallPluginAction.OFFICIAL_PLUGINS.contains(eachPluginId)) {
                PluginDescriptor info = existingPlugins.stream().filter(each -> each.getName().equals(eachPluginId)).findFirst().orElseThrow(() -> {
                    throw new RuntimeException("Couldn't find a PluginInfo for [" + eachPluginId + "], which should be impossible");
                });
                if (!info.getElasticsearchVersion().toString().equals(Build.current().version())) {
                    this.terminal.println(Terminal.Verbosity.VERBOSE, (CharSequence)String.format(Locale.ROOT, "Official plugin [%s] is out-of-sync (%s versus %s), upgrading", eachPluginId, info.getElasticsearchVersion(), Build.current().version()));
                    return true;
                }
                return false;
            }
            return false;
        }).collect(Collectors.toList());
    }

    private List<PluginDescriptor> getExistingPlugins() throws PluginSyncException {
        ArrayList<PluginDescriptor> plugins = new ArrayList<PluginDescriptor>();
        try (DirectoryStream<Path> paths = Files.newDirectoryStream(this.env.pluginsDir());){
            for (Path pluginPath : paths) {
                String filename = pluginPath.getFileName().toString();
                if (filename.startsWith(".")) continue;
                PluginDescriptor info = PluginDescriptor.readFromProperties((Path)this.env.pluginsDir().resolve(pluginPath));
                plugins.add(info);
                if (!InstallPluginAction.OFFICIAL_PLUGINS.contains(info.getName()) || info.getElasticsearchVersion().toString().equals(Build.current().version())) continue;
                this.terminal.errorPrintln(String.format(Locale.ROOT, "WARNING: plugin [%s] was built for Elasticsearch version %s but version %s is required", info.getName(), info.getElasticsearchVersion(), Build.current().version()));
            }
        }
        catch (IOException e) {
            throw new PluginSyncException("Failed to list existing plugins", e);
        }
        return plugins;
    }

    private static List<InstallablePlugin> difference(List<InstallablePlugin> left, List<InstallablePlugin> right) {
        return left.stream().filter(eachDescriptor -> {
            String id = eachDescriptor.getId();
            return !right.stream().anyMatch(p -> p.getId().equals(id));
        }).collect(Collectors.toList());
    }

    private void logRequiredChanges(PluginChanges changes) {
        BiConsumer<String, List> printSummary = (action, plugins) -> {
            if (!plugins.isEmpty()) {
                List<String> pluginIds = plugins.stream().map(InstallablePlugin::getId).toList();
                this.terminal.errorPrintln(String.format(Locale.ROOT, "Plugins to be %s: %s", action, pluginIds));
            }
        };
        printSummary.accept("removed", changes.remove);
        printSummary.accept("installed", changes.install);
        printSummary.accept("upgraded", changes.upgrade);
    }

    static class PluginChanges {
        final List<InstallablePlugin> remove;
        final List<InstallablePlugin> install;
        final List<InstallablePlugin> upgrade;

        PluginChanges(List<InstallablePlugin> remove, List<InstallablePlugin> install, List<InstallablePlugin> upgrade) {
            this.remove = Objects.requireNonNull(remove);
            this.install = Objects.requireNonNull(install);
            this.upgrade = Objects.requireNonNull(upgrade);
        }

        boolean isEmpty() {
            return this.remove.isEmpty() && this.install.isEmpty() && this.upgrade.isEmpty();
        }
    }
}

