/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.admin.cluster.repositories.delete.DeleteRepositoryRequest;
import org.elasticsearch.action.admin.cluster.repositories.put.PutRepositoryRequest;
import org.elasticsearch.action.admin.cluster.stats.RepositoryUsageStats;
import org.elasticsearch.action.support.SubscribableListener;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.internal.node.NodeClient;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.RepositoryCleanupInProgress;
import org.elasticsearch.cluster.RestoreInProgress;
import org.elasticsearch.cluster.SnapshotDeletionsInProgress;
import org.elasticsearch.cluster.SnapshotsInProgress;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.RepositoriesMetadata;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.ListenableFuture;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.repositories.InvalidRepository;
import org.elasticsearch.repositories.RepositoriesStats;
import org.elasticsearch.repositories.RepositoriesStatsArchive;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryConflictException;
import org.elasticsearch.repositories.RepositoryData;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositoryMissingException;
import org.elasticsearch.repositories.RepositoryStatsSnapshot;
import org.elasticsearch.repositories.UnknownTypeRepository;
import org.elasticsearch.repositories.VerifyNodeRepositoryAction;
import org.elasticsearch.repositories.VerifyNodeRepositoryCoordinationAction;
import org.elasticsearch.repositories.blobstore.MeteredBlobStoreRepository;
import org.elasticsearch.snapshots.Snapshot;
import org.elasticsearch.threadpool.ThreadPool;

public class RepositoriesService
extends AbstractLifecycleComponent
implements ClusterStateApplier {
    private static final Logger logger = LogManager.getLogger(RepositoriesService.class);
    public static final Setting<TimeValue> REPOSITORIES_STATS_ARCHIVE_RETENTION_PERIOD = Setting.positiveTimeSetting("repositories.stats.archive.retention_period", TimeValue.timeValueHours((long)2L), Setting.Property.NodeScope);
    public static final Setting<Integer> REPOSITORIES_STATS_ARCHIVE_MAX_ARCHIVED_STATS = Setting.intSetting("repositories.stats.archive.max_archived_stats", 100, 0, Setting.Property.NodeScope);
    private final Map<String, Repository.Factory> typesRegistry;
    private final Map<String, Repository.Factory> internalTypesRegistry;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final NodeClient client;
    private final Map<String, Repository> internalRepositories = ConcurrentCollections.newConcurrentMap();
    private volatile Map<String, Repository> repositories = Collections.emptyMap();
    private final RepositoriesStatsArchive repositoriesStatsArchive;
    private final List<BiConsumer<Snapshot, IndexVersion>> preRestoreChecks;
    public static String COUNT_USAGE_STATS_NAME = "count";

    public RepositoriesService(Settings settings, ClusterService clusterService, Map<String, Repository.Factory> typesRegistry, Map<String, Repository.Factory> internalTypesRegistry, ThreadPool threadPool, NodeClient client, List<BiConsumer<Snapshot, IndexVersion>> preRestoreChecks) {
        this.typesRegistry = typesRegistry;
        this.internalTypesRegistry = internalTypesRegistry;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.client = client;
        if ((DiscoveryNode.canContainData(settings) || DiscoveryNode.isMasterNode(settings)) && !RepositoriesService.isDedicatedVotingOnlyNode(DiscoveryNode.getRolesFromSettings(settings))) {
            clusterService.addHighPriorityApplier(this);
        }
        this.repositoriesStatsArchive = new RepositoriesStatsArchive(REPOSITORIES_STATS_ARCHIVE_RETENTION_PERIOD.get(settings), REPOSITORIES_STATS_ARCHIVE_MAX_ARCHIVED_STATS.get(settings), threadPool.relativeTimeInMillisSupplier());
        this.preRestoreChecks = preRestoreChecks;
    }

    public void registerRepository(final PutRepositoryRequest request, ActionListener<AcknowledgedResponse> responseListener) {
        record RegisterRepositoryTaskResult(AcknowledgedResponse ackResponse, boolean changed) {
        }
        assert (this.lifecycle.started()) : "Trying to register new repository but service is in state [" + this.lifecycle.state() + "]";
        RepositoriesService.validateRepositoryName(request.name());
        SubscribableListener.newForked(validationStep -> this.validatePutRepositoryRequest(request, (ActionListener<Void>)validationStep)).andThen(clusterUpdateStep -> {
            ListenableFuture acknowledgementStep = new ListenableFuture();
            final ListenableFuture<Boolean> publicationStep = new ListenableFuture<Boolean>();
            this.submitUnbatchedTask("put_repository [" + request.name() + "]", new RegisterRepositoryTask(this, request, acknowledgementStep){

                @Override
                public void onFailure(Exception e) {
                    logger.warn(() -> "failed to create repository [" + request.name() + "]", (Throwable)e);
                    publicationStep.onFailure(e);
                    super.onFailure(e);
                }

                @Override
                public boolean mustAck(DiscoveryNode discoveryNode) {
                    return discoveryNode.isMasterNode() || discoveryNode.canContainData();
                }

                @Override
                public void clusterStateProcessed(ClusterState oldState, ClusterState newState) {
                    if (this.changed) {
                        if (this.found) {
                            logger.info("updated repository [{}]", (Object)request.name());
                        } else {
                            logger.info("put repository [{}]", (Object)request.name());
                        }
                    }
                    publicationStep.onResponse(oldState != newState);
                }
            });
            publicationStep.addListener(clusterUpdateStep.delegateFailureAndWrap((stateChangeListener, changed) -> acknowledgementStep.addListener(stateChangeListener.map(acknowledgedResponse -> new RegisterRepositoryTaskResult((AcknowledgedResponse)acknowledgedResponse, (boolean)changed)))));
        }).andThen((verificationStep, taskResult) -> {
            if (!request.verify()) {
                verificationStep.onResponse(taskResult.ackResponse);
            } else {
                SubscribableListener.newForked(verifyRepositoryStep -> {
                    if (taskResult.ackResponse.isAcknowledged() && taskResult.changed) {
                        this.verifyRepository(request.name(), (ActionListener<List<DiscoveryNode>>)verifyRepositoryStep);
                    } else {
                        verifyRepositoryStep.onResponse(null);
                    }
                }).andThen(getRepositoryDataStep -> this.threadPool.generic().execute(ActionRunnable.wrap(getRepositoryDataStep, ll -> this.repository(request.name()).getRepositoryData(EsExecutors.DIRECT_EXECUTOR_SERVICE, (ActionListener<RepositoryData>)ll)))).andThen((updateRepoUuidStep, repositoryData) -> RepositoriesService.updateRepositoryUuidInMetadata(this.clusterService, request.name(), repositoryData, updateRepoUuidStep)).andThenApply(uuidUpdated -> taskResult.ackResponse).addListener(verificationStep);
            }
        }).addListener(responseListener);
    }

    public void validateRepositoryCanBeCreated(PutRepositoryRequest request) {
        RepositoryMetadata newRepositoryMetadata = new RepositoryMetadata(request.name(), request.type(), request.settings());
        RepositoriesService.closeRepository(this.createRepository(newRepositoryMetadata));
    }

    private void validatePutRepositoryRequest(PutRepositoryRequest request, ActionListener<Void> resultListener) {
        RepositoryMetadata newRepositoryMetadata = new RepositoryMetadata(request.name(), request.type(), request.settings());
        try {
            Repository repository = this.createRepository(newRepositoryMetadata);
            if (request.verify()) {
                this.threadPool.executor("snapshot").execute(ActionRunnable.run(ActionListener.runBefore(resultListener, () -> RepositoriesService.closeRepository(repository)), (CheckedRunnable<Exception>)((CheckedRunnable)() -> {
                    String token = repository.startVerification();
                    if (token != null) {
                        repository.verify(token, this.clusterService.localNode());
                        repository.endVerification(token);
                    }
                })));
            } else {
                RepositoriesService.closeRepository(repository);
                resultListener.onResponse(null);
            }
        }
        catch (Exception e) {
            resultListener.onFailure(e);
        }
    }

    private void submitUnbatchedTask(String source, ClusterStateUpdateTask task) {
        RepositoriesService.submitUnbatchedTask(this.clusterService, source, task);
    }

    @SuppressForbidden(reason="legacy usage of unbatched task")
    private static void submitUnbatchedTask(ClusterService clusterService, String source, ClusterStateUpdateTask task) {
        clusterService.submitUnbatchedStateUpdateTask(source, task);
    }

    public static void updateRepositoryUuidInMetadata(ClusterService clusterService, final String repositoryName, RepositoryData repositoryData, final ActionListener<Void> listener) {
        final String repositoryUuid = repositoryData.getUuid();
        if (repositoryUuid.equals("_na_")) {
            listener.onResponse(null);
            return;
        }
        RepositoryMetadata repositoryMetadata = RepositoriesMetadata.get(clusterService.state()).repository(repositoryName);
        if (repositoryMetadata == null || repositoryMetadata.uuid().equals(repositoryUuid)) {
            listener.onResponse(null);
            return;
        }
        logger.info(org.elasticsearch.common.Strings.format("Registering repository [%s] with repository UUID [%s] and generation [%d]", repositoryName, repositoryData.getUuid(), repositoryData.getGenId()));
        RepositoriesService.submitUnbatchedTask(clusterService, "update repository UUID [" + repositoryName + "] to [" + repositoryUuid + "]", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                RepositoriesMetadata currentReposMetadata = RepositoriesMetadata.get(currentState);
                RepositoryMetadata repositoryMetadata = currentReposMetadata.repository(repositoryName);
                if (repositoryMetadata == null || repositoryMetadata.uuid().equals(repositoryUuid)) {
                    return currentState;
                }
                RepositoriesMetadata newReposMetadata = currentReposMetadata.withUuid(repositoryName, repositoryUuid);
                Metadata.Builder metadata = Metadata.builder(currentState.metadata()).putCustom("repositories", newReposMetadata);
                return ClusterState.builder(currentState).metadata(metadata).build();
            }

            @Override
            public void onFailure(Exception e) {
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(ClusterState oldState, ClusterState newState) {
                listener.onResponse(null);
            }
        });
    }

    public void unregisterRepository(DeleteRepositoryRequest request, ActionListener<AcknowledgedResponse> listener) {
        this.submitUnbatchedTask("delete_repository [" + request.name() + "]", new UnregisterRepositoryTask(request, listener){

            @Override
            public void clusterStateProcessed(ClusterState oldState, ClusterState newState) {
                if (!this.deletedRepositories.isEmpty()) {
                    logger.info("deleted repositories [{}]", (Object)this.deletedRepositories);
                }
            }

            @Override
            public boolean mustAck(DiscoveryNode discoveryNode) {
                return discoveryNode.isMasterNode() || discoveryNode.canContainData();
            }
        });
    }

    public void verifyRepository(final String repositoryName, ActionListener<List<DiscoveryNode>> listener) {
        final Repository repository = this.repository(repositoryName);
        this.threadPool.executor("snapshot").execute(new ActionRunnable<List<DiscoveryNode>>(listener){

            @Override
            protected void doRun() {
                String verificationToken = repository.startVerification();
                if (verificationToken != null) {
                    try {
                        VerifyNodeRepositoryAction.Request nodeRequest = new VerifyNodeRepositoryAction.Request(repositoryName, verificationToken);
                        RepositoriesService.this.client.execute(VerifyNodeRepositoryCoordinationAction.TYPE, nodeRequest, this.listener.delegateFailure((delegatedListener, response) -> RepositoriesService.this.threadPool.executor("snapshot").execute(() -> {
                            try {
                                repository.endVerification(verificationToken);
                            }
                            catch (Exception e) {
                                logger.warn(() -> "[" + repositoryName + "] failed to finish repository verification", (Throwable)e);
                                delegatedListener.onFailure(e);
                                return;
                            }
                            delegatedListener.onResponse(response.nodes);
                        })));
                    }
                    catch (Exception e) {
                        RepositoriesService.this.threadPool.executor("snapshot").execute(() -> {
                            try {
                                repository.endVerification(verificationToken);
                            }
                            catch (Exception inner) {
                                inner.addSuppressed(e);
                                logger.warn(() -> "[" + repositoryName + "] failed to finish repository verification", (Throwable)inner);
                            }
                            this.listener.onFailure(e);
                        });
                    }
                } else {
                    this.listener.onResponse(Collections.emptyList());
                }
            }
        });
    }

    public static boolean isDedicatedVotingOnlyNode(Set<DiscoveryNodeRole> roles) {
        return roles.contains(DiscoveryNodeRole.MASTER_ROLE) && roles.stream().noneMatch(DiscoveryNodeRole::canContainData) && roles.contains(DiscoveryNodeRole.VOTING_ONLY_NODE_ROLE);
    }

    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        try {
            RepositoriesMetadata newMetadata;
            ClusterState state = event.state();
            assert (RepositoriesService.assertReadonlyRepositoriesNotInUseForWrites(state));
            RepositoriesMetadata oldMetadata = RepositoriesMetadata.get(event.previousState());
            if (oldMetadata.equalsIgnoreGenerations(newMetadata = RepositoriesMetadata.get(state))) {
                for (Repository repo : this.repositories.values()) {
                    repo.updateState(state);
                }
                return;
            }
            logger.trace("processing new index repositories for state version [{}]", (Object)event.state().version());
            HashMap<String, Repository> survivors = new HashMap<String, Repository>();
            for (Map.Entry<String, Repository> entry : this.repositories.entrySet()) {
                if (newMetadata.repository(entry.getKey()) == null) {
                    logger.debug("unregistering repository [{}]", (Object)entry.getKey());
                    Repository repository = entry.getValue();
                    RepositoriesService.closeRepository(repository);
                    this.archiveRepositoryStats(repository, state.version());
                    continue;
                }
                survivors.put(entry.getKey(), entry.getValue());
            }
            HashMap<String, Repository> builder = new HashMap<String, Repository>();
            for (RepositoryMetadata repositoryMetadata : newMetadata.repositories()) {
                Repository repository = (Repository)survivors.get(repositoryMetadata.name());
                if (repository != null) {
                    if (!RepositoriesService.canUpdateInPlace(repositoryMetadata, repository)) {
                        logger.debug("updating repository [{}]", (Object)repositoryMetadata.name());
                        RepositoriesService.closeRepository(repository);
                        this.archiveRepositoryStats(repository, state.version());
                        repository = null;
                        try {
                            repository = RepositoriesService.createRepository(repositoryMetadata, this.typesRegistry, RepositoriesService::createUnknownTypeRepository);
                        }
                        catch (RepositoryException ex) {
                            logger.warn(() -> "failed to change repository [" + repositoryMetadata.name() + "]", (Throwable)ex);
                            repository = new InvalidRepository(repositoryMetadata, ex);
                        }
                    }
                } else {
                    try {
                        repository = RepositoriesService.createRepository(repositoryMetadata, this.typesRegistry, RepositoriesService::createUnknownTypeRepository);
                    }
                    catch (RepositoryException ex) {
                        logger.warn(() -> "failed to create repository [" + repositoryMetadata.name() + "]", (Throwable)ex);
                        repository = new InvalidRepository(repositoryMetadata, ex);
                    }
                }
                assert (repository != null) : "repository should not be null here";
                logger.debug("registering repository [{}]", (Object)repositoryMetadata.name());
                builder.put(repositoryMetadata.name(), repository);
            }
            for (Repository repo : builder.values()) {
                repo.updateState(state);
            }
            this.repositories = Collections.unmodifiableMap(builder);
        }
        catch (Exception ex) {
            assert (false) : new AssertionError((Object)ex);
            logger.warn("failure updating cluster state ", (Throwable)ex);
        }
    }

    private static boolean canUpdateInPlace(RepositoryMetadata updatedMetadata, Repository repository) {
        assert (updatedMetadata.name().equals(repository.getMetadata().name()));
        return repository.getMetadata().type().equals(updatedMetadata.type()) && repository.canUpdateInPlace(updatedMetadata.settings(), Collections.emptySet());
    }

    public void getRepositoryData(String repositoryName, ActionListener<RepositoryData> listener) {
        try {
            Repository repository = this.repository(repositoryName);
            assert (repository != null);
            repository.getRepositoryData(EsExecutors.DIRECT_EXECUTOR_SERVICE, listener);
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    public Repository repository(String repositoryName) {
        Repository repository = this.repositories.get(repositoryName);
        if (repository != null) {
            return repository;
        }
        repository = this.internalRepositories.get(repositoryName);
        if (repository != null) {
            return repository;
        }
        throw new RepositoryMissingException(repositoryName);
    }

    public Map<String, Repository> getRepositories() {
        return Collections.unmodifiableMap(this.repositories);
    }

    public List<RepositoryStatsSnapshot> repositoriesStats() {
        List<RepositoryStatsSnapshot> archivedRepoStats = this.repositoriesStatsArchive.getArchivedStats();
        List<RepositoryStatsSnapshot> activeRepoStats = this.getRepositoryStatsForActiveRepositories();
        ArrayList<RepositoryStatsSnapshot> repositoriesStats = new ArrayList<RepositoryStatsSnapshot>(archivedRepoStats);
        repositoriesStats.addAll(activeRepoStats);
        return repositoriesStats;
    }

    public RepositoriesStats getRepositoriesThrottlingStats() {
        return new RepositoriesStats(this.repositories.values().stream().collect(Collectors.toMap(r -> r.getMetadata().name(), r -> new RepositoriesStats.ThrottlingStats(r.getRestoreThrottleTimeInNanos(), r.getSnapshotThrottleTimeInNanos()))));
    }

    private List<RepositoryStatsSnapshot> getRepositoryStatsForActiveRepositories() {
        return Stream.concat(this.repositories.values().stream(), this.internalRepositories.values().stream()).filter(r -> r instanceof MeteredBlobStoreRepository).map(r -> (MeteredBlobStoreRepository)r).map(MeteredBlobStoreRepository::statsSnapshot).toList();
    }

    public List<RepositoryStatsSnapshot> clearRepositoriesStatsArchive(long maxVersionToClear) {
        return this.repositoriesStatsArchive.clear(maxVersionToClear);
    }

    public void registerInternalRepository(String name, String type) {
        RepositoryMetadata metadata = new RepositoryMetadata(name, type, Settings.EMPTY);
        Repository repository = this.internalRepositories.computeIfAbsent(name, n -> {
            logger.debug("put internal repository [{}][{}]", (Object)name, (Object)type);
            return RepositoriesService.createRepository(metadata, this.internalTypesRegistry, RepositoriesService::throwRepositoryTypeDoesNotExists);
        });
        if (!type.equals(repository.getMetadata().type())) {
            logger.warn(() -> Strings.format((String)"internal repository [%s][%s] already registered. this prevented the registration of internal repository [%s][%s].", (Object[])new Object[]{name, repository.getMetadata().type(), name, type}));
        } else if (this.repositories.containsKey(name)) {
            logger.warn(() -> Strings.format((String)"non-internal repository [%s] already registered. this repository will block the usage of internal repository [%s][%s].", (Object[])new Object[]{name, metadata.type(), name}));
        }
    }

    public void unregisterInternalRepository(String name) {
        Repository repository = this.internalRepositories.remove(name);
        if (repository != null) {
            RepositoryMetadata metadata = repository.getMetadata();
            logger.debug(() -> Strings.format((String)"delete internal repository [%s][%s].", (Object[])new Object[]{metadata.type(), name}));
            RepositoriesService.closeRepository(repository);
        }
    }

    private static void closeRepository(Repository repository) {
        logger.debug("closing repository [{}][{}]", (Object)repository.getMetadata().type(), (Object)repository.getMetadata().name());
        repository.close();
    }

    private void archiveRepositoryStats(Repository repository, long clusterStateVersion) {
        RepositoryStatsSnapshot stats;
        if (repository instanceof MeteredBlobStoreRepository && !this.repositoriesStatsArchive.archive(stats = ((MeteredBlobStoreRepository)repository).statsSnapshotForArchival(clusterStateVersion))) {
            logger.warn("Unable to archive the repository stats [{}] as the archive is full.", (Object)stats);
        }
    }

    private static Repository createRepository(RepositoryMetadata repositoryMetadata, Map<String, Repository.Factory> factories, Function<RepositoryMetadata, Repository> defaultFactory) {
        logger.debug("creating repository [{}][{}]", (Object)repositoryMetadata.type(), (Object)repositoryMetadata.name());
        Repository.Factory factory = factories.get(repositoryMetadata.type());
        if (factory == null) {
            return defaultFactory.apply(repositoryMetadata);
        }
        Repository repository = null;
        try {
            repository = factory.create(repositoryMetadata, factories::get);
            repository.start();
            return repository;
        }
        catch (Exception e) {
            IOUtils.closeWhileHandlingException(repository);
            logger.warn(() -> Strings.format((String)"failed to create repository [%s][%s]", (Object[])new Object[]{repositoryMetadata.type(), repositoryMetadata.name()}), (Throwable)e);
            throw new RepositoryException(repositoryMetadata.name(), "failed to create repository", e, new Object[0]);
        }
    }

    public Repository createRepository(RepositoryMetadata repositoryMetadata) {
        return RepositoriesService.createRepository(repositoryMetadata, this.typesRegistry, RepositoriesService::throwRepositoryTypeDoesNotExists);
    }

    private static Repository throwRepositoryTypeDoesNotExists(RepositoryMetadata repositoryMetadata) {
        throw new RepositoryException(repositoryMetadata.name(), "repository type [" + repositoryMetadata.type() + "] does not exist", new Object[0]);
    }

    private static Repository createUnknownTypeRepository(RepositoryMetadata repositoryMetadata) {
        logger.warn("[{}] repository type [{}] is unknown; ensure that all required plugins are installed on this node", (Object)repositoryMetadata.name(), (Object)repositoryMetadata.type());
        return new UnknownTypeRepository(repositoryMetadata);
    }

    public static void validateRepositoryName(String repositoryName) {
        if (!org.elasticsearch.common.Strings.hasLength(repositoryName)) {
            throw new RepositoryException(repositoryName, "cannot be empty", new Object[0]);
        }
        if (repositoryName.contains("#")) {
            throw new RepositoryException(repositoryName, "must not contain '#'", new Object[0]);
        }
        if (!org.elasticsearch.common.Strings.validFileName(repositoryName)) {
            throw new RepositoryException(repositoryName, "must not contain the following characters " + org.elasticsearch.common.Strings.INVALID_FILENAME_CHARS, new Object[0]);
        }
    }

    private static void ensureRepositoryNotInUseForWrites(ClusterState clusterState, String repository) {
        if (!SnapshotsInProgress.get(clusterState).forRepo(repository).isEmpty()) {
            throw RepositoriesService.newRepositoryConflictException(repository, "snapshot is in progress");
        }
        for (SnapshotDeletionsInProgress.Entry entry : SnapshotDeletionsInProgress.get(clusterState).getEntries()) {
            if (!entry.repository().equals(repository)) continue;
            throw RepositoriesService.newRepositoryConflictException(repository, "snapshot deletion is in progress");
        }
        for (RepositoryCleanupInProgress.Entry entry : RepositoryCleanupInProgress.get(clusterState).entries()) {
            if (!entry.repository().equals(repository)) continue;
            throw RepositoriesService.newRepositoryConflictException(repository, "repository clean up is in progress");
        }
    }

    private static void ensureRepositoryNotInUse(ClusterState clusterState, String repository) {
        RepositoriesService.ensureRepositoryNotInUseForWrites(clusterState, repository);
        for (RestoreInProgress.Entry entry : RestoreInProgress.get(clusterState)) {
            if (!repository.equals(entry.snapshot().getRepository())) continue;
            throw RepositoriesService.newRepositoryConflictException(repository, "snapshot restore is in progress");
        }
    }

    public static boolean isReadOnly(Settings repositorySettings) {
        return Boolean.TRUE.equals(repositorySettings.getAsBoolean("readonly", null));
    }

    private static boolean assertReadonlyRepositoriesNotInUseForWrites(ClusterState clusterState) {
        for (RepositoryMetadata repositoryMetadata : RepositoriesMetadata.get(clusterState).repositories()) {
            if (!RepositoriesService.isReadOnly(repositoryMetadata.settings())) continue;
            try {
                RepositoriesService.ensureRepositoryNotInUseForWrites(clusterState, repositoryMetadata.name());
            }
            catch (Exception e) {
                throw new AssertionError("repository [" + repositoryMetadata + "] is readonly but still in use", e);
            }
        }
        return true;
    }

    private static void rejectInvalidReadonlyFlagChange(RepositoryMetadata existingRepositoryMetadata, Settings newSettings) {
        if (RepositoriesService.isReadOnly(newSettings) && !RepositoriesService.isReadOnly(existingRepositoryMetadata.settings()) && existingRepositoryMetadata.generation() >= -1L && existingRepositoryMetadata.generation() != existingRepositoryMetadata.pendingGeneration()) {
            throw RepositoriesService.newRepositoryConflictException(existingRepositoryMetadata.name(), org.elasticsearch.common.Strings.format("currently updating root blob generation from [%d] to [%d], cannot update readonly flag", existingRepositoryMetadata.generation(), existingRepositoryMetadata.pendingGeneration()));
        }
    }

    private static void ensureNoSearchableSnapshotsIndicesInUse(ClusterState clusterState, RepositoryMetadata repositoryMetadata) {
        long count = 0L;
        ArrayList<Index> indices = null;
        for (IndexMetadata indexMetadata : clusterState.metadata()) {
            if (!RepositoriesService.indexSettingsMatchRepositoryMetadata(indexMetadata, repositoryMetadata)) continue;
            if (indices == null) {
                indices = new ArrayList<Index>();
            }
            if (indices.size() < 5) {
                indices.add(indexMetadata.getIndex());
            }
            ++count;
        }
        if (indices != null && !indices.isEmpty()) {
            throw RepositoriesService.newRepositoryConflictException(repositoryMetadata.name(), "found " + count + " searchable snapshots indices that use the repository: " + org.elasticsearch.common.Strings.collectionToCommaDelimitedString(indices) + (count > (long)indices.size() ? ",..." : ""));
        }
    }

    private static boolean indexSettingsMatchRepositoryMetadata(IndexMetadata indexMetadata, RepositoryMetadata repositoryMetadata) {
        if (indexMetadata.isSearchableSnapshot()) {
            Settings indexSettings = indexMetadata.getSettings();
            String indexRepositoryUuid = indexSettings.get("index.store.snapshot.repository_uuid");
            if (org.elasticsearch.common.Strings.hasLength(indexRepositoryUuid)) {
                return Objects.equals(repositoryMetadata.uuid(), indexRepositoryUuid);
            }
            return Objects.equals(repositoryMetadata.name(), indexSettings.get("index.store.snapshot.repository_name"));
        }
        return false;
    }

    private static RepositoryConflictException newRepositoryConflictException(String repository, String reason) {
        return new RepositoryConflictException(repository, "trying to modify or unregister repository that is currently used (" + reason + ")", "trying to modify or unregister repository [" + repository + "] that is currently used (" + reason + ")");
    }

    public List<BiConsumer<Snapshot, IndexVersion>> getPreRestoreVersionChecks() {
        return this.preRestoreChecks;
    }

    public RepositoryUsageStats getUsageStats() {
        if (this.repositories.isEmpty()) {
            return RepositoryUsageStats.EMPTY;
        }
        HashMap<String, Map> statsByType = new HashMap<String, Map>();
        for (Repository repository : this.repositories.values()) {
            String repositoryType = repository.getMetadata().type();
            Map typeStats = statsByType.computeIfAbsent(repositoryType, ignored -> new HashMap());
            typeStats.compute(COUNT_USAGE_STATS_NAME, (k, count) -> (count == null ? 0L : count) + 1L);
            Set<String> repositoryUsageTags = repository.getUsageFeatures();
            assert (!repositoryUsageTags.contains(COUNT_USAGE_STATS_NAME)) : repositoryUsageTags;
            for (String repositoryUsageTag : repositoryUsageTags) {
                typeStats.compute(repositoryUsageTag, (k, count) -> (count == null ? 0L : count) + 1L);
            }
        }
        return new RepositoryUsageStats(statsByType.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> Map.copyOf((Map)e.getValue()))));
    }

    @Override
    protected void doStart() {
    }

    @Override
    protected void doStop() {
    }

    @Override
    protected void doClose() throws IOException {
        this.clusterService.removeApplier(this);
        ArrayList<Repository> repos = new ArrayList<Repository>();
        repos.addAll(this.internalRepositories.values());
        repos.addAll(this.repositories.values());
        IOUtils.close(repos);
        for (Repository repo : repos) {
            repo.awaitIdle();
        }
    }

    public static class UnregisterRepositoryTask
    extends AckedClusterStateUpdateTask {
        protected final List<String> deletedRepositories = new ArrayList<String>();
        private final DeleteRepositoryRequest request;

        UnregisterRepositoryTask(DeleteRepositoryRequest request, ActionListener<AcknowledgedResponse> listener) {
            super(request, listener);
            this.request = request;
        }

        public UnregisterRepositoryTask(TimeValue dummyTimeout, String name) {
            this(new DeleteRepositoryRequest(dummyTimeout, dummyTimeout, name), (ActionListener<AcknowledgedResponse>)null);
        }

        @Override
        public ClusterState execute(ClusterState currentState) {
            Metadata.Builder mdBuilder = Metadata.builder(currentState.metadata());
            RepositoriesMetadata repositories = RepositoriesMetadata.get(currentState);
            if (repositories.repositories().size() > 0) {
                ArrayList<RepositoryMetadata> repositoriesMetadata = new ArrayList<RepositoryMetadata>(repositories.repositories().size());
                boolean changed = false;
                for (RepositoryMetadata repositoryMetadata : repositories.repositories()) {
                    if (Regex.simpleMatch(this.request.name(), repositoryMetadata.name())) {
                        RepositoriesService.ensureRepositoryNotInUse(currentState, repositoryMetadata.name());
                        RepositoriesService.ensureNoSearchableSnapshotsIndicesInUse(currentState, repositoryMetadata);
                        this.deletedRepositories.add(repositoryMetadata.name());
                        changed = true;
                        continue;
                    }
                    repositoriesMetadata.add(repositoryMetadata);
                }
                if (changed) {
                    repositories = new RepositoriesMetadata(repositoriesMetadata);
                    mdBuilder.putCustom("repositories", repositories);
                    return ClusterState.builder(currentState).metadata(mdBuilder).build();
                }
            }
            if (Regex.isMatchAllPattern(this.request.name())) {
                return currentState;
            }
            throw new RepositoryMissingException(this.request.name());
        }
    }

    public static class RegisterRepositoryTask
    extends AckedClusterStateUpdateTask {
        protected boolean found = false;
        protected boolean changed = false;
        private final PutRepositoryRequest request;
        private final RepositoriesService repositoriesService;

        RegisterRepositoryTask(RepositoriesService repositoriesService, PutRepositoryRequest request, ListenableFuture<AcknowledgedResponse> acknowledgementStep) {
            super(request, acknowledgementStep);
            this.repositoriesService = repositoriesService;
            this.request = request;
        }

        public RegisterRepositoryTask(RepositoriesService repositoriesService, PutRepositoryRequest request) {
            this(repositoriesService, request, null);
        }

        @Override
        public ClusterState execute(ClusterState currentState) {
            Metadata.Builder mdBuilder = Metadata.builder(currentState.metadata());
            RepositoriesMetadata repositories = RepositoriesMetadata.get(currentState);
            ArrayList<RepositoryMetadata> repositoriesMetadata = new ArrayList<RepositoryMetadata>(repositories.repositories().size() + 1);
            for (RepositoryMetadata repositoryMetadata : repositories.repositories()) {
                if (repositoryMetadata.name().equals(this.request.name())) {
                    RepositoryMetadata updatedMetadata;
                    RepositoriesService.rejectInvalidReadonlyFlagChange(repositoryMetadata, this.request.settings());
                    RepositoryMetadata newRepositoryMetadata = new RepositoryMetadata(this.request.name(), repositoryMetadata.uuid(), this.request.type(), this.request.settings());
                    Repository existing = this.repositoriesService.repositories.get(this.request.name());
                    if (existing == null) {
                        existing = this.repositoriesService.internalRepositories.get(this.request.name());
                    }
                    assert (existing != null) : "repository [" + newRepositoryMetadata.name() + "] must exist";
                    assert (existing.getMetadata() == repositoryMetadata);
                    if (RepositoriesService.canUpdateInPlace(newRepositoryMetadata, existing)) {
                        if (repositoryMetadata.settings().equals(newRepositoryMetadata.settings())) {
                            if (repositoryMetadata.generation() == -3L) {
                                RepositoriesService.ensureRepositoryNotInUse(currentState, this.request.name());
                                logger.info("repository [{}/{}] is marked as corrupted, resetting the corruption marker", (Object)repositoryMetadata.name(), (Object)repositoryMetadata.uuid());
                                repositoryMetadata = repositoryMetadata.withGeneration(-2L, repositoryMetadata.pendingGeneration());
                            } else {
                                return currentState;
                            }
                        }
                        updatedMetadata = repositoryMetadata.withSettings(newRepositoryMetadata.settings());
                    } else {
                        RepositoriesService.ensureRepositoryNotInUse(currentState, this.request.name());
                        updatedMetadata = newRepositoryMetadata;
                    }
                    this.found = true;
                    repositoriesMetadata.add(updatedMetadata);
                    continue;
                }
                repositoriesMetadata.add(repositoryMetadata);
            }
            if (!this.found) {
                repositoriesMetadata.add(new RepositoryMetadata(this.request.name(), this.request.type(), this.request.settings()));
            }
            repositories = new RepositoriesMetadata(repositoriesMetadata);
            mdBuilder.putCustom("repositories", repositories);
            this.changed = true;
            return ClusterState.builder(currentState).metadata(mdBuilder).build();
        }
    }
}

