/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.health.node;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.util.CollectionUtils;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.features.FeatureService;
import org.elasticsearch.health.Diagnosis;
import org.elasticsearch.health.HealthFeatures;
import org.elasticsearch.health.HealthIndicatorDetails;
import org.elasticsearch.health.HealthIndicatorImpact;
import org.elasticsearch.health.HealthIndicatorResult;
import org.elasticsearch.health.HealthIndicatorService;
import org.elasticsearch.health.HealthStatus;
import org.elasticsearch.health.ImpactArea;
import org.elasticsearch.health.node.DiskHealthInfo;
import org.elasticsearch.health.node.HealthIndicatorDisplayValues;
import org.elasticsearch.health.node.HealthInfo;
import org.elasticsearch.index.Index;

public class DiskHealthIndicatorService
implements HealthIndicatorService {
    public static final String NAME = "disk";
    private static final Logger logger = LogManager.getLogger(DiskHealthIndicatorService.class);
    private static final String IMPACT_INGEST_UNAVAILABLE_ID = "ingest_capability_unavailable";
    private static final String IMPACT_INGEST_AT_RISK_ID = "ingest_capability_at_risk";
    private static final String IMPACT_CLUSTER_STABILITY_AT_RISK_ID = "cluster_stability_at_risk";
    private static final String IMPACT_CLUSTER_FUNCTIONALITY_UNAVAILABLE_ID = "cluster_functionality_unavailable";
    private final ClusterService clusterService;
    private final FeatureService featureService;

    public DiskHealthIndicatorService(ClusterService clusterService, FeatureService featureService) {
        this.clusterService = clusterService;
        this.featureService = featureService;
    }

    @Override
    public String name() {
        return NAME;
    }

    @Override
    public HealthIndicatorResult calculate(boolean verbose, int maxAffectedResourcesCount, HealthInfo healthInfo) {
        ClusterState clusterState = this.clusterService.state();
        Map<String, DiskHealthInfo> diskHealthInfoMap = healthInfo.diskInfoByNode();
        if (diskHealthInfoMap == null || diskHealthInfoMap.isEmpty()) {
            if (!this.featureService.clusterHasFeature(clusterState, HealthFeatures.SUPPORTS_HEALTH)) {
                return this.createIndicator(HealthStatus.GREEN, "No disk usage data available. The cluster currently has mixed versions (an upgrade may be in progress).", HealthIndicatorDetails.EMPTY, List.of(), List.of());
            }
            return this.createIndicator(HealthStatus.UNKNOWN, "No disk usage data.", HealthIndicatorDetails.EMPTY, Collections.emptyList(), Collections.emptyList());
        }
        DiskHealthIndicatorService.logNodesMissingHealthInfo(diskHealthInfoMap, clusterState);
        DiskHealthAnalyzer diskHealthAnalyzer = new DiskHealthAnalyzer(diskHealthInfoMap, clusterState);
        return this.createIndicator(diskHealthAnalyzer.getHealthStatus(), diskHealthAnalyzer.getSymptom(), diskHealthAnalyzer.getDetails(verbose), diskHealthAnalyzer.getImpacts(), diskHealthAnalyzer.getDiagnoses(verbose, maxAffectedResourcesCount));
    }

    private static void logNodesMissingHealthInfo(Map<String, DiskHealthInfo> diskHealthInfoMap, ClusterState clusterState) {
        String nodesMissingHealthInfo;
        if (logger.isDebugEnabled() && !(nodesMissingHealthInfo = HealthIndicatorDisplayValues.getSortedUniqueValuesString(clusterState.getNodes().getAllNodes(), node -> !diskHealthInfoMap.containsKey(node.getId()), HealthIndicatorDisplayValues::getNodeName)).isBlank()) {
            logger.debug("The following nodes are in the cluster state but not reporting health data: [{}]", (Object)nodesMissingHealthInfo);
        }
    }

    static class DiskHealthAnalyzer {
        public static final String INDICES_WITH_READONLY_BLOCK = "indices_with_readonly_block";
        public static final String NODES_WITH_ENOUGH_DISK_SPACE = "nodes_with_enough_disk_space";
        public static final String NODES_OVER_FLOOD_STAGE_WATERMARK = "nodes_over_flood_stage_watermark";
        public static final String NODES_OVER_HIGH_WATERMARK = "nodes_over_high_watermark";
        public static final String NODES_WITH_UNKNOWN_DISK_STATUS = "nodes_with_unknown_disk_status";
        private final ClusterState clusterState;
        private final Set<String> blockedIndices;
        private final List<DiscoveryNode> dataNodes = new ArrayList<DiscoveryNode>();
        private final Map<HealthStatus, List<DiscoveryNode>> masterNodes = new EnumMap<HealthStatus, List<DiscoveryNode>>(HealthStatus.class);
        private final Map<HealthStatus, List<DiscoveryNode>> otherNodes = new EnumMap<HealthStatus, List<DiscoveryNode>>(HealthStatus.class);
        private final Set<DiscoveryNodeRole> affectedRoles = new HashSet<DiscoveryNodeRole>();
        private final Set<String> indicesAtRisk;
        private final HealthStatus healthStatus;
        private final Map<HealthStatus, Integer> healthStatusNodeCount;

        DiskHealthAnalyzer(Map<String, DiskHealthInfo> diskHealthByNode, ClusterState clusterState) {
            this.clusterState = clusterState;
            this.blockedIndices = clusterState.blocks().indices().entrySet().stream().filter(entry -> ((Set)entry.getValue()).contains(IndexMetadata.INDEX_READ_ONLY_ALLOW_DELETE_BLOCK)).map(Map.Entry::getKey).collect(Collectors.toSet());
            HealthStatus mostSevereStatusSoFar = this.blockedIndices.isEmpty() ? HealthStatus.GREEN : HealthStatus.RED;
            for (String string : diskHealthByNode.keySet()) {
                DiscoveryNode node = clusterState.getNodes().get(string);
                HealthStatus healthStatus = diskHealthByNode.get(string).healthStatus();
                if (node == null || !healthStatus.indicatesHealthProblem()) continue;
                if (mostSevereStatusSoFar.value() < healthStatus.value()) {
                    mostSevereStatusSoFar = healthStatus;
                }
                this.affectedRoles.addAll(node.getRoles());
                if (node.canContainData()) {
                    this.dataNodes.add(node);
                    continue;
                }
                if (node.isMasterNode()) {
                    this.masterNodes.computeIfAbsent(healthStatus, ignored -> new ArrayList()).add(node);
                    continue;
                }
                this.otherNodes.computeIfAbsent(healthStatus, ignored -> new ArrayList()).add(node);
            }
            this.dataNodes.sort(DiscoveryNode.DISCOVERY_NODE_COMPARATOR);
            for (List list : this.masterNodes.values()) {
                list.sort(DiscoveryNode.DISCOVERY_NODE_COMPARATOR);
            }
            for (List list : this.otherNodes.values()) {
                list.sort(DiscoveryNode.DISCOVERY_NODE_COMPARATOR);
            }
            this.indicesAtRisk = DiskHealthAnalyzer.getIndicesForNodes(this.dataNodes, clusterState);
            this.healthStatus = mostSevereStatusSoFar;
            this.healthStatusNodeCount = DiskHealthAnalyzer.countNodesByHealthStatus(diskHealthByNode, clusterState);
        }

        public HealthStatus getHealthStatus() {
            return this.healthStatus;
        }

        String getSymptom() {
            Object symptom;
            if (this.healthStatus == HealthStatus.GREEN) {
                return "The cluster has enough available disk space.";
            }
            if (this.hasBlockedIndices()) {
                symptom = String.format(Locale.ROOT, "%d %s %s not allowed to be updated.", this.blockedIndices.size(), HealthIndicatorDisplayValues.indices(this.blockedIndices.size()), HealthIndicatorDisplayValues.are(this.blockedIndices.size()));
                symptom = this.hasUnhealthyDataNodes() ? (String)symptom + String.format(Locale.ROOT, " %d %s %s out of disk or running low on disk space.", this.dataNodes.size(), HealthIndicatorDisplayValues.regularNoun("node", this.dataNodes.size()), HealthIndicatorDisplayValues.are(this.dataNodes.size())) : (String)symptom + " The cluster is recovering and ingest capabilities should be restored within a few minutes.";
                if (this.hasUnhealthyMasterNodes() || this.hasUnhealthyOtherNodes()) {
                    String roles = Stream.concat(this.masterNodes.values().stream(), this.otherNodes.values().stream()).flatMap(Collection::stream).flatMap(node -> node.getRoles().stream()).map(DiscoveryNodeRole::roleName).distinct().sorted().collect(Collectors.joining(", "));
                    int unhealthyNodesCount = DiskHealthAnalyzer.getUnhealthyNodeSize(this.masterNodes) + DiskHealthAnalyzer.getUnhealthyNodeSize(this.otherNodes);
                    symptom = (String)symptom + String.format(Locale.ROOT, " %d %s with roles: [%s] %s out of disk or running low on disk space.", unhealthyNodesCount, HealthIndicatorDisplayValues.regularNoun("node", unhealthyNodesCount), roles, HealthIndicatorDisplayValues.are(unhealthyNodesCount));
                }
            } else {
                String roles = HealthIndicatorDisplayValues.getSortedUniqueValuesString(this.affectedRoles, DiscoveryNodeRole::roleName);
                int unhealthyNodesCount = this.dataNodes.size() + DiskHealthAnalyzer.getUnhealthyNodeSize(this.masterNodes) + DiskHealthAnalyzer.getUnhealthyNodeSize(this.otherNodes);
                symptom = String.format(Locale.ROOT, "%d %s with roles: [%s] %s out of disk or running low on disk space.", unhealthyNodesCount, HealthIndicatorDisplayValues.regularNoun("node", unhealthyNodesCount), roles, HealthIndicatorDisplayValues.are(unhealthyNodesCount));
            }
            return symptom;
        }

        List<HealthIndicatorImpact> getImpacts() {
            String impactedOtherRoles;
            if (this.healthStatus == HealthStatus.GREEN) {
                return List.of();
            }
            ArrayList<HealthIndicatorImpact> impacts = new ArrayList<HealthIndicatorImpact>();
            if (this.hasBlockedIndices()) {
                impacts.add(new HealthIndicatorImpact(DiskHealthIndicatorService.NAME, DiskHealthIndicatorService.IMPACT_INGEST_UNAVAILABLE_ID, 1, String.format(Locale.ROOT, "Cannot insert or update documents in the affected indices [%s].", HealthIndicatorDisplayValues.getTruncatedIndices(this.blockedIndices, this.clusterState.getMetadata())), List.of(ImpactArea.INGEST)));
            } else {
                if (!this.indicesAtRisk.isEmpty()) {
                    impacts.add(new HealthIndicatorImpact(DiskHealthIndicatorService.NAME, DiskHealthIndicatorService.IMPACT_INGEST_AT_RISK_ID, 1, String.format(Locale.ROOT, "The cluster is at risk of not being able to insert or update documents in the affected indices [%s].", HealthIndicatorDisplayValues.getTruncatedIndices(this.indicesAtRisk, this.clusterState.metadata())), List.of(ImpactArea.INGEST)));
                }
                if (this.hasUnhealthyDataNodes()) {
                    impacts.add(new HealthIndicatorImpact(DiskHealthIndicatorService.NAME, DiskHealthIndicatorService.IMPACT_INGEST_AT_RISK_ID, 2, String.format(Locale.ROOT, "%d %s %s out of disk or running low on disk space. %s %s cannot be used to store data anymore.", this.dataNodes.size(), HealthIndicatorDisplayValues.regularNoun("node", this.dataNodes.size()), HealthIndicatorDisplayValues.are(this.dataNodes.size()), HealthIndicatorDisplayValues.these(this.dataNodes.size()), HealthIndicatorDisplayValues.regularNoun("node", this.dataNodes.size())), List.of(ImpactArea.DEPLOYMENT_MANAGEMENT)));
                }
            }
            if (this.affectedRoles.contains(DiscoveryNodeRole.MASTER_ROLE)) {
                impacts.add(new HealthIndicatorImpact(DiskHealthIndicatorService.NAME, DiskHealthIndicatorService.IMPACT_CLUSTER_STABILITY_AT_RISK_ID, 1, "Cluster stability might be impaired.", List.of(ImpactArea.DEPLOYMENT_MANAGEMENT)));
            }
            if (!(impactedOtherRoles = HealthIndicatorDisplayValues.getSortedUniqueValuesString(this.affectedRoles, role -> !role.canContainData() && !role.equals(DiscoveryNodeRole.MASTER_ROLE), DiscoveryNodeRole::roleName)).isBlank()) {
                impacts.add(new HealthIndicatorImpact(DiskHealthIndicatorService.NAME, DiskHealthIndicatorService.IMPACT_CLUSTER_FUNCTIONALITY_UNAVAILABLE_ID, 3, String.format(Locale.ROOT, "The [%s] functionality might be impaired.", impactedOtherRoles), List.of(ImpactArea.DEPLOYMENT_MANAGEMENT)));
            }
            return impacts;
        }

        private List<Diagnosis> getDiagnoses(boolean verbose, int size) {
            if (!verbose || this.healthStatus == HealthStatus.GREEN) {
                return List.of();
            }
            ArrayList<Diagnosis> diagnosisList = new ArrayList<Diagnosis>();
            if (this.hasBlockedIndices() || this.hasUnhealthyDataNodes()) {
                Set<String> affectedIndices = Sets.union(this.blockedIndices, this.indicesAtRisk);
                ArrayList<Diagnosis.Resource> affectedResources = new ArrayList<Diagnosis.Resource>();
                if (this.dataNodes.size() > 0) {
                    Diagnosis.Resource nodeResources = new Diagnosis.Resource(CollectionUtils.limitSize(this.dataNodes, size));
                    affectedResources.add(nodeResources);
                }
                if (affectedIndices.size() > 0) {
                    Diagnosis.Resource indexResources = new Diagnosis.Resource(Diagnosis.Resource.Type.INDEX, affectedIndices.stream().sorted(HealthIndicatorDisplayValues.indicesComparatorByPriorityAndName(this.clusterState.metadata())).limit(Math.min(affectedIndices.size(), size)).collect(Collectors.toList()));
                    affectedResources.add(indexResources);
                }
                diagnosisList.add(DiskHealthAnalyzer.createDataNodeDiagnosis(affectedIndices.size(), affectedResources));
            }
            if (this.masterNodes.containsKey(HealthStatus.RED)) {
                diagnosisList.add(DiskHealthAnalyzer.createNonDataNodeDiagnosis(HealthStatus.RED, this.masterNodes.get(HealthStatus.RED), size, true));
            }
            if (this.masterNodes.containsKey(HealthStatus.YELLOW)) {
                diagnosisList.add(DiskHealthAnalyzer.createNonDataNodeDiagnosis(HealthStatus.YELLOW, this.masterNodes.get(HealthStatus.YELLOW), size, true));
            }
            if (this.otherNodes.containsKey(HealthStatus.RED)) {
                diagnosisList.add(DiskHealthAnalyzer.createNonDataNodeDiagnosis(HealthStatus.RED, this.otherNodes.get(HealthStatus.RED), size, false));
            }
            if (this.otherNodes.containsKey(HealthStatus.YELLOW)) {
                diagnosisList.add(DiskHealthAnalyzer.createNonDataNodeDiagnosis(HealthStatus.YELLOW, this.otherNodes.get(HealthStatus.YELLOW), size, false));
            }
            return diagnosisList;
        }

        HealthIndicatorDetails getDetails(boolean verbose) {
            if (!verbose) {
                return HealthIndicatorDetails.EMPTY;
            }
            return (builder, params) -> {
                builder.startObject();
                builder.field(INDICES_WITH_READONLY_BLOCK, this.blockedIndices.size());
                for (HealthStatus healthStatus : HealthStatus.values()) {
                    builder.field(DiskHealthAnalyzer.getDetailsDisplayKey(healthStatus), this.healthStatusNodeCount.get(healthStatus));
                }
                return builder.endObject();
            };
        }

        static Map<HealthStatus, Integer> countNodesByHealthStatus(Map<String, DiskHealthInfo> diskHealthInfoMap, ClusterState clusterState) {
            EnumMap<HealthStatus, Integer> counts = new EnumMap<HealthStatus, Integer>(HealthStatus.class);
            for (HealthStatus healthStatus : HealthStatus.values()) {
                counts.put(healthStatus, 0);
            }
            for (DiscoveryNode node : clusterState.getNodes()) {
                HealthStatus healthStatus = diskHealthInfoMap.containsKey(node.getId()) ? diskHealthInfoMap.get(node.getId()).healthStatus() : HealthStatus.UNKNOWN;
                counts.computeIfPresent(healthStatus, (ignored, count) -> count + 1);
            }
            return counts;
        }

        private static String getDetailsDisplayKey(HealthStatus status) {
            return switch (status) {
                default -> throw new IncompatibleClassChangeError();
                case HealthStatus.GREEN -> NODES_WITH_ENOUGH_DISK_SPACE;
                case HealthStatus.UNKNOWN -> NODES_WITH_UNKNOWN_DISK_STATUS;
                case HealthStatus.YELLOW -> NODES_OVER_HIGH_WATERMARK;
                case HealthStatus.RED -> NODES_OVER_FLOOD_STAGE_WATERMARK;
            };
        }

        private boolean hasUnhealthyDataNodes() {
            return !this.dataNodes.isEmpty();
        }

        private boolean hasUnhealthyMasterNodes() {
            return !this.masterNodes.isEmpty();
        }

        private boolean hasUnhealthyOtherNodes() {
            return !this.otherNodes.isEmpty();
        }

        private boolean hasBlockedIndices() {
            return !this.blockedIndices.isEmpty();
        }

        static Set<String> getIndicesForNodes(List<DiscoveryNode> nodes, ClusterState clusterState) {
            RoutingNodes routingNodes = clusterState.getRoutingNodes();
            return nodes.stream().map(node -> routingNodes.node(node.getId())).filter(Objects::nonNull).flatMap(routingNode -> Arrays.stream(routingNode.copyIndices())).map(Index::getName).collect(Collectors.toSet());
        }

        static Diagnosis createDataNodeDiagnosis(int numberOfAffectedIndices, List<Diagnosis.Resource> affectedResources) {
            String message = numberOfAffectedIndices == 0 ? "Disk is almost full." : String.format(Locale.ROOT, "%d %s %s on nodes that have run or are likely to run out of disk space, this can temporarily disable writing on %s %s.", numberOfAffectedIndices, HealthIndicatorDisplayValues.indices(numberOfAffectedIndices), HealthIndicatorDisplayValues.regularVerb("reside", numberOfAffectedIndices), HealthIndicatorDisplayValues.these(numberOfAffectedIndices), HealthIndicatorDisplayValues.indices(numberOfAffectedIndices));
            return new Diagnosis(new Diagnosis.Definition(DiskHealthIndicatorService.NAME, "add_disk_capacity_data_nodes", message, "Enable autoscaling (if applicable), add disk capacity or free up disk space to resolve this. If you have already taken action please wait for the rebalancing to complete.", "https://ela.st/fix-data-disk"), affectedResources);
        }

        static Diagnosis createNonDataNodeDiagnosis(HealthStatus healthStatus, List<DiscoveryNode> nodes, int size, boolean isMaster) {
            return new Diagnosis(new Diagnosis.Definition(DiskHealthIndicatorService.NAME, isMaster ? "add_disk_capacity_master_nodes" : "add_disk_capacity", healthStatus == HealthStatus.RED ? "Disk is full." : "The cluster is running low on disk space.", "Please add capacity to the current nodes, or replace them with ones with higher capacity.", isMaster ? "https://ela.st/fix-master-disk" : "https://ela.st/fix-disk-space"), List.of(new Diagnosis.Resource(CollectionUtils.limitSize(nodes, size))));
        }

        private static int getUnhealthyNodeSize(Map<HealthStatus, List<DiscoveryNode>> nodes) {
            return (nodes.containsKey(HealthStatus.RED) ? nodes.get(HealthStatus.RED).size() : 0) + (nodes.containsKey(HealthStatus.YELLOW) ? nodes.get(HealthStatus.YELLOW).size() : 0);
        }
    }
}

