/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.time;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.elasticsearch.common.ReferenceDocs;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.time.DateUtilsRounding;
import org.elasticsearch.core.Predicates;

public class DateUtils {
    public static final long MAX_MILLIS_BEFORE_9999 = 253402300799999L;
    public static final long MAX_MILLIS_BEFORE_MINUS_9999 = -377705116800000L;
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(DateUtils.class);
    static final Map<String, String> DEPRECATED_SHORT_TIMEZONES;
    public static final Map<String, String> DEPRECATED_LONG_TIMEZONES;
    public static final Instant MAX_NANOSECOND_INSTANT;
    static final long MAX_NANOSECOND_IN_MILLIS;
    public static final long MAX_NANOSECOND;
    private static final boolean USES_COMPAT;
    private static final Predicate<String> LEGACY_DATE_FORMAT_MATCHER;
    private static final Predicate<String> CONTAINS_CHANGING_TEXT_SPECIFIERS;
    private static final Predicate<String> CONTAINS_WEEK_DATE_SPECIFIERS;

    public static ZoneId of(String zoneId) {
        String deprecatedId = DEPRECATED_SHORT_TIMEZONES.get(zoneId);
        if (deprecatedId != null) {
            deprecationLogger.critical(DeprecationCategory.PARSING, "timezone", "Use of short timezone id " + zoneId + " is deprecated. Use " + deprecatedId + " instead", new Object[0]);
            return ZoneId.of(deprecatedId);
        }
        return ZoneId.of(zoneId).normalized();
    }

    public static long toLong(Instant instant) {
        if (instant.isBefore(Instant.EPOCH)) {
            throw new IllegalArgumentException("date[" + instant + "] is before the epoch in 1970 and cannot be stored in nanosecond resolution");
        }
        if (instant.isAfter(MAX_NANOSECOND_INSTANT)) {
            throw new IllegalArgumentException("date[" + instant + "] is after 2262-04-11T23:47:16.854775807 and cannot be stored in nanosecond resolution");
        }
        return instant.getEpochSecond() * 1000000000L + (long)instant.getNano();
    }

    public static long toLongMillis(Instant instant) {
        try {
            return instant.toEpochMilli();
        }
        catch (ArithmeticException e) {
            if (instant.isAfter(Instant.now())) {
                throw new IllegalArgumentException("date[" + instant + "] is too far in the future to be represented in a long milliseconds variable", e);
            }
            throw new IllegalArgumentException("date[" + instant + "] is too far in the past to be represented in a long milliseconds variable", e);
        }
    }

    public static Instant clampToNanosRange(Instant instant) {
        if (instant.isBefore(Instant.EPOCH)) {
            return Instant.EPOCH;
        }
        if (instant.isAfter(MAX_NANOSECOND_INSTANT)) {
            return MAX_NANOSECOND_INSTANT;
        }
        return instant;
    }

    public static Instant toInstant(long nanoSecondsSinceEpoch) {
        if (nanoSecondsSinceEpoch < 0L) {
            throw new IllegalArgumentException("nanoseconds [" + nanoSecondsSinceEpoch + "] are before the epoch in 1970 and cannot be processed in nanosecond resolution");
        }
        if (nanoSecondsSinceEpoch == 0L) {
            return Instant.EPOCH;
        }
        long seconds = nanoSecondsSinceEpoch / 1000000000L;
        long nanos = nanoSecondsSinceEpoch % 1000000000L;
        return Instant.ofEpochSecond(seconds, nanos);
    }

    public static long toNanoSeconds(long milliSecondsSinceEpoch) {
        if (milliSecondsSinceEpoch < 0L) {
            throw new IllegalArgumentException("milliSeconds [" + milliSecondsSinceEpoch + "] are before the epoch in 1970 and cannot be converted to nanoseconds");
        }
        if (milliSecondsSinceEpoch > MAX_NANOSECOND_IN_MILLIS) {
            throw new IllegalArgumentException("milliSeconds [" + milliSecondsSinceEpoch + "] are after 2262-04-11T23:47:16.854775807 and cannot be converted to nanoseconds");
        }
        return milliSecondsSinceEpoch * 1000000L;
    }

    public static long toMilliSeconds(long nanoSecondsSinceEpoch) {
        if (nanoSecondsSinceEpoch < 0L) {
            throw new IllegalArgumentException("nanoseconds are [" + nanoSecondsSinceEpoch + "] are before the epoch in 1970 and cannot be converted to milliseconds");
        }
        if (nanoSecondsSinceEpoch == 0L) {
            return 0L;
        }
        return nanoSecondsSinceEpoch / 1000000L;
    }

    public static int compareNanosToMillis(long nanos, long millis) {
        assert (nanos >= 0L);
        if (millis < 0L) {
            return 1;
        }
        if (millis > MAX_NANOSECOND_IN_MILLIS) {
            return -1;
        }
        long diff = nanos - millis * 1000000L;
        return diff == 0L ? 0 : (diff < 0L ? -1 : 1);
    }

    public static long roundFloor(long utcMillis, long unitMillis) {
        if (utcMillis >= 0L) {
            return utcMillis - utcMillis % unitMillis;
        }
        return ++utcMillis - utcMillis % unitMillis - unitMillis;
    }

    public static long roundQuarterOfYear(long utcMillis) {
        int year = DateUtilsRounding.getYear(utcMillis);
        int month = DateUtilsRounding.getMonthOfYear(utcMillis, year);
        int firstMonthOfQuarter = (month - 1) / 3 * 3 + 1;
        return DateUtils.of(year, firstMonthOfQuarter);
    }

    public static long roundMonthOfYear(long utcMillis) {
        int year = DateUtilsRounding.getYear(utcMillis);
        int month = DateUtilsRounding.getMonthOfYear(utcMillis, year);
        return DateUtils.of(year, month);
    }

    public static long roundIntervalMonthOfYear(long utcMillis, int monthInterval) {
        if (monthInterval <= 0) {
            throw new IllegalArgumentException("month interval must be strictly positive, got [" + monthInterval + "]");
        }
        int year = DateUtilsRounding.getYear(utcMillis);
        int month = DateUtilsRounding.getMonthOfYear(utcMillis, year);
        int totalMonths = (year - 1) * 12 + (month - 1);
        int quotient = Math.floorDiv(totalMonths, monthInterval);
        int firstMonthOfInterval = quotient * monthInterval;
        int monthInYear = (firstMonthOfInterval % 12 + 12) % 12 + 1;
        int yearResult = (firstMonthOfInterval - (monthInYear - 1)) / 12 + 1;
        return DateUtils.of(yearResult, monthInYear);
    }

    public static long roundYear(long utcMillis) {
        int year = DateUtilsRounding.getYear(utcMillis);
        return DateUtilsRounding.utcMillisAtStartOfYear(year);
    }

    public static long roundYearInterval(long utcMillis, int yearInterval) {
        if (yearInterval <= 0) {
            throw new IllegalArgumentException("year interval must be strictly positive, got [" + yearInterval + "]");
        }
        int year = DateUtilsRounding.getYear(utcMillis);
        int totalYears = year - 1;
        int quotient = Math.floorDiv(totalYears, yearInterval);
        int startTotalYears = quotient * yearInterval;
        int startYear = startTotalYears + 1;
        return DateUtilsRounding.utcMillisAtStartOfYear(startYear);
    }

    public static long roundWeekOfWeekYear(long utcMillis) {
        return DateUtils.roundFloor(utcMillis + 259200000L, 604800000L) - 259200000L;
    }

    private static long of(int year, int month) {
        long millis = DateUtilsRounding.utcMillisAtStartOfYear(year);
        return millis += DateUtilsRounding.getTotalMillisByYearMonth(year, month);
    }

    public static ZonedDateTime nowWithMillisResolution() {
        return DateUtils.nowWithMillisResolution(Clock.systemUTC());
    }

    public static ZonedDateTime nowWithMillisResolution(Clock clock) {
        Clock millisResolutionClock = Clock.tick(clock, Duration.ofMillis(1L));
        return ZonedDateTime.now(millisResolutionClock);
    }

    static void checkTextualDateFormats(String format) {
        if (CONTAINS_CHANGING_TEXT_SPECIFIERS.test(format)) {
            deprecationLogger.warn(DeprecationCategory.PARSING, "cldr_date_formats_" + format, "Date format [{}] contains textual field specifiers that could change in JDK 23. See [{}] for more information.", new Object[]{format, ReferenceDocs.JDK_LOCALE_DIFFERENCES});
        }
        if (CONTAINS_WEEK_DATE_SPECIFIERS.test(format)) {
            deprecationLogger.warn(DeprecationCategory.PARSING, "cldr_week_dates_" + format, "Date format [{}] contains week-date field specifiers that are changing in JDK 23. See [{}] for more information.", new Object[]{format, ReferenceDocs.JDK_LOCALE_DIFFERENCES});
        }
    }

    public static boolean containsCompatOnlyDateFormat(String format) {
        return LEGACY_DATE_FORMAT_MATCHER.test(format);
    }

    static {
        HashMap<String, String> tzs = new HashMap<String, String>();
        tzs.put("EST", "-05:00");
        tzs.put("HST", "-10:00");
        tzs.put("MST", "-07:00");
        tzs.put("ROC", "Asia/Taipei");
        tzs.put("Eire", "Europe/London");
        DEPRECATED_SHORT_TIMEZONES = Collections.unmodifiableMap(tzs);
        DEPRECATED_LONG_TIMEZONES = Map.ofEntries(Map.entry("Africa/Asmera", "Africa/Nairobi"), Map.entry("Africa/Timbuktu", "Africa/Abidjan"), Map.entry("America/Argentina/ComodRivadavia", "America/Argentina/Catamarca"), Map.entry("America/Atka", "America/Adak"), Map.entry("America/Buenos_Aires", "America/Argentina/Buenos_Aires"), Map.entry("America/Catamarca", "America/Argentina/Catamarca"), Map.entry("America/Coral_Harbour", "America/Atikokan"), Map.entry("America/Cordoba", "America/Argentina/Cordoba"), Map.entry("America/Ensenada", "America/Tijuana"), Map.entry("America/Fort_Wayne", "America/Indiana/Indianapolis"), Map.entry("America/Indianapolis", "America/Indiana/Indianapolis"), Map.entry("America/Jujuy", "America/Argentina/Jujuy"), Map.entry("America/Knox_IN", "America/Indiana/Knox"), Map.entry("America/Louisville", "America/Kentucky/Louisville"), Map.entry("America/Mendoza", "America/Argentina/Mendoza"), Map.entry("America/Montreal", "America/Toronto"), Map.entry("America/Porto_Acre", "America/Rio_Branco"), Map.entry("America/Rosario", "America/Argentina/Cordoba"), Map.entry("America/Santa_Isabel", "America/Tijuana"), Map.entry("America/Shiprock", "America/Denver"), Map.entry("America/Virgin", "America/Port_of_Spain"), Map.entry("Antarctica/South_Pole", "Pacific/Auckland"), Map.entry("Asia/Ashkhabad", "Asia/Ashgabat"), Map.entry("Asia/Calcutta", "Asia/Kolkata"), Map.entry("Asia/Chongqing", "Asia/Shanghai"), Map.entry("Asia/Chungking", "Asia/Shanghai"), Map.entry("Asia/Dacca", "Asia/Dhaka"), Map.entry("Asia/Harbin", "Asia/Shanghai"), Map.entry("Asia/Kashgar", "Asia/Urumqi"), Map.entry("Asia/Katmandu", "Asia/Kathmandu"), Map.entry("Asia/Macao", "Asia/Macau"), Map.entry("Asia/Rangoon", "Asia/Yangon"), Map.entry("Asia/Saigon", "Asia/Ho_Chi_Minh"), Map.entry("Asia/Tel_Aviv", "Asia/Jerusalem"), Map.entry("Asia/Thimbu", "Asia/Thimphu"), Map.entry("Asia/Ujung_Pandang", "Asia/Makassar"), Map.entry("Asia/Ulan_Bator", "Asia/Ulaanbaatar"), Map.entry("Atlantic/Faeroe", "Atlantic/Faroe"), Map.entry("Atlantic/Jan_Mayen", "Europe/Oslo"), Map.entry("Australia/ACT", "Australia/Sydney"), Map.entry("Australia/Canberra", "Australia/Sydney"), Map.entry("Australia/LHI", "Australia/Lord_Howe"), Map.entry("Australia/NSW", "Australia/Sydney"), Map.entry("Australia/North", "Australia/Darwin"), Map.entry("Australia/Queensland", "Australia/Brisbane"), Map.entry("Australia/South", "Australia/Adelaide"), Map.entry("Australia/Tasmania", "Australia/Hobart"), Map.entry("Australia/Victoria", "Australia/Melbourne"), Map.entry("Australia/West", "Australia/Perth"), Map.entry("Australia/Yancowinna", "Australia/Broken_Hill"), Map.entry("Brazil/Acre", "America/Rio_Branco"), Map.entry("Brazil/DeNoronha", "America/Noronha"), Map.entry("Brazil/East", "America/Sao_Paulo"), Map.entry("Brazil/West", "America/Manaus"), Map.entry("Canada/Atlantic", "America/Halifax"), Map.entry("Canada/Central", "America/Winnipeg"), Map.entry("Canada/East-Saskatchewan", "America/Regina"), Map.entry("Canada/Eastern", "America/Toronto"), Map.entry("Canada/Mountain", "America/Edmonton"), Map.entry("Canada/Newfoundland", "America/St_Johns"), Map.entry("Canada/Pacific", "America/Vancouver"), Map.entry("Canada/Yukon", "America/Whitehorse"), Map.entry("Chile/Continental", "America/Santiago"), Map.entry("Chile/EasterIsland", "Pacific/Easter"), Map.entry("Cuba", "America/Havana"), Map.entry("Egypt", "Africa/Cairo"), Map.entry("Eire", "Europe/Dublin"), Map.entry("Europe/Belfast", "Europe/London"), Map.entry("Europe/Tiraspol", "Europe/Chisinau"), Map.entry("GB", "Europe/London"), Map.entry("GB-Eire", "Europe/London"), Map.entry("Greenwich", "Etc/GMT"), Map.entry("Hongkong", "Asia/Hong_Kong"), Map.entry("Iceland", "Atlantic/Reykjavik"), Map.entry("Iran", "Asia/Tehran"), Map.entry("Israel", "Asia/Jerusalem"), Map.entry("Jamaica", "America/Jamaica"), Map.entry("Japan", "Asia/Tokyo"), Map.entry("Kwajalein", "Pacific/Kwajalein"), Map.entry("Libya", "Africa/Tripoli"), Map.entry("Mexico/BajaNorte", "America/Tijuana"), Map.entry("Mexico/BajaSur", "America/Mazatlan"), Map.entry("Mexico/General", "America/Mexico_City"), Map.entry("NZ", "Pacific/Auckland"), Map.entry("NZ-CHAT", "Pacific/Chatham"), Map.entry("Navajo", "America/Denver"), Map.entry("PRC", "Asia/Shanghai"), Map.entry("Pacific/Johnston", "Pacific/Honolulu"), Map.entry("Pacific/Ponape", "Pacific/Pohnpei"), Map.entry("Pacific/Samoa", "Pacific/Pago_Pago"), Map.entry("Pacific/Truk", "Pacific/Chuuk"), Map.entry("Pacific/Yap", "Pacific/Chuuk"), Map.entry("Poland", "Europe/Warsaw"), Map.entry("Portugal", "Europe/Lisbon"), Map.entry("ROC", "Asia/Taipei"), Map.entry("ROK", "Asia/Seoul"), Map.entry("Singapore", "Asia/Singapore"), Map.entry("Turkey", "Europe/Istanbul"), Map.entry("UCT", "Etc/UCT"), Map.entry("US/Alaska", "America/Anchorage"), Map.entry("US/Aleutian", "America/Adak"), Map.entry("US/Arizona", "America/Phoenix"), Map.entry("US/Central", "America/Chicago"), Map.entry("US/East-Indiana", "America/Indiana/Indianapolis"), Map.entry("US/Eastern", "America/New_York"), Map.entry("US/Hawaii", "Pacific/Honolulu"), Map.entry("US/Indiana-Starke", "America/Indiana/Knox"), Map.entry("US/Michigan", "America/Detroit"), Map.entry("US/Mountain", "America/Denver"), Map.entry("US/Pacific", "America/Los_Angeles"), Map.entry("US/Samoa", "Pacific/Pago_Pago"), Map.entry("Universal", "Etc/UTC"), Map.entry("W-SU", "Europe/Moscow"), Map.entry("Zulu", "Etc/UTC"));
        MAX_NANOSECOND_INSTANT = Instant.parse("2262-04-11T23:47:16.854775807Z");
        MAX_NANOSECOND_IN_MILLIS = MAX_NANOSECOND_INSTANT.toEpochMilli();
        MAX_NANOSECOND = DateUtils.toLong(MAX_NANOSECOND_INSTANT);
        USES_COMPAT = System.getProperty("java.locale.providers", "").contains("COMPAT");
        LEGACY_DATE_FORMAT_MATCHER = Pattern.compile("[BEGOavz]|LLL|MMM|QQQ|qqq|ccc|eee|(?<!Z)Z{4}(?!Z)").asPredicate();
        CONTAINS_CHANGING_TEXT_SPECIFIERS = USES_COMPAT ? LEGACY_DATE_FORMAT_MATCHER : Predicates.never();
        CONTAINS_WEEK_DATE_SPECIFIERS = USES_COMPAT ? Pattern.compile("[YWw]").asPredicate() : Predicates.never();
    }
}

