/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.coordination.CoordinationMetadata;
import org.elasticsearch.cluster.coordination.Coordinator;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.ReferenceDocs;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.discovery.DiscoveryModule;
import org.elasticsearch.discovery.SettingsBasedSeedHostsProvider;
import org.elasticsearch.node.Node;
import org.elasticsearch.transport.TransportService;

public class ClusterBootstrapService
implements Coordinator.PeerFinderListener {
    public static final Setting<List<String>> INITIAL_MASTER_NODES_SETTING = Setting.stringListSetting("cluster.initial_master_nodes", Setting.Property.NodeScope);
    public static final Setting<TimeValue> UNCONFIGURED_BOOTSTRAP_TIMEOUT_SETTING = Setting.timeSetting("discovery.unconfigured_bootstrap_timeout", TimeValue.timeValueSeconds((long)3L), TimeValue.timeValueMillis((long)1L), Setting.Property.NodeScope);
    static final String BOOTSTRAP_PLACEHOLDER_PREFIX = "{bootstrap-placeholder}-";
    private static final Logger logger = LogManager.getLogger(ClusterBootstrapService.class);
    private final Set<String> bootstrapRequirements;
    @Nullable
    private final TimeValue unconfiguredBootstrapTimeout;
    private final TransportService transportService;
    private final Supplier<Iterable<DiscoveryNode>> discoveredNodesSupplier;
    private final BooleanSupplier isBootstrappedSupplier;
    private final Consumer<CoordinationMetadata.VotingConfiguration> votingConfigurationConsumer;
    private final AtomicBoolean bootstrappingPermitted = new AtomicBoolean(true);
    private final boolean singleNodeDiscovery;

    public ClusterBootstrapService(Settings settings, TransportService transportService, Supplier<Iterable<DiscoveryNode>> discoveredNodesSupplier, BooleanSupplier isBootstrappedSupplier, Consumer<CoordinationMetadata.VotingConfiguration> votingConfigurationConsumer) {
        this.singleNodeDiscovery = DiscoveryModule.isSingleNodeDiscovery(settings);
        if (this.singleNodeDiscovery) {
            if (INITIAL_MASTER_NODES_SETTING.exists(settings)) {
                throw new IllegalArgumentException("setting [" + INITIAL_MASTER_NODES_SETTING.getKey() + "] is not allowed when [" + DiscoveryModule.DISCOVERY_TYPE_SETTING.getKey() + "] is set to [single-node]");
            }
            if (!DiscoveryNode.isMasterNode(settings)) {
                throw new IllegalArgumentException("node with [" + DiscoveryModule.DISCOVERY_TYPE_SETTING.getKey() + "] set to [single-node] must be master-eligible");
            }
            this.bootstrapRequirements = Collections.singleton(Node.NODE_NAME_SETTING.get(settings));
            this.unconfiguredBootstrapTimeout = null;
        } else {
            List<String> initialMasterNodes = INITIAL_MASTER_NODES_SETTING.get(settings);
            this.bootstrapRequirements = Collections.unmodifiableSet(new LinkedHashSet<String>(initialMasterNodes));
            if (this.bootstrapRequirements.size() != initialMasterNodes.size()) {
                throw new IllegalArgumentException("setting [" + INITIAL_MASTER_NODES_SETTING.getKey() + "] contains duplicates: " + initialMasterNodes);
            }
            this.unconfiguredBootstrapTimeout = ClusterBootstrapService.discoveryIsConfigured(settings) ? null : UNCONFIGURED_BOOTSTRAP_TIMEOUT_SETTING.get(settings);
        }
        this.transportService = transportService;
        this.discoveredNodesSupplier = discoveredNodesSupplier;
        this.isBootstrappedSupplier = isBootstrappedSupplier;
        this.votingConfigurationConsumer = votingConfigurationConsumer;
    }

    public static boolean discoveryIsConfigured(Settings settings) {
        return Stream.of(DiscoveryModule.DISCOVERY_SEED_PROVIDERS_SETTING, SettingsBasedSeedHostsProvider.DISCOVERY_SEED_HOSTS_SETTING, INITIAL_MASTER_NODES_SETTING).anyMatch(s -> s.exists(settings));
    }

    void logBootstrapState(Metadata metadata) {
        if (metadata.clusterUUIDCommitted()) {
            String clusterUUID = metadata.clusterUUID();
            if (this.singleNodeDiscovery || this.bootstrapRequirements.isEmpty()) {
                logger.info("this node is locked into cluster UUID [{}] and will not attempt further cluster bootstrapping", (Object)clusterUUID);
            } else {
                this.transportService.getThreadPool().scheduleWithFixedDelay(() -> this.logRemovalWarning(clusterUUID), TimeValue.timeValueHours((long)12L), EsExecutors.DIRECT_EXECUTOR_SERVICE);
                this.logRemovalWarning(clusterUUID);
            }
        } else {
            logger.info("this node has not joined a bootstrapped cluster yet; [{}] is set to {}", (Object)INITIAL_MASTER_NODES_SETTING.getKey(), this.bootstrapRequirements);
        }
    }

    private void logRemovalWarning(String clusterUUID) {
        logger.warn("this node is locked into cluster UUID [{}] but [{}] is set to {}; remove this setting to avoid possible data loss caused by subsequent cluster bootstrap attempts; for further information see {}", (Object)clusterUUID, (Object)INITIAL_MASTER_NODES_SETTING.getKey(), this.bootstrapRequirements, (Object)ReferenceDocs.INITIAL_MASTER_NODES);
    }

    @Override
    public void onFoundPeersUpdated() {
        Set<DiscoveryNode> nodes = this.getDiscoveredNodes();
        if (this.bootstrappingPermitted.get() && this.transportService.getLocalNode().isMasterNode() && !this.bootstrapRequirements.isEmpty() && !this.isBootstrappedSupplier.getAsBoolean()) {
            Tuple<Set<DiscoveryNode>, List<String>> requirementMatchingResult;
            try {
                requirementMatchingResult = this.checkRequirements(nodes);
            }
            catch (IllegalStateException e) {
                logger.warn("bootstrapping cancelled", (Throwable)e);
                this.bootstrappingPermitted.set(false);
                return;
            }
            Set nodesMatchingRequirements = (Set)requirementMatchingResult.v1();
            List unsatisfiedRequirements = (List)requirementMatchingResult.v2();
            logger.trace("nodesMatchingRequirements={}, unsatisfiedRequirements={}, bootstrapRequirements={}", (Object)nodesMatchingRequirements, (Object)unsatisfiedRequirements, this.bootstrapRequirements);
            if (!nodesMatchingRequirements.contains(this.transportService.getLocalNode())) {
                logger.info("skipping cluster bootstrapping as local node does not match bootstrap requirements: {}", this.bootstrapRequirements);
                this.bootstrappingPermitted.set(false);
                return;
            }
            if (nodesMatchingRequirements.size() * 2 > this.bootstrapRequirements.size()) {
                this.startBootstrap(nodesMatchingRequirements, unsatisfiedRequirements);
            }
        }
    }

    void scheduleUnconfiguredBootstrap() {
        if (this.unconfiguredBootstrapTimeout == null) {
            return;
        }
        if (!this.transportService.getLocalNode().isMasterNode()) {
            return;
        }
        logger.info("no discovery configuration found, will perform best-effort cluster bootstrapping after [{}] unless existing master is discovered", (Object)this.unconfiguredBootstrapTimeout);
        this.transportService.getThreadPool().scheduleUnlessShuttingDown(this.unconfiguredBootstrapTimeout, this.transportService.getThreadPool().generic(), new Runnable(){

            @Override
            public void run() {
                Set<DiscoveryNode> discoveredNodes = ClusterBootstrapService.this.getDiscoveredNodes();
                logger.debug("performing best-effort cluster bootstrapping with {}", discoveredNodes);
                ClusterBootstrapService.this.startBootstrap(discoveredNodes, Collections.emptyList());
            }

            public String toString() {
                return "unconfigured-discovery delayed bootstrap";
            }
        });
    }

    private Set<DiscoveryNode> getDiscoveredNodes() {
        return Stream.concat(Stream.of(this.transportService.getLocalNode()), StreamSupport.stream(this.discoveredNodesSupplier.get().spliterator(), false)).collect(Collectors.toSet());
    }

    private void startBootstrap(Set<DiscoveryNode> discoveryNodes, List<String> unsatisfiedRequirements) {
        assert (discoveryNodes.stream().allMatch(DiscoveryNode::isMasterNode)) : discoveryNodes;
        assert (unsatisfiedRequirements.size() < discoveryNodes.size()) : discoveryNodes + " smaller than " + unsatisfiedRequirements;
        if (this.bootstrappingPermitted.compareAndSet(true, false)) {
            this.doBootstrap(new CoordinationMetadata.VotingConfiguration(Stream.concat(discoveryNodes.stream().map(DiscoveryNode::getId), unsatisfiedRequirements.stream().map(s -> BOOTSTRAP_PLACEHOLDER_PREFIX + s)).collect(Collectors.toSet())));
        }
    }

    public static boolean isBootstrapPlaceholder(String nodeId) {
        return nodeId.startsWith(BOOTSTRAP_PLACEHOLDER_PREFIX);
    }

    private void doBootstrap(final CoordinationMetadata.VotingConfiguration votingConfiguration) {
        assert (this.transportService.getLocalNode().isMasterNode());
        try {
            this.votingConfigurationConsumer.accept(votingConfiguration);
        }
        catch (Exception e) {
            logger.warn(() -> "exception when bootstrapping with " + votingConfiguration + ", rescheduling", (Throwable)e);
            this.transportService.getThreadPool().scheduleUnlessShuttingDown(TimeValue.timeValueSeconds((long)10L), this.transportService.getThreadPool().generic(), new Runnable(){

                @Override
                public void run() {
                    ClusterBootstrapService.this.doBootstrap(votingConfiguration);
                }

                public String toString() {
                    return "retry of failed bootstrapping with " + votingConfiguration;
                }
            });
        }
    }

    private static boolean matchesRequirement(DiscoveryNode discoveryNode, String requirement) {
        return discoveryNode.getName().equals(requirement) || discoveryNode.getAddress().toString().equals(requirement) || discoveryNode.getAddress().getAddress().equals(requirement);
    }

    private Tuple<Set<DiscoveryNode>, List<String>> checkRequirements(Set<DiscoveryNode> nodes) {
        HashSet<DiscoveryNode> selectedNodes = new HashSet<DiscoveryNode>();
        ArrayList<String> unmatchedRequirements = new ArrayList<String>();
        for (String bootstrapRequirement : this.bootstrapRequirements) {
            Set matchingNodes = nodes.stream().filter(n -> ClusterBootstrapService.matchesRequirement(n, bootstrapRequirement)).collect(Collectors.toSet());
            if (matchingNodes.size() == 0) {
                unmatchedRequirements.add(bootstrapRequirement);
            }
            if (matchingNodes.size() > 1) {
                throw new IllegalStateException("requirement [" + bootstrapRequirement + "] matches multiple nodes: " + matchingNodes);
            }
            for (DiscoveryNode matchingNode : matchingNodes) {
                if (selectedNodes.add(matchingNode)) continue;
                throw new IllegalStateException("node [" + matchingNode + "] matches multiple requirements: " + this.bootstrapRequirements.stream().filter(r -> ClusterBootstrapService.matchesRequirement(matchingNode, r)).toList());
            }
        }
        return Tuple.tuple(selectedNodes, unmatchedRequirements);
    }
}

