/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.snapshots;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.RepositoriesMetadata;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.util.CollectionUtils;
import org.elasticsearch.features.FeatureService;
import org.elasticsearch.health.Diagnosis;
import org.elasticsearch.health.HealthFeatures;
import org.elasticsearch.health.HealthIndicatorDetails;
import org.elasticsearch.health.HealthIndicatorImpact;
import org.elasticsearch.health.HealthIndicatorResult;
import org.elasticsearch.health.HealthIndicatorService;
import org.elasticsearch.health.HealthStatus;
import org.elasticsearch.health.ImpactArea;
import org.elasticsearch.health.SimpleHealthIndicatorDetails;
import org.elasticsearch.health.node.HealthInfo;
import org.elasticsearch.health.node.RepositoriesHealthInfo;

public class RepositoryIntegrityHealthIndicatorService
implements HealthIndicatorService {
    public static final String NAME = "repository_integrity";
    private static final String HELP_URL = "https://ela.st/fix-repository-integrity";
    public static final String NO_REPOS_CONFIGURED = "No snapshot repositories configured.";
    public static final String ALL_REPOS_HEALTHY = "All repositories are healthy.";
    public static final String NO_REPO_HEALTH_INFO = "No repository health info.";
    public static final String MIXED_VERSIONS = "No repository health info. The cluster currently has mixed versions (an upgrade may be in progress).";
    public static final List<HealthIndicatorImpact> IMPACTS = List.of(new HealthIndicatorImpact("repository_integrity", "backups_at_risk", 2, "Data in the affected snapshot repositories may be lost and cannot be restored.", List.of(ImpactArea.BACKUP)));
    public static final Diagnosis.Definition CORRUPTED_DEFINITION = new Diagnosis.Definition("repository_integrity", "corrupt_repo_integrity", "Multiple clusters are writing to the same repository.", "Remove the repository from the other cluster(s), or mark it as read-only in the other cluster(s), and then re-add the repository to this cluster.", "https://ela.st/fix-repository-integrity");
    public static final Diagnosis.Definition UNKNOWN_DEFINITION = new Diagnosis.Definition("repository_integrity", "unknown_repository", "The repository uses an unknown type.", "Ensure that all required plugins are installed on the affected nodes.", "https://ela.st/fix-repository-integrity");
    public static final Diagnosis.Definition INVALID_DEFINITION = new Diagnosis.Definition("repository_integrity", "invalid_repository", "An exception occurred while trying to initialize the repository.", "Make sure all nodes in the cluster are in sync with each other.Refer to the nodes\u2019 logs for detailed information on why the repository initialization failed.", "https://ela.st/fix-repository-integrity");
    private final ClusterService clusterService;
    private final FeatureService featureService;

    public RepositoryIntegrityHealthIndicatorService(ClusterService clusterService, FeatureService featureService) {
        this.clusterService = clusterService;
        this.featureService = featureService;
    }

    @Override
    public String name() {
        return NAME;
    }

    @Override
    public HealthIndicatorResult calculate(boolean verbose, int maxAffectedResourcesCount, HealthInfo healthInfo) {
        ClusterState clusterState = this.clusterService.state();
        RepositoriesMetadata snapshotMetadata = RepositoriesMetadata.get(this.clusterService.state());
        List<RepositoryMetadata> repositories = snapshotMetadata.repositories();
        if (repositories.isEmpty()) {
            return this.createIndicator(HealthStatus.GREEN, NO_REPOS_CONFIGURED, HealthIndicatorDetails.EMPTY, List.of(), List.of());
        }
        RepositoryHealthAnalyzer repositoryHealthAnalyzer = new RepositoryHealthAnalyzer(clusterState, repositories, healthInfo.repositoriesInfoByNode());
        return this.createIndicator(repositoryHealthAnalyzer.getHealthStatus(), repositoryHealthAnalyzer.getSymptom(), repositoryHealthAnalyzer.getDetails(verbose), repositoryHealthAnalyzer.getImpacts(), repositoryHealthAnalyzer.getDiagnoses(verbose, maxAffectedResourcesCount));
    }

    class RepositoryHealthAnalyzer {
        private final ClusterState clusterState;
        private final int totalRepositories;
        private final List<String> corruptedRepositories;
        private final Set<String> unknownRepositories = new HashSet<String>();
        private final Set<String> nodesWithUnknownRepos = new HashSet<String>();
        private final Set<String> invalidRepositories = new HashSet<String>();
        private final Set<String> nodesWithInvalidRepos = new HashSet<String>();
        private final HealthStatus healthStatus;
        private boolean clusterHasFeature = true;

        private RepositoryHealthAnalyzer(ClusterState clusterState, List<RepositoryMetadata> repositories, Map<String, RepositoriesHealthInfo> repositoriesHealthByNode) {
            this.clusterState = clusterState;
            this.totalRepositories = repositories.size();
            this.corruptedRepositories = repositories.stream().filter(repository -> repository.generation() == -3L).map(RepositoryMetadata::name).sorted().toList();
            repositoriesHealthByNode.forEach((nodeId, healthInfo) -> {
                this.unknownRepositories.addAll(healthInfo.unknownRepositories());
                if (!healthInfo.unknownRepositories().isEmpty()) {
                    this.nodesWithUnknownRepos.add((String)nodeId);
                }
                this.invalidRepositories.addAll(healthInfo.invalidRepositories());
                if (!healthInfo.invalidRepositories().isEmpty()) {
                    this.nodesWithInvalidRepos.add((String)nodeId);
                }
            });
            if (!(this.corruptedRepositories.isEmpty() && this.unknownRepositories.isEmpty() && this.invalidRepositories.isEmpty())) {
                this.healthStatus = HealthStatus.YELLOW;
            } else if (repositoriesHealthByNode.isEmpty()) {
                boolean bl = this.clusterHasFeature = !RepositoryIntegrityHealthIndicatorService.this.featureService.clusterHasFeature(clusterState, HealthFeatures.SUPPORTS_EXTENDED_REPOSITORY_INDICATOR);
                this.healthStatus = this.clusterHasFeature ? HealthStatus.GREEN : HealthStatus.UNKNOWN;
            } else {
                this.healthStatus = HealthStatus.GREEN;
            }
        }

        public HealthStatus getHealthStatus() {
            return this.healthStatus;
        }

        public String getSymptom() {
            if (this.healthStatus == HealthStatus.GREEN) {
                return this.clusterHasFeature ? RepositoryIntegrityHealthIndicatorService.ALL_REPOS_HEALTHY : RepositoryIntegrityHealthIndicatorService.MIXED_VERSIONS;
            }
            if (this.healthStatus == HealthStatus.UNKNOWN) {
                return RepositoryIntegrityHealthIndicatorService.NO_REPO_HEALTH_INFO;
            }
            return "Detected " + Stream.of(RepositoryHealthAnalyzer.generateSymptomString("corrupted", this.corruptedRepositories.size()), RepositoryHealthAnalyzer.generateSymptomString("unknown", this.unknownRepositories.size()), RepositoryHealthAnalyzer.generateSymptomString("invalid", this.invalidRepositories.size())).filter(Objects::nonNull).collect(Collectors.joining(", and ")) + ".";
        }

        private static String generateSymptomString(String type, long size) {
            if (size == 0L) {
                return null;
            }
            return String.format(Locale.ROOT, "[%d] %s snapshot repositor%s", size, type, size > 1L ? "ies" : "y");
        }

        public HealthIndicatorDetails getDetails(boolean verbose) {
            if (!verbose) {
                return HealthIndicatorDetails.EMPTY;
            }
            HashMap<String, Object> map = new HashMap<String, Object>();
            map.put("total_repositories", this.totalRepositories);
            if (this.healthStatus != HealthStatus.GREEN) {
                map.put("corrupted_repositories", this.corruptedRepositories.size());
                map.put("corrupted", CollectionUtils.limitSize(this.corruptedRepositories, 10));
                if (this.healthStatus != HealthStatus.UNKNOWN) {
                    map.put("unknown_repositories", this.unknownRepositories.size());
                    map.put("invalid_repositories", this.invalidRepositories.size());
                }
            }
            return new SimpleHealthIndicatorDetails(map);
        }

        public List<HealthIndicatorImpact> getImpacts() {
            if (this.healthStatus == HealthStatus.GREEN || this.healthStatus == HealthStatus.UNKNOWN) {
                return List.of();
            }
            return IMPACTS;
        }

        public List<Diagnosis> getDiagnoses(boolean verbose, int maxAffectedResourcesCount) {
            if (!verbose) {
                return List.of();
            }
            ArrayList<Diagnosis> diagnoses = new ArrayList<Diagnosis>();
            if (!this.corruptedRepositories.isEmpty()) {
                diagnoses.add(new Diagnosis(CORRUPTED_DEFINITION, List.of(new Diagnosis.Resource(Diagnosis.Resource.Type.SNAPSHOT_REPOSITORY, CollectionUtils.limitSize(this.corruptedRepositories, maxAffectedResourcesCount)))));
            }
            if (!this.unknownRepositories.isEmpty()) {
                diagnoses.add(this.createDiagnosis(UNKNOWN_DEFINITION, this.unknownRepositories, this.nodesWithUnknownRepos, maxAffectedResourcesCount));
            }
            if (!this.invalidRepositories.isEmpty()) {
                diagnoses.add(this.createDiagnosis(INVALID_DEFINITION, this.invalidRepositories, this.nodesWithInvalidRepos, maxAffectedResourcesCount));
            }
            return diagnoses;
        }

        private Diagnosis createDiagnosis(Diagnosis.Definition definition, Set<String> repos, Set<String> nodes, int maxAffectedResourcesCount) {
            List<String> reposView = repos.stream().sorted().limit(maxAffectedResourcesCount).toList();
            List<DiscoveryNode> nodesView = nodes.stream().map(nodeId -> this.clusterState.nodes().get((String)nodeId)).sorted(DiscoveryNode.DISCOVERY_NODE_COMPARATOR).limit(maxAffectedResourcesCount).toList();
            return new Diagnosis(definition, List.of(new Diagnosis.Resource(Diagnosis.Resource.Type.SNAPSHOT_REPOSITORY, reposView), new Diagnosis.Resource(nodesView)));
        }
    }
}

