/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.env;

import java.io.Closeable;
import java.io.IOException;
import java.io.PrintStream;
import java.io.UncheckedIOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.store.NativeFSLockFactory;
import org.elasticsearch.Build;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.ReferenceDocs;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.CheckedFunction;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Predicates;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.env.BuildVersion;
import org.elasticsearch.env.Environment;
import org.elasticsearch.env.NodeMetadata;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.env.ShardLockObtainFailedException;
import org.elasticsearch.gateway.CorruptStateException;
import org.elasticsearch.gateway.PersistedClusterStateService;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.FsDirectoryFactory;
import org.elasticsearch.monitor.fs.FsInfo;
import org.elasticsearch.monitor.fs.FsProbe;
import org.elasticsearch.monitor.jvm.HotThreads;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.xcontent.NamedXContentRegistry;

public final class NodeEnvironment
implements Closeable {
    private final Logger logger = LogManager.getLogger(NodeEnvironment.class);
    private final DataPath[] dataPaths;
    private final Path sharedDataPath;
    private final Lock[] locks;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final Map<ShardId, InternalShardLock> shardLocks = new HashMap<ShardId, InternalShardLock>();
    private final NodeMetadata nodeMetadata;
    public static final Setting<Long> NODE_ID_SEED_SETTING = Setting.longSetting("node.id.seed", 0L, Long.MIN_VALUE, Setting.Property.NodeScope);
    public static final Setting<Boolean> ENABLE_LUCENE_SEGMENT_INFOS_TRACE_SETTING = Setting.boolSetting("node.enable_lucene_segment_infos_trace", false, Setting.Property.NodeScope);
    public static final String INDICES_FOLDER = "indices";
    public static final String NODE_LOCK_FILENAME = "node.lock";
    private static final String SNAPSHOT_CACHE_FOLDER = "snapshot_cache";
    static final String SEARCHABLE_SHARED_CACHE_FILE = "shared_snapshot_cache";
    private final Semaphore shardLockHotThreadsPermit = new Semaphore(1);
    private long nextShardLockHotThreadsNanos = Long.MIN_VALUE;
    static final String TEMP_FILE_NAME = ".es_temp_file";

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NodeEnvironment(Settings settings, Environment environment) throws IOException {
        boolean success = false;
        try {
            NodeLock nodeLock;
            this.sharedDataPath = environment.sharedDataDir();
            for (Path path : environment.dataDirs()) {
                if (Files.exists(path, new LinkOption[0])) {
                    path = path.toRealPath(new LinkOption[0]);
                }
                Files.createDirectories(path, new FileAttribute[0]);
            }
            try {
                nodeLock = new NodeLock(this.logger, environment, (CheckedFunction<Path, Boolean, IOException>)((CheckedFunction)dir -> true));
            }
            catch (IOException e) {
                String message = String.format(Locale.ROOT, "failed to obtain node locks, tried %s; maybe these locations are not writable or multiple nodes were started on the same data path?", Arrays.toString(environment.dataDirs()));
                throw new IllegalStateException(message, e);
            }
            this.locks = nodeLock.locks;
            this.dataPaths = nodeLock.dataPaths;
            this.logger.debug("using node location {}", (Object)Arrays.toString(this.dataPaths));
            this.maybeLogPathDetails();
            this.maybeLogHeapDetails();
            NodeEnvironment.applySegmentInfosTrace(settings);
            this.assertCanWrite();
            NodeEnvironment.ensureAtomicMoveSupported(this.dataPaths);
            if (NodeEnvironment.upgradeLegacyNodeFolders(this.logger, settings, environment, nodeLock)) {
                this.assertCanWrite();
            }
            for (Path dataPath : environment.dataDirs()) {
                Path legacyNodesPath = dataPath.resolve("nodes");
                if (Files.isRegularFile(legacyNodesPath, new LinkOption[0])) continue;
                String content = "written by Elasticsearch " + Build.current().version() + " to prevent a downgrade to a version prior to v8.0.0 which would result in data loss";
                Files.writeString(legacyNodesPath, (CharSequence)content, new OpenOption[0]);
                IOUtils.fsync((Path)legacyNodesPath, (boolean)false);
                IOUtils.fsync((Path)dataPath, (boolean)true);
            }
            if (!DiscoveryNode.canContainData(settings)) {
                if (!DiscoveryNode.isMasterNode(settings)) {
                    NodeEnvironment.ensureNoIndexMetadata(this.dataPaths);
                }
                NodeEnvironment.ensureNoShardData(this.dataPaths);
            }
            this.nodeMetadata = NodeEnvironment.loadNodeMetadata(settings, this.logger, this.dataPaths);
            success = true;
        }
        finally {
            if (!success) {
                this.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean upgradeLegacyNodeFolders(Logger logger, Settings settings, Environment environment, NodeLock nodeLock) throws IOException {
        NodeLock legacyNodeLock;
        boolean upgradeNeeded = false;
        for (Path path2 : environment.dataDirs()) {
            Path nodesFolderPath = path2.resolve("nodes");
            if (!Files.isDirectory(nodesFolderPath, new LinkOption[0])) continue;
            ArrayList<Integer> nodeLockIds = new ArrayList<Integer>();
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(nodesFolderPath);){
                for (Path nodeLockIdPath : stream) {
                    String fileName = nodeLockIdPath.getFileName().toString();
                    if (Files.isDirectory(nodeLockIdPath, new LinkOption[0]) && fileName.chars().allMatch(Character::isDigit)) {
                        int nodeLockId = Integer.parseInt(fileName);
                        nodeLockIds.add(nodeLockId);
                        continue;
                    }
                    if (FileSystemUtils.isDesktopServicesStore(nodeLockIdPath)) continue;
                    throw new IllegalStateException("unexpected file/folder encountered during data folder upgrade: " + String.valueOf(nodeLockIdPath));
                }
            }
            if (nodeLockIds.isEmpty()) continue;
            upgradeNeeded = true;
            if (nodeLockIds.equals(Arrays.asList(0))) continue;
            throw new IllegalStateException("data path " + String.valueOf(nodesFolderPath) + " cannot be upgraded automatically because it contains data from nodes with ordinals " + String.valueOf(nodeLockIds) + ", due to previous use of the now obsolete [node.max_local_storage_nodes] setting. Please check the breaking changes docs for the current version of Elasticsearch to find an upgrade path");
        }
        if (!upgradeNeeded) {
            logger.trace("data folder upgrade not required");
            return false;
        }
        logger.info("upgrading legacy data folders: {}", (Object)Arrays.toString(environment.dataDirs()));
        try {
            legacyNodeLock = new NodeLock(logger, environment, (CheckedFunction<Path, Boolean, IOException>)((CheckedFunction)dir -> true), path -> path.resolve("nodes").resolve("0"));
        }
        catch (IOException e) {
            String message = String.format(Locale.ROOT, "failed to obtain legacy node locks, tried %s; maybe these locations are not writable or multiple nodes were started on the same data path?", Arrays.toString(environment.dataDirs()));
            throw new IllegalStateException(message, e);
        }
        assert (nodeLock.getDataPaths().length == legacyNodeLock.getDataPaths().length);
        try {
            NodeEnvironment.checkForIndexCompatibility(logger, legacyNodeLock.getDataPaths());
            ArrayList<CheckedRunnable> upgradeActions = new ArrayList<CheckedRunnable>();
            for (int i = 0; i < legacyNodeLock.getDataPaths().length; ++i) {
                DataPath legacyDataPath = legacyNodeLock.getDataPaths()[i];
                DataPath dataPath = nodeLock.getDataPaths()[i];
                HashSet<String> folderNames = new HashSet<String>();
                HashSet<String> expectedFolderNames = new HashSet<String>(Arrays.asList("_state", INDICES_FOLDER, SNAPSHOT_CACHE_FOLDER));
                HashSet<String> ignoredFileNames = new HashSet<String>(Arrays.asList(NODE_LOCK_FILENAME, TEMP_FILE_NAME, ".es_temp_file.tmp", ".es_temp_file.final", SEARCHABLE_SHARED_CACHE_FILE));
                try (DirectoryStream<Path> stream = Files.newDirectoryStream(legacyDataPath.path);){
                    for (Path subFolderPath : stream) {
                        String fileName = subFolderPath.getFileName().toString();
                        if (FileSystemUtils.isDesktopServicesStore(subFolderPath)) continue;
                        if (FileSystemUtils.isAccessibleDirectory(subFolderPath, logger)) {
                            if (!expectedFolderNames.contains(fileName)) {
                                throw new IllegalStateException("unexpected folder encountered during data folder upgrade: " + String.valueOf(subFolderPath));
                            }
                            Path targetSubFolderPath = dataPath.path.resolve(fileName);
                            if (Files.exists(targetSubFolderPath, new LinkOption[0])) {
                                throw new IllegalStateException("target folder already exists during data folder upgrade: " + String.valueOf(targetSubFolderPath));
                            }
                            folderNames.add(fileName);
                            continue;
                        }
                        if (ignoredFileNames.contains(fileName)) continue;
                        throw new IllegalStateException("unexpected file/folder encountered during data folder upgrade: " + String.valueOf(subFolderPath));
                    }
                }
                assert (Sets.difference(folderNames, expectedFolderNames).isEmpty()) : "expected indices and/or state dir folder but was " + String.valueOf(folderNames);
                upgradeActions.add(() -> {
                    for (String folderName : folderNames) {
                        Path sourceSubFolderPath = legacyDataPath.path.resolve(folderName);
                        Path targetSubFolderPath = dataPath.path.resolve(folderName);
                        Files.move(sourceSubFolderPath, targetSubFolderPath, StandardCopyOption.ATOMIC_MOVE);
                        logger.info("data folder upgrade: moved from [{}] to [{}]", (Object)sourceSubFolderPath, (Object)targetSubFolderPath);
                    }
                    IOUtils.fsync((Path)dataPath.path, (boolean)true);
                });
            }
            for (CheckedRunnable upgradeAction : upgradeActions) {
                upgradeAction.run();
            }
        }
        finally {
            legacyNodeLock.close();
        }
        IOUtils.rm((Path[])((Path[])Stream.of(environment.dataDirs()).map(path -> path.resolve("nodes")).toArray(Path[]::new)));
        return true;
    }

    static void checkForIndexCompatibility(Logger logger, DataPath ... dataPaths) throws IOException {
        Path[] paths = (Path[])Arrays.stream(dataPaths).map(np -> np.path).toArray(Path[]::new);
        NodeMetadata metadata = PersistedClusterStateService.nodeMetadata(paths);
        if (metadata == null) {
            throw new CorruptStateException("Format version is not supported. Upgrading to [" + Build.current().version() + "] is only supported from version [" + Build.current().minWireCompatVersion() + "].");
        }
        metadata.verifyUpgradeToCurrentVersion();
        logger.info("oldest index version recorded in NodeMetadata {}", (Object)metadata.oldestIndexVersion());
        if (metadata.oldestIndexVersion().isLegacyIndexVersion()) {
            String bestDowngradeVersion = NodeEnvironment.getBestDowngradeVersion(metadata.previousNodeVersion().toString());
            throw new IllegalStateException("Cannot start this node because it holds metadata for indices with version [" + metadata.oldestIndexVersion().toReleaseVersion() + "] with which this node of version [" + Build.current().version() + "] is incompatible. Revert this node to version [" + bestDowngradeVersion + "] and delete any indices with versions earlier than [" + IndexVersions.MINIMUM_COMPATIBLE.toReleaseVersion() + "] before upgrading to version [" + Build.current().version() + "]. If all such indices have already been deleted, revert this node to version [" + bestDowngradeVersion + "] and wait for it to join the cluster to clean up any older indices from its metadata.");
        }
    }

    private void maybeLogPathDetails() throws IOException {
        if (this.logger.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder();
            for (DataPath dataPath : this.dataPaths) {
                sb.append('\n').append(" -> ").append(dataPath.path.toAbsolutePath());
                FsInfo.Path fsPath = FsProbe.getFSInfo(dataPath);
                sb.append(", free_space [").append(fsPath.getFree()).append("], usable_space [").append(fsPath.getAvailable()).append("], total_space [").append(fsPath.getTotal()).append("], mount [").append(fsPath.getMount()).append("], type [").append(fsPath.getType()).append(']');
            }
            this.logger.debug("node data locations details:{}", (Object)sb);
        } else if (this.logger.isInfoEnabled()) {
            FsInfo.Path totFSPath = new FsInfo.Path();
            HashSet<String> allTypes = new HashSet<String>();
            HashSet<String> allMounts = new HashSet<String>();
            for (DataPath dataPath : this.dataPaths) {
                FsInfo.Path fsPath = FsProbe.getFSInfo(dataPath);
                String mount = fsPath.getMount();
                if (allMounts.contains(mount)) continue;
                allMounts.add(mount);
                String type = fsPath.getType();
                if (type != null) {
                    allTypes.add(type);
                }
                totFSPath.add(fsPath);
            }
            this.logger.info("using [{}] data paths, mounts [{}], net usable_space [{}], net total_space [{}], types [{}]", (Object)this.dataPaths.length, allMounts, (Object)totFSPath.getAvailable(), (Object)totFSPath.getTotal(), (Object)NodeEnvironment.toString(allTypes));
        }
    }

    private void maybeLogHeapDetails() {
        JvmInfo jvmInfo = JvmInfo.jvmInfo();
        ByteSizeValue maxHeapSize = jvmInfo.getMem().getHeapMax();
        String useCompressedOops = jvmInfo.useCompressedOops();
        this.logger.info("heap size [{}], compressed ordinary object pointers [{}]", (Object)maxHeapSize, (Object)useCompressedOops);
    }

    private static NodeMetadata loadNodeMetadata(Settings settings, Logger logger, DataPath ... dataPaths) throws IOException {
        Object[] paths = (Path[])Arrays.stream(dataPaths).map(np -> np.path).toArray(Path[]::new);
        NodeMetadata metadata = PersistedClusterStateService.nodeMetadata((Path[])paths);
        if (metadata == null) {
            HashSet<String> nodeIds = new HashSet<String>();
            for (Path path : paths) {
                NodeMetadata oldStyleMetadata = NodeMetadata.FORMAT.loadLatestState(logger, NamedXContentRegistry.EMPTY, path);
                if (oldStyleMetadata == null) continue;
                nodeIds.add(oldStyleMetadata.nodeId());
            }
            if (nodeIds.size() > 1) {
                throw new IllegalStateException("data paths " + Arrays.toString(paths) + " belong to multiple nodes with IDs " + String.valueOf(nodeIds));
            }
            NodeMetadata legacyMetadata = NodeMetadata.FORMAT.loadLatestState(logger, NamedXContentRegistry.EMPTY, (Path[])paths);
            if (legacyMetadata == null) {
                assert (nodeIds.isEmpty()) : nodeIds;
                metadata = new NodeMetadata(NodeEnvironment.generateNodeId(settings), BuildVersion.current(), IndexVersion.current());
            } else {
                assert (nodeIds.equals(Collections.singleton(legacyMetadata.nodeId()))) : String.valueOf(nodeIds) + " doesn't match " + String.valueOf(legacyMetadata);
                metadata = legacyMetadata;
            }
        }
        metadata = metadata.upgradeToCurrentVersion();
        assert (metadata.nodeVersion().equals(BuildVersion.current())) : String.valueOf(metadata.nodeVersion()) + " != " + String.valueOf(Build.current());
        return metadata;
    }

    public static String generateNodeId(Settings settings) {
        Random random = Randomness.get(settings, NODE_ID_SEED_SETTING);
        return UUIDs.randomBase64UUID(random);
    }

    @SuppressForbidden(reason="System.out.*")
    static void applySegmentInfosTrace(Settings settings) {
        if (ENABLE_LUCENE_SEGMENT_INFOS_TRACE_SETTING.get(settings).booleanValue()) {
            SegmentInfos.setInfoStream((PrintStream)System.out);
        }
    }

    private static String toString(Collection<String> items) {
        StringBuilder b = new StringBuilder();
        for (String item : items) {
            if (b.length() > 0) {
                b.append(", ");
            }
            b.append(item);
        }
        return b.toString();
    }

    public void deleteShardDirectorySafe(ShardId shardId, IndexSettings indexSettings, Consumer<Path[]> listener) throws IOException, ShardLockObtainFailedException {
        Path[] paths = this.availableShardPaths(shardId);
        this.logger.trace("deleting shard {} directory, paths: [{}]", (Object)shardId, (Object)paths);
        try (ShardLock lock = this.shardLock(shardId, "shard deletion under lock");){
            this.deleteShardDirectoryUnderLock(lock, indexSettings, listener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void acquireFSLockForPaths(IndexSettings indexSettings, Path ... shardPaths) throws IOException {
        Lock[] locks = new Lock[shardPaths.length];
        Directory[] dirs = new Directory[shardPaths.length];
        try {
            for (int i = 0; i < shardPaths.length; ++i) {
                Path p = shardPaths[i].resolve("index");
                dirs[i] = new NIOFSDirectory(p, indexSettings.getValue(FsDirectoryFactory.INDEX_LOCK_FACTOR_SETTING));
                try {
                    locks[i] = dirs[i].obtainLock("write.lock");
                    continue;
                }
                catch (IOException ex) {
                    throw new LockObtainFailedException("unable to acquire write.lock for " + String.valueOf(p), (Throwable)ex);
                }
            }
        }
        finally {
            IOUtils.closeWhileHandlingException((Closeable[])locks);
            IOUtils.closeWhileHandlingException((Closeable[])dirs);
        }
    }

    public void deleteShardDirectoryUnderLock(ShardLock lock, IndexSettings indexSettings, Consumer<Path[]> listener) throws IOException {
        ShardId shardId = lock.getShardId();
        assert (this.isShardLocked(shardId)) : "shard " + String.valueOf(shardId) + " is not locked";
        Path[] paths = this.availableShardPaths(shardId);
        this.logger.trace("acquiring locks for {}, paths: [{}]", (Object)shardId, (Object)paths);
        NodeEnvironment.acquireFSLockForPaths(indexSettings, paths);
        listener.accept(paths);
        IOUtils.rm((Path[])paths);
        if (indexSettings.hasCustomDataPath()) {
            Path customLocation = this.resolveCustomLocation(indexSettings.customDataPath(), shardId);
            this.logger.trace("acquiring lock for {}, custom path: [{}]", (Object)shardId, (Object)customLocation);
            NodeEnvironment.acquireFSLockForPaths(indexSettings, customLocation);
            this.logger.trace("deleting custom shard {} directory [{}]", (Object)shardId, (Object)customLocation);
            listener.accept(new Path[]{customLocation});
            IOUtils.rm((Path[])new Path[]{customLocation});
        }
        this.logger.trace("deleted shard {} directory, paths: [{}]", (Object)shardId, (Object)paths);
        assert (NodeEnvironment.assertPathsDoNotExist(paths));
    }

    private static boolean assertPathsDoNotExist(Path[] paths) {
        Set existingPaths = Stream.of(paths).filter(xva$0 -> FileSystemUtils.exists(xva$0)).filter(leftOver -> {
            try (DirectoryStream<Path> children = Files.newDirectoryStream(leftOver);){
                boolean bl;
                block18: {
                    Iterator<Path> iter = children.iterator();
                    if (!iter.hasNext()) {
                        boolean bl2 = true;
                        return bl2;
                    }
                    Path maybeState = iter.next();
                    if (iter.hasNext() || !maybeState.equals(leftOver.resolve("_state"))) {
                        boolean bl3 = true;
                        return bl3;
                    }
                    DirectoryStream<Path> stateChildren = Files.newDirectoryStream(maybeState);
                    try {
                        bl = stateChildren.iterator().hasNext();
                        if (stateChildren == null) break block18;
                    }
                    catch (Throwable throwable) {
                        if (stateChildren != null) {
                            try {
                                stateChildren.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    stateChildren.close();
                }
                return bl;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }).collect(Collectors.toSet());
        assert (existingPaths.size() == 0) : "Paths exist that should have been deleted: " + String.valueOf(existingPaths);
        return existingPaths.size() == 0;
    }

    private boolean isShardLocked(ShardId id) {
        try {
            this.shardLock(id, "checking if shard is locked").close();
            return false;
        }
        catch (ShardLockObtainFailedException ex) {
            return true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteIndexDirectorySafe(Index index, long lockTimeoutMS, IndexSettings indexSettings, Consumer<Path[]> listener) throws IOException, ShardLockObtainFailedException {
        List<ShardLock> locks = this.lockAllForIndex(index, indexSettings, "deleting index directory", lockTimeoutMS);
        try {
            this.deleteIndexDirectoryUnderLock(index, indexSettings, listener);
        }
        finally {
            IOUtils.closeWhileHandlingException(locks);
        }
    }

    public void deleteIndexDirectoryUnderLock(Index index, IndexSettings indexSettings, Consumer<Path[]> listener) throws IOException {
        Path[] indexPaths = this.indexPaths(index);
        this.logger.trace("deleting index {} directory, paths({}): [{}]", (Object)index, (Object)indexPaths.length, (Object)indexPaths);
        listener.accept(indexPaths);
        IOUtils.rm((Path[])indexPaths);
        if (indexSettings.hasCustomDataPath()) {
            Path customLocation = this.resolveIndexCustomLocation(indexSettings.customDataPath(), index.getUUID());
            this.logger.trace("deleting custom index {} directory [{}]", (Object)index, (Object)customLocation);
            listener.accept(new Path[]{customLocation});
            IOUtils.rm((Path[])new Path[]{customLocation});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<ShardLock> lockAllForIndex(Index index, IndexSettings settings, String lockDetails, long lockTimeoutMS) throws ShardLockObtainFailedException {
        int numShards = settings.getNumberOfShards();
        if (numShards <= 0) {
            throw new IllegalArgumentException("settings must contain a non-null > 0 number of shards");
        }
        this.logger.trace("locking all shards for index {} - [{}]", (Object)index, (Object)numShards);
        ArrayList<ShardLock> allLocks = new ArrayList<ShardLock>(numShards);
        boolean success = false;
        long startTimeNS = System.nanoTime();
        try {
            for (int i = 0; i < numShards; ++i) {
                long timeoutLeftMS = Math.max(0L, lockTimeoutMS - TimeValue.nsecToMSec((long)(System.nanoTime() - startTimeNS)));
                allLocks.add(this.shardLock(new ShardId(index, i), lockDetails, timeoutLeftMS));
            }
            success = true;
        }
        finally {
            if (!success) {
                this.logger.trace("unable to lock all shards for index {}", (Object)index);
                IOUtils.closeWhileHandlingException(allLocks);
            }
        }
        return allLocks;
    }

    public ShardLock shardLock(ShardId id, String details) throws ShardLockObtainFailedException {
        return this.shardLock(id, details, 0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ShardLock shardLock(final ShardId shardId, String details, long lockTimeoutMS) throws ShardLockObtainFailedException {
        boolean acquired;
        InternalShardLock shardLock;
        this.logger.trace("acquiring node shardlock on [{}], timeout [{}], details [{}]", (Object)shardId, (Object)lockTimeoutMS, (Object)details);
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            InternalShardLock found = this.shardLocks.get(shardId);
            if (found != null) {
                shardLock = found;
                shardLock.incWaitCount();
                acquired = false;
            } else {
                shardLock = new InternalShardLock(shardId, details);
                this.shardLocks.put(shardId, shardLock);
                acquired = true;
            }
        }
        if (!acquired) {
            boolean success = false;
            try {
                shardLock.acquire(lockTimeoutMS, details);
                success = true;
            }
            finally {
                if (!success) {
                    shardLock.decWaitCount();
                }
            }
        }
        this.logger.trace("successfully acquired shardlock for [{}]", (Object)shardId);
        return new ShardLock(shardId){

            @Override
            protected void closeInternal() {
                shardLock.release();
                NodeEnvironment.this.logger.trace("released shard lock for [{}]", (Object)shardId);
            }

            @Override
            public void setDetails(String details) {
                shardLock.setDetails(details);
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<ShardId> lockedShards() {
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            return Set.copyOf(this.shardLocks.keySet());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maybeLogThreadDump(ShardId shardId, String message) {
        if (!this.logger.isDebugEnabled()) {
            return;
        }
        String prefix = Strings.format((String)"hot threads while failing to obtain shard lock for %s: %s", (Object[])new Object[]{shardId, message});
        if (this.shardLockHotThreadsPermit.tryAcquire()) {
            try {
                long now = System.nanoTime();
                if (now <= this.nextShardLockHotThreadsNanos) {
                    return;
                }
                this.nextShardLockHotThreadsNanos = now + TimeUnit.SECONDS.toNanos(60L);
                HotThreads.logLocalHotThreads(this.logger, Level.DEBUG, prefix, ReferenceDocs.SHARD_LOCK_TROUBLESHOOTING);
            }
            catch (Exception e) {
                this.logger.error(Strings.format((String)"could not obtain %s", (Object[])new Object[]{prefix}), (Throwable)e);
            }
            finally {
                this.shardLockHotThreadsPermit.release();
            }
        }
    }

    public boolean hasNodeFile() {
        return this.dataPaths != null && this.locks != null;
    }

    public Path[] nodeDataPaths() {
        this.assertEnvIsLocked();
        Path[] paths = new Path[this.dataPaths.length];
        for (int i = 0; i < paths.length; ++i) {
            paths[i] = this.dataPaths[i].path;
        }
        return paths;
    }

    public Path sharedDataPath() {
        return this.sharedDataPath;
    }

    public String nodeId() {
        return this.nodeMetadata.nodeId();
    }

    public NodeMetadata nodeMetadata() {
        return this.nodeMetadata;
    }

    public DataPath[] dataPaths() {
        this.assertEnvIsLocked();
        if (this.dataPaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        return this.dataPaths;
    }

    public Path[] indexPaths(Index index) {
        this.assertEnvIsLocked();
        Path[] indexPaths = new Path[this.dataPaths.length];
        for (int i = 0; i < this.dataPaths.length; ++i) {
            indexPaths[i] = this.dataPaths[i].resolve(index);
        }
        return indexPaths;
    }

    public Path[] availableShardPaths(ShardId shardId) {
        this.assertEnvIsLocked();
        DataPath[] dataPaths = this.dataPaths();
        Path[] shardLocations = new Path[dataPaths.length];
        for (int i = 0; i < dataPaths.length; ++i) {
            shardLocations[i] = dataPaths[i].resolve(shardId);
        }
        return shardLocations;
    }

    public Set<String> availableIndexFolders() throws IOException {
        return this.availableIndexFolders(Predicates.never());
    }

    public Set<String> availableIndexFolders(Predicate<String> excludeIndexPathIdsPredicate) throws IOException {
        if (this.dataPaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<String> indexFolders = new HashSet<String>();
        for (DataPath dataPath : this.dataPaths) {
            indexFolders.addAll(this.availableIndexFoldersForPath(dataPath, excludeIndexPathIdsPredicate));
        }
        return indexFolders;
    }

    public Set<String> availableIndexFoldersForPath(DataPath dataPath) throws IOException {
        return this.availableIndexFoldersForPath(dataPath, Predicates.never());
    }

    public Set<String> availableIndexFoldersForPath(DataPath dataPath, Predicate<String> excludeIndexPathIdsPredicate) throws IOException {
        if (this.dataPaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<String> indexFolders = new HashSet<String>();
        Path indicesLocation = dataPath.indicesPath;
        if (Files.isDirectory(indicesLocation, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indicesLocation);){
                for (Path index : stream) {
                    String fileName = index.getFileName().toString();
                    if (excludeIndexPathIdsPredicate.test(fileName) || !Files.isDirectory(index, new LinkOption[0])) continue;
                    indexFolders.add(fileName);
                }
            }
        }
        return indexFolders;
    }

    public Path[] resolveIndexFolder(String indexFolderName) {
        if (this.dataPaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        ArrayList<Path> paths = new ArrayList<Path>(this.dataPaths.length);
        for (DataPath dataPath : this.dataPaths) {
            Path indexFolder = dataPath.indicesPath.resolve(indexFolderName);
            if (!Files.exists(indexFolder, new LinkOption[0])) continue;
            paths.add(indexFolder);
        }
        return (Path[])paths.toArray(Path[]::new);
    }

    public Set<ShardId> findAllShardIds(Index index) throws IOException {
        assert (index != null);
        if (this.dataPaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        String indexUniquePathId = index.getUUID();
        for (DataPath dataPath : this.dataPaths) {
            shardIds.addAll(NodeEnvironment.findAllShardsForIndex(dataPath.indicesPath.resolve(indexUniquePathId), index));
        }
        return shardIds;
    }

    public Map<DataPath, Long> shardCountPerPath(Index index) throws IOException {
        assert (index != null);
        if (this.dataPaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashMap<DataPath, Long> shardCountPerPath = new HashMap<DataPath, Long>();
        String indexUniquePathId = index.getUUID();
        for (DataPath dataPath : this.dataPaths) {
            Path indexLocation = dataPath.indicesPath.resolve(indexUniquePathId);
            if (!Files.isDirectory(indexLocation, new LinkOption[0])) continue;
            shardCountPerPath.put(dataPath, Long.valueOf(NodeEnvironment.findAllShardsForIndex(indexLocation, index).size()));
        }
        return shardCountPerPath;
    }

    private static Set<ShardId> findAllShardsForIndex(Path indexPath, Index index) throws IOException {
        assert (indexPath.getFileName().toString().equals(index.getUUID()));
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        if (Files.isDirectory(indexPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                for (Path shardPath : stream) {
                    String fileName = shardPath.getFileName().toString();
                    if (!Files.isDirectory(shardPath, new LinkOption[0]) || !fileName.chars().allMatch(Character::isDigit)) continue;
                    int shardId = Integer.parseInt(fileName);
                    ShardId id = new ShardId(index, shardId);
                    shardIds.add(id);
                }
            }
        }
        return shardIds;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void close() {
        if (!this.closed.compareAndSet(false, true) || this.locks == null) return;
        Lock[] lockArray = this.locks;
        synchronized (this.locks) {
            for (Lock lock : this.locks) {
                try {
                    this.logger.trace("releasing lock [{}]", (Object)lock);
                    lock.close();
                }
                catch (IOException e) {
                    this.logger.trace(() -> "failed to release lock [" + String.valueOf(lock) + "]", (Throwable)e);
                }
            }
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void assertEnvIsLocked() {
        if (this.closed.get() || this.locks == null) return;
        Lock[] lockArray = this.locks;
        synchronized (this.locks) {
            if (this.closed.get()) {
                // ** MonitorExit[var1_1] (shouldn't be in output)
                return;
            }
            for (Lock lock : this.locks) {
                try {
                    lock.ensureValid();
                }
                catch (IOException e) {
                    this.logger.warn("lock assertion failed", (Throwable)e);
                    throw new IllegalStateException("environment is not locked", e);
                }
            }
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void ensureAtomicMoveSupported(DataPath[] dataPaths) throws IOException {
        for (DataPath dataPath : dataPaths) {
            assert (Files.isDirectory(dataPath.path, new LinkOption[0])) : String.valueOf(dataPath.path) + " is not a directory";
            Path src = dataPath.path.resolve(".es_temp_file.tmp");
            Path target = dataPath.path.resolve(".es_temp_file.final");
            try {
                Files.deleteIfExists(src);
                Files.createFile(src, new FileAttribute[0]);
                Files.move(src, target, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (AtomicMoveNotSupportedException ex) {
                throw new IllegalStateException("atomic_move is not supported by the filesystem on path [" + String.valueOf(dataPath.path) + "] atomic_move is required for elasticsearch to work correctly.", ex);
            }
            finally {
                try {
                    Files.deleteIfExists(src);
                }
                finally {
                    Files.deleteIfExists(target);
                }
            }
        }
    }

    private static void ensureNoShardData(DataPath[] dataPaths) throws IOException {
        List<Path> shardDataPaths = NodeEnvironment.collectShardDataPaths(dataPaths);
        if (!shardDataPaths.isEmpty()) {
            String message = String.format(Locale.ROOT, "node does not have the %s role but has shard data: %s. Use 'elasticsearch-node repurpose' tool to clean up", DiscoveryNodeRole.DATA_ROLE.roleName(), shardDataPaths);
            throw new IllegalStateException(message);
        }
    }

    private static void ensureNoIndexMetadata(DataPath[] dataPaths) throws IOException {
        List<Path> indexMetadataPaths = NodeEnvironment.collectIndexMetadataPaths(dataPaths);
        if (!indexMetadataPaths.isEmpty()) {
            String message = String.format(Locale.ROOT, "node does not have the %s and %s roles but has index metadata: %s. Use 'elasticsearch-node repurpose' tool to clean up", DiscoveryNodeRole.DATA_ROLE.roleName(), DiscoveryNodeRole.MASTER_ROLE.roleName(), indexMetadataPaths);
            throw new IllegalStateException(message);
        }
    }

    static List<Path> collectShardDataPaths(DataPath[] dataPaths) throws IOException {
        return NodeEnvironment.collectIndexSubPaths(dataPaths, NodeEnvironment::isShardPath);
    }

    static List<Path> collectIndexMetadataPaths(DataPath[] dataPaths) throws IOException {
        return NodeEnvironment.collectIndexSubPaths(dataPaths, NodeEnvironment::isIndexMetadataPath);
    }

    private static List<Path> collectIndexSubPaths(DataPath[] dataPaths, Predicate<Path> subPathPredicate) throws IOException {
        ArrayList<Path> indexSubPaths = new ArrayList<Path>();
        for (DataPath dataPath : dataPaths) {
            Path indicesPath = dataPath.indicesPath;
            if (!Files.isDirectory(indicesPath, new LinkOption[0])) continue;
            try (DirectoryStream<Path> indexStream = Files.newDirectoryStream(indicesPath);){
                for (Path indexPath : indexStream) {
                    if (!Files.isDirectory(indexPath, new LinkOption[0])) continue;
                    Stream<Path> shardStream = Files.list(indexPath);
                    try {
                        shardStream.filter(subPathPredicate).map(Path::toAbsolutePath).forEach(indexSubPaths::add);
                    }
                    finally {
                        if (shardStream == null) continue;
                        shardStream.close();
                    }
                }
            }
        }
        return indexSubPaths;
    }

    private static boolean isShardPath(Path path) {
        return Files.isDirectory(path, new LinkOption[0]) && path.getFileName().toString().chars().allMatch(Character::isDigit);
    }

    private static boolean isIndexMetadataPath(Path path) {
        return Files.isDirectory(path, new LinkOption[0]) && path.getFileName().toString().equals("_state");
    }

    public static Path resolveBaseCustomLocation(String customDataPath, Path sharedDataPath) {
        if (org.apache.logging.log4j.util.Strings.isNotEmpty((CharSequence)customDataPath)) {
            assert (sharedDataPath != null);
            return sharedDataPath.resolve(customDataPath).resolve("0");
        }
        throw new IllegalArgumentException("no custom index.data_path setting available");
    }

    private Path resolveIndexCustomLocation(String customDataPath, String indexUUID) {
        return NodeEnvironment.resolveIndexCustomLocation(customDataPath, indexUUID, this.sharedDataPath);
    }

    private static Path resolveIndexCustomLocation(String customDataPath, String indexUUID, Path sharedDataPath) {
        return NodeEnvironment.resolveBaseCustomLocation(customDataPath, sharedDataPath).resolve(indexUUID);
    }

    public Path resolveCustomLocation(String customDataPath, ShardId shardId) {
        return NodeEnvironment.resolveCustomLocation(customDataPath, shardId, this.sharedDataPath);
    }

    public static Path resolveCustomLocation(String customDataPath, ShardId shardId, Path sharedDataPath) {
        return NodeEnvironment.resolveIndexCustomLocation(customDataPath, shardId.getIndex().getUUID(), sharedDataPath).resolve(Integer.toString(shardId.id()));
    }

    public static Path shardStatePathToDataPath(Path shardPath) {
        int count = shardPath.getNameCount();
        assert (Integer.parseInt(shardPath.getName(count - 1).toString()) >= 0);
        assert (INDICES_FOLDER.equals(shardPath.getName(count - 3).toString()));
        return shardPath.getParent().getParent().getParent();
    }

    private void assertCanWrite() throws IOException {
        for (Path path : this.nodeDataPaths()) {
            NodeEnvironment.tryWriteTempFile(path);
        }
        for (String indexFolderName : this.availableIndexFolders()) {
            for (Path indexPath : this.resolveIndexFolder(indexFolderName)) {
                Path indexStatePath = indexPath.resolve("_state");
                NodeEnvironment.tryWriteTempFile(indexStatePath);
                NodeEnvironment.tryWriteTempFile(indexPath);
                try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                    for (Path shardPath : stream) {
                        String fileName = shardPath.getFileName().toString();
                        if (!Files.isDirectory(shardPath, new LinkOption[0]) || !fileName.chars().allMatch(Character::isDigit)) continue;
                        Path indexDir = shardPath.resolve("index");
                        Path statePath = shardPath.resolve("_state");
                        Path translogDir = shardPath.resolve("translog");
                        NodeEnvironment.tryWriteTempFile(indexDir);
                        NodeEnvironment.tryWriteTempFile(translogDir);
                        NodeEnvironment.tryWriteTempFile(statePath);
                        NodeEnvironment.tryWriteTempFile(shardPath);
                    }
                }
            }
        }
    }

    private static void tryWriteTempFile(Path path) throws IOException {
        if (Files.exists(path, new LinkOption[0])) {
            Path resolve = path.resolve(TEMP_FILE_NAME);
            try {
                Files.deleteIfExists(resolve);
                Files.createFile(resolve, new FileAttribute[0]);
                Files.delete(resolve);
            }
            catch (IOException ex) {
                throw new IOException("failed to test writes in data directory [" + String.valueOf(path) + "] write permission is required", ex);
            }
        }
    }

    static String getBestDowngradeVersion(String previousNodeVersion) {
        assert (!Build.current().version().startsWith("9."));
        Pattern pattern = Pattern.compile("^7\\.(\\d+)\\.\\d+$");
        Matcher matcher = pattern.matcher(previousNodeVersion);
        if (matcher.matches()) {
            try {
                int minorVersion = Integer.parseInt(matcher.group(1));
                if (minorVersion >= 17) {
                    return previousNodeVersion;
                }
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return "7.17.0";
    }

    public static final class NodeLock
    implements Releasable {
        private final Lock[] locks;
        private final DataPath[] dataPaths;

        public NodeLock(Logger logger, Environment environment, CheckedFunction<Path, Boolean, IOException> pathFunction) throws IOException {
            this(logger, environment, pathFunction, Function.identity());
        }

        public NodeLock(Logger logger, Environment environment, CheckedFunction<Path, Boolean, IOException> pathFunction, Function<Path, Path> subPathMapping) throws IOException {
            this.dataPaths = new DataPath[environment.dataDirs().length];
            this.locks = new Lock[this.dataPaths.length];
            try {
                Path[] dataPaths = environment.dataDirs();
                for (int dirIndex = 0; dirIndex < dataPaths.length; ++dirIndex) {
                    Path dataDir = dataPaths[dirIndex];
                    Path dir = subPathMapping.apply(dataDir);
                    if (!((Boolean)pathFunction.apply((Object)dir)).booleanValue()) continue;
                    try (FSDirectory luceneDir = FSDirectory.open((Path)dir, (LockFactory)NativeFSLockFactory.INSTANCE);){
                        logger.trace("obtaining node lock on {} ...", (Object)dir.toAbsolutePath());
                        this.locks[dirIndex] = luceneDir.obtainLock(NodeEnvironment.NODE_LOCK_FILENAME);
                        this.dataPaths[dirIndex] = new DataPath(dir);
                        continue;
                    }
                    catch (IOException e) {
                        logger.trace(() -> Strings.format((String)"failed to obtain node lock on %s", (Object[])new Object[]{dir.toAbsolutePath()}), (Throwable)e);
                        throw e instanceof LockObtainFailedException ? e : new IOException("failed to obtain lock on " + String.valueOf(dir.toAbsolutePath()), e);
                    }
                }
            }
            catch (IOException e) {
                this.close();
                throw e;
            }
        }

        public DataPath[] getDataPaths() {
            return this.dataPaths;
        }

        public void close() {
            for (int i = 0; i < this.locks.length; ++i) {
                if (this.locks[i] != null) {
                    IOUtils.closeWhileHandlingException((Closeable)this.locks[i]);
                }
                this.locks[i] = null;
            }
        }
    }

    public static class DataPath {
        public final Path path;
        public final Path indicesPath;
        public final FileStore fileStore;
        public final int majorDeviceNumber;
        public final int minorDeviceNumber;

        public DataPath(Path path) throws IOException {
            this.path = path;
            this.indicesPath = path.resolve(NodeEnvironment.INDICES_FOLDER);
            this.fileStore = Environment.getFileStore(path);
            if (this.fileStore.supportsFileAttributeView("lucene")) {
                this.majorDeviceNumber = (Integer)this.fileStore.getAttribute("lucene:major_device_number");
                this.minorDeviceNumber = (Integer)this.fileStore.getAttribute("lucene:minor_device_number");
            } else {
                this.majorDeviceNumber = -1;
                this.minorDeviceNumber = -1;
            }
        }

        public Path resolve(ShardId shardId) {
            return this.resolve(shardId.getIndex()).resolve(Integer.toString(shardId.id()));
        }

        public Path resolve(Index index) {
            return this.resolve(index.getUUID());
        }

        Path resolve(String uuid) {
            return this.indicesPath.resolve(uuid);
        }

        public String toString() {
            return "DataPath{path=" + String.valueOf(this.path) + ", indicesPath=" + String.valueOf(this.indicesPath) + ", fileStore=" + String.valueOf(this.fileStore) + ", majorDeviceNumber=" + this.majorDeviceNumber + ", minorDeviceNumber=" + this.minorDeviceNumber + "}";
        }
    }

    private final class InternalShardLock {
        private final Semaphore mutex = new Semaphore(1);
        private int waitCount = 1;
        private final ShardId shardId;
        private volatile Tuple<Long, String> lockDetails;

        InternalShardLock(ShardId shardId, String details) {
            this.shardId = shardId;
            this.mutex.acquireUninterruptibly();
            this.lockDetails = Tuple.tuple((Object)System.nanoTime(), (Object)details);
        }

        private void release() {
            this.mutex.release();
            this.decWaitCount();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void incWaitCount() {
            Map<ShardId, InternalShardLock> map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                ++this.waitCount;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void decWaitCount() {
            Map<ShardId, InternalShardLock> map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                --this.waitCount;
                NodeEnvironment.this.logger.trace("shard lock wait count for {} is now [{}]", (Object)this.shardId, (Object)this.waitCount);
                if (this.waitCount == 0) {
                    NodeEnvironment.this.logger.trace("last shard lock wait decremented, removing lock for {}", (Object)this.shardId);
                    InternalShardLock remove = NodeEnvironment.this.shardLocks.remove(this.shardId);
                    assert (remove != null) : "Removed lock was null";
                }
            }
        }

        void acquire(long timeoutInMillis, String details) throws ShardLockObtainFailedException {
            try {
                if (!this.mutex.tryAcquire(timeoutInMillis, TimeUnit.MILLISECONDS)) {
                    Tuple<Long, String> lockDetails = this.lockDetails;
                    String message = Strings.format((String)"obtaining shard lock for [%s] timed out after [%dms], lock already held for [%s] with age [%dms]", (Object[])new Object[]{details, timeoutInMillis, lockDetails.v2(), TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - (Long)lockDetails.v1())});
                    NodeEnvironment.this.maybeLogThreadDump(this.shardId, message);
                    throw new ShardLockObtainFailedException(this.shardId, message);
                }
                this.setDetails(details);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ShardLockObtainFailedException(this.shardId, "thread interrupted while trying to obtain shard lock", e);
            }
        }

        public void setDetails(String details) {
            this.lockDetails = Tuple.tuple((Object)System.nanoTime(), (Object)details);
        }
    }

    @FunctionalInterface
    public static interface ShardLocker {
        public ShardLock lock(ShardId var1, String var2, long var3) throws ShardLockObtainFailedException;
    }
}

