/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.inference.registry;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.action.support.SubscribableListener;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.OriginSettingClient;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.core.Strings;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.index.query.ConstantScoreQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.reindex.DeleteByQueryAction;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.inference.InferenceService;
import org.elasticsearch.inference.Model;
import org.elasticsearch.inference.TaskType;
import org.elasticsearch.inference.UnparsedModel;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.sort.SortOrder;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentObject;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentFactory;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xpack.inference.services.ServiceUtils;

public class ModelRegistry {
    private static final String TASK_TYPE_FIELD = "task_type";
    private static final String MODEL_ID_FIELD = "model_id";
    private static final Logger logger = LogManager.getLogger(ModelRegistry.class);
    private final OriginSettingClient client;
    private final List<InferenceService.DefaultConfigId> defaultConfigIds;
    private final Set<String> preventDeletionLock = Collections.newSetFromMap(new ConcurrentHashMap());

    public static UnparsedModel unparsedModelFromMap(ModelConfigMap modelConfigMap) {
        if (modelConfigMap.config() == null) {
            throw new ElasticsearchStatusException("Missing config map", RestStatus.BAD_REQUEST, new Object[0]);
        }
        String inferenceEntityId = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), MODEL_ID_FIELD);
        String service = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), "service");
        String taskTypeStr = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), TaskType.NAME);
        TaskType taskType = TaskType.fromString((String)taskTypeStr);
        return new UnparsedModel(inferenceEntityId, taskType, service, modelConfigMap.config(), modelConfigMap.secrets());
    }

    public ModelRegistry(Client client) {
        this.client = new OriginSettingClient(client, "inference");
        this.defaultConfigIds = new ArrayList<InferenceService.DefaultConfigId>();
    }

    public void addDefaultIds(InferenceService.DefaultConfigId defaultConfigIds) {
        Optional<InferenceService.DefaultConfigId> matched = ModelRegistry.idMatchedDefault(defaultConfigIds.inferenceId(), this.defaultConfigIds);
        if (matched.isPresent()) {
            throw new IllegalStateException("Cannot add default endpoint to the inference endpoint registry with duplicate inference id [" + defaultConfigIds.inferenceId() + "] declared by service [" + defaultConfigIds.service().name() + "]. The inference Id is already use by [" + matched.get().service().name() + "] service.");
        }
        this.defaultConfigIds.add(defaultConfigIds);
    }

    public void getModelWithSecrets(String inferenceEntityId, ActionListener<UnparsedModel> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            if (searchResponse.getHits().getHits().length == 0) {
                Optional<InferenceService.DefaultConfigId> maybeDefault = ModelRegistry.idMatchedDefault(inferenceEntityId, this.defaultConfigIds);
                if (maybeDefault.isPresent()) {
                    this.getDefaultConfig(true, maybeDefault.get(), listener);
                } else {
                    delegate.onFailure((Exception)this.inferenceNotFoundException(inferenceEntityId));
                }
                return;
            }
            delegate.onResponse((Object)ModelRegistry.unparsedModelFromMap(this.createModelConfigMap(searchResponse.getHits(), inferenceEntityId)));
        });
        QueryBuilder queryBuilder = this.documentIdQuery(inferenceEntityId);
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*", ".secrets-inference*"}).setQuery(queryBuilder).setSize(2).request();
        this.client.search(modelSearch, searchListener);
    }

    public void getModel(String inferenceEntityId, ActionListener<UnparsedModel> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            if (searchResponse.getHits().getHits().length == 0) {
                Optional<InferenceService.DefaultConfigId> maybeDefault = ModelRegistry.idMatchedDefault(inferenceEntityId, this.defaultConfigIds);
                if (maybeDefault.isPresent()) {
                    this.getDefaultConfig(true, maybeDefault.get(), listener);
                } else {
                    delegate.onFailure((Exception)this.inferenceNotFoundException(inferenceEntityId));
                }
                return;
            }
            List<UnparsedModel> modelConfigs = this.parseHitsAsModels(searchResponse.getHits()).stream().map(ModelRegistry::unparsedModelFromMap).toList();
            assert (modelConfigs.size() == 1);
            delegate.onResponse((Object)modelConfigs.get(0));
        });
        QueryBuilder queryBuilder = this.documentIdQuery(inferenceEntityId);
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery(queryBuilder).setSize(1).setTrackTotalHits(false).request();
        this.client.search(modelSearch, searchListener);
    }

    private ResourceNotFoundException inferenceNotFoundException(String inferenceEntityId) {
        return new ResourceNotFoundException("Inference endpoint not found [{}]", new Object[]{inferenceEntityId});
    }

    public void getModelsByTaskType(TaskType taskType, ActionListener<List<UnparsedModel>> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            List<UnparsedModel> modelConfigs = this.parseHitsAsModels(searchResponse.getHits()).stream().map(ModelRegistry::unparsedModelFromMap).toList();
            List<InferenceService.DefaultConfigId> defaultConfigsForTaskType = ModelRegistry.taskTypeMatchedDefaults(taskType, this.defaultConfigIds);
            this.addAllDefaultConfigsIfMissing(true, modelConfigs, defaultConfigsForTaskType, (ActionListener<List<UnparsedModel>>)delegate);
        });
        ConstantScoreQueryBuilder queryBuilder = QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.termsQuery((String)TASK_TYPE_FIELD, (String[])new String[]{taskType.toString()}));
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery((QueryBuilder)queryBuilder).setSize(10000).setTrackTotalHits(false).addSort(MODEL_ID_FIELD, SortOrder.ASC).request();
        this.client.search(modelSearch, searchListener);
    }

    public void getAllModels(boolean persistDefaultEndpoints, ActionListener<List<UnparsedModel>> listener) {
        ActionListener searchListener = listener.delegateFailureAndWrap((delegate, searchResponse) -> {
            List<UnparsedModel> foundConfigs = this.parseHitsAsModels(searchResponse.getHits()).stream().map(ModelRegistry::unparsedModelFromMap).toList();
            this.addAllDefaultConfigsIfMissing(persistDefaultEndpoints, foundConfigs, this.defaultConfigIds, (ActionListener<List<UnparsedModel>>)delegate);
        });
        ConstantScoreQueryBuilder queryBuilder = QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.existsQuery((String)TASK_TYPE_FIELD));
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery((QueryBuilder)queryBuilder).setSize(10000).setTrackTotalHits(false).addSort(MODEL_ID_FIELD, SortOrder.ASC).request();
        this.client.search(modelSearch, searchListener);
    }

    private void addAllDefaultConfigsIfMissing(boolean persistDefaultEndpoints, List<UnparsedModel> foundConfigs, List<InferenceService.DefaultConfigId> matchedDefaults, ActionListener<List<UnparsedModel>> listener) {
        Set foundIds = foundConfigs.stream().map(UnparsedModel::inferenceEntityId).collect(Collectors.toSet());
        List<InferenceService.DefaultConfigId> missing = matchedDefaults.stream().filter(d -> !foundIds.contains(d.inferenceId())).toList();
        if (missing.isEmpty()) {
            listener.onResponse(foundConfigs);
        } else {
            GroupedActionListener groupedListener = new GroupedActionListener(missing.size(), listener.delegateFailure((delegate, listOfModels) -> {
                ArrayList<UnparsedModel> allConfigs = new ArrayList<UnparsedModel>();
                allConfigs.addAll(foundConfigs);
                allConfigs.addAll((Collection<UnparsedModel>)listOfModels);
                allConfigs.sort(Comparator.comparing(UnparsedModel::inferenceEntityId));
                delegate.onResponse(allConfigs);
            }));
            for (InferenceService.DefaultConfigId required : missing) {
                this.getDefaultConfig(persistDefaultEndpoints, required, (ActionListener<UnparsedModel>)groupedListener);
            }
        }
    }

    private void getDefaultConfig(boolean persistDefaultEndpoints, InferenceService.DefaultConfigId defaultConfig, ActionListener<UnparsedModel> listener) {
        defaultConfig.service().defaultConfigs(listener.delegateFailureAndWrap((delegate, models) -> {
            boolean foundModel = false;
            for (Model m : models) {
                if (!m.getInferenceEntityId().equals(defaultConfig.inferenceId())) continue;
                foundModel = true;
                if (persistDefaultEndpoints) {
                    this.storeDefaultEndpoint(m, () -> listener.onResponse((Object)ModelRegistry.modelToUnparsedModel(m)));
                    break;
                }
                listener.onResponse((Object)ModelRegistry.modelToUnparsedModel(m));
                break;
            }
            if (!foundModel) {
                listener.onFailure((Exception)new IllegalStateException("Configuration not found for default inference id [" + defaultConfig.inferenceId() + "]"));
            }
        }));
    }

    private void storeDefaultEndpoint(Model preconfigured, Runnable runAfter) {
        ActionListener responseListener = ActionListener.wrap(success -> logger.debug("Added default inference endpoint [{}]", (Object)preconfigured.getInferenceEntityId()), exception -> {
            if (exception instanceof ResourceAlreadyExistsException) {
                logger.debug("Default inference id [{}] already exists", (Object)preconfigured.getInferenceEntityId());
            } else {
                logger.error("Failed to store default inference id [" + preconfigured.getInferenceEntityId() + "]", (Throwable)exception);
            }
        });
        this.storeModel(preconfigured, (ActionListener<Boolean>)ActionListener.runAfter((ActionListener)responseListener, (Runnable)runAfter));
    }

    private ArrayList<ModelConfigMap> parseHitsAsModels(SearchHits hits) {
        ArrayList<ModelConfigMap> modelConfigs = new ArrayList<ModelConfigMap>();
        for (SearchHit hit : hits) {
            modelConfigs.add(new ModelConfigMap(hit.getSourceAsMap(), Map.of()));
        }
        return modelConfigs;
    }

    private ModelConfigMap createModelConfigMap(SearchHits hits, String inferenceEntityId) {
        Map mappedHits = Arrays.stream(hits.getHits()).collect(Collectors.toMap(hit -> {
            if (hit.getIndex().startsWith(".inference")) {
                return ".inference";
            }
            if (hit.getIndex().startsWith(".secrets-inference")) {
                return ".secrets-inference";
            }
            logger.warn(Strings.format((String)"Found invalid index for inference endpoint [%s] at index [%s]", (Object[])new Object[]{inferenceEntityId, hit.getIndex()}));
            throw new IllegalArgumentException(Strings.format((String)"Invalid result while loading inference endpoint [%s] index: [%s]. Try deleting and reinitializing the service", (Object[])new Object[]{inferenceEntityId, hit.getIndex()}));
        }, Function.identity()));
        if (!mappedHits.containsKey(".inference") || !mappedHits.containsKey(".secrets-inference") || mappedHits.size() > 2) {
            logger.warn(Strings.format((String)"Failed to load inference endpoint [%s], found endpoint parts from index prefixes: [%s]", (Object[])new Object[]{inferenceEntityId, mappedHits.keySet()}));
            throw new IllegalStateException(Strings.format((String)"Failed to load inference endpoint [%s]. Endpoint is in an invalid state, try deleting and reinitializing the service", (Object[])new Object[]{inferenceEntityId}));
        }
        return new ModelConfigMap(((SearchHit)mappedHits.get(".inference")).getSourceAsMap(), ((SearchHit)mappedHits.get(".secrets-inference")).getSourceAsMap());
    }

    public void updateModelTransaction(Model newModel, Model existingModel, ActionListener<Boolean> finalListener) {
        String inferenceEntityId = newModel.getConfigurations().getInferenceEntityId();
        logger.info("Attempting to store update to inference endpoint [{}]", (Object)inferenceEntityId);
        if (this.preventDeletionLock.contains(inferenceEntityId)) {
            logger.warn(Strings.format((String)"Attempted to update endpoint [{}] that is already being updated", (Object[])new Object[]{inferenceEntityId}));
            finalListener.onFailure((Exception)new ElasticsearchStatusException("Endpoint [{}] is currently being updated. Try again once the update completes", RestStatus.CONFLICT, new Object[]{inferenceEntityId}));
            return;
        }
        this.preventDeletionLock.add(inferenceEntityId);
        SubscribableListener.newForked(subListener -> {
            IndexRequest configRequest = ModelRegistry.createIndexRequest(Model.documentId((String)inferenceEntityId), ".inference", (ToXContentObject)newModel.getConfigurations(), true);
            ActionListener storeConfigListener = subListener.delegateResponse((l, e) -> {
                this.preventDeletionLock.remove(inferenceEntityId);
                l.onFailure(e);
            });
            this.client.prepareBulk().add(configRequest).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(storeConfigListener);
        }).andThen((subListener, configResponse) -> {
            if (configResponse.hasFailures()) {
                logger.error(Strings.format((String)"Failed to update inference endpoint [%s] due to [%s]", (Object[])new Object[]{inferenceEntityId, configResponse.buildFailureMessage()}));
                this.preventDeletionLock.remove(inferenceEntityId);
                finalListener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to update inference endpoint [%s] due to [%s]", (Object[])new Object[]{inferenceEntityId}), RestStatus.INTERNAL_SERVER_ERROR, new Object[]{configResponse.buildFailureMessage()}));
            } else {
                IndexRequest secretsRequest = ModelRegistry.createIndexRequest(Model.documentId((String)newModel.getConfigurations().getInferenceEntityId()), ".secrets-inference", (ToXContentObject)newModel.getSecrets(), true);
                ActionListener storeSecretsListener = subListener.delegateResponse((l, e) -> {
                    this.preventDeletionLock.remove(inferenceEntityId);
                    l.onFailure(e);
                });
                this.client.prepareBulk().add(secretsRequest).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(storeSecretsListener);
            }
        }).andThen((subListener, secretsResponse) -> {
            if (secretsResponse.hasFailures()) {
                IndexRequest configRequest = ModelRegistry.createIndexRequest(Model.documentId((String)inferenceEntityId), ".inference", (ToXContentObject)existingModel.getConfigurations(), true);
                logger.error("Failed to update inference endpoint secrets [{}], attempting rolling back to previous state", (Object)inferenceEntityId);
                ActionListener rollbackConfigListener = subListener.delegateResponse((l, e) -> {
                    this.preventDeletionLock.remove(inferenceEntityId);
                    l.onFailure(e);
                });
                this.client.prepareBulk().add(configRequest).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(rollbackConfigListener);
            } else {
                this.preventDeletionLock.remove(inferenceEntityId);
                finalListener.onResponse((Object)true);
            }
        }).andThen((subListener, configResponse) -> {
            this.preventDeletionLock.remove(inferenceEntityId);
            if (configResponse.hasFailures()) {
                logger.error(Strings.format((String)"Failed to update inference endpoint [%s] due to [%s]", (Object[])new Object[]{inferenceEntityId, configResponse.buildFailureMessage()}));
                finalListener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to rollback while handling failure to update inference endpoint [%s]. Endpoint may be in an inconsistent state due to [%s]", (Object[])new Object[]{inferenceEntityId, configResponse.buildFailureMessage()}), RestStatus.INTERNAL_SERVER_ERROR, new Object[0]));
            } else {
                logger.warn("Failed to update inference endpoint [{}], successfully rolled back to previous state", (Object)inferenceEntityId);
                finalListener.onResponse((Object)false);
            }
        });
    }

    public void storeModel(Model model, ActionListener<Boolean> listener) {
        ActionListener<BulkResponse> bulkResponseActionListener = ModelRegistry.getStoreModelListener(model, listener);
        IndexRequest configRequest = ModelRegistry.createIndexRequest(Model.documentId((String)model.getConfigurations().getInferenceEntityId()), ".inference", (ToXContentObject)model.getConfigurations(), false);
        IndexRequest secretsRequest = ModelRegistry.createIndexRequest(Model.documentId((String)model.getConfigurations().getInferenceEntityId()), ".secrets-inference", (ToXContentObject)model.getSecrets(), false);
        this.client.prepareBulk().add(configRequest).add(secretsRequest).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE).execute(bulkResponseActionListener);
    }

    private static ActionListener<BulkResponse> getStoreModelListener(Model model, ActionListener<Boolean> listener) {
        return ActionListener.wrap(bulkItemResponses -> {
            String inferenceEntityId = model.getConfigurations().getInferenceEntityId();
            if (bulkItemResponses.getItems().length == 0) {
                logger.warn(Strings.format((String)"Storing inference endpoint [%s] failed, no items were received from the bulk response", (Object[])new Object[]{inferenceEntityId}));
                listener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to store inference endpoint [%s], invalid bulk response received. Try reinitializing the service", (Object[])new Object[]{inferenceEntityId}), RestStatus.INTERNAL_SERVER_ERROR, new Object[0]));
                return;
            }
            BulkItemResponse.Failure failure = ModelRegistry.getFirstBulkFailure(bulkItemResponses);
            if (failure == null) {
                listener.onResponse((Object)true);
                return;
            }
            ModelRegistry.logBulkFailures(model.getConfigurations().getInferenceEntityId(), bulkItemResponses);
            if (ExceptionsHelper.unwrapCause((Throwable)failure.getCause()) instanceof VersionConflictEngineException) {
                listener.onFailure((Exception)new ResourceAlreadyExistsException("Inference endpoint [{}] already exists", new Object[]{inferenceEntityId}));
                return;
            }
            listener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to store inference endpoint [%s]", (Object[])new Object[]{inferenceEntityId}), RestStatus.INTERNAL_SERVER_ERROR, (Throwable)failure.getCause(), new Object[0]));
        }, e -> {
            String errorMessage = Strings.format((String)"Failed to store inference endpoint [%s]", (Object[])new Object[]{model.getConfigurations().getInferenceEntityId()});
            logger.warn(errorMessage, (Throwable)e);
            listener.onFailure((Exception)new ElasticsearchStatusException(errorMessage, RestStatus.INTERNAL_SERVER_ERROR, (Throwable)e, new Object[0]));
        });
    }

    private static void logBulkFailures(String inferenceEntityId, BulkResponse bulkResponse) {
        for (BulkItemResponse item : bulkResponse.getItems()) {
            if (!item.isFailed()) continue;
            logger.warn(Strings.format((String)"Failed to store inference endpoint [%s] index: [%s] bulk failure message [%s]", (Object[])new Object[]{inferenceEntityId, item.getIndex(), item.getFailureMessage()}));
        }
    }

    private static BulkItemResponse.Failure getFirstBulkFailure(BulkResponse bulkResponse) {
        for (BulkItemResponse item : bulkResponse.getItems()) {
            if (!item.isFailed()) continue;
            return item.getFailure();
        }
        return null;
    }

    public void deleteModel(String inferenceEntityId, ActionListener<Boolean> listener) {
        if (this.preventDeletionLock.contains(inferenceEntityId)) {
            listener.onFailure((Exception)new ElasticsearchStatusException("Model is currently being updated, you may delete the model once the update completes", RestStatus.CONFLICT, new Object[0]));
            return;
        }
        DeleteByQueryRequest request = (DeleteByQueryRequest)new DeleteByQueryRequest().setAbortOnVersionConflict(false);
        request.indices(new String[]{".inference*", ".secrets-inference*"});
        request.setQuery(this.documentIdQuery(inferenceEntityId));
        request.setRefresh(true);
        this.client.execute((ActionType)DeleteByQueryAction.INSTANCE, (ActionRequest)request, listener.delegateFailureAndWrap((l, r) -> l.onResponse((Object)Boolean.TRUE)));
    }

    private static IndexRequest createIndexRequest(String docId, String indexName, ToXContentObject body, boolean allowOverwriting) {
        IndexRequest indexRequest;
        block8: {
            XContentBuilder builder = XContentFactory.jsonBuilder();
            try {
                IndexRequest request = new IndexRequest(indexName);
                XContentBuilder source = body.toXContent(builder, (ToXContent.Params)new ToXContent.MapParams(Map.of("for_index", Boolean.TRUE.toString())));
                DocWriteRequest.OpType operation = allowOverwriting ? DocWriteRequest.OpType.INDEX : DocWriteRequest.OpType.CREATE;
                indexRequest = request.opType(operation).id(docId).source(source);
                if (builder == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (builder != null) {
                        try {
                            builder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new ElasticsearchException(Strings.format((String)"Unexpected serialization exception for index [%s] doc [%s]", (Object[])new Object[]{indexName, docId}), (Throwable)ex, new Object[0]);
                }
            }
            builder.close();
        }
        return indexRequest;
    }

    private static UnparsedModel modelToUnparsedModel(Model model) {
        UnparsedModel unparsedModel;
        block8: {
            XContentBuilder builder = XContentFactory.jsonBuilder();
            try {
                model.getConfigurations().toXContent(builder, (ToXContent.Params)new ToXContent.MapParams(Map.of("for_index", Boolean.TRUE.toString())));
                Map modelConfigMap = (Map)XContentHelper.convertToMap((BytesReference)BytesReference.bytes((XContentBuilder)builder), (boolean)false, (XContentType)builder.contentType()).v2();
                unparsedModel = ModelRegistry.unparsedModelFromMap(new ModelConfigMap(modelConfigMap, new HashMap<String, Object>()));
                if (builder == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (builder != null) {
                        try {
                            builder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new ElasticsearchException("[{}] Error serializing inference endpoint configuration", new Object[]{model.getInferenceEntityId(), ex});
                }
            }
            builder.close();
        }
        return unparsedModel;
    }

    private QueryBuilder documentIdQuery(String inferenceEntityId) {
        return QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.idsQuery().addIds(new String[]{Model.documentId((String)inferenceEntityId)}));
    }

    static Optional<InferenceService.DefaultConfigId> idMatchedDefault(String inferenceId, List<InferenceService.DefaultConfigId> defaultConfigIds) {
        return defaultConfigIds.stream().filter(defaultConfigId -> defaultConfigId.inferenceId().equals(inferenceId)).findFirst();
    }

    static List<InferenceService.DefaultConfigId> taskTypeMatchedDefaults(TaskType taskType, List<InferenceService.DefaultConfigId> defaultConfigIds) {
        return defaultConfigIds.stream().filter(defaultConfigId -> defaultConfigId.taskType().equals((Object)taskType)).collect(Collectors.toList());
    }

    public record ModelConfigMap(Map<String, Object> config, Map<String, Object> secrets) {
    }
}

