/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.settings;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.AccessDeniedException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import javax.crypto.AEADBadTagException;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.lucene.backward_codecs.store.EndiannessReverserUtil;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.IndexFormatTooNewException;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.apache.lucene.store.ByteArrayDataInput;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.cli.UserException;
import org.elasticsearch.common.CheckedSupplier;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.settings.SecureSetting;
import org.elasticsearch.common.settings.SecureSettings;
import org.elasticsearch.common.settings.SecureString;
import org.elasticsearch.common.settings.Setting;

public class KeyStoreWrapper
implements SecureSettings {
    public static final String PROMPT = "Enter password for the elasticsearch keystore : ";
    private static final Pattern ALLOWED_SETTING_NAME = Pattern.compile("[A-Za-z0-9_\\-.]+");
    public static final Setting<SecureString> SEED_SETTING = SecureSetting.secureString("keystore.seed", null, new Setting.Property[0]);
    private static final char[] SEED_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789~!@#$%^&*-_=+?".toCharArray();
    public static final String KEYSTORE_FILENAME = "elasticsearch.keystore";
    private static final int MIN_FORMAT_VERSION = 3;
    public static final int V3_VERSION = 3;
    public static final int V4_VERSION = 4;
    public static final int LE_VERSION = 5;
    public static final int HIGHER_KDF_ITERATION_COUNT_VERSION = 6;
    public static final int CURRENT_VERSION = 6;
    private static final String KDF_ALGO = "PBKDF2WithHmacSHA512";
    private static final int KDF_ITERS = 210000;
    private static final int CIPHER_KEY_BITS = 128;
    private static final int GCM_TAG_BITS = 128;
    private static final String CIPHER_ALGO = "AES";
    private static final String CIPHER_MODE = "GCM";
    private static final String CIPHER_PADDING = "NoPadding";
    private final int formatVersion;
    private final boolean hasPassword;
    private final byte[] dataBytes;
    private final SetOnce<Map<String, Entry>> entries = new SetOnce();
    private volatile boolean closed;

    private KeyStoreWrapper(int formatVersion, boolean hasPassword, byte[] dataBytes) {
        this.formatVersion = formatVersion;
        this.hasPassword = hasPassword;
        this.dataBytes = dataBytes;
    }

    public KeyStoreWrapper(StreamInput input) throws IOException {
        this.formatVersion = input.readInt();
        this.hasPassword = input.readBoolean();
        this.dataBytes = input.readOptionalByteArray();
        this.entries.set(input.readMap(Entry::new));
        this.closed = input.readBoolean();
    }

    public int getFormatVersion() {
        return this.formatVersion;
    }

    public static Path keystorePath(Path configDir) {
        return configDir.resolve(KEYSTORE_FILENAME);
    }

    public static KeyStoreWrapper create() {
        KeyStoreWrapper wrapper = new KeyStoreWrapper(6, false, null);
        wrapper.entries.set(new HashMap());
        KeyStoreWrapper.addBootstrapSeed(wrapper);
        return wrapper;
    }

    public static void addBootstrapSeed(KeyStoreWrapper wrapper) {
        assert (!wrapper.getSettingNames().contains(SEED_SETTING.getKey()));
        SecureRandom random = Randomness.createSecure();
        int passwordLength = 20;
        char[] characters = new char[passwordLength];
        for (int i = 0; i < passwordLength; ++i) {
            characters[i] = SEED_CHARS[random.nextInt(SEED_CHARS.length)];
        }
        wrapper.setString(SEED_SETTING.getKey(), characters);
        Arrays.fill(characters, '\u0000');
    }

    public static KeyStoreWrapper bootstrap(Path configDir, CheckedSupplier<SecureString, Exception> passwordSupplier) throws Exception {
        KeyStoreWrapper keystore = KeyStoreWrapper.load(configDir);
        SecureString password = keystore != null && keystore.hasPassword() ? passwordSupplier.get() : new SecureString(new char[0]);
        try (SecureString secureString = password;){
            if (keystore == null) {
                KeyStoreWrapper keyStoreWrapper = KeyStoreWrapper.create();
                keyStoreWrapper.save(configDir, new char[0]);
                KeyStoreWrapper keyStoreWrapper2 = keyStoreWrapper;
                return keyStoreWrapper2;
            }
            keystore.decrypt(password.getChars());
            KeyStoreWrapper.upgrade(keystore, configDir, password.getChars());
        }
        return keystore;
    }

    public static KeyStoreWrapper load(Path configDir) throws IOException {
        Path keystoreFile = KeyStoreWrapper.keystorePath(configDir);
        if (!Files.exists(keystoreFile, new LinkOption[0])) {
            return null;
        }
        NIOFSDirectory directory = new NIOFSDirectory(configDir);
        try (ChecksumIndexInput input = directory.openChecksumInput(KEYSTORE_FILENAME, IOContext.READONCE);){
            boolean hasPassword;
            int formatVersion;
            try {
                formatVersion = CodecUtil.checkHeader((DataInput)input, (String)KEYSTORE_FILENAME, (int)3, (int)6);
            }
            catch (IndexFormatTooOldException e) {
                throw new IllegalStateException("The Elasticsearch keystore [" + String.valueOf(keystoreFile) + "] format is too old. You should delete and recreate it in order to upgrade.", e);
            }
            catch (IndexFormatTooNewException e) {
                throw new IllegalStateException("The Elasticsearch keystore [" + String.valueOf(keystoreFile) + "] format is too new. Are you trying to downgrade? You should delete and recreate it in order to downgrade.", e);
            }
            byte hasPasswordByte = input.readByte();
            boolean bl = hasPassword = hasPasswordByte == 1;
            if (!hasPassword && hasPasswordByte != 0) {
                throw new IllegalStateException("hasPassword boolean is corrupt: " + String.format(Locale.ROOT, "%02x", hasPasswordByte));
            }
            int dataBytesLen = formatVersion < 5 ? Integer.reverseBytes(input.readInt()) : input.readInt();
            byte[] dataBytes = new byte[dataBytesLen];
            input.readBytes(dataBytes, 0, dataBytesLen);
            CodecUtil.checkFooter((ChecksumIndexInput)input);
            KeyStoreWrapper keyStoreWrapper = new KeyStoreWrapper(formatVersion, hasPassword, dataBytes);
            return keyStoreWrapper;
        }
    }

    public static void upgrade(KeyStoreWrapper wrapper, Path configDir, char[] password) throws Exception {
        if (wrapper.getFormatVersion() == 6 && wrapper.getSettingNames().contains(SEED_SETTING.getKey())) {
            return;
        }
        if (!wrapper.getSettingNames().contains(SEED_SETTING.getKey())) {
            KeyStoreWrapper.addBootstrapSeed(wrapper);
        }
        wrapper.save(configDir, password);
    }

    @Override
    public boolean isLoaded() {
        return this.entries.get() != null;
    }

    public boolean hasPassword() {
        return this.hasPassword;
    }

    private static Cipher createCipher(int opmode, char[] password, byte[] salt, byte[] iv, int kdfIters) throws GeneralSecurityException {
        SecretKey secretKey;
        PBEKeySpec keySpec = new PBEKeySpec(password, salt, kdfIters, 128);
        SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(KDF_ALGO);
        try {
            secretKey = keyFactory.generateSecret(keySpec);
        }
        catch (Error e) {
            throw new GeneralSecurityException("Error generating an encryption key from the provided password", e);
        }
        SecretKeySpec secret = new SecretKeySpec(secretKey.getEncoded(), CIPHER_ALGO);
        GCMParameterSpec spec = new GCMParameterSpec(128, iv);
        Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
        cipher.init(opmode, (Key)secret, spec);
        cipher.updateAAD(salt);
        return cipher;
    }

    private static int getKdfIterationCountForVersion(int formatVersion) {
        return formatVersion < 6 ? 10000 : 210000;
    }

    public void decrypt(char[] password) throws GeneralSecurityException, IOException {
        byte[] encryptedBytes;
        byte[] iv;
        byte[] salt;
        if (this.entries.get() != null) {
            throw new IllegalStateException("Keystore has already been decrypted");
        }
        try {
            ByteArrayDataInput input = new ByteArrayDataInput(this.dataBytes);
            ByteArrayDataInput maybeWrappedInput = this.formatVersion < 5 ? EndiannessReverserUtil.wrapDataInput((DataInput)input) : input;
            salt = KeyStoreWrapper.readByteArray((DataInput)maybeWrappedInput);
            iv = KeyStoreWrapper.readByteArray((DataInput)maybeWrappedInput);
            encryptedBytes = KeyStoreWrapper.readByteArray((DataInput)maybeWrappedInput);
            if (!input.eof()) {
                throw new SecurityException("Keystore has been corrupted or tampered with");
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new SecurityException("Keystore has been corrupted or tampered with", e);
        }
        Cipher cipher = KeyStoreWrapper.createCipher(2, password, salt, iv, KeyStoreWrapper.getKdfIterationCountForVersion(this.formatVersion));
        try (ByteArrayInputStream bytesStream = new ByteArrayInputStream(encryptedBytes);
             CipherInputStream cipherStream = new CipherInputStream(bytesStream, cipher);
             DataInputStream input = new DataInputStream(cipherStream);){
            this.entries.set(new HashMap());
            int numEntries = input.readInt();
            while (numEntries-- > 0) {
                String setting = input.readUTF();
                if (this.formatVersion == 3) {
                    input.readUTF();
                }
                int entrySize = input.readInt();
                byte[] entryBytes = new byte[entrySize];
                input.readFully(entryBytes);
                ((Map)this.entries.get()).put(setting, new Entry(entryBytes));
            }
            if (input.read() != -1) {
                throw new SecurityException("Keystore has been corrupted or tampered with");
            }
        }
        catch (IOException e) {
            if (e.getCause() instanceof AEADBadTagException) {
                throw new SecurityException("Provided keystore password was incorrect", e);
            }
            throw new SecurityException("Keystore has been corrupted or tampered with", e);
        }
    }

    private static byte[] readByteArray(DataInput input) throws IOException {
        int len = input.readInt();
        byte[] b = new byte[len];
        input.readBytes(b, 0, len);
        return b;
    }

    private byte[] encrypt(char[] password, byte[] salt, byte[] iv, int kdfIterationCount) throws GeneralSecurityException, IOException {
        assert (this.isLoaded());
        ByteArrayOutputStream bytes = new ByteArrayOutputStream();
        Cipher cipher = KeyStoreWrapper.createCipher(1, password, salt, iv, kdfIterationCount);
        try (CipherOutputStream cipherStream = new CipherOutputStream(bytes, cipher);
             DataOutputStream output = new DataOutputStream(cipherStream);){
            output.writeInt(((Map)this.entries.get()).size());
            for (Map.Entry mapEntry : ((Map)this.entries.get()).entrySet()) {
                output.writeUTF((String)mapEntry.getKey());
                byte[] entryBytes = ((Entry)mapEntry.getValue()).bytes;
                output.writeInt(entryBytes.length);
                output.write(entryBytes);
            }
        }
        return bytes.toByteArray();
    }

    public synchronized void save(Path configDir, char[] password) throws Exception {
        this.save(configDir, password, true);
    }

    public synchronized void save(Path configDir, char[] password, boolean preservePermissions) throws Exception {
        this.ensureOpen();
        NIOFSDirectory directory = new NIOFSDirectory(configDir);
        String tmpFile = "elasticsearch.keystore.tmp";
        Path keystoreTempFile = configDir.resolve(tmpFile);
        try (IndexOutput output = directory.createOutput(tmpFile, IOContext.DEFAULT);){
            CodecUtil.writeHeader((DataOutput)output, (String)KEYSTORE_FILENAME, (int)6);
            output.writeByte(password.length == 0 ? (byte)0 : 1);
            SecureRandom random = Randomness.createSecure();
            byte[] salt = new byte[64];
            random.nextBytes(salt);
            byte[] iv = new byte[12];
            random.nextBytes(iv);
            byte[] encryptedBytes = this.encrypt(password, salt, iv, KeyStoreWrapper.getKdfIterationCountForVersion(6));
            output.writeInt(4 + salt.length + 4 + iv.length + 4 + encryptedBytes.length);
            output.writeInt(salt.length);
            output.writeBytes(salt, salt.length);
            output.writeInt(iv.length);
            output.writeBytes(iv, iv.length);
            output.writeInt(encryptedBytes.length);
            output.writeBytes(encryptedBytes, encryptedBytes.length);
            CodecUtil.writeFooter((IndexOutput)output);
        }
        catch (AccessDeniedException e) {
            String message = String.format(Locale.ROOT, "unable to create temporary keystore at [%s], write permissions required for [%s] or run [elasticsearch-keystore upgrade]", keystoreTempFile, configDir);
            throw new UserException(78, message, (Throwable)e);
        }
        catch (Exception e) {
            try {
                Files.deleteIfExists(keystoreTempFile);
            }
            catch (Exception ex) {
                e.addSuppressed(e);
            }
            throw e;
        }
        Path keystoreFile = KeyStoreWrapper.keystorePath(configDir);
        if (preservePermissions) {
            try {
                if (Files.exists(keystoreFile, LinkOption.NOFOLLOW_LINKS) && !Files.getOwner(keystoreTempFile, LinkOption.NOFOLLOW_LINKS).equals(Files.getOwner(keystoreFile, LinkOption.NOFOLLOW_LINKS))) {
                    String message = String.format(Locale.ROOT, "will not overwrite keystore at [%s], because this incurs changing the file owner", keystoreFile);
                    throw new UserException(78, message);
                }
                PosixFileAttributeView attrs = Files.getFileAttributeView(keystoreTempFile, PosixFileAttributeView.class, new LinkOption[0]);
                if (attrs != null) {
                    attrs.setPermissions(PosixFilePermissions.fromString("rw-rw----"));
                }
            }
            catch (Exception e) {
                try {
                    Files.deleteIfExists(keystoreTempFile);
                }
                catch (Exception ex) {
                    e.addSuppressed(ex);
                }
                throw e;
            }
        }
        try {
            Files.move(keystoreTempFile, keystoreFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (Exception e) {
            try {
                Files.deleteIfExists(keystoreTempFile);
            }
            catch (Exception ex) {
                e.addSuppressed(ex);
            }
            throw e;
        }
    }

    @Override
    public Set<String> getSettingNames() {
        assert (this.entries.get() != null) : "Keystore is not loaded";
        return ((Map)this.entries.get()).keySet();
    }

    @Override
    public synchronized SecureString getString(String setting) {
        this.ensureOpen();
        Entry entry = (Entry)((Map)this.entries.get()).get(setting);
        ByteBuffer byteBuffer = ByteBuffer.wrap(entry.bytes);
        CharBuffer charBuffer = StandardCharsets.UTF_8.decode(byteBuffer);
        return new SecureString(Arrays.copyOfRange(charBuffer.array(), charBuffer.position(), charBuffer.limit()));
    }

    @Override
    public synchronized InputStream getFile(String setting) {
        this.ensureOpen();
        Entry entry = (Entry)((Map)this.entries.get()).get(setting);
        return new ByteArrayInputStream(entry.bytes);
    }

    @Override
    public byte[] getSHA256Digest(String setting) {
        assert (this.entries.get() != null) : "Keystore is not loaded";
        Entry entry = (Entry)((Map)this.entries.get()).get(setting);
        return entry.sha256Digest;
    }

    public static void validateSettingName(String setting) {
        if (!ALLOWED_SETTING_NAME.matcher(setting).matches()) {
            throw new IllegalArgumentException("Setting name [" + setting + "] does not match the allowed setting name pattern [" + ALLOWED_SETTING_NAME.pattern() + "]");
        }
    }

    public synchronized void setString(String setting, char[] value) {
        this.ensureOpen();
        KeyStoreWrapper.validateSettingName(setting);
        ByteBuffer byteBuffer = StandardCharsets.UTF_8.encode(CharBuffer.wrap(value));
        byte[] bytes = Arrays.copyOfRange(byteBuffer.array(), byteBuffer.position(), byteBuffer.limit());
        Entry oldEntry = ((Map)this.entries.get()).put(setting, new Entry(bytes));
        if (oldEntry != null) {
            Arrays.fill(oldEntry.bytes, (byte)0);
        }
    }

    public synchronized void setFile(String setting, byte[] bytes) {
        this.ensureOpen();
        KeyStoreWrapper.validateSettingName(setting);
        Entry oldEntry = ((Map)this.entries.get()).put(setting, new Entry(Arrays.copyOf(bytes, bytes.length)));
        if (oldEntry != null) {
            Arrays.fill(oldEntry.bytes, (byte)0);
        }
    }

    public void remove(String setting) {
        this.ensureOpen();
        Entry oldEntry = (Entry)((Map)this.entries.get()).remove(setting);
        if (oldEntry != null) {
            Arrays.fill(oldEntry.bytes, (byte)0);
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            throw new IllegalStateException("Keystore is closed");
        }
        assert (this.isLoaded()) : "Keystore is not loaded";
    }

    @Override
    public synchronized void close() {
        this.closed = true;
        if (null != this.entries.get() && !((Map)this.entries.get()).isEmpty()) {
            for (Entry entry : ((Map)this.entries.get()).values()) {
                Arrays.fill(entry.bytes, (byte)0);
            }
        }
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        out.writeInt(this.formatVersion);
        out.writeBoolean(this.hasPassword);
        out.writeOptionalByteArray(this.dataBytes);
        Map entriesMap = (Map)this.entries.get();
        out.writeMap(entriesMap == null ? Map.of() : entriesMap, StreamOutput::writeWriteable);
        out.writeBoolean(this.closed);
    }

    private static class Entry
    implements Writeable {
        final byte[] bytes;
        final byte[] sha256Digest;

        Entry(byte[] bytes) {
            this.bytes = bytes;
            this.sha256Digest = MessageDigests.sha256().digest(bytes);
        }

        Entry(StreamInput input) throws IOException {
            this.bytes = input.readByteArray();
            this.sha256Digest = input.readByteArray();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeByteArray(this.bytes);
            out.writeByteArray(this.sha256Digest);
        }
    }
}

