/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonWebServiceRequest;
import com.amazonaws.Request;
import com.amazonaws.Response;
import com.amazonaws.metrics.MetricType;
import com.amazonaws.metrics.RequestMetricCollector;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.StorageClass;
import com.amazonaws.util.AWSRequestMetrics;
import com.amazonaws.util.TimingInfo;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.LongAdder;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobContainer;
import org.elasticsearch.repositories.s3.S3RepositoriesMetrics;
import org.elasticsearch.repositories.s3.S3Repository;
import org.elasticsearch.repositories.s3.S3Service;
import org.elasticsearch.repositories.s3.SocketAccess;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.threadpool.ThreadPool;

class S3BlobStore
implements BlobStore {
    public static final String CUSTOM_QUERY_PARAMETER_PURPOSE = "x-purpose";
    static final int MAX_BULK_DELETES = 1000;
    private static final Logger logger = LogManager.getLogger(S3BlobStore.class);
    private final S3Service service;
    private final BigArrays bigArrays;
    private final String bucket;
    private final ByteSizeValue bufferSize;
    private final boolean serverSideEncryption;
    private final CannedAccessControlList cannedACL;
    private final StorageClass storageClass;
    private final RepositoryMetadata repositoryMetadata;
    private final ThreadPool threadPool;
    private final Executor snapshotExecutor;
    private final S3RepositoriesMetrics s3RepositoriesMetrics;
    private final StatsCollectors statsCollectors = new StatsCollectors();
    private final int bulkDeletionBatchSize;

    S3BlobStore(S3Service service, String bucket, boolean serverSideEncryption, ByteSizeValue bufferSize, String cannedACL, String storageClass, RepositoryMetadata repositoryMetadata, BigArrays bigArrays, ThreadPool threadPool, S3RepositoriesMetrics s3RepositoriesMetrics) {
        this.service = service;
        this.bigArrays = bigArrays;
        this.bucket = bucket;
        this.serverSideEncryption = serverSideEncryption;
        this.bufferSize = bufferSize;
        this.cannedACL = S3BlobStore.initCannedACL(cannedACL);
        this.storageClass = S3BlobStore.initStorageClass(storageClass);
        this.repositoryMetadata = repositoryMetadata;
        this.threadPool = threadPool;
        this.snapshotExecutor = threadPool.executor("snapshot");
        this.s3RepositoriesMetrics = s3RepositoriesMetrics;
        this.bulkDeletionBatchSize = (Integer)S3Repository.DELETION_BATCH_SIZE_SETTING.get(repositoryMetadata.settings());
    }

    RequestMetricCollector getMetricCollector(Operation operation, OperationPurpose purpose) {
        return this.statsCollectors.getMetricCollector(operation, purpose);
    }

    public Executor getSnapshotExecutor() {
        return this.snapshotExecutor;
    }

    public TimeValue getCompareAndExchangeTimeToLive() {
        return this.service.compareAndExchangeTimeToLive;
    }

    public TimeValue getCompareAndExchangeAntiContentionDelay() {
        return this.service.compareAndExchangeAntiContentionDelay;
    }

    private static long getCountForMetric(TimingInfo info, AWSRequestMetrics.Field field) {
        Number count = info.getCounter(field.name());
        if (count == null) {
            if (field == AWSRequestMetrics.Field.RequestCount) {
                String message = "Expected request count to be tracked but found not count.";
                assert (false) : "Expected request count to be tracked but found not count.";
                logger.warn("Expected request count to be tracked but found not count.");
            }
            return 0L;
        }
        return count.longValue();
    }

    private static long getTotalTimeInNanos(List<TimingInfo> requestTimesIncludingRetries) {
        long totalTimeInNanos = 0L;
        for (TimingInfo timingInfo : requestTimesIncludingRetries) {
            Long endTimeInNanos = timingInfo.getEndTimeNanoIfKnown();
            if (endTimeInNanos == null) continue;
            totalTimeInNanos += endTimeInNanos - timingInfo.getStartTimeNano();
        }
        return totalTimeInNanos;
    }

    public String toString() {
        return this.bucket;
    }

    public AmazonS3Reference clientReference() {
        return this.service.client(this.repositoryMetadata);
    }

    final int getMaxRetries() {
        return this.service.settings((RepositoryMetadata)this.repositoryMetadata).maxRetries;
    }

    public String bucket() {
        return this.bucket;
    }

    public BigArrays bigArrays() {
        return this.bigArrays;
    }

    public boolean serverSideEncryption() {
        return this.serverSideEncryption;
    }

    public long bufferSizeInBytes() {
        return this.bufferSize.getBytes();
    }

    public RepositoryMetadata getRepositoryMetadata() {
        return this.repositoryMetadata;
    }

    public S3RepositoriesMetrics getS3RepositoriesMetrics() {
        return this.s3RepositoriesMetrics;
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new S3BlobContainer(path, this);
    }

    public void deleteBlobsIgnoringIfNotExists(OperationPurpose purpose, Iterator<String> blobNames) throws IOException {
        if (!blobNames.hasNext()) {
            return;
        }
        ArrayList<String> partition = new ArrayList<String>();
        try (AmazonS3Reference clientReference = this.clientReference();){
            AtomicReference<Exception> aex = new AtomicReference<Exception>();
            blobNames.forEachRemaining(key -> {
                partition.add((String)key);
                if (partition.size() == this.bulkDeletionBatchSize) {
                    this.deletePartition(purpose, clientReference, partition, aex);
                    partition.clear();
                }
            });
            if (!partition.isEmpty()) {
                this.deletePartition(purpose, clientReference, partition, aex);
            }
            if (aex.get() != null) {
                throw (Exception)aex.get();
            }
        }
        catch (Exception e) {
            throw new IOException("Failed to delete blobs " + partition.stream().limit(10L).toList(), e);
        }
    }

    private void deletePartition(OperationPurpose purpose, AmazonS3Reference clientReference, List<String> partition, AtomicReference<Exception> aex) {
        try {
            SocketAccess.doPrivilegedVoid(() -> clientReference.client().deleteObjects(S3BlobStore.bulkDelete(purpose, this, partition)));
        }
        catch (MultiObjectDeleteException e) {
            logger.warn(() -> Strings.format((String)"Failed to delete some blobs %s", (Object[])new Object[]{e.getErrors().stream().map(err -> "[" + err.getKey() + "][" + err.getCode() + "][" + err.getMessage() + "]").toList()}), (Throwable)e);
            aex.set((Exception)ExceptionsHelper.useOrSuppress((Throwable)aex.get(), (Throwable)e));
        }
        catch (AmazonClientException e) {
            aex.set((Exception)ExceptionsHelper.useOrSuppress((Throwable)aex.get(), (Throwable)e));
        }
    }

    private static DeleteObjectsRequest bulkDelete(OperationPurpose purpose, S3BlobStore blobStore, List<String> blobs) {
        DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(blobStore.bucket()).withKeys(blobs.toArray(org.elasticsearch.common.Strings.EMPTY_ARRAY)).withQuiet(true);
        S3BlobStore.configureRequestForMetrics((AmazonWebServiceRequest)deleteObjectsRequest, blobStore, Operation.DELETE_OBJECTS, purpose);
        return deleteObjectsRequest;
    }

    public void close() throws IOException {
        this.service.close();
    }

    public Map<String, Long> stats() {
        return this.statsCollectors.statsMap(this.service.isStateless);
    }

    StatsCollectors getStatsCollectors() {
        return this.statsCollectors;
    }

    public CannedAccessControlList getCannedACL() {
        return this.cannedACL;
    }

    public StorageClass getStorageClass() {
        return this.storageClass;
    }

    public static StorageClass initStorageClass(String storageClass) {
        if (storageClass == null || storageClass.equals("")) {
            return StorageClass.Standard;
        }
        try {
            StorageClass _storageClass = StorageClass.fromValue((String)storageClass.toUpperCase(Locale.ENGLISH));
            if (_storageClass.equals((Object)StorageClass.Glacier)) {
                throw new BlobStoreException("Glacier storage class is not supported");
            }
            return _storageClass;
        }
        catch (IllegalArgumentException illegalArgumentException) {
            throw new BlobStoreException("`" + storageClass + "` is not a valid S3 Storage Class.");
        }
    }

    public static CannedAccessControlList initCannedACL(String cannedACL) {
        if (cannedACL == null || cannedACL.equals("")) {
            return CannedAccessControlList.Private;
        }
        for (CannedAccessControlList cur : CannedAccessControlList.values()) {
            if (!cur.toString().equalsIgnoreCase(cannedACL)) continue;
            return cur;
        }
        throw new BlobStoreException("cannedACL is not valid: [" + cannedACL + "]");
    }

    ThreadPool getThreadPool() {
        return this.threadPool;
    }

    static void configureRequestForMetrics(AmazonWebServiceRequest request, S3BlobStore blobStore, Operation operation, OperationPurpose purpose) {
        request.setRequestMetricCollector(blobStore.getMetricCollector(operation, purpose));
        request.putCustomQueryParameter(CUSTOM_QUERY_PARAMETER_PURPOSE, purpose.getKey());
    }

    class StatsCollectors {
        final Map<StatsKey, IgnoreNoResponseMetricsCollector> collectors = new ConcurrentHashMap<StatsKey, IgnoreNoResponseMetricsCollector>();

        StatsCollectors() {
        }

        RequestMetricCollector getMetricCollector(Operation operation, OperationPurpose purpose) {
            return this.collectors.computeIfAbsent(new StatsKey(operation, purpose), k -> this.buildMetricCollector(k.operation(), k.purpose()));
        }

        Map<String, Long> statsMap(boolean isStateless) {
            if (isStateless) {
                return this.collectors.entrySet().stream().collect(Collectors.toUnmodifiableMap(entry -> ((StatsKey)entry.getKey()).toString(), entry -> ((IgnoreNoResponseMetricsCollector)((Object)((Object)entry.getValue()))).counter.sum()));
            }
            Map<String, Long> m = Arrays.stream(Operation.values()).collect(Collectors.toMap(Operation::getKey, e -> 0L));
            this.collectors.forEach((sk, v) -> m.compute(sk.operation().getKey(), (k, c) -> Objects.requireNonNull(c) + v.counter.sum()));
            return Map.copyOf(m);
        }

        IgnoreNoResponseMetricsCollector buildMetricCollector(Operation operation, OperationPurpose purpose) {
            return new IgnoreNoResponseMetricsCollector(operation, purpose);
        }
    }

    static enum Operation {
        HEAD_OBJECT("HeadObject"),
        GET_OBJECT("GetObject"),
        LIST_OBJECTS("ListObjects"),
        PUT_OBJECT("PutObject"),
        PUT_MULTIPART_OBJECT("PutMultipartObject"),
        DELETE_OBJECTS("DeleteObjects"),
        ABORT_MULTIPART_OBJECT("AbortMultipartObject");

        private final String key;

        String getKey() {
            return this.key;
        }

        private Operation(String key) {
            this.key = key;
        }

        static Operation parse(String s) {
            for (Operation operation : Operation.values()) {
                if (!operation.key.equals(s)) continue;
                return operation;
            }
            throw new IllegalArgumentException(org.elasticsearch.common.Strings.format((String)"invalid operation [%s] expected one of [%s]", (Object[])new Object[]{s, org.elasticsearch.common.Strings.arrayToCommaDelimitedString((Object[])Operation.values())}));
        }
    }

    record StatsKey(Operation operation, OperationPurpose purpose) {
        @Override
        public String toString() {
            return this.purpose.getKey() + "_" + this.operation.getKey();
        }
    }

    class IgnoreNoResponseMetricsCollector
    extends RequestMetricCollector {
        final LongAdder counter = new LongAdder();
        private final Operation operation;
        private final Map<String, Object> attributes;

        private IgnoreNoResponseMetricsCollector(Operation operation, OperationPurpose purpose) {
            this.operation = operation;
            this.attributes = Map.of("repo_type", "s3", "repo_name", S3BlobStore.this.repositoryMetadata.name(), "operation", operation.getKey(), "purpose", purpose.getKey());
        }

        public final void collectMetrics(Request<?> request, Response<?> response) {
            List statusCodes;
            long amountOfRequestRangeNotSatisfiedErrors;
            assert (this.assertConsistencyBetweenHttpRequestAndOperation(request, this.operation));
            AWSRequestMetrics awsRequestMetrics = request.getAWSRequestMetrics();
            TimingInfo timingInfo = awsRequestMetrics.getTimingInfo();
            long requestCount = S3BlobStore.getCountForMetric(timingInfo, AWSRequestMetrics.Field.RequestCount);
            long exceptionCount = S3BlobStore.getCountForMetric(timingInfo, AWSRequestMetrics.Field.Exception);
            long throttleCount = S3BlobStore.getCountForMetric(timingInfo, AWSRequestMetrics.Field.ThrottleException);
            if (response != null) {
                this.counter.add(requestCount);
            }
            int numberOfAwsErrors = Optional.ofNullable(awsRequestMetrics.getProperty((MetricType)AWSRequestMetrics.Field.AWSErrorCode)).map(List::size).orElse(0);
            if (exceptionCount > 0L && (amountOfRequestRangeNotSatisfiedErrors = (statusCodes = Objects.requireNonNullElse(awsRequestMetrics.getProperty((MetricType)AWSRequestMetrics.Field.StatusCode), List.of())).stream().filter(e -> ((Integer)e).intValue() == RestStatus.REQUESTED_RANGE_NOT_SATISFIED.getStatus()).count()) > 0L) {
                S3BlobStore.this.s3RepositoriesMetrics.common().requestRangeNotSatisfiedExceptionCounter().incrementBy(amountOfRequestRangeNotSatisfiedErrors, this.attributes);
            }
            S3BlobStore.this.s3RepositoriesMetrics.common().operationCounter().incrementBy(1L, this.attributes);
            if ((long)numberOfAwsErrors == requestCount) {
                S3BlobStore.this.s3RepositoriesMetrics.common().unsuccessfulOperationCounter().incrementBy(1L, this.attributes);
            }
            S3BlobStore.this.s3RepositoriesMetrics.common().requestCounter().incrementBy(requestCount, this.attributes);
            if (exceptionCount > 0L) {
                S3BlobStore.this.s3RepositoriesMetrics.common().exceptionCounter().incrementBy(exceptionCount, this.attributes);
                S3BlobStore.this.s3RepositoriesMetrics.common().exceptionHistogram().record(exceptionCount, this.attributes);
            }
            if (throttleCount > 0L) {
                S3BlobStore.this.s3RepositoriesMetrics.common().throttleCounter().incrementBy(throttleCount, this.attributes);
                S3BlobStore.this.s3RepositoriesMetrics.common().throttleHistogram().record(throttleCount, this.attributes);
            }
            this.maybeRecordHttpRequestTime(request);
        }

        private void maybeRecordHttpRequestTime(Request<?> request) {
            List requestTimesIncludingRetries = request.getAWSRequestMetrics().getTimingInfo().getAllSubMeasurements(AWSRequestMetrics.Field.HttpRequestTime.name());
            if (requestTimesIncludingRetries == null) {
                return;
            }
            long totalTimeInNanos = S3BlobStore.getTotalTimeInNanos(requestTimesIncludingRetries);
            if (totalTimeInNanos == 0L) {
                logger.warn("Expected HttpRequestTime to be tracked for request [{}] but found no count.", request);
            } else {
                S3BlobStore.this.s3RepositoriesMetrics.common().httpRequestTimeInMillisHistogram().record(TimeUnit.NANOSECONDS.toMillis(totalTimeInNanos), this.attributes);
            }
        }

        private boolean assertConsistencyBetweenHttpRequestAndOperation(Request<?> request, Operation operation) {
            switch (operation) {
                case HEAD_OBJECT: {
                    return request.getHttpMethod().name().equals("HEAD");
                }
                case GET_OBJECT: 
                case LIST_OBJECTS: {
                    return request.getHttpMethod().name().equals("GET");
                }
                case PUT_OBJECT: {
                    return request.getHttpMethod().name().equals("PUT");
                }
                case PUT_MULTIPART_OBJECT: {
                    return request.getHttpMethod().name().equals("PUT") || request.getHttpMethod().name().equals("POST");
                }
                case DELETE_OBJECTS: {
                    return request.getHttpMethod().name().equals("POST");
                }
                case ABORT_MULTIPART_OBJECT: {
                    return request.getHttpMethod().name().equals("DELETE");
                }
            }
            throw new AssertionError((Object)("unknown operation [" + operation + "]"));
        }
    }
}

