/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.readiness;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.security.AccessController;
import java.util.Collection;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.metadata.ReservedStateMetadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.CheckedSupplier;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.network.NetworkAddress;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.BoundTransportAddress;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.env.Environment;
import org.elasticsearch.reservedstate.service.FileSettingsFeatures;
import org.elasticsearch.shutdown.PluginShutdownService;
import org.elasticsearch.transport.BindTransportException;

public class ReadinessService
extends AbstractLifecycleComponent
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(ReadinessService.class);
    private final Environment environment;
    private final CheckedSupplier<ServerSocketChannel, IOException> socketChannelFactory;
    private volatile boolean active;
    private volatile ServerSocketChannel serverChannel = null;
    volatile CountDownLatch listenerThreadLatch = new CountDownLatch(0);
    final AtomicReference<InetSocketAddress> boundSocket = new AtomicReference();
    private final Collection<BoundAddressListener> boundAddressListeners = new CopyOnWriteArrayList<BoundAddressListener>();
    public static final Setting<Integer> PORT = Setting.intSetting("readiness.port", -1, Setting.Property.NodeScope);

    public ReadinessService(ClusterService clusterService, Environment environment) {
        this(clusterService, environment, ServerSocketChannel::open);
    }

    ReadinessService(ClusterService clusterService, Environment environment, CheckedSupplier<ServerSocketChannel, IOException> socketChannelFactory) {
        this.environment = environment;
        this.socketChannelFactory = socketChannelFactory;
        clusterService.addListener(this);
    }

    boolean ready() {
        return this.serverChannel != null;
    }

    public static boolean enabled(Environment environment) {
        return PORT.get(environment.settings()) != -1;
    }

    ServerSocketChannel serverChannel() {
        return this.serverChannel;
    }

    public BoundTransportAddress boundAddress() {
        InetSocketAddress boundAddress = this.boundSocket.get();
        if (boundAddress == null) {
            return null;
        }
        TransportAddress publishAddress = new TransportAddress(boundAddress);
        return new BoundTransportAddress(new TransportAddress[]{publishAddress}, publishAddress);
    }

    InetSocketAddress socketAddress(InetAddress host, int portNumber) {
        InetSocketAddress socketAddress = this.boundSocket.get();
        if (socketAddress == null) {
            socketAddress = new InetSocketAddress(host, portNumber);
        }
        return socketAddress;
    }

    ServerSocketChannel setupSocket() {
        Settings settings = this.environment.settings();
        int portNumber = PORT.get(settings);
        assert (portNumber >= 0);
        InetSocketAddress socketAddress = AccessController.doPrivileged(() -> {
            try {
                return this.socketAddress(InetAddress.getByName("0"), portNumber);
            }
            catch (IOException e) {
                throw new IllegalArgumentException("Failed to resolve readiness host address", e);
            }
        });
        try {
            this.serverChannel = this.socketChannelFactory.get();
            AccessController.doPrivileged(() -> {
                try {
                    this.serverChannel.bind(socketAddress);
                }
                catch (IOException e) {
                    throw new BindTransportException("Failed to bind to " + NetworkAddress.format(socketAddress), e);
                }
                return null;
            });
            if (this.boundSocket.get() == null) {
                this.boundSocket.set((InetSocketAddress)this.serverChannel.getLocalAddress());
                BoundTransportAddress boundAddress = this.boundAddress();
                for (BoundAddressListener listener : this.boundAddressListeners) {
                    listener.addressBound(boundAddress);
                }
            }
        }
        catch (Exception e) {
            throw new BindTransportException("Failed to open socket channel " + NetworkAddress.format(socketAddress), e);
        }
        return this.serverChannel;
    }

    @Override
    protected void doStart() {
        this.active = true;
    }

    synchronized void startListener() {
        assert (ReadinessService.enabled(this.environment));
        if (this.serverChannel != null || !this.active) {
            return;
        }
        this.serverChannel = this.setupSocket();
        this.listenerThreadLatch = new CountDownLatch(1);
        new Thread(() -> {
            assert (this.serverChannel != null);
            try {
                while (this.serverChannel.isOpen()) {
                    AccessController.doPrivileged(() -> {
                        try {
                            SocketChannel channel = this.serverChannel.accept();
                            if (channel != null) {
                                channel.close();
                            }
                        }
                        catch (IOException e) {
                            logger.debug("encountered exception while responding to readiness check request", (Throwable)e);
                        }
                        catch (Exception other) {
                            logger.warn("encountered unknown exception while responding to readiness check request", (Throwable)other);
                        }
                        return null;
                    });
                }
            }
            finally {
                this.listenerThreadLatch.countDown();
            }
        }, "elasticsearch[readiness-service]").start();
        logger.info("readiness service up and running on {}", (Object)this.boundAddress().publishAddress());
    }

    @Override
    protected void doStop() {
        this.active = false;
        this.stopListener();
    }

    synchronized void stopListener() {
        assert (ReadinessService.enabled(this.environment));
        if (!this.ready()) {
            return;
        }
        try {
            logger.info("stopping readiness service on channel {}", this.serverChannel == null ? "None" : this.serverChannel.getLocalAddress());
            if (this.serverChannel != null) {
                this.serverChannel.close();
                this.listenerThreadLatch.await();
            }
        }
        catch (IOException | InterruptedException e) {
            logger.warn("error closing readiness service channel", (Throwable)e);
        }
        finally {
            this.serverChannel = null;
            logger.info("readiness service stopped");
        }
    }

    @Override
    protected void doClose() {
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState clusterState = event.state();
        Set<String> shutdownNodeIds = PluginShutdownService.shutdownNodes(clusterState);
        boolean shuttingDown = shutdownNodeIds.contains(clusterState.nodes().getLocalNodeId());
        if (shuttingDown) {
            if (this.ready()) {
                this.setReady(false);
                logger.info("marking node as not ready because it's shutting down");
            }
        } else {
            boolean masterElected = this.getReadinessState(clusterState, event.previousState(), this::isMasterElected, "masterElected");
            boolean fileSettingsApplied = this.getReadinessState(clusterState, event.previousState(), this::areFileSettingsApplied, "fileSettingsApplied");
            this.setReady(masterElected && fileSettingsApplied);
        }
    }

    private boolean getReadinessState(ClusterState clusterState, ClusterState previousState, Function<ClusterState, Boolean> accessor, String description) {
        boolean newStateValue = accessor.apply(clusterState);
        boolean oldStateValue = accessor.apply(previousState);
        if (oldStateValue != newStateValue) {
            logger.info("readiness change: {}={}", (Object)description, (Object)newStateValue);
        }
        return newStateValue;
    }

    private boolean isMasterElected(ClusterState clusterState) {
        return clusterState.nodes().getMasterNodeId() != null;
    }

    protected boolean areFileSettingsApplied(ClusterState clusterState) {
        ReservedStateMetadata fileSettingsMetadata = clusterState.metadata().reservedStateMetadata().get("file_settings");
        if (fileSettingsMetadata == null) {
            return !ReadinessService.supportsFileSettings(clusterState);
        }
        return !fileSettingsMetadata.version().equals(ReservedStateMetadata.NO_VERSION);
    }

    @SuppressForbidden(reason="need to check file settings support on exact cluster state")
    private static boolean supportsFileSettings(ClusterState clusterState) {
        return clusterState.clusterFeatures().clusterHasFeature(FileSettingsFeatures.FILE_SETTINGS_SUPPORTED);
    }

    private void setReady(boolean ready) {
        if (ready) {
            this.startListener();
        } else {
            this.stopListener();
        }
    }

    public synchronized void addBoundAddressListener(BoundAddressListener listener) {
        BoundTransportAddress b = this.boundAddress();
        if (b != null) {
            listener.addressBound(b);
        }
        this.boundAddressListeners.add(listener);
    }

    public static interface BoundAddressListener {
        public void addressBound(BoundTransportAddress var1);
    }
}

