/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.file;

import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.reservedstate.service.FileChangedListener;

public abstract class AbstractFileWatchingService
extends AbstractLifecycleComponent {
    private static final Logger logger = LogManager.getLogger(AbstractFileWatchingService.class);
    private static final int REGISTER_RETRY_COUNT = 5;
    private final Path watchedFileDir;
    private final Path watchedFile;
    private final List<FileChangedListener> eventListeners;
    private WatchService watchService;
    private Thread watcherThread;
    private FileUpdateState fileUpdateState;
    private WatchKey settingsDirWatchKey;
    private WatchKey configDirWatchKey;

    public AbstractFileWatchingService(Path watchedFile) {
        this.watchedFile = watchedFile;
        this.watchedFileDir = watchedFile.getParent();
        this.eventListeners = new CopyOnWriteArrayList<FileChangedListener>();
    }

    protected abstract void processFileChanges() throws InterruptedException, ExecutionException, IOException;

    protected abstract void processInitialFileMissing() throws InterruptedException, ExecutionException, IOException;

    protected void processFileOnServiceStart() throws IOException, ExecutionException, InterruptedException {
        this.processFileChanges();
    }

    public final void addFileChangedListener(FileChangedListener listener) {
        this.eventListeners.add(listener);
    }

    public final Path watchedFileDir() {
        return this.watchedFileDir;
    }

    public final Path watchedFile() {
        return this.watchedFile;
    }

    @Override
    protected void doStart() {
        this.startWatcher();
    }

    @Override
    protected void doStop() {
        logger.debug("Stopping file watching service");
        this.stopWatcher();
    }

    @Override
    protected final void doClose() {
    }

    public final boolean watching() {
        return this.watcherThread != null;
    }

    final boolean watchedFileChanged(Path path) throws IOException {
        if (!Files.exists(path, new LinkOption[0])) {
            return false;
        }
        FileUpdateState previousUpdateState = this.fileUpdateState;
        BasicFileAttributes attr = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
        this.fileUpdateState = new FileUpdateState(attr.lastModifiedTime().toMillis(), path.toRealPath(new LinkOption[0]).toString(), attr.fileKey());
        return previousUpdateState == null || !previousUpdateState.equals(this.fileUpdateState);
    }

    protected final synchronized void startWatcher() {
        if (!Files.exists(this.watchedFileDir.getParent(), new LinkOption[0])) {
            logger.warn("File watcher for [{}] cannot start because grandparent directory does not exist", (Object)this.watchedFile);
            return;
        }
        logger.info("starting file watcher ...");
        try {
            Path settingsDirPath = this.watchedFileDir();
            this.watchService = settingsDirPath.getParent().getFileSystem().newWatchService();
            if (Files.exists(settingsDirPath, new LinkOption[0])) {
                this.settingsDirWatchKey = this.enableDirectoryWatcher(this.settingsDirWatchKey, settingsDirPath);
            } else {
                logger.debug("watched directory [{}] not found, will watch for its creation...", (Object)settingsDirPath);
            }
            this.configDirWatchKey = this.enableDirectoryWatcher(this.configDirWatchKey, settingsDirPath.getParent());
        }
        catch (Exception e) {
            if (this.watchService != null) {
                try {
                    this.watchService.close();
                }
                catch (Exception ce) {
                    e.addSuppressed(ce);
                }
                finally {
                    this.watchService = null;
                }
            }
            throw new IllegalStateException("unable to launch a new watch service", e);
        }
        this.watcherThread = new Thread(this::watcherThread, "elasticsearch[file-watcher[" + this.watchedFile + "]]");
        this.watcherThread.start();
    }

    protected final void watcherThread() {
        try {
            WatchKey key;
            logger.info("file settings service up and running [tid={}]", (Object)Thread.currentThread().getId());
            Path path = this.watchedFile();
            if (Files.exists(path, new LinkOption[0])) {
                logger.debug("found initial operator settings file [{}], applying...", (Object)path);
                this.processSettingsOnServiceStartAndNotifyListeners();
            } else {
                this.processInitialFileMissing();
                for (FileChangedListener listener : this.eventListeners) {
                    listener.watchedFileChanged();
                }
            }
            while ((key = this.watchService.take()) != null) {
                Path settingsPath = this.watchedFileDir();
                if (Files.exists(settingsPath, new LinkOption[0])) {
                    try {
                        if (logger.isDebugEnabled()) {
                            key.pollEvents().forEach(e -> logger.debug("{}:{}", (Object)e.kind().toString(), (Object)e.context().toString()));
                        } else {
                            key.pollEvents();
                        }
                        key.reset();
                        this.settingsDirWatchKey = this.enableDirectoryWatcher(this.settingsDirWatchKey, settingsPath);
                        if (!this.watchedFileChanged(path)) continue;
                        this.processSettingsAndNotifyListeners();
                    }
                    catch (IOException e2) {
                        logger.warn("encountered I/O error while watching file settings", (Throwable)e2);
                    }
                    continue;
                }
                key.pollEvents();
                key.reset();
            }
        }
        catch (InterruptedException | ClosedWatchServiceException expected) {
            logger.info("shutting down watcher thread");
        }
        catch (Exception e3) {
            logger.error("shutting down watcher thread with exception", (Throwable)e3);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    protected final synchronized void stopWatcher() {
        if (this.watching()) {
            logger.debug("stopping watcher ...");
            try (WatchService ws = this.watchService;){
                this.watcherThread.interrupt();
                this.watcherThread.join();
                if (this.configDirWatchKey != null) {
                    this.configDirWatchKey.cancel();
                }
                if (this.settingsDirWatchKey == null) return;
                this.settingsDirWatchKey.cancel();
                return;
            }
            catch (IOException e) {
                logger.warn("encountered exception while closing watch service", (Throwable)e);
                return;
            }
            catch (InterruptedException interruptedException) {
                logger.info("interrupted while closing the watch service", (Throwable)interruptedException);
                return;
            }
            finally {
                this.watcherThread = null;
                this.settingsDirWatchKey = null;
                this.configDirWatchKey = null;
                this.watchService = null;
                logger.info("watcher service stopped");
            }
        } else {
            logger.trace("file watch service already stopped");
        }
    }

    final WatchKey enableDirectoryWatcher(WatchKey previousKey, Path settingsDir) throws IOException, InterruptedException {
        if (previousKey != null) {
            previousKey.cancel();
        }
        int retryCount = 0;
        while (true) {
            try {
                return settingsDir.register(this.watchService, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE);
            }
            catch (IOException e) {
                if (retryCount == 4) {
                    throw e;
                }
                Thread.sleep(this.retryDelayMillis(retryCount));
                ++retryCount;
                continue;
            }
            break;
        }
    }

    void processSettingsOnServiceStartAndNotifyListeners() throws InterruptedException {
        try {
            this.processFileOnServiceStart();
            for (FileChangedListener listener : this.eventListeners) {
                listener.watchedFileChanged();
            }
        }
        catch (IOException | ExecutionException e) {
            logger.error(() -> "Error processing watched file: " + this.watchedFile(), (Throwable)e);
        }
    }

    void processSettingsAndNotifyListeners() throws InterruptedException {
        try {
            this.processFileChanges();
            for (FileChangedListener listener : this.eventListeners) {
                listener.watchedFileChanged();
            }
        }
        catch (IOException | ExecutionException e) {
            logger.error(() -> "Error processing watched file: " + this.watchedFile(), (Throwable)e);
        }
    }

    long retryDelayMillis(int failedCount) {
        assert (failedCount < 31);
        return 100 * (1 << failedCount) + Randomness.get().nextInt(10);
    }

    private record FileUpdateState(long timestamp, String path, Object fileKey) {
    }
}

