/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.geometry.simplify;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.elasticsearch.geometry.Circle;
import org.elasticsearch.geometry.Geometry;
import org.elasticsearch.geometry.GeometryCollection;
import org.elasticsearch.geometry.Line;
import org.elasticsearch.geometry.LinearRing;
import org.elasticsearch.geometry.MultiPoint;
import org.elasticsearch.geometry.MultiPolygon;
import org.elasticsearch.geometry.Point;
import org.elasticsearch.geometry.Polygon;
import org.elasticsearch.geometry.Rectangle;
import org.elasticsearch.geometry.simplify.SimplificationErrorCalculator;
import org.elasticsearch.geometry.simplify.StreamingGeometrySimplifier;

public abstract class GeometrySimplifier<T extends Geometry> {
    protected final int maxPoints;
    protected final SimplificationErrorCalculator calculator;
    protected final StreamingGeometrySimplifier.Monitor monitor;
    protected String description;
    protected final StreamingGeometrySimplifier<T> innerSimplifier;

    protected GeometrySimplifier(String description, int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor, StreamingGeometrySimplifier<T> innerSimplifier) {
        this.description = description;
        this.maxPoints = maxPoints;
        this.calculator = calculator;
        this.monitor = monitor;
        this.innerSimplifier = innerSimplifier;
    }

    public abstract T simplify(T var1);

    public void reset() {
        if (this.innerSimplifier != null) {
            this.innerSimplifier.reset();
        }
    }

    protected void notifyMonitorSimplificationStart() {
        if (this.monitor != null) {
            this.monitor.startSimplification(this.description, this.maxPoints);
        }
    }

    protected void notifyMonitorSimplificationEnd() {
        if (this.monitor != null) {
            this.monitor.endSimplification(this.description, this.getCurrentPoints());
        }
    }

    protected List<SimplificationErrorCalculator.PointLike> getCurrentPoints() {
        ArrayList<SimplificationErrorCalculator.PointLike> simplification = new ArrayList<SimplificationErrorCalculator.PointLike>();
        if (this.innerSimplifier != null) {
            simplification.addAll(Arrays.asList(this.innerSimplifier.points).subList(0, this.innerSimplifier.length));
        }
        return simplification;
    }

    public static <G extends Geometry> GeometrySimplifier<G> simplifierFor(G geometry, int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor) {
        if (geometry instanceof Point || geometry instanceof Circle || geometry instanceof Rectangle || geometry instanceof MultiPoint) {
            return new Identity(maxPoints, calculator, monitor);
        }
        throw new IllegalArgumentException("Unsupported geometry type: " + geometry.type());
    }

    static int lengthOf(Geometry geometry) {
        if (geometry instanceof Polygon) {
            Polygon polygon = (Polygon)geometry;
            return polygon.getPolygon().length();
        }
        if (geometry instanceof Point) {
            return 1;
        }
        if (geometry instanceof Line) {
            Line line = (Line)geometry;
            return line.length();
        }
        if (geometry instanceof MultiPolygon) {
            MultiPolygon multiPolygon = (MultiPolygon)geometry;
            int maxPolyLength = 0;
            for (int i = 0; i < multiPolygon.size(); ++i) {
                Polygon polygon = (Polygon)multiPolygon.get(i);
                maxPolyLength = Math.max(maxPolyLength, polygon.getPolygon().length());
            }
            return maxPolyLength;
        }
        if (geometry instanceof GeometryCollection) {
            GeometryCollection collection = (GeometryCollection)geometry;
            return GeometrySimplifier.maxLengthOf(collection);
        }
        throw new IllegalArgumentException("Unsupported geometry type: " + geometry.type());
    }

    private static int maxLengthOf(GeometryCollection<?> collection) {
        int maxLength = 0;
        for (int i = 0; i < collection.size(); ++i) {
            maxLength = Math.max(maxLength, GeometrySimplifier.lengthOf(collection.get(i)));
        }
        return maxLength;
    }

    private static int maxLengthOf(MultiPolygon polygons) {
        int maxLength = 0;
        for (int i = 0; i < polygons.size(); ++i) {
            maxLength = Math.max(maxLength, GeometrySimplifier.lengthOf(polygons.get(i)));
        }
        return maxLength;
    }

    public static class Identity<G extends Geometry>
    extends GeometrySimplifier<G> {
        public Identity(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public Identity(int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor) {
            super("Identity", maxPoints, calculator, monitor, null);
        }

        @Override
        public G simplify(G geometry) {
            this.notifyMonitorSimplificationStart();
            try {
                G g = geometry;
                return g;
            }
            finally {
                this.notifyMonitorSimplificationEnd();
            }
        }
    }

    public static class GeometryCollections
    extends GeometrySimplifier<GeometryCollection<?>> {
        public GeometryCollections(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public GeometryCollections(int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor) {
            super("GeometryCollection", maxPoints, calculator, monitor, null);
        }

        @Override
        public GeometryCollection<?> simplify(GeometryCollection<?> collection) {
            ArrayList<Geometry> geometries = new ArrayList<Geometry>(collection.size());
            int maxGeometryLength = GeometrySimplifier.maxLengthOf(collection);
            this.notifyMonitorSimplificationStart();
            for (int i = 0; i < collection.size(); ++i) {
                Object geometry = collection.get(i);
                double simplificationFactor = (double)this.maxPoints / (double)maxGeometryLength;
                int maxLength = GeometryCollections.lengthOf(geometry);
                int maxPolyPoints = Math.max(4, (int)(simplificationFactor * (double)maxLength));
                if (geometry instanceof Point) {
                    Point point = (Point)geometry;
                    Identity<Point> pointSimplifier = new Identity<Point>(maxPolyPoints, this.calculator, this.monitor);
                    pointSimplifier.description = "GeometryCollection.Point[" + i + "]";
                    geometries.add(pointSimplifier.simplify(point));
                    continue;
                }
                if (geometry instanceof Line) {
                    Line line = (Line)geometry;
                    LineSimplifier lineSimplifier = new LineSimplifier(maxPolyPoints, this.calculator, this.monitor);
                    lineSimplifier.description = "GeometryCollection.Line[" + i + "]";
                    geometries.add(lineSimplifier.simplify(line));
                    continue;
                }
                if (geometry instanceof Polygon) {
                    Polygon polygon = (Polygon)geometry;
                    PolygonSimplifier polygonSimplifier = new PolygonSimplifier(maxPolyPoints, this.calculator, this.monitor);
                    polygonSimplifier.description = "GeometryCollection.Polygon[" + i + "]";
                    geometries.add(polygonSimplifier.simplify(polygon));
                    continue;
                }
                if (geometry instanceof MultiPolygon) {
                    MultiPolygon multiPolygon = (MultiPolygon)geometry;
                    MultiPolygonSimplifier multiPolygonSimplifier = new MultiPolygonSimplifier(maxPolyPoints, this.calculator, this.monitor);
                    multiPolygonSimplifier.description = "GeometryCollection.MultiPolygon[" + i + "]";
                    geometries.add(multiPolygonSimplifier.simplify(multiPolygon));
                    continue;
                }
                if (geometry instanceof GeometryCollection) {
                    GeometryCollection g = (GeometryCollection)geometry;
                    GeometryCollections collectionSimplifier = new GeometryCollections(maxPolyPoints, this.calculator, this.monitor);
                    collectionSimplifier.description = "GeometryCollection.GeometryCollection[" + i + "]";
                    geometries.add(collectionSimplifier.simplify(g));
                    continue;
                }
                throw new IllegalArgumentException("Unsupported geometry type: " + geometry.type());
            }
            this.notifyMonitorSimplificationEnd();
            return new GeometryCollection(geometries);
        }
    }

    public static class MultiPolygonSimplifier
    extends GeometrySimplifier<MultiPolygon> {
        ArrayList<Integer> indexes = new ArrayList();

        public MultiPolygonSimplifier(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public MultiPolygonSimplifier(int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor) {
            super("MultiPolygon", maxPoints, calculator, monitor, null);
        }

        @Override
        public void reset() {
            super.reset();
            this.indexes.clear();
        }

        @Override
        public MultiPolygon simplify(MultiPolygon geometry) {
            ArrayList<Polygon> polygons = new ArrayList<Polygon>(geometry.size());
            int maxPolyLength = GeometrySimplifier.maxLengthOf(geometry);
            this.notifyMonitorSimplificationStart();
            for (int i = 0; i < geometry.size(); ++i) {
                Polygon polygon = (Polygon)geometry.get(i);
                double simplificationFactor = (double)this.maxPoints / (double)maxPolyLength;
                int maxPolyPoints = Math.max(4, (int)(simplificationFactor * (double)polygon.getPolygon().length()));
                PolygonSimplifier simplifier = new PolygonSimplifier(maxPolyPoints, this.calculator, this.monitor);
                simplifier.description = "MultiPolygon.Polygon[" + i + "]";
                Polygon simplified = simplifier.simplify(polygon);
                if (simplified.getPolygon().length() <= 0) continue;
                polygons.add(simplified);
                this.indexes.add(i);
            }
            this.notifyMonitorSimplificationEnd();
            return new MultiPolygon((List<Polygon>)polygons);
        }

        public int indexOf(int simplified) {
            return this.indexes.get(simplified);
        }
    }

    public static class PolygonSimplifier
    extends GeometrySimplifier<Polygon> {
        public PolygonSimplifier(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public PolygonSimplifier(int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor) {
            super("Polygon", maxPoints, calculator, monitor, new StreamingGeometrySimplifier.PolygonSimplifier(maxPoints, calculator, monitor));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Polygon simplify(Polygon geometry) {
            this.reset();
            this.notifyMonitorSimplificationStart();
            try {
                LinearRing ring = geometry.getPolygon();
                if (ring.length() <= this.maxPoints) {
                    Polygon polygon = geometry;
                    return polygon;
                }
                for (int i = 0; i < ring.length(); ++i) {
                    this.innerSimplifier.consume(ring.getX(i), ring.getY(i));
                }
                ArrayList<LinearRing> holes = new ArrayList<LinearRing>(geometry.getNumberOfHoles());
                for (int i = 0; i < geometry.getNumberOfHoles(); ++i) {
                    LinearRing hole = geometry.getHole(i);
                    double simplificationFactor = (double)this.maxPoints / (double)ring.length();
                    int maxHolePoints = Math.max(4, (int)(simplificationFactor * (double)hole.length()));
                    LinearRingSimplifier holeSimplifier = new LinearRingSimplifier(maxHolePoints, this.calculator, this.monitor);
                    holeSimplifier.description = "Polygon.Hole";
                    holes.add(holeSimplifier.simplify(hole));
                }
                Polygon polygon = new Polygon(StreamingGeometrySimplifier.produceLinearRing(this.innerSimplifier), holes);
                return polygon;
            }
            finally {
                this.notifyMonitorSimplificationEnd();
            }
        }
    }

    public static class LinearRingSimplifier
    extends GeometrySimplifier<LinearRing> {
        public LinearRingSimplifier(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public LinearRingSimplifier(int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor) {
            super("LinearRing", maxPoints, calculator, monitor, new StreamingGeometrySimplifier.LinearRingSimplifier(maxPoints, calculator, monitor));
            assert (maxPoints >= 4);
        }

        @Override
        public LinearRing simplify(LinearRing ring) {
            this.reset();
            this.notifyMonitorSimplificationStart();
            try {
                if (ring.length() <= this.maxPoints) {
                    LinearRing linearRing = ring;
                    return linearRing;
                }
                for (int i = 0; i < ring.length(); ++i) {
                    this.innerSimplifier.consume(ring.getX(i), ring.getY(i));
                }
                LinearRing linearRing = (LinearRing)this.innerSimplifier.produce();
                return linearRing;
            }
            finally {
                this.notifyMonitorSimplificationEnd();
            }
        }
    }

    public static class LineSimplifier
    extends GeometrySimplifier<Line> {
        public LineSimplifier(int maxPoints, SimplificationErrorCalculator calculator) {
            this(maxPoints, calculator, null);
        }

        public LineSimplifier(int maxPoints, SimplificationErrorCalculator calculator, StreamingGeometrySimplifier.Monitor monitor) {
            super("Line", maxPoints, calculator, monitor, new StreamingGeometrySimplifier.LineSimplifier(maxPoints, calculator, monitor));
        }

        @Override
        public Line simplify(Line line) {
            this.reset();
            this.notifyMonitorSimplificationStart();
            try {
                if (line.length() <= this.maxPoints) {
                    Line line2 = line;
                    return line2;
                }
                for (int i = 0; i < line.length(); ++i) {
                    this.innerSimplifier.consume(line.getX(i), line.getY(i));
                }
                Line line3 = (Line)this.innerSimplifier.produce();
                return line3;
            }
            finally {
                this.notifyMonitorSimplificationEnd();
            }
        }
    }
}

