/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.blobcache.shared;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.function.IntConsumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.blobcache.BlobCacheUtils;
import org.elasticsearch.blobcache.common.ByteBufferReference;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.AbstractRefCounted;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Streams;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.env.Environment;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.preallocate.Preallocate;

public class SharedBytes
extends AbstractRefCounted {
    public static final int MAX_BYTES_PER_WRITE = StrictMath.toIntExact(ByteSizeValue.parseBytesSizeValue((String)System.getProperty("es.searchable.snapshot.shared_cache.write_buffer.size", "2m"), (String)"es.searchable.snapshot.shared_cache.write_buffer.size").getBytes());
    private static final Logger logger = LogManager.getLogger(SharedBytes.class);
    public static int PAGE_SIZE = 4096;
    private static final String CACHE_FILE_NAME = "shared_snapshot_cache";
    private static final StandardOpenOption[] OPEN_OPTIONS = new StandardOpenOption[]{StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.CREATE};
    private static final long MAX_BYTES_PER_MAP = ByteSizeValue.ofGb((long)1L).getBytes();
    final int numRegions;
    private final IO[] ios;
    final int regionSize;
    private final FileChannel fileChannel;
    private final Path path;
    private final IntConsumer writeBytes;
    private final IntConsumer readBytes;
    private final boolean mmap;

    SharedBytes(int numRegions, int regionSize, NodeEnvironment environment, IntConsumer writeBytes, IntConsumer readBytes, boolean mmap) throws IOException {
        this.numRegions = numRegions;
        this.regionSize = regionSize;
        long fileSize = (long)numRegions * (long)regionSize;
        Path cacheFile = null;
        if (fileSize > 0L) {
            cacheFile = SharedBytes.findCacheSnapshotCacheFilePath(environment, fileSize);
            Preallocate.preallocate((Path)cacheFile, (long)fileSize);
            this.fileChannel = FileChannel.open(cacheFile, OPEN_OPTIONS);
            assert (this.fileChannel.size() == fileSize) : "expected file size " + fileSize + " but was " + this.fileChannel.size();
        } else {
            this.fileChannel = null;
            for (Path path : environment.nodeDataPaths()) {
                Files.deleteIfExists(path.resolve(CACHE_FILE_NAME));
            }
        }
        this.path = cacheFile;
        this.mmap = mmap;
        this.ios = new IO[numRegions];
        if (mmap && fileSize > 0L) {
            int i;
            int regionsPerMmap = Math.toIntExact(MAX_BYTES_PER_MAP / (long)regionSize);
            int mapSize = regionsPerMmap * regionSize;
            int lastMapSize = Math.toIntExact(fileSize % (long)mapSize);
            int mapCount = Math.toIntExact(fileSize / (long)mapSize) + (lastMapSize == 0 ? 0 : 1);
            MappedByteBuffer[] mmaps = new MappedByteBuffer[mapCount];
            for (i = 0; i < mapCount - 1; ++i) {
                mmaps[i] = this.fileChannel.map(FileChannel.MapMode.READ_ONLY, (long)mapSize * (long)i, mapSize);
            }
            mmaps[mapCount - 1] = this.fileChannel.map(FileChannel.MapMode.READ_ONLY, (long)mapSize * (long)(mapCount - 1), lastMapSize == 0 ? (long)mapSize : (long)lastMapSize);
            for (i = 0; i < numRegions; ++i) {
                this.ios[i] = new IO(i, mmaps[i / regionsPerMmap].slice(i % regionsPerMmap * regionSize, regionSize));
            }
        } else {
            for (int i = 0; i < numRegions; ++i) {
                this.ios[i] = new IO(i, null);
            }
        }
        this.writeBytes = writeBytes;
        this.readBytes = readBytes;
    }

    public static Path findCacheSnapshotCacheFilePath(NodeEnvironment environment, long fileSize) throws IOException {
        assert (environment.nodeDataPaths().length == 1);
        Path path = environment.nodeDataPaths()[0];
        Files.createDirectories(path, new FileAttribute[0]);
        long usableSpace = Environment.getUsableSpace((Path)path);
        Path p = path.resolve(CACHE_FILE_NAME);
        if (Files.exists(p, new LinkOption[0])) {
            usableSpace += Files.size(p);
        }
        if (usableSpace > fileSize) {
            return p;
        }
        throw new IOException("Not enough free space [" + usableSpace + "] for cache file of size [" + fileSize + "] in path [" + path + "]");
    }

    public static void copyToCacheFileAligned(IO fc, InputStream input, int fileChannelPos, int relativePos, int length, IntConsumer progressUpdater, ByteBuffer buf) throws IOException {
        long remaining;
        int bytesRead;
        int bytesCopied = 0;
        for (remaining = (long)length; remaining > 0L; remaining -= (long)bytesRead) {
            bytesRead = BlobCacheUtils.readSafe(input, buf, relativePos, remaining);
            if (buf.hasRemaining()) break;
            bytesCopied += SharedBytes.positionalWrite(fc, fileChannelPos + bytesCopied, buf);
            progressUpdater.accept(bytesCopied);
        }
        if (remaining > 0L) {
            int remainder = buf.position() % PAGE_SIZE;
            int adjustment = remainder == 0 ? 0 : PAGE_SIZE - remainder;
            buf.position(buf.position() + adjustment);
            bytesCopied += SharedBytes.positionalWrite(fc, fileChannelPos + bytesCopied, buf);
            int adjustedBytesCopied = bytesCopied - adjustment;
            assert (adjustedBytesCopied == length) : adjustedBytesCopied + " vs " + length;
            progressUpdater.accept(adjustedBytesCopied);
        }
    }

    public static int copyToCacheFileAligned(IO fc, InputStream input, int fileChannelPos, IntConsumer progressUpdater, ByteBuffer buffer) throws IOException {
        int bytesRead;
        int bytesCopied = 0;
        while ((bytesRead = Streams.read((InputStream)input, (ByteBuffer)buffer, (int)buffer.remaining())) > 0) {
            bytesCopied += SharedBytes.copyBufferToCacheFileAligned(fc, fileChannelPos + bytesCopied, buffer);
            progressUpdater.accept(bytesCopied);
        }
        return bytesCopied;
    }

    public static int copyBufferToCacheFileAligned(IO fc, int fileChannelPos, ByteBuffer buffer) throws IOException {
        if (buffer.hasRemaining()) {
            int remainder = buffer.position() % PAGE_SIZE;
            int adjustment = remainder == 0 ? 0 : PAGE_SIZE - remainder;
            buffer.position(buffer.position() + adjustment);
        }
        return SharedBytes.positionalWrite(fc, fileChannelPos, buffer);
    }

    private static int positionalWrite(IO fc, int start, ByteBuffer byteBuffer) throws IOException {
        byteBuffer.flip();
        int written = fc.write(byteBuffer, start);
        assert (!byteBuffer.hasRemaining());
        byteBuffer.clear();
        return written;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static int readCacheFile(IO fc, int channelPos, int relativePos, int length, ByteBufferReference byteBufferReference) throws IOException {
        if ((long)length == 0L) {
            return 0;
        }
        ByteBuffer dup = byteBufferReference.tryAcquire(relativePos, length);
        if (dup == null) return length;
        try {
            int bytesRead = fc.read(dup, channelPos);
            if (bytesRead != -1) return bytesRead;
            BlobCacheUtils.throwEOF(channelPos, dup.remaining());
            return bytesRead;
        }
        finally {
            byteBufferReference.release();
        }
    }

    protected void closeInternal() {
        try {
            IOUtils.close((Closeable[])new Closeable[]{this.fileChannel, this.path == null ? null : () -> Files.deleteIfExists(this.path)});
        }
        catch (IOException e) {
            logger.warn("Failed to clean up shared bytes file", (Throwable)e);
        }
    }

    public IO getFileChannel(int sharedBytesPos) {
        assert (this.fileChannel != null);
        return this.ios[sharedBytesPos];
    }

    public final class IO {
        private final long pageStart;
        private final MappedByteBuffer mappedByteBuffer;

        private IO(int sharedBytesPos, MappedByteBuffer mappedByteBuffer) {
            long physicalOffset = (long)sharedBytesPos * (long)SharedBytes.this.regionSize;
            assert (physicalOffset <= (long)SharedBytes.this.numRegions * (long)SharedBytes.this.regionSize);
            this.pageStart = physicalOffset;
            this.mappedByteBuffer = mappedByteBuffer;
        }

        @SuppressForbidden(reason="Use positional reads on purpose")
        public int read(ByteBuffer dst, int position) throws IOException {
            int bytesRead;
            int remaining = dst.remaining();
            this.checkOffsets(position, remaining);
            if (SharedBytes.this.mmap) {
                bytesRead = remaining;
                int startPosition = dst.position();
                dst.put(startPosition, this.mappedByteBuffer, position, bytesRead).position(startPosition + bytesRead);
            } else {
                bytesRead = SharedBytes.this.fileChannel.read(dst, this.pageStart + (long)position);
            }
            SharedBytes.this.readBytes.accept(bytesRead);
            return bytesRead;
        }

        @SuppressForbidden(reason="Use positional writes on purpose")
        public int write(ByteBuffer src, int position) throws IOException {
            assert (position % PAGE_SIZE == 0);
            assert (src.remaining() % PAGE_SIZE == 0);
            this.checkOffsets(position, src.remaining());
            int bytesWritten = SharedBytes.this.fileChannel.write(src, this.pageStart + (long)position);
            SharedBytes.this.writeBytes.accept(bytesWritten);
            return bytesWritten;
        }

        private void checkOffsets(int position, int length) {
            if (position < 0 || position + length > SharedBytes.this.regionSize) {
                IO.offsetCheckFailed();
            }
        }

        private static void offsetCheckFailed() {
            assert (false);
            throw new IllegalArgumentException("bad access");
        }
    }
}

