/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery.plan;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.Lock;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.Version;
import org.elasticsearch.TransportVersions;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.snapshots.get.shard.GetShardSnapshotRequest;
import org.elasticsearch.action.admin.cluster.snapshots.get.shard.GetShardSnapshotResponse;
import org.elasticsearch.action.admin.cluster.snapshots.get.shard.TransportGetShardSnapshotAction;
import org.elasticsearch.action.support.ThreadedActionListener;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.metadata.RepositoriesMetadata;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.lucene.store.ByteArrayIndexInput;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.index.store.StoreFileMetadata;
import org.elasticsearch.indices.recovery.plan.ShardSnapshot;
import org.elasticsearch.repositories.RepositoriesService;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.ShardSnapshotInfo;
import org.elasticsearch.repositories.blobstore.BlobStoreRepository;
import org.elasticsearch.snapshots.Snapshot;
import org.elasticsearch.threadpool.ThreadPool;

public class ShardSnapshotsService {
    private static final Logger logger = LogManager.getLogger(ShardSnapshotsService.class);
    private final Client client;
    private final RepositoriesService repositoriesService;
    private final ThreadPool threadPool;
    private final ClusterService clusterService;

    @Inject
    public ShardSnapshotsService(Client client, RepositoriesService repositoriesService, ThreadPool threadPool, ClusterService clusterService) {
        this.client = client;
        this.repositoriesService = repositoriesService;
        this.threadPool = threadPool;
        this.clusterService = clusterService;
    }

    public void fetchLatestSnapshotsForShard(ShardId shardId, ActionListener<Optional<ShardSnapshot>> listener) {
        assert (shardId != null) : "ShardId was null but a value was expected";
        List<String> repositories = RepositoriesMetadata.get(this.clusterService.state()).repositories().stream().filter(repositoryMetadata -> BlobStoreRepository.USE_FOR_PEER_RECOVERY_SETTING.get(repositoryMetadata.settings())).map(RepositoryMetadata::name).toList();
        if (repositories.isEmpty()) {
            logger.debug("Unable to use snapshots during peer recovery use_for_peer_recovery_repositories=[{}]", repositories);
            listener.onResponse(Optional.empty());
            return;
        }
        logger.debug("Searching for peer recovery compatible snapshots in [{}]", repositories);
        GetShardSnapshotRequest request = GetShardSnapshotRequest.latestSnapshotInRepositories(this.clusterService.state().getMinTransportVersion().onOrAfter(TransportVersions.SNAPSHOT_REQUEST_TIMEOUTS) ? TimeValue.MINUS_ONE : TimeValue.MAX_VALUE, shardId, repositories);
        this.client.execute(TransportGetShardSnapshotAction.TYPE, request, new ThreadedActionListener(this.threadPool.generic(), listener.map(shardSnapshotResponse -> this.fetchSnapshotFiles(shardId, (GetShardSnapshotResponse)shardSnapshotResponse))));
    }

    private Optional<ShardSnapshot> fetchSnapshotFiles(ShardId shardId, GetShardSnapshotResponse shardSnapshotResponse) {
        assert (ThreadPool.assertCurrentThreadPool("generic"));
        Optional<ShardSnapshotInfo> latestShardSnapshotOpt = shardSnapshotResponse.getLatestShardSnapshot();
        if (latestShardSnapshotOpt.isEmpty()) {
            logger.debug("{} no latest shard snapshot found", (Object)shardId);
            return Optional.empty();
        }
        ShardSnapshotInfo latestShardSnapshot = latestShardSnapshotOpt.get();
        try {
            Snapshot snapshot = latestShardSnapshot.getSnapshot();
            logger.debug("{} considering recovery from [{}][{}]", (Object)shardId, (Object)snapshot.getRepository(), (Object)snapshot.getSnapshotId());
            Repository repository = this.repositoriesService.repository(snapshot.getRepository());
            if (!(repository instanceof BlobStoreRepository)) {
                logger.debug("{} not recovering from snapshot in non-blobstore repository [{}]", (Object)shardId, (Object)snapshot.getRepository());
                return Optional.empty();
            }
            BlobStoreRepository blobStoreRepository = (BlobStoreRepository)repository;
            BlobContainer blobContainer = blobStoreRepository.shardContainer(latestShardSnapshot.getIndexId(), latestShardSnapshot.getShardId().getId());
            BlobStoreIndexShardSnapshot blobStoreIndexShardSnapshot = blobStoreRepository.loadShardSnapshot(blobContainer, snapshot.getSnapshotId());
            Map<String, StoreFileMetadata> snapshotFiles = blobStoreIndexShardSnapshot.indexFiles().stream().map(BlobStoreIndexShardSnapshot.FileInfo::metadata).collect(Collectors.toMap(StoreFileMetadata::name, Function.identity()));
            StoreFileMetadataDirectory directory = new StoreFileMetadataDirectory(snapshotFiles);
            SegmentInfos segmentCommitInfos = Lucene.readSegmentInfos(directory);
            Map userData = segmentCommitInfos.userData;
            Version commitLuceneVersion = segmentCommitInfos.getCommitLuceneVersion();
            return Optional.of(new ShardSnapshot(latestShardSnapshot, blobStoreIndexShardSnapshot.indexFiles(), userData, commitLuceneVersion));
        }
        catch (Exception e) {
            logger.warn(() -> Strings.format((String)"Unable to fetch shard snapshot files for %s", (Object[])new Object[]{latestShardSnapshot}), (Throwable)e);
            return Optional.empty();
        }
    }

    private static final class StoreFileMetadataDirectory
    extends Directory {
        private final Map<String, StoreFileMetadata> files;

        private StoreFileMetadataDirectory(Map<String, StoreFileMetadata> files) {
            this.files = files;
        }

        public String[] listAll() {
            return this.files.keySet().toArray(new String[0]);
        }

        public IndexInput openInput(String name, IOContext context) throws IOException {
            StoreFileMetadata storeFileMetadata = this.getStoreFileMetadata(name);
            if (!storeFileMetadata.hashEqualsContents()) {
                throw new IOException("Unable to open " + name);
            }
            BytesRef data = storeFileMetadata.hash();
            return new ByteArrayIndexInput(name, data.bytes, data.offset, data.length);
        }

        public void deleteFile(String name) {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        public long fileLength(String name) throws IOException {
            StoreFileMetadata storeFileMetadata = this.getStoreFileMetadata(name);
            return storeFileMetadata.length();
        }

        public IndexOutput createOutput(String name, IOContext context) {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        public IndexOutput createTempOutput(String prefix, String suffix, IOContext context) {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        public void sync(Collection<String> names) {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        public void syncMetaData() {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        public void rename(String source, String dest) {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        public void close() {
        }

        public Set<String> getPendingDeletions() {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        public Lock obtainLock(String name) {
            throw new UnsupportedOperationException("this directory is read-only");
        }

        private StoreFileMetadata getStoreFileMetadata(String name) throws IOException {
            StoreFileMetadata storeFileMetadata = this.files.get(name);
            if (storeFileMetadata == null) {
                throw new IOException("Unable to find " + name);
            }
            return storeFileMetadata;
        }
    }
}

