/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.autoscaling;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.TemporalAccessor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.xcontent.ChunkedToXContent;
import org.elasticsearch.common.xcontent.XContentElasticsearchExtension;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Predicates;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.xpack.autoscaling.capacity.AutoscalingDeciderContext;
import org.elasticsearch.xpack.core.ml.MachineLearningField;
import org.elasticsearch.xpack.core.ml.MlTasks;
import org.elasticsearch.xpack.core.ml.action.StartDatafeedAction;
import org.elasticsearch.xpack.core.ml.inference.assignment.TrainedModelAssignment;
import org.elasticsearch.xpack.ml.MachineLearning;
import org.elasticsearch.xpack.ml.autoscaling.MlAutoscalingContext;
import org.elasticsearch.xpack.ml.autoscaling.MlAutoscalingDeciderService;
import org.elasticsearch.xpack.ml.autoscaling.MlMemoryAutoscalingCapacity;
import org.elasticsearch.xpack.ml.autoscaling.NativeMemoryCapacity;
import org.elasticsearch.xpack.ml.autoscaling.NodeAvailabilityZoneMapper;
import org.elasticsearch.xpack.ml.autoscaling.ScaleTimer;
import org.elasticsearch.xpack.ml.job.NodeLoad;
import org.elasticsearch.xpack.ml.job.NodeLoadDetector;
import org.elasticsearch.xpack.ml.process.MlMemoryTracker;
import org.elasticsearch.xpack.ml.utils.MlProcessors;
import org.elasticsearch.xpack.ml.utils.NativeMemoryCalculator;

class MlMemoryAutoscalingDecider {
    private static final Logger logger = LogManager.getLogger(MlMemoryAutoscalingDecider.class);
    private static final String MEMORY_STALE = "unable to make scaling decision as job memory requirements are stale";
    private static final long ACCEPTABLE_DIFFERENCE = ByteSizeValue.ofMb((long)1L).getBytes();
    private final MlMemoryTracker mlMemoryTracker;
    private final NodeAvailabilityZoneMapper nodeAvailabilityZoneMapper;
    private final NodeLoadDetector nodeLoadDetector;
    private final ScaleTimer scaleTimer;
    private volatile int maxMachineMemoryPercent;
    private volatile int maxOpenJobs;
    private volatile boolean useAuto;
    private volatile long mlNativeMemoryForLargestMlNode;

    MlMemoryAutoscalingDecider(Settings settings, ClusterService clusterService, NodeAvailabilityZoneMapper nodeAvailabilityZoneMapper, NodeLoadDetector nodeLoadDetector, ScaleTimer scaleTimer) {
        this.nodeAvailabilityZoneMapper = Objects.requireNonNull(nodeAvailabilityZoneMapper);
        this.nodeLoadDetector = Objects.requireNonNull(nodeLoadDetector);
        this.mlMemoryTracker = Objects.requireNonNull(nodeLoadDetector.getMlMemoryTracker());
        this.scaleTimer = Objects.requireNonNull(scaleTimer);
        this.maxMachineMemoryPercent = (Integer)MachineLearning.MAX_MACHINE_MEMORY_PERCENT.get(settings);
        this.maxOpenJobs = (Integer)MachineLearning.MAX_OPEN_JOBS_PER_NODE.get(settings);
        this.useAuto = (Boolean)MachineLearningField.USE_AUTO_MACHINE_MEMORY_PERCENT.get(settings);
        this.setMaxMlNodeSize((ByteSizeValue)MachineLearning.MAX_ML_NODE_SIZE.get(settings));
        clusterService.getClusterSettings().addSettingsUpdateConsumer(MachineLearning.MAX_MACHINE_MEMORY_PERCENT, this::setMaxMachineMemoryPercent);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(MachineLearning.MAX_OPEN_JOBS_PER_NODE, this::setMaxOpenJobs);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(MachineLearningField.USE_AUTO_MACHINE_MEMORY_PERCENT, this::setUseAuto);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(MachineLearning.MAX_ML_NODE_SIZE, this::setMaxMlNodeSize);
    }

    void setMaxMachineMemoryPercent(int maxMachineMemoryPercent) {
        this.maxMachineMemoryPercent = maxMachineMemoryPercent;
    }

    void setMaxOpenJobs(int maxOpenJobs) {
        this.maxOpenJobs = maxOpenJobs;
    }

    void setUseAuto(boolean useAuto) {
        this.useAuto = useAuto;
    }

    void setMaxMlNodeSize(ByteSizeValue maxMlNodeSize) {
        long maxMlNodeSizeBytes = maxMlNodeSize.getBytes();
        this.mlNativeMemoryForLargestMlNode = maxMlNodeSizeBytes <= 0L ? Long.MAX_VALUE : NativeMemoryCalculator.allowedBytesForMl(maxMlNodeSizeBytes, this.maxMachineMemoryPercent, this.useAuto);
    }

    public MlMemoryAutoscalingCapacity scale(Settings configuration, AutoscalingDeciderContext context, MlAutoscalingContext mlContext, int allocatedProcessorsScale) {
        ClusterState clusterState = context.state();
        this.scaleTimer.lastScaleToScaleIntervalMillis().ifPresent(scaleInterval -> this.mlMemoryTracker.setAutoscalingCheckInterval(Duration.ofMillis(scaleInterval)));
        int numAnalyticsJobsInQueue = (Integer)MlAutoscalingDeciderService.NUM_ANALYTICS_JOBS_IN_QUEUE.get(configuration);
        int numAnomalyJobsInQueue = (Integer)MlAutoscalingDeciderService.NUM_ANOMALY_JOBS_IN_QUEUE.get(configuration);
        NativeMemoryCapacity currentScale = this.currentScale(mlContext.mlNodes);
        if (mlContext.mlNodes.isEmpty() && mlContext.hasWaitingTasks()) {
            return this.scaleUpFromZero(mlContext);
        }
        if (!this.mlMemoryTracker.isRecentlyRefreshed()) {
            logger.debug("view of job memory is stale given duration [{}]. Not attempting to make scaling decision", new Object[]{this.mlMemoryTracker.getStalenessDuration()});
            return this.refreshMemoryTrackerAndBuildEmptyDecision(MEMORY_STALE);
        }
        ArrayList<NodeLoad> nodeLoads = new ArrayList<NodeLoad>(mlContext.mlNodes.size());
        boolean nodeLoadIsMemoryAccurate = true;
        for (DiscoveryNode node : mlContext.mlNodes) {
            NodeLoad nodeLoad = this.nodeLoadDetector.detectNodeLoad(clusterState, node, this.maxOpenJobs, this.maxMachineMemoryPercent, this.useAuto);
            if (nodeLoad.getError() != null) {
                logger.warn("[{}] failed to gather node load limits, failure [{}]. Returning no scale", new Object[]{node.getId(), nodeLoad.getError()});
                return this.refreshMemoryTrackerAndBuildEmptyDecision("Passing currently perceived capacity as there was a failure gathering node limits [" + nodeLoad.getError() + "]");
            }
            nodeLoads.add(nodeLoad);
            if (nodeLoad.isUseMemory()) continue;
            nodeLoadIsMemoryAccurate = false;
            logger.debug("[{}] failed to gather node load - memory usage for one or more tasks not available.", new Object[]{node.getId()});
        }
        if (!nodeLoadIsMemoryAccurate) {
            return this.refreshMemoryTrackerAndBuildEmptyDecision("Passing currently perceived capacity as nodes were unable to provide an accurate view of their memory usage");
        }
        Optional<MlMemoryAutoscalingCapacity> scaleUpDecision = this.checkForScaleUp(numAnomalyJobsInQueue, numAnalyticsJobsInQueue, nodeLoads, mlContext.waitingAnomalyJobs, mlContext.waitingSnapshotUpgrades, mlContext.waitingAnalyticsJobs, mlContext.waitingAllocatedModels, this.calculateFutureAvailableCapacity(mlContext.persistentTasks, nodeLoads).orElse(null), currentScale);
        if (scaleUpDecision.isPresent()) {
            this.scaleTimer.resetScaleDownCoolDown();
            return scaleUpDecision.get();
        }
        List<String> partiallyAllocatedModels = mlContext.findPartiallyAllocatedModels();
        if (!(mlContext.waitingAnalyticsJobs.isEmpty() && mlContext.waitingSnapshotUpgrades.isEmpty() && mlContext.waitingAnomalyJobs.isEmpty() && partiallyAllocatedModels.isEmpty())) {
            this.scaleTimer.resetScaleDownCoolDown();
            return MlMemoryAutoscalingCapacity.from(context.currentCapacity()).setReason(String.format(Locale.ROOT, "Passing currently perceived capacity as there are [%d] model snapshot upgrades, [%d] analytics and [%d] anomaly detection jobs in the queue, [%d] trained models not fully-allocated, but the number in the queue is less than the configured maximum allowed or the queued jobs will eventually be assignable at the current size.", mlContext.waitingSnapshotUpgrades.size(), mlContext.waitingAnalyticsJobs.size(), mlContext.waitingAnomalyJobs.size(), partiallyAllocatedModels.size())).build();
        }
        long maxTaskMemoryBytes = this.maxMemoryBytes(mlContext);
        if (maxTaskMemoryBytes == 0L) {
            assert (!mlContext.isEmpty()) : "No tasks or models at all should have put us in the scale down to zero branch";
            logger.warn("The calculated minimum required node size was unexpectedly [0] as there are [{}] anomaly job tasks, [{}] model snapshot upgrade tasks, [{}] data frame analytics tasks and [{}] model assignments", new Object[]{mlContext.anomalyDetectionTasks.size(), mlContext.snapshotUpgradeTasks.size(), mlContext.dataframeAnalyticsTasks.size(), mlContext.modelAssignments.size()});
            logger.debug(() -> Strings.format((String)"persistent tasks that caused unexpected scaling situation: [%s]", (Object[])new Object[]{mlContext.persistentTasks == null ? "null" : org.elasticsearch.common.Strings.toString((ChunkedToXContent)mlContext.persistentTasks)}));
            return this.refreshMemoryTrackerAndBuildEmptyDecision("Passing currently perceived capacity as there are running analytics and anomaly jobs or deployed models, but their assignment explanations are unexpected or their memory usage estimates are inaccurate.");
        }
        Optional<MlMemoryAutoscalingCapacity> maybeScaleDown = this.checkForScaleDown(nodeLoads, maxTaskMemoryBytes, currentScale).map(result -> {
            MlMemoryAutoscalingCapacity capacity = MlMemoryAutoscalingDecider.ensureScaleDown(result, MlMemoryAutoscalingCapacity.from(context.currentCapacity()).build());
            if (capacity == null) {
                return null;
            }
            if (MlMemoryAutoscalingDecider.modelAssignmentsRequireMoreThanHalfCpu(mlContext.modelAssignments.values(), mlContext.mlNodes, allocatedProcessorsScale)) {
                logger.debug("not down-scaling; model assignments require more than half of the ML tier's allocated processors");
                return null;
            }
            return capacity;
        });
        if (maybeScaleDown.isPresent()) {
            long maxOpenJobsCopy;
            long totalAssignedJobs;
            MlMemoryAutoscalingCapacity scaleDownDecisionResult = maybeScaleDown.get();
            if (nodeLoads.size() > 1 && (totalAssignedJobs = nodeLoads.stream().mapToLong(NodeLoad::getNumAssignedJobsAndModels).sum()) > (maxOpenJobsCopy = (long)this.maxOpenJobs)) {
                String msg = String.format(Locale.ROOT, "not scaling down as the total number of jobs [%d] exceeds the setting [%s (%d)]. To allow a scale down [%s] must be increased.", totalAssignedJobs, MachineLearning.MAX_OPEN_JOBS_PER_NODE.getKey(), maxOpenJobsCopy, MachineLearning.MAX_OPEN_JOBS_PER_NODE.getKey());
                logger.info(() -> Strings.format((String)"%s Calculated potential scaled down capacity [%s]", (Object[])new Object[]{msg, scaleDownDecisionResult}));
                return MlMemoryAutoscalingCapacity.from(context.currentCapacity()).setReason(msg).build();
            }
            long msLeftToScale = this.scaleTimer.markDownScaleAndGetMillisLeftFromDelay(configuration);
            if (msLeftToScale <= 0L) {
                return scaleDownDecisionResult;
            }
            TimeValue downScaleDelay = (TimeValue)MlAutoscalingDeciderService.DOWN_SCALE_DELAY.get(configuration);
            logger.debug(() -> Strings.format((String)"not scaling down as the current scale down delay [%s] is not satisfied. The last time scale down was detected [%s]. Calculated scaled down capacity [%s] ", (Object[])new Object[]{downScaleDelay.getStringRep(), XContentElasticsearchExtension.DEFAULT_FORMATTER.format((TemporalAccessor)Instant.ofEpochMilli(this.scaleTimer.downScaleDetectedMillis())), scaleDownDecisionResult}));
            return MlMemoryAutoscalingCapacity.from(context.currentCapacity()).setReason(String.format(Locale.ROOT, "Passing currently perceived capacity as down scale delay has not been satisfied; configured delay [%s] last detected scale down event [%s]. Will request scale down in approximately [%s]", downScaleDelay.getStringRep(), XContentElasticsearchExtension.DEFAULT_FORMATTER.format((TemporalAccessor)Instant.ofEpochMilli(this.scaleTimer.downScaleDetectedMillis())), TimeValue.timeValueMillis((long)msLeftToScale).getStringRep())).build();
        }
        return MlMemoryAutoscalingCapacity.from(context.currentCapacity()).setReason("Passing currently perceived capacity as no scaling changes are necessary").build();
    }

    NativeMemoryCapacity currentScale(List<DiscoveryNode> machineLearningNodes) {
        return NativeMemoryCapacity.currentScale(machineLearningNodes, this.maxMachineMemoryPercent, this.useAuto);
    }

    MlMemoryAutoscalingCapacity capacityFromNativeMemory(NativeMemoryCapacity nativeMemoryCapacity) {
        return nativeMemoryCapacity.autoscalingCapacity(this.maxMachineMemoryPercent, this.useAuto, this.mlNativeMemoryForLargestMlNode, this.nodeAvailabilityZoneMapper.getNumMlAvailabilityZones().orElse(1)).build();
    }

    private MlMemoryAutoscalingCapacity refreshMemoryTrackerAndBuildEmptyDecision(String reason) {
        this.mlMemoryTracker.asyncRefresh();
        return MlMemoryAutoscalingCapacity.builder(null, null).setReason(reason).build();
    }

    private long maxMemoryBytes(MlAutoscalingContext mlContext) {
        long maxMemoryBytes = Math.max(mlContext.anomalyDetectionTasks.stream().filter(PersistentTasksCustomMetadata.PersistentTask::isAssigned).mapToLong(t -> {
            Long mem = this.getAnomalyMemoryRequirement((PersistentTasksCustomMetadata.PersistentTask<?>)t);
            if (mem == null) {
                logger.warn("unexpected null for anomaly detection memory requirement for [{}]", new Object[]{MlTasks.jobId((String)t.getId())});
            }
            assert (mem != null) : "unexpected null for anomaly memory requirement after recent stale check";
            return mem == null ? 0L : mem;
        }).max().orElse(0L), mlContext.snapshotUpgradeTasks.stream().filter(PersistentTasksCustomMetadata.PersistentTask::isAssigned).mapToLong(t -> {
            Long mem = this.getAnomalyMemoryRequirement((PersistentTasksCustomMetadata.PersistentTask<?>)t);
            if (mem == null) {
                logger.warn("unexpected null for snapshot upgrade memory requirement for [{}]", new Object[]{MlTasks.jobId((String)t.getId())});
            }
            assert (mem != null) : "unexpected null for anomaly memory requirement after recent stale check";
            return mem == null ? 0L : mem;
        }).max().orElse(0L));
        maxMemoryBytes = Math.max(maxMemoryBytes, mlContext.dataframeAnalyticsTasks.stream().filter(PersistentTasksCustomMetadata.PersistentTask::isAssigned).mapToLong(t -> {
            Long mem = this.getAnalyticsMemoryRequirement((PersistentTasksCustomMetadata.PersistentTask<?>)t);
            if (mem == null) {
                logger.warn("unexpected null for analytics memory requirement for [{}]", new Object[]{MlTasks.dataFrameAnalyticsId((String)t.getId())});
            }
            assert (mem != null) : "unexpected null for analytics memory requirement after recent stale check";
            return mem == null ? 0L : mem;
        }).max().orElse(0L));
        maxMemoryBytes = Math.max(maxMemoryBytes, mlContext.modelAssignments.values().stream().mapToLong(t -> t.getTaskParams().estimateMemoryUsageBytes()).max().orElse(0L));
        return maxMemoryBytes;
    }

    MlMemoryAutoscalingCapacity scaleUpFromZero(MlAutoscalingContext mlContext) {
        Optional<NativeMemoryCapacity> analyticsCapacity = MlMemoryAutoscalingDecider.requiredCapacityExcludingPerNodeOverheadForUnassignedJobs(mlContext.waitingAnalyticsJobs, this::getAnalyticsMemoryRequirement, 0);
        Optional<NativeMemoryCapacity> anomalyCapacity = MlMemoryAutoscalingDecider.requiredCapacityExcludingPerNodeOverheadForUnassignedJobs(mlContext.waitingAnomalyJobs, this::getAnomalyMemoryRequirement, 0);
        Optional<NativeMemoryCapacity> snapshotUpgradeCapacity = MlMemoryAutoscalingDecider.requiredCapacityExcludingPerNodeOverheadForUnassignedJobs(mlContext.waitingSnapshotUpgrades, this::getAnomalyMemoryRequirement, 0);
        Optional<NativeMemoryCapacity> allocatedModelCapacity = MlMemoryAutoscalingDecider.requiredCapacityExcludingPerNodeOverheadForUnassignedJobs(mlContext.waitingAllocatedModels, this::getAllocatedModelRequirement, 0);
        NativeMemoryCapacity updatedCapacity = anomalyCapacity.orElse(NativeMemoryCapacity.ZERO).merge(snapshotUpgradeCapacity.orElse(NativeMemoryCapacity.ZERO)).merge(analyticsCapacity.orElse(NativeMemoryCapacity.ZERO)).merge(allocatedModelCapacity.orElse(NativeMemoryCapacity.ZERO));
        if (updatedCapacity.getNodeMlNativeMemoryRequirementExcludingOverhead() == 0L) {
            updatedCapacity = updatedCapacity.merge(new NativeMemoryCapacity(ByteSizeValue.ofMb((long)1024L).getBytes(), ByteSizeValue.ofMb((long)1024L).getBytes()));
        }
        MlMemoryAutoscalingCapacity.Builder requiredCapacity = updatedCapacity.autoscalingCapacity(this.maxMachineMemoryPercent, this.useAuto, this.mlNativeMemoryForLargestMlNode, this.nodeAvailabilityZoneMapper.getNumMlAvailabilityZones().orElse(1));
        return requiredCapacity.setReason("requesting scale up as number of jobs in queues exceeded configured limit and there are no machine learning nodes").build();
    }

    static Optional<NativeMemoryCapacity> requiredCapacityExcludingPerNodeOverheadForUnassignedJobs(List<String> unassignedJobs, Function<String, Long> sizeFunction, int maxNumInQueue) {
        if (unassignedJobs.isEmpty()) {
            return Optional.empty();
        }
        List<Long> jobSizes = MlMemoryAutoscalingDecider.computeJobSizes(unassignedJobs, sizeFunction);
        long tierMemory = 0L;
        long nodeMemory = jobSizes.get(0);
        Iterator<Long> iter = jobSizes.iterator();
        while (jobSizes.size() > maxNumInQueue && iter.hasNext()) {
            tierMemory += iter.next().longValue();
            iter.remove();
        }
        return Optional.of(new NativeMemoryCapacity(tierMemory, nodeMemory));
    }

    Optional<MlMemoryAutoscalingCapacity> checkForScaleUp(int numAnomalyJobsInQueue, int numAnalyticsJobsInQueue, List<NodeLoad> nodeLoads, List<String> waitingAnomalyJobs, List<String> waitingSnapshotUpgrades, List<String> waitingAnalyticsJobs, List<String> waitingAllocatedModels, @Nullable NativeMemoryCapacity futureFreedCapacity, NativeMemoryCapacity currentScale) {
        logger.debug(() -> Strings.format((String)"Checking for scale up - waiting data frame analytics jobs [%s] data frame analytics jobs allowed to queue [%s] waiting anomaly detection jobs (including model snapshot upgrades) [%s] anomaly detection jobs allowed to queue [%s] waiting models [%s] future freed capacity [%s] current scale [%s]", (Object[])new Object[]{waitingAnalyticsJobs.size(), numAnalyticsJobsInQueue, waitingAnomalyJobs.size() + waitingSnapshotUpgrades.size(), numAnomalyJobsInQueue, waitingAllocatedModels.size(), futureFreedCapacity, currentScale}));
        if (waitingAnalyticsJobs.size() > numAnalyticsJobsInQueue || waitingAnomalyJobs.size() + waitingSnapshotUpgrades.size() > numAnomalyJobsInQueue || waitingAllocatedModels.size() > 0) {
            Tuple<NativeMemoryCapacity, List<NodeLoad>> anomalyCapacityAndNewLoad = MlMemoryAutoscalingDecider.determineUnassignableJobs(Stream.concat(waitingAnomalyJobs.stream(), waitingSnapshotUpgrades.stream()).toList(), this::getAnomalyMemoryRequirement, NodeLoad.Builder::incNumAssignedAnomalyDetectorJobs, numAnomalyJobsInQueue, nodeLoads).orElse((Tuple<NativeMemoryCapacity, List<NodeLoad>>)Tuple.tuple((Object)NativeMemoryCapacity.ZERO, nodeLoads));
            Tuple<NativeMemoryCapacity, List<NodeLoad>> analyticsCapacityAndNewLoad = MlMemoryAutoscalingDecider.determineUnassignableJobs(waitingAnalyticsJobs, this::getAnalyticsMemoryRequirement, NodeLoad.Builder::incNumAssignedDataFrameAnalyticsJobs, numAnalyticsJobsInQueue, (List)anomalyCapacityAndNewLoad.v2()).orElse((Tuple<NativeMemoryCapacity, List<NodeLoad>>)Tuple.tuple((Object)NativeMemoryCapacity.ZERO, (Object)((List)anomalyCapacityAndNewLoad.v2())));
            Tuple<NativeMemoryCapacity, List<NodeLoad>> modelCapacityAndNewLoad = MlMemoryAutoscalingDecider.determineUnassignableJobs(waitingAllocatedModels, this::getAllocatedModelRequirement, NodeLoad.Builder::incNumAssignedNativeInferenceModels, 0, (List)analyticsCapacityAndNewLoad.v2()).orElse((Tuple<NativeMemoryCapacity, List<NodeLoad>>)Tuple.tuple((Object)NativeMemoryCapacity.ZERO, (Object)((List)analyticsCapacityAndNewLoad.v2())));
            if (((NativeMemoryCapacity)analyticsCapacityAndNewLoad.v1()).equals(NativeMemoryCapacity.ZERO) && ((NativeMemoryCapacity)anomalyCapacityAndNewLoad.v1()).equals(NativeMemoryCapacity.ZERO) && ((NativeMemoryCapacity)modelCapacityAndNewLoad.v1()).equals(NativeMemoryCapacity.ZERO)) {
                logger.debug("no_scale event as current capacity, even though there are waiting jobs, is adequate to run the queued jobs");
                return Optional.empty();
            }
            long maxFreeNodeMemAfterPossibleAssignments = ((List)modelCapacityAndNewLoad.v2()).stream().filter(nodeLoad -> nodeLoad.getError() == null && nodeLoad.isUseMemory()).map(NodeLoad::getFreeMemoryExcludingPerNodeOverhead).max(Long::compareTo).orElse(0L);
            if (maxFreeNodeMemAfterPossibleAssignments > currentScale.getNodeMlNativeMemoryRequirementExcludingOverhead() || maxFreeNodeMemAfterPossibleAssignments > currentScale.getTierMlNativeMemoryRequirementExcludingOverhead()) {
                assert (false) : "highest free node memory after possible assignments [" + maxFreeNodeMemAfterPossibleAssignments + "] greater than current scale [" + currentScale + "]";
                logger.warn("Highest free node memory after possible assignments [" + maxFreeNodeMemAfterPossibleAssignments + "] greater than current scale [" + currentScale + "] - will scale up without considering current free memory");
                maxFreeNodeMemAfterPossibleAssignments = 0L;
            }
            NativeMemoryCapacity updatedCapacity = new NativeMemoryCapacity(-maxFreeNodeMemAfterPossibleAssignments, 0L).merge(currentScale).merge((NativeMemoryCapacity)analyticsCapacityAndNewLoad.v1()).merge((NativeMemoryCapacity)anomalyCapacityAndNewLoad.v1()).merge((NativeMemoryCapacity)modelCapacityAndNewLoad.v1());
            MlMemoryAutoscalingCapacity requiredCapacity = updatedCapacity.autoscalingCapacity(this.maxMachineMemoryPercent, this.useAuto, this.mlNativeMemoryForLargestMlNode, this.nodeAvailabilityZoneMapper.getNumMlAvailabilityZones().orElse(1)).setReason("requesting scale up as number of jobs in queues exceeded configured limit or there is at least one trained model waiting for assignment and current capacity is not large enough for waiting jobs or models").build();
            return Optional.of(requiredCapacity);
        }
        if (!(waitingAnalyticsJobs.isEmpty() && waitingSnapshotUpgrades.isEmpty() && waitingAnomalyJobs.isEmpty())) {
            Long requiredMemory;
            if (futureFreedCapacity == null) {
                Optional<Long> maxSize = Stream.concat(waitingAnalyticsJobs.stream().map(this::getAnalyticsMemoryRequirement), Stream.concat(waitingAnomalyJobs.stream().map(this::getAnomalyMemoryRequirement), waitingSnapshotUpgrades.stream().map(this::getAnomalyMemoryRequirement))).filter(Objects::nonNull).max(Long::compareTo);
                if (maxSize.isPresent() && maxSize.get() > currentScale.getNodeMlNativeMemoryRequirementExcludingOverhead()) {
                    MlMemoryAutoscalingCapacity requiredCapacity = new NativeMemoryCapacity(Math.max(currentScale.getTierMlNativeMemoryRequirementExcludingOverhead(), maxSize.get()), maxSize.get()).autoscalingCapacity(this.maxMachineMemoryPercent, this.useAuto, this.mlNativeMemoryForLargestMlNode, this.nodeAvailabilityZoneMapper.getNumMlAvailabilityZones().orElse(1)).setReason("requesting scale up as there is no node large enough to handle queued jobs").build();
                    return Optional.of(requiredCapacity);
                }
                logger.debug("Cannot make a scaling decision as future freed capacity is not known and largest job could fit on an existing node");
                return Optional.empty();
            }
            long newTierNeeded = -futureFreedCapacity.getTierMlNativeMemoryRequirementExcludingOverhead();
            long newNodeMax = currentScale.getNodeMlNativeMemoryRequirementExcludingOverhead();
            for (String analyticsJob : waitingAnalyticsJobs) {
                requiredMemory = this.getAnalyticsMemoryRequirement(analyticsJob);
                if (requiredMemory == null) continue;
                newTierNeeded += requiredMemory.longValue();
                newNodeMax = Math.max(newNodeMax, requiredMemory);
            }
            for (String anomalyJob : waitingAnomalyJobs) {
                requiredMemory = this.getAnomalyMemoryRequirement(anomalyJob);
                if (requiredMemory == null) continue;
                newTierNeeded += requiredMemory.longValue();
                newNodeMax = Math.max(newNodeMax, requiredMemory);
            }
            for (String snapshotUpgrade : waitingSnapshotUpgrades) {
                requiredMemory = this.getAnomalyMemoryRequirement(snapshotUpgrade);
                if (requiredMemory == null) continue;
                newTierNeeded += requiredMemory.longValue();
                newNodeMax = Math.max(newNodeMax, requiredMemory);
            }
            if (newNodeMax > currentScale.getNodeMlNativeMemoryRequirementExcludingOverhead() || newTierNeeded > 0L) {
                NativeMemoryCapacity newCapacity = new NativeMemoryCapacity(Math.max(0L, newTierNeeded), newNodeMax);
                MlMemoryAutoscalingCapacity requiredCapacity = currentScale.merge(newCapacity).autoscalingCapacity(this.maxMachineMemoryPercent, this.useAuto, this.mlNativeMemoryForLargestMlNode, this.nodeAvailabilityZoneMapper.getNumMlAvailabilityZones().orElse(1)).setReason("scaling up as adequate space would not automatically become available when running jobs finish").build();
                return Optional.of(requiredCapacity);
            }
        }
        return Optional.empty();
    }

    static Optional<Tuple<NativeMemoryCapacity, List<NodeLoad>>> determineUnassignableJobs(List<String> unassignedJobs, Function<String, Long> sizeFunction, Consumer<NodeLoad.Builder> incrementCountFunction, int maxNumInQueue, List<NodeLoad> nodeLoads) {
        if (unassignedJobs.isEmpty()) {
            return Optional.empty();
        }
        if (unassignedJobs.size() < maxNumInQueue) {
            return Optional.empty();
        }
        PriorityQueue<NodeLoad.Builder> mostFreeMemoryFirst = new PriorityQueue<NodeLoad.Builder>(nodeLoads.size(), Comparator.comparingLong(v -> v.remainingJobs() == 0 ? 0L : v.getFreeMemory()).reversed());
        for (NodeLoad load : nodeLoads) {
            mostFreeMemoryFirst.add(NodeLoad.builder(load));
        }
        List<Long> jobSizes = MlMemoryAutoscalingDecider.computeJobSizes(unassignedJobs, sizeFunction);
        Iterator<Long> assignmentIter = jobSizes.iterator();
        while (jobSizes.size() > maxNumInQueue && assignmentIter.hasNext()) {
            long requiredMemory = assignmentIter.next();
            long requiredNativeCodeOverhead = 0L;
            NodeLoad.Builder nodeLoad = mostFreeMemoryFirst.peek();
            assert (nodeLoad != null) : "unexpected null value while calculating assignable memory";
            if (nodeLoad.getNumAssignedJobs() == 0) {
                requiredNativeCodeOverhead = MachineLearning.NATIVE_EXECUTABLE_CODE_OVERHEAD.getBytes();
            }
            if (nodeLoad.getFreeMemory() < requiredMemory + requiredNativeCodeOverhead) continue;
            assignmentIter.remove();
            nodeLoad = mostFreeMemoryFirst.poll();
            incrementCountFunction.accept(nodeLoad);
            mostFreeMemoryFirst.add(nodeLoad.incAssignedNativeCodeOverheadMemory(requiredNativeCodeOverhead).incAssignedAnomalyDetectorMemory(requiredMemory));
        }
        List<NodeLoad> adjustedLoads = mostFreeMemoryFirst.stream().map(NodeLoad.Builder::build).toList();
        ArrayList<Long> unassignableMemory = new ArrayList<Long>();
        Iterator<Long> unassignableIter = jobSizes.iterator();
        while (jobSizes.size() > maxNumInQueue && unassignableIter.hasNext()) {
            unassignableMemory.add(unassignableIter.next());
            unassignableIter.remove();
        }
        if (unassignableMemory.isEmpty()) {
            return Optional.of(Tuple.tuple((Object)NativeMemoryCapacity.ZERO, adjustedLoads));
        }
        return Optional.of(Tuple.tuple((Object)new NativeMemoryCapacity(unassignableMemory.stream().mapToLong(Long::longValue).sum(), (Long)unassignableMemory.get(0)), adjustedLoads));
    }

    Optional<MlMemoryAutoscalingCapacity> checkForScaleDown(List<NodeLoad> nodeLoads, long largestJob, NativeMemoryCapacity currentCapacity) {
        long currentlyNecessaryTier = nodeLoads.stream().mapToLong(NodeLoad::getAssignedJobMemoryExcludingPerNodeOverhead).sum();
        if (currentlyNecessaryTier < currentCapacity.getTierMlNativeMemoryRequirementExcludingOverhead() || largestJob < currentCapacity.getNodeMlNativeMemoryRequirementExcludingOverhead()) {
            NativeMemoryCapacity nativeMemoryCapacity = new NativeMemoryCapacity(Math.min(currentlyNecessaryTier, currentCapacity.getTierMlNativeMemoryRequirementExcludingOverhead()), Math.min(largestJob, currentCapacity.getNodeMlNativeMemoryRequirementExcludingOverhead()), null);
            MlMemoryAutoscalingCapacity requiredCapacity = nativeMemoryCapacity.autoscalingCapacity(this.maxMachineMemoryPercent, this.useAuto, this.mlNativeMemoryForLargestMlNode, this.nodeAvailabilityZoneMapper.getNumMlAvailabilityZones().orElse(1)).setReason("Requesting scale down as tier and/or node size could be smaller").build();
            return Optional.of(requiredCapacity);
        }
        return Optional.empty();
    }

    static MlMemoryAutoscalingCapacity ensureScaleDown(MlMemoryAutoscalingCapacity scaleDownResult, MlMemoryAutoscalingCapacity currentCapacity) {
        if (scaleDownResult == null || currentCapacity == null) {
            return null;
        }
        MlMemoryAutoscalingCapacity newCapacity = MlMemoryAutoscalingCapacity.builder(ByteSizeValue.ofBytes((long)Math.min(scaleDownResult.nodeSize().getBytes(), currentCapacity.nodeSize().getBytes())), ByteSizeValue.ofBytes((long)Math.min(scaleDownResult.tierSize().getBytes(), currentCapacity.tierSize().getBytes()))).setReason(scaleDownResult.reason()).build();
        if (scaleDownResult.nodeSize().getBytes() - newCapacity.nodeSize().getBytes() > ACCEPTABLE_DIFFERENCE || scaleDownResult.tierSize().getBytes() - newCapacity.tierSize().getBytes() > ACCEPTABLE_DIFFERENCE) {
            logger.warn("scale down accidentally requested a scale up, auto-corrected; initial scaling [{}], corrected [{}]", new Object[]{scaleDownResult, newCapacity});
        }
        return newCapacity;
    }

    static boolean modelAssignmentsRequireMoreThanHalfCpu(Collection<TrainedModelAssignment> assignments, List<DiscoveryNode> mlNodes, int allocatedProcessorsScale) {
        int totalMlProcessors;
        int totalRequiredProcessors = assignments.stream().mapToInt(t -> t.getTaskParams().getNumberOfAllocations() * t.getTaskParams().getThreadsPerAllocation()).sum();
        return totalRequiredProcessors * 2 > (totalMlProcessors = mlNodes.stream().mapToInt(node -> MlProcessors.get(node, allocatedProcessorsScale).roundUp()).sum());
    }

    Optional<NativeMemoryCapacity> calculateFutureAvailableCapacity(Collection<DiscoveryNode> mlNodes, ClusterState clusterState) {
        return this.calculateFutureAvailableCapacity((PersistentTasksCustomMetadata)clusterState.metadata().custom("persistent_tasks"), mlNodes.stream().map(node -> this.nodeLoadDetector.detectNodeLoad(clusterState, (DiscoveryNode)node, this.maxOpenJobs, this.maxMachineMemoryPercent, this.useAuto)).toList());
    }

    Optional<NativeMemoryCapacity> calculateFutureAvailableCapacity(PersistentTasksCustomMetadata tasks, List<NodeLoad> nodeLoads) {
        Long jobSize;
        List<PersistentTasksCustomMetadata.PersistentTask> jobsWithLookbackDatafeeds = MlMemoryAutoscalingDecider.datafeedTasks(tasks).stream().filter(t -> ((StartDatafeedAction.DatafeedParams)t.getParams()).getEndTime() != null && t.getExecutorNode() != null).toList();
        List<PersistentTasksCustomMetadata.PersistentTask> assignedAnalyticsJobs = MlAutoscalingContext.dataframeAnalyticsTasks(tasks).stream().filter(t -> t.getExecutorNode() != null).toList();
        HashMap<String, Long> freeMemoryByNodeId = new HashMap<String, Long>();
        for (NodeLoad nodeLoad : nodeLoads) {
            if (nodeLoad.getError() != null || !nodeLoad.isUseMemory()) {
                logger.debug("[{}] node free memory not available", new Object[]{nodeLoad.getNodeId()});
                return Optional.empty();
            }
            freeMemoryByNodeId.put(nodeLoad.getNodeId(), nodeLoad.getFreeMemoryExcludingPerNodeOverhead());
        }
        for (PersistentTasksCustomMetadata.PersistentTask lookbackOnlyDf : jobsWithLookbackDatafeeds) {
            jobSize = this.getAnomalyMemoryRequirement(((StartDatafeedAction.DatafeedParams)lookbackOnlyDf.getParams()).getJobId());
            if (jobSize == null) {
                return Optional.empty();
            }
            freeMemoryByNodeId.compute(lookbackOnlyDf.getExecutorNode(), (k, v) -> v == null ? jobSize : jobSize + v);
        }
        for (PersistentTasksCustomMetadata.PersistentTask task : assignedAnalyticsJobs) {
            jobSize = this.getAnalyticsMemoryRequirement(MlTasks.dataFrameAnalyticsId((String)task.getId()));
            if (jobSize == null) {
                return Optional.empty();
            }
            freeMemoryByNodeId.compute(task.getExecutorNode(), (k, v) -> v == null ? jobSize : jobSize + v);
        }
        return Optional.of(new NativeMemoryCapacity(freeMemoryByNodeId.values().stream().mapToLong(Long::longValue).sum(), freeMemoryByNodeId.values().stream().mapToLong(Long::longValue).max().orElse(0L)));
    }

    private static Collection<PersistentTasksCustomMetadata.PersistentTask<StartDatafeedAction.DatafeedParams>> datafeedTasks(PersistentTasksCustomMetadata tasksCustomMetadata) {
        if (tasksCustomMetadata == null) {
            return List.of();
        }
        return tasksCustomMetadata.findTasks("xpack/ml/datafeed", Predicates.always()).stream().map(p -> p).toList();
    }

    private Long getAnalyticsMemoryRequirement(String analyticsId) {
        Long mem = this.mlMemoryTracker.getDataFrameAnalyticsJobMemoryRequirement(analyticsId);
        if (mem == null) {
            logger.debug("[{}] data frame analytics job memory requirement not available", new Object[]{analyticsId});
        }
        return mem;
    }

    private Long getAllocatedModelRequirement(String modelId) {
        Long mem = this.mlMemoryTracker.getTrainedModelAssignmentMemoryRequirement(modelId);
        if (mem == null) {
            logger.debug("[{}] trained model memory requirement not available", new Object[]{modelId});
        }
        return mem;
    }

    private Long getAnalyticsMemoryRequirement(PersistentTasksCustomMetadata.PersistentTask<?> task) {
        return this.getAnalyticsMemoryRequirement(MlTasks.dataFrameAnalyticsId((String)task.getId()));
    }

    private Long getAnomalyMemoryRequirement(String anomalyId) {
        Long mem = this.mlMemoryTracker.getAnomalyDetectorJobMemoryRequirement(anomalyId);
        if (mem == null) {
            logger.debug("[{}] anomaly detection job memory requirement not available", new Object[]{anomalyId});
        }
        return mem;
    }

    private Long getAnomalyMemoryRequirement(PersistentTasksCustomMetadata.PersistentTask<?> task) {
        return this.getAnomalyMemoryRequirement(MlTasks.jobId((String)task.getId()));
    }

    private static List<Long> computeJobSizes(List<String> unassignedJobs, Function<String, Long> sizeFunction) {
        ArrayList<Long> jobSizes = new ArrayList<Long>(unassignedJobs.size());
        for (String unassignedJob : unassignedJobs) {
            jobSizes.add(Objects.requireNonNullElse(sizeFunction.apply(unassignedJob), 0L));
        }
        jobSizes.sort(Comparator.comparingLong(Long::longValue).reversed());
        return jobSizes;
    }
}

