/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.transform.transforms.scheduling;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.transform.transforms.TransformTaskParams;
import org.elasticsearch.xpack.transform.Transform;
import org.elasticsearch.xpack.transform.transforms.scheduling.MonotonicClock;
import org.elasticsearch.xpack.transform.transforms.scheduling.TransformScheduledTask;
import org.elasticsearch.xpack.transform.transforms.scheduling.TransformScheduledTaskQueue;

public final class TransformScheduler {
    private static final Logger logger = LogManager.getLogger(TransformScheduler.class);
    private final Clock clock;
    private final ThreadPool threadPool;
    private final TimeValue schedulerFrequency;
    private final TransformScheduledTaskQueue scheduledTasks;
    private final AtomicBoolean isProcessingActive;
    private final TimeValue minFrequency;
    private Scheduler.Cancellable scheduledFuture;

    public TransformScheduler(Clock clock, ThreadPool threadPool, Settings settings, TimeValue minFrequency) {
        this.clock = new MonotonicClock(Objects.requireNonNull(clock));
        this.threadPool = Objects.requireNonNull(threadPool);
        this.schedulerFrequency = (TimeValue)Transform.SCHEDULER_FREQUENCY.get(settings);
        this.scheduledTasks = new TransformScheduledTaskQueue();
        this.isProcessingActive = new AtomicBoolean();
        this.minFrequency = minFrequency;
    }

    public void start() {
        if (this.scheduledFuture == null) {
            this.scheduledFuture = this.threadPool.scheduleWithFixedDelay(this::processScheduledTasks, this.schedulerFrequency, (Executor)this.threadPool.generic());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void processScheduledTasks() {
        boolean taskWasProcessed;
        if (!this.isProcessingActive.compareAndSet(false, true)) {
            return;
        }
        logger.trace("Processing scheduled tasks started");
        boolean isTraceEnabled = logger.isTraceEnabled();
        Instant processingStarted = isTraceEnabled ? this.clock.instant() : null;
        try {
            taskWasProcessed = this.processScheduledTasksInternal();
        }
        finally {
            this.isProcessingActive.set(false);
        }
        if (isTraceEnabled) {
            Instant processingFinished = this.clock.instant();
            long tookMs = Duration.between(processingStarted, processingFinished).toMillis();
            if (taskWasProcessed) {
                logger.trace(Strings.format((String)"Processing one scheduled task finished, took %dms", (Object[])new Object[]{tookMs}));
            } else {
                logger.trace(Strings.format((String)"Looking for scheduled tasks to process finished, took %dms", (Object[])new Object[]{tookMs}));
            }
        }
        if (!taskWasProcessed) {
            return;
        }
        this.processScheduledTasks();
    }

    private boolean processScheduledTasksInternal() {
        TransformScheduledTask scheduledTask = this.scheduledTasks.first();
        if (scheduledTask == null) {
            return false;
        }
        long currentTimeMillis = this.clock.millis();
        if (currentTimeMillis < scheduledTask.getNextScheduledTimeMillis()) {
            return false;
        }
        Event event = new Event(scheduledTask.getTransformId(), scheduledTask.getNextScheduledTimeMillis(), currentTimeMillis);
        scheduledTask.getListener().triggered(event);
        this.scheduledTasks.update(scheduledTask.getTransformId(), task -> {
            if (!task.equals(scheduledTask)) {
                logger.debug(() -> Strings.format((String)"[%s] task object got modified while processing. Expected: %s, was: %s", (Object[])new Object[]{scheduledTask.getTransformId(), scheduledTask, task}));
            }
            return new TransformScheduledTask(task.getTransformId(), this.getFrequency(task.getFrequency()), currentTimeMillis, task.getFailureCount(), task.getListener());
        });
        return true;
    }

    public void stop() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel();
            this.scheduledFuture = null;
        }
    }

    public void registerTransform(TransformTaskParams transformTaskParams, Listener listener) {
        String transformId = transformTaskParams.getId();
        logger.trace(() -> Strings.format((String)"[%s] register the transform", (Object[])new Object[]{transformId}));
        long currentTimeMillis = this.clock.millis();
        TransformScheduledTask transformScheduledTask = new TransformScheduledTask(transformId, this.getFrequency(transformTaskParams.getFrequency()), null, 0, currentTimeMillis, listener);
        this.scheduledTasks.add(transformScheduledTask);
        this.processScheduledTasks();
    }

    public void handleTransformFailureCountChanged(String transformId, int failureCount) {
        logger.trace(() -> Strings.format((String)"[%s] handle transform failure count change to %d", (Object[])new Object[]{transformId, failureCount}));
        this.scheduledTasks.update(transformId, task -> new TransformScheduledTask(task.getTransformId(), this.getFrequency(task.getFrequency()), task.getLastTriggeredTimeMillis(), failureCount, task.getListener()));
    }

    public void scheduleNow(String transformId) {
        logger.trace(() -> Strings.format((String)"[%s] schedule_now transform", (Object[])new Object[]{transformId}));
        long currentTimeMillis = this.clock.millis();
        this.scheduledTasks.update(transformId, task -> new TransformScheduledTask(task.getTransformId(), this.getFrequency(task.getFrequency()), task.getLastTriggeredTimeMillis(), task.getFailureCount(), currentTimeMillis, task.getListener()));
        this.processScheduledTasks();
    }

    public void deregisterTransform(String transformId) {
        Objects.requireNonNull(transformId);
        logger.trace(() -> Strings.format((String)"[%s] de-register the transform", (Object[])new Object[]{transformId}));
        this.scheduledTasks.remove(transformId);
    }

    List<TransformScheduledTask> getTransformScheduledTasks() {
        return this.scheduledTasks.listScheduledTasks();
    }

    private TimeValue getFrequency(TimeValue frequency) {
        if (frequency == null) {
            frequency = Transform.DEFAULT_TRANSFORM_FREQUENCY;
        }
        return frequency.compareTo(this.minFrequency) >= 0 ? frequency : this.minFrequency;
    }

    public record Event(String transformId, long scheduledTime, long triggeredTime) {
        @Override
        public String toString() {
            return "Event[" + "transformId=" + this.transformId + ",scheduledTime=" + Instant.ofEpochMilli(this.scheduledTime) + ",triggeredTime=" + Instant.ofEpochMilli(this.triggeredTime) + "]";
        }
    }

    public static interface Listener {
        public void triggered(Event var1);
    }
}

