/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.IndexingOperationListener;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryController
implements IndexingOperationListener,
Closeable {
    private static final Logger logger = LogManager.getLogger(IndexingMemoryController.class);
    public static final Setting<ByteSizeValue> INDEX_BUFFER_SIZE_SETTING = Setting.memorySizeSetting("indices.memory.index_buffer_size", "10%", Setting.Property.NodeScope);
    public static final Setting<ByteSizeValue> MIN_INDEX_BUFFER_SIZE_SETTING = Setting.byteSizeSetting("indices.memory.min_index_buffer_size", new ByteSizeValue(48L, ByteSizeUnit.MB), ByteSizeValue.ZERO, ByteSizeValue.ofBytes(Long.MAX_VALUE), Setting.Property.NodeScope);
    public static final Setting<ByteSizeValue> MAX_INDEX_BUFFER_SIZE_SETTING = Setting.byteSizeSetting("indices.memory.max_index_buffer_size", ByteSizeValue.MINUS_ONE, ByteSizeValue.MINUS_ONE, ByteSizeValue.ofBytes(Long.MAX_VALUE), Setting.Property.NodeScope);
    public static final Setting<TimeValue> SHARD_INACTIVE_TIME_SETTING = Setting.positiveTimeSetting("indices.memory.shard_inactive_time", TimeValue.timeValueMinutes((long)5L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> SHARD_MEMORY_INTERVAL_TIME_SETTING = Setting.positiveTimeSetting("indices.memory.interval", TimeValue.timeValueSeconds((long)5L), Setting.Property.NodeScope);
    private final ThreadPool threadPool;
    private final Iterable<IndexShard> indexShards;
    private final ByteSizeValue indexingBuffer;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private final Set<IndexShard> throttled = new HashSet<IndexShard>();
    private final Scheduler.Cancellable scheduler;
    private static final EnumSet<IndexShardState> CAN_WRITE_INDEX_BUFFER_STATES = EnumSet.of(IndexShardState.RECOVERING, IndexShardState.POST_RECOVERY, IndexShardState.STARTED);
    private final ShardsIndicesStatusChecker statusChecker;
    private final Set<IndexShard> pendingWriteIndexingBufferSet = ConcurrentCollections.newConcurrentSet();
    private final Deque<IndexShard> pendingWriteIndexingBufferQueue = new ConcurrentLinkedDeque<IndexShard>();

    IndexingMemoryController(Settings settings, ThreadPool threadPool, Iterable<IndexShard> indexServices) {
        this.indexShards = indexServices;
        ByteSizeValue indexingBuffer = INDEX_BUFFER_SIZE_SETTING.get(settings);
        String indexingBufferSetting = settings.get(INDEX_BUFFER_SIZE_SETTING.getKey());
        if (indexingBufferSetting == null || indexingBufferSetting.endsWith("%")) {
            ByteSizeValue minIndexingBuffer = MIN_INDEX_BUFFER_SIZE_SETTING.get(settings);
            ByteSizeValue maxIndexingBuffer = MAX_INDEX_BUFFER_SIZE_SETTING.get(settings);
            if (indexingBuffer.getBytes() < minIndexingBuffer.getBytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer.getBytes() != -1L && indexingBuffer.getBytes() > maxIndexingBuffer.getBytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        }
        this.indexingBuffer = indexingBuffer;
        this.inactiveTime = SHARD_INACTIVE_TIME_SETTING.get(settings);
        this.interval = SHARD_MEMORY_INTERVAL_TIME_SETTING.get(settings);
        this.statusChecker = new ShardsIndicesStatusChecker();
        logger.debug("using indexing buffer size [{}] with {} [{}], {} [{}]", (Object)this.indexingBuffer, (Object)SHARD_INACTIVE_TIME_SETTING.getKey(), (Object)this.inactiveTime, (Object)SHARD_MEMORY_INTERVAL_TIME_SETTING.getKey(), (Object)this.interval);
        this.scheduler = this.scheduleTask(threadPool);
        this.threadPool = threadPool;
    }

    protected Scheduler.Cancellable scheduleTask(ThreadPool threadPool) {
        return threadPool.scheduleWithFixedDelay(this.statusChecker, this.interval, EsExecutors.DIRECT_EXECUTOR_SERVICE);
    }

    @Override
    public void close() {
        this.scheduler.cancel();
    }

    ByteSizeValue indexingBufferSize() {
        return this.indexingBuffer;
    }

    protected List<IndexShard> availableShards() {
        ArrayList<IndexShard> availableShards = new ArrayList<IndexShard>();
        for (IndexShard shard : this.indexShards) {
            if (!CAN_WRITE_INDEX_BUFFER_STATES.contains((Object)shard.state())) continue;
            availableShards.add(shard);
        }
        return availableShards;
    }

    protected long getIndexBufferRAMBytesUsed(IndexShard shard) {
        return shard.getIndexBufferRAMBytesUsed();
    }

    protected long getShardWritingBytes(IndexShard shard) {
        return shard.getWritingBytes();
    }

    protected void enqueueWriteIndexingBuffer(IndexShard shard) {
        if (this.pendingWriteIndexingBufferSet.add(shard)) {
            this.pendingWriteIndexingBufferQueue.addLast(shard);
        }
    }

    private boolean writePendingIndexingBuffers() {
        boolean wrotePendingIndexingBuffer = false;
        IndexShard shard = this.pendingWriteIndexingBufferQueue.pollFirst();
        while (shard != null) {
            this.pendingWriteIndexingBufferSet.remove(shard);
            shard.writeIndexingBuffer();
            wrotePendingIndexingBuffer = true;
            shard = this.pendingWriteIndexingBufferQueue.pollFirst();
        }
        return wrotePendingIndexingBuffer;
    }

    private void writePendingIndexingBuffersAsync() {
        IndexShard shard = this.pendingWriteIndexingBufferQueue.pollFirst();
        while (shard != null) {
            IndexShard finalShard = shard;
            this.threadPool.executor("refresh").execute(() -> {
                this.pendingWriteIndexingBufferSet.remove(finalShard);
                finalShard.writeIndexingBuffer();
            });
            shard = this.pendingWriteIndexingBufferQueue.pollFirst();
        }
    }

    void forceCheck() {
        this.statusChecker.run();
    }

    protected void activateThrottling(IndexShard shard) {
        shard.activateThrottling();
    }

    protected void deactivateThrottling(IndexShard shard) {
        shard.deactivateThrottling();
    }

    @Override
    public void postIndex(ShardId shardId, Engine.Index index, Engine.IndexResult result) {
        this.postOperation(shardId, index, result);
    }

    @Override
    public void postDelete(ShardId shardId, Engine.Delete delete, Engine.DeleteResult result) {
        this.postOperation(shardId, delete, result);
    }

    private void postOperation(ShardId shardId, Engine.Operation operation, Engine.Result result) {
        this.recordOperationBytes(operation, result);
        while (this.writePendingIndexingBuffers() && this.statusChecker.tryRun()) {
        }
    }

    private void recordOperationBytes(Engine.Operation operation, Engine.Result result) {
        if (result.getResultType() == Engine.Result.Type.SUCCESS) {
            this.statusChecker.bytesWritten(operation.estimatedSizeInBytes());
        }
    }

    protected void checkIdle(IndexShard shard, long inactiveTimeNS) {
        try {
            shard.flushOnIdle(inactiveTimeNS);
        }
        catch (AlreadyClosedException e) {
            logger.trace(() -> "ignore exception while checking if shard " + shard.shardId() + " is inactive", (Throwable)e);
        }
    }

    final class ShardsIndicesStatusChecker
    implements Runnable {
        final AtomicLong bytesWrittenSinceCheck = new AtomicLong();
        final ReentrantLock runLock = new ReentrantLock();
        private ShardId lastShardId = null;

        ShardsIndicesStatusChecker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void bytesWritten(int bytes) {
            long totalBytes = this.bytesWrittenSinceCheck.addAndGet(bytes);
            assert (totalBytes >= 0L);
            while (totalBytes > IndexingMemoryController.this.indexingBuffer.getBytes() / 128L && this.runLock.tryLock()) {
                try {
                    totalBytes = this.bytesWrittenSinceCheck.get();
                    if (totalBytes > IndexingMemoryController.this.indexingBuffer.getBytes() / 128L) {
                        this.bytesWrittenSinceCheck.addAndGet(-totalBytes);
                        this.runUnlocked();
                    }
                }
                finally {
                    this.runLock.unlock();
                }
                totalBytes = this.bytesWrittenSinceCheck.get();
            }
        }

        public boolean tryRun() {
            if (this.runLock.tryLock()) {
                try {
                    this.runUnlocked();
                }
                finally {
                    this.runLock.unlock();
                }
                return true;
            }
            return false;
        }

        @Override
        public void run() {
            IndexingMemoryController.this.writePendingIndexingBuffersAsync();
            this.runLock.lock();
            try {
                this.runUnlocked();
            }
            finally {
                this.runLock.unlock();
            }
        }

        private void runUnlocked() {
            boolean doThrottle;
            assert (this.runLock.isHeldByCurrentThread()) : "ShardsIndicesStatusChecker#runUnlocked must always run under the run lock";
            long totalBytesUsed = 0L;
            long totalBytesWriting = 0L;
            for (IndexShard shard : IndexingMemoryController.this.availableShards()) {
                IndexingMemoryController.this.checkIdle(shard, IndexingMemoryController.this.inactiveTime.nanos());
                long shardWritingBytes = IndexingMemoryController.this.getShardWritingBytes(shard);
                long shardBytesUsed = IndexingMemoryController.this.getIndexBufferRAMBytesUsed(shard);
                totalBytesWriting += shardWritingBytes;
                if ((shardBytesUsed -= shardWritingBytes) < 0L) continue;
                totalBytesUsed += shardBytesUsed;
            }
            if (logger.isTraceEnabled()) {
                logger.trace("total indexing heap bytes used [{}] vs {} [{}], currently writing bytes [{}]", (Object)ByteSizeValue.ofBytes(totalBytesUsed), (Object)INDEX_BUFFER_SIZE_SETTING.getKey(), (Object)IndexingMemoryController.this.indexingBuffer, (Object)ByteSizeValue.ofBytes(totalBytesWriting));
            }
            boolean bl = doThrottle = (double)(totalBytesWriting + totalBytesUsed) > 1.5 * (double)IndexingMemoryController.this.indexingBuffer.getBytes();
            if (totalBytesUsed > IndexingMemoryController.this.indexingBuffer.getBytes()) {
                ArrayList<ShardAndBytesUsed> queue = new ArrayList<ShardAndBytesUsed>();
                for (IndexShard indexShard : IndexingMemoryController.this.availableShards()) {
                    long shardWritingBytes = IndexingMemoryController.this.getShardWritingBytes(indexShard);
                    long shardBytesUsed = IndexingMemoryController.this.getIndexBufferRAMBytesUsed(indexShard);
                    if ((shardBytesUsed -= shardWritingBytes) < 0L || shardBytesUsed <= 0L) continue;
                    if (logger.isTraceEnabled()) {
                        if (shardWritingBytes != 0L) {
                            logger.trace("shard [{}] is using [{}] heap, writing [{}] heap", (Object)indexShard.shardId(), (Object)shardBytesUsed, (Object)shardWritingBytes);
                        } else {
                            logger.trace("shard [{}] is using [{}] heap, not writing any bytes", (Object)indexShard.shardId(), (Object)shardBytesUsed);
                        }
                    }
                    queue.add(new ShardAndBytesUsed(shardBytesUsed, indexShard));
                }
                logger.debug("now write some indexing buffers: total indexing heap bytes used [{}] vs {} [{}], currently writing bytes [{}], [{}] shards with non-zero indexing buffer", (Object)ByteSizeValue.ofBytes(totalBytesUsed), (Object)INDEX_BUFFER_SIZE_SETTING.getKey(), (Object)IndexingMemoryController.this.indexingBuffer, (Object)ByteSizeValue.ofBytes(totalBytesWriting), (Object)queue.size());
                queue.sort(Comparator.comparing(shardAndBytes -> shardAndBytes.shard.shardId()));
                if (this.lastShardId != null) {
                    int nextShardIdIndex = 0;
                    for (ShardAndBytesUsed shardAndBytes2 : queue) {
                        if (shardAndBytes2.shard.shardId().compareTo(this.lastShardId) > 0) break;
                        ++nextShardIdIndex;
                    }
                    Collections.rotate(queue, -nextShardIdIndex);
                }
                for (ShardAndBytesUsed shardAndBytesUsed : queue) {
                    logger.debug("write indexing buffer to disk for shard [{}] to free up its [{}] indexing buffer", (Object)shardAndBytesUsed.shard.shardId(), (Object)ByteSizeValue.ofBytes(shardAndBytesUsed.bytesUsed));
                    IndexingMemoryController.this.enqueueWriteIndexingBuffer(shardAndBytesUsed.shard);
                    totalBytesUsed -= shardAndBytesUsed.bytesUsed;
                    this.lastShardId = shardAndBytesUsed.shard.shardId();
                    if (doThrottle && !IndexingMemoryController.this.throttled.contains(shardAndBytesUsed.shard)) {
                        logger.debug("now throttling indexing for shard [{}]: segment writing can't keep up", (Object)shardAndBytesUsed.shard.shardId());
                        IndexingMemoryController.this.throttled.add(shardAndBytesUsed.shard);
                        IndexingMemoryController.this.activateThrottling(shardAndBytesUsed.shard);
                    }
                    if (totalBytesUsed > IndexingMemoryController.this.indexingBuffer.getBytes()) continue;
                    break;
                }
            }
            if (!doThrottle) {
                for (IndexShard shard : IndexingMemoryController.this.throttled) {
                    logger.info("stop throttling indexing for shard [{}]", (Object)shard.shardId());
                    IndexingMemoryController.this.deactivateThrottling(shard);
                }
                IndexingMemoryController.this.throttled.clear();
            }
        }
    }

    private static final class ShardAndBytesUsed {
        final long bytesUsed;
        final IndexShard shard;

        ShardAndBytesUsed(long bytesUsed, IndexShard shard) {
            this.bytesUsed = bytesUsed;
            this.shard = shard;
        }
    }
}

