/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.scheduler;

import java.time.Clock;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.FutureUtils;

public class SchedulerEngine {
    private final Map<String, ActiveSchedule> schedules = ConcurrentCollections.newConcurrentMap();
    private final Clock clock;
    private final ScheduledExecutorService scheduler;
    private final Logger logger;
    private final List<Listener> listeners = new CopyOnWriteArrayList<Listener>();

    public SchedulerEngine(Settings settings, Clock clock) {
        this(settings, clock, LogManager.getLogger(SchedulerEngine.class));
    }

    SchedulerEngine(Settings settings, Clock clock, Logger logger) {
        this.clock = Objects.requireNonNull(clock, "clock");
        this.scheduler = Executors.newScheduledThreadPool(1, EsExecutors.daemonThreadFactory(Objects.requireNonNull(settings, "settings"), "trigger_engine_scheduler"));
        this.logger = Objects.requireNonNull(logger, "logger");
    }

    public void register(Listener listener) {
        this.listeners.add(listener);
    }

    public void unregister(Listener listener) {
        this.listeners.remove(listener);
    }

    public void start(Collection<Job> jobs) {
        jobs.forEach(this::add);
    }

    public void stop() {
        this.scheduler.shutdownNow();
        try {
            boolean terminated = this.scheduler.awaitTermination(5L, TimeUnit.SECONDS);
            if (!terminated) {
                this.logger.warn("scheduler engine was not terminated after waiting 5s");
            }
        }
        catch (InterruptedException e) {
            this.logger.warn("interrupted while waiting for scheduler engine termination");
            Thread.currentThread().interrupt();
        }
    }

    public Set<String> scheduledJobIds() {
        return Collections.unmodifiableSet(new HashSet<String>(this.schedules.keySet()));
    }

    public void add(Job job) {
        ActiveSchedule schedule = new ActiveSchedule(job.getId(), job.getSchedule(), this.clock.millis());
        this.schedules.compute(schedule.name, (name, previousSchedule) -> {
            if (previousSchedule != null) {
                previousSchedule.cancel();
            }
            this.logger.debug(() -> "added job [" + job.getId() + "]");
            return schedule;
        });
    }

    public boolean remove(String jobId) {
        ActiveSchedule removedSchedule = this.schedules.remove(jobId);
        if (removedSchedule != null) {
            this.logger.debug(() -> "removed job [" + jobId + "]");
            removedSchedule.cancel();
        }
        return removedSchedule != null;
    }

    public int jobCount() {
        return this.schedules.size();
    }

    protected void notifyListeners(String name, long triggeredTime, long scheduledTime) {
        Event event = new Event(name, triggeredTime, scheduledTime);
        for (Listener listener : this.listeners) {
            try {
                listener.triggered(event);
            }
            catch (Exception e) {
                this.logger.warn(() -> "listener failed while handling triggered event [" + name + "]", (Throwable)e);
            }
        }
    }

    ActiveSchedule getSchedule(String jobId) {
        return this.schedules.get(jobId);
    }

    class ActiveSchedule
    implements Runnable {
        private final String name;
        private final Schedule schedule;
        private final long startTime;
        private ScheduledFuture<?> future;
        private long scheduledTime = -1L;

        ActiveSchedule(String name, Schedule schedule, long startTime) {
            this.name = name;
            this.schedule = schedule;
            this.startTime = startTime;
            this.scheduleNextRun(startTime);
        }

        @Override
        public void run() {
            long triggeredTime = SchedulerEngine.this.clock.millis();
            try {
                SchedulerEngine.this.logger.debug(() -> "job [" + this.name + "] triggered with triggeredTime=[" + triggeredTime + "]");
                SchedulerEngine.this.notifyListeners(this.name, triggeredTime, this.scheduledTime);
            }
            catch (Throwable t) {
                ExceptionsHelper.maybeDieOnAnotherThread(t);
                throw t;
            }
            this.scheduleNextRun(triggeredTime);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void scheduleNextRun(long triggeredTime) {
            block7: {
                this.scheduledTime = this.computeNextScheduledTime(triggeredTime);
                if (this.scheduledTime != -1L) {
                    long delay = Math.max(0L, this.scheduledTime - SchedulerEngine.this.clock.millis());
                    try {
                        ActiveSchedule activeSchedule = this;
                        synchronized (activeSchedule) {
                            if (this.future == null || !this.future.isCancelled()) {
                                SchedulerEngine.this.logger.debug(() -> "schedule job [" + this.name + "] with scheduleTime=[" + this.scheduledTime + "] and delay=[" + delay + "]");
                                this.future = SchedulerEngine.this.scheduler.schedule(this, delay, TimeUnit.MILLISECONDS);
                            }
                        }
                    }
                    catch (RejectedExecutionException e) {
                        if (SchedulerEngine.this.scheduler.isShutdown()) break block7;
                        throw e;
                    }
                }
            }
        }

        long getScheduledTime() {
            return this.scheduledTime;
        }

        long computeNextScheduledTime(long triggeredTime) {
            long scheduleAfterTime = this.scheduledTime != -1L && triggeredTime < this.scheduledTime ? this.scheduledTime : triggeredTime;
            return this.schedule.nextScheduledTimeAfter(this.startTime, scheduleAfterTime);
        }

        public synchronized void cancel() {
            FutureUtils.cancel(this.future);
        }
    }

    public static class Job {
        private final String id;
        private final Schedule schedule;

        public Job(String id, Schedule schedule) {
            this.id = id;
            this.schedule = schedule;
        }

        public String getId() {
            return this.id;
        }

        public Schedule getSchedule() {
            return this.schedule;
        }
    }

    public static interface Schedule {
        public long nextScheduledTimeAfter(long var1, long var3);
    }

    public static class Event {
        private final String jobName;
        private final long triggeredTime;
        private final long scheduledTime;

        public Event(String jobName, long triggeredTime, long scheduledTime) {
            this.jobName = jobName;
            this.triggeredTime = triggeredTime;
            this.scheduledTime = scheduledTime;
        }

        public String getJobName() {
            return this.jobName;
        }

        public long getTriggeredTime() {
            return this.triggeredTime;
        }

        public long getScheduledTime() {
            return this.scheduledTime;
        }

        public String toString() {
            return "Event[jobName=" + this.jobName + ",triggeredTime=" + this.triggeredTime + ",scheduledTime=" + this.scheduledTime + "]";
        }
    }

    public static interface Listener {
        public void triggered(Event var1);
    }
}

