/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.inference.external.http.sender;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.ContextPreservingActionListener;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.inference.InferenceServiceResults;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.inference.common.AdjustableCapacityBlockingQueue;
import org.elasticsearch.xpack.inference.external.http.RequestExecutor;
import org.elasticsearch.xpack.inference.external.http.sender.InferenceInputs;
import org.elasticsearch.xpack.inference.external.http.sender.NoopTask;
import org.elasticsearch.xpack.inference.external.http.sender.RejectableTask;
import org.elasticsearch.xpack.inference.external.http.sender.RequestExecutorServiceSettings;
import org.elasticsearch.xpack.inference.external.http.sender.RequestManager;
import org.elasticsearch.xpack.inference.external.http.sender.RequestTask;
import org.elasticsearch.xpack.inference.external.http.sender.SingleRequestManager;

class RequestExecutorService
implements RequestExecutor {
    private static final AdjustableCapacityBlockingQueue.QueueCreator<RejectableTask> QUEUE_CREATOR = new AdjustableCapacityBlockingQueue.QueueCreator<RejectableTask>(){

        @Override
        public BlockingQueue<RejectableTask> create(int capacity) {
            BlockingQueue<RejectableTask> queue = capacity <= 0 ? this.create() : new LinkedBlockingQueue<RejectableTask>(capacity);
            return queue;
        }

        @Override
        public BlockingQueue<RejectableTask> create() {
            return new LinkedBlockingQueue<RejectableTask>();
        }
    };
    private static final Logger logger = LogManager.getLogger(RequestExecutorService.class);
    private final String serviceName;
    private final AdjustableCapacityBlockingQueue<RejectableTask> queue;
    private final AtomicBoolean running = new AtomicBoolean(true);
    private final CountDownLatch terminationLatch = new CountDownLatch(1);
    private final HttpClientContext httpContext;
    private final ThreadPool threadPool;
    private final CountDownLatch startupLatch;
    private final BlockingQueue<Runnable> controlQueue = new LinkedBlockingQueue<Runnable>();
    private final SingleRequestManager requestManager;

    RequestExecutorService(String serviceName, ThreadPool threadPool, @Nullable CountDownLatch startupLatch, RequestExecutorServiceSettings settings, SingleRequestManager requestManager) {
        this(serviceName, threadPool, QUEUE_CREATOR, startupLatch, settings, requestManager);
    }

    RequestExecutorService(String serviceName, ThreadPool threadPool, AdjustableCapacityBlockingQueue.QueueCreator<RejectableTask> createQueue, @Nullable CountDownLatch startupLatch, RequestExecutorServiceSettings settings, SingleRequestManager requestManager) {
        this.serviceName = Objects.requireNonNull(serviceName);
        this.threadPool = Objects.requireNonNull(threadPool);
        this.httpContext = HttpClientContext.create();
        this.queue = new AdjustableCapacityBlockingQueue<RejectableTask>(createQueue, settings.getQueueCapacity());
        this.startupLatch = startupLatch;
        this.requestManager = Objects.requireNonNull(requestManager);
        Objects.requireNonNull(settings);
        settings.registerQueueCapacityCallback(this::onCapacityChange);
    }

    private void onCapacityChange(int capacity) {
        logger.debug(() -> Strings.format((String)"Setting queue capacity to [%s]", (Object[])new Object[]{capacity}));
        boolean enqueuedCapacityCommand = this.controlQueue.offer(() -> this.updateCapacity(capacity));
        if (!enqueuedCapacityCommand) {
            logger.warn("Failed to change request batching service queue capacity. Control queue was full, please try again later.");
        } else {
            this.queue.offer(new NoopTask());
        }
    }

    private void updateCapacity(int newCapacity) {
        try {
            this.queue.setCapacity(newCapacity);
        }
        catch (Exception e) {
            logger.warn(Strings.format((String)"Failed to set the capacity of the task queue to [%s] for request batching service [%s]", (Object[])new Object[]{newCapacity, this.serviceName}), (Throwable)e);
        }
    }

    @Override
    public void start() {
        try {
            this.signalStartInitiated();
            while (this.running.get()) {
                this.handleTasks();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        finally {
            this.running.set(false);
            this.notifyRequestsOfShutdown();
            this.terminationLatch.countDown();
        }
    }

    private void signalStartInitiated() {
        if (this.startupLatch != null) {
            this.startupLatch.countDown();
        }
    }

    private void handleTasks() throws InterruptedException {
        try {
            RejectableTask task = this.queue.take();
            Runnable command = (Runnable)this.controlQueue.poll();
            if (command != null) {
                command.run();
            }
            if (!this.running.get()) {
                logger.debug(() -> Strings.format((String)"Http executor service [%s] exiting", (Object[])new Object[]{this.serviceName}));
                this.rejectTaskBecauseOfShutdown(task);
            } else {
                this.executeTask(task);
            }
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch (Exception e) {
            logger.warn(Strings.format((String)"Http executor service [%s] failed while retrieving task for execution", (Object[])new Object[]{this.serviceName}), (Throwable)e);
        }
    }

    private void executeTask(RejectableTask task) {
        try {
            this.requestManager.execute(task, this.httpContext);
        }
        catch (Exception e) {
            logger.warn(Strings.format((String)"Http executor service [%s] failed to execute request [%s]", (Object[])new Object[]{this.serviceName, task}), (Throwable)e);
        }
    }

    private synchronized void notifyRequestsOfShutdown() {
        assert (this.isShutdown()) : "Requests should only be notified if the executor is shutting down";
        try {
            ArrayList<RejectableTask> notExecuted = new ArrayList<RejectableTask>();
            this.queue.drainTo(notExecuted);
            this.rejectTasks(notExecuted, this::rejectTaskBecauseOfShutdown);
        }
        catch (Exception e) {
            logger.warn(Strings.format((String)"Failed to notify tasks of queuing service [%s] shutdown", (Object[])new Object[]{this.serviceName}));
        }
    }

    private void rejectTaskBecauseOfShutdown(RejectableTask task) {
        try {
            task.onRejection((Exception)new EsRejectedExecutionException(Strings.format((String)"Failed to send request, queue service [%s] has shutdown prior to executing request", (Object[])new Object[]{this.serviceName}), true));
        }
        catch (Exception e) {
            logger.warn(Strings.format((String)"Failed to notify request [%s] for service [%s] of rejection after queuing service shutdown", (Object[])new Object[]{task, this.serviceName}));
        }
    }

    private void rejectTasks(List<RejectableTask> tasks, Consumer<RejectableTask> rejectionFunction) {
        for (RejectableTask task : tasks) {
            rejectionFunction.accept(task);
        }
    }

    public int queueSize() {
        return this.queue.size();
    }

    @Override
    public void shutdown() {
        if (this.running.compareAndSet(true, false)) {
            this.queue.offer(new NoopTask());
        }
    }

    @Override
    public boolean isShutdown() {
        return !this.running.get();
    }

    @Override
    public boolean isTerminated() {
        return this.terminationLatch.getCount() == 0L;
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return this.terminationLatch.await(timeout, unit);
    }

    @Override
    public void execute(RequestManager requestCreator, InferenceInputs inferenceInputs, @Nullable TimeValue timeout, ActionListener<InferenceServiceResults> listener) {
        RequestTask task = new RequestTask(requestCreator, inferenceInputs, timeout, this.threadPool, (ActionListener<InferenceServiceResults>)ContextPreservingActionListener.wrapPreservingContext(listener, (ThreadContext)this.threadPool.getThreadContext()));
        this.completeExecution(task);
    }

    private void completeExecution(RequestTask task) {
        if (this.isShutdown()) {
            EsRejectedExecutionException rejected = new EsRejectedExecutionException(Strings.format((String)"Failed to enqueue task because the http executor service [%s] has already shutdown", (Object[])new Object[]{this.serviceName}), true);
            task.onRejection((Exception)rejected);
            return;
        }
        boolean added = this.queue.offer(task);
        if (!added) {
            EsRejectedExecutionException rejected = new EsRejectedExecutionException(Strings.format((String)"Failed to execute task because the http executor service [%s] queue is full", (Object[])new Object[]{this.serviceName}), false);
            task.onRejection((Exception)rejected);
        } else if (this.isShutdown()) {
            this.notifyRequestsOfShutdown();
        }
    }

    int remainingQueueCapacity() {
        return this.queue.remainingCapacity();
    }
}

