/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.inference.common;

import java.time.Clock;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.elasticsearch.common.Strings;

public class RateLimiter {
    private double tokensPerMicros;
    private double accumulatedTokensLimit;
    private double accumulatedTokens;
    private Instant nextTokenAvailability;
    private final Sleeper sleeper;
    private final Clock clock;

    public RateLimiter(double accumulatedTokensLimit, double tokensPerTimeUnit, TimeUnit unit) {
        this(accumulatedTokensLimit, tokensPerTimeUnit, unit, new TimeUnitSleeper(), Clock.systemUTC());
    }

    RateLimiter(double accumulatedTokensLimit, double tokensPerTimeUnit, TimeUnit unit, Sleeper sleeper, Clock clock) {
        this.sleeper = Objects.requireNonNull(sleeper);
        this.clock = Objects.requireNonNull(clock);
        this.nextTokenAvailability = Instant.MIN;
        this.setRate(accumulatedTokensLimit, tokensPerTimeUnit, unit);
    }

    public final synchronized void setRate(double newAccumulatedTokensLimit, double newTokensPerTimeUnit, TimeUnit newUnit) {
        Objects.requireNonNull(newUnit);
        if (newAccumulatedTokensLimit < 0.0) {
            throw new IllegalArgumentException("Accumulated tokens limit must be greater than or equal to 0");
        }
        if (Double.isInfinite(newAccumulatedTokensLimit)) {
            throw new IllegalArgumentException(Strings.format((String)"Accumulated tokens limit must be less than or equal to %s", (Object[])new Object[]{Double.MAX_VALUE}));
        }
        if (newTokensPerTimeUnit <= 0.0) {
            throw new IllegalArgumentException("Tokens per time unit must be greater than 0");
        }
        if (newTokensPerTimeUnit == Double.POSITIVE_INFINITY) {
            throw new IllegalArgumentException(Strings.format((String)"Tokens per time unit must be less than or equal to %s", (Object[])new Object[]{Double.MAX_VALUE}));
        }
        this.accumulatedTokens = Math.min(this.accumulatedTokens, newAccumulatedTokensLimit);
        this.accumulatedTokensLimit = newAccumulatedTokensLimit;
        long unitsInMicros = newUnit.toMicros(1L);
        this.tokensPerMicros = newTokensPerTimeUnit / (double)unitsInMicros;
        assert (!Double.isInfinite(this.tokensPerMicros)) : "Tokens per microsecond should not be infinity";
        this.accumulateTokens();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void acquire(int tokens) throws InterruptedException {
        double microsToWait;
        if (tokens <= 0) {
            throw new IllegalArgumentException("Requested tokens must be positive");
        }
        RateLimiter rateLimiter = this;
        synchronized (rateLimiter) {
            this.accumulateTokens();
            double accumulatedTokensToUse = Math.min((double)tokens, this.accumulatedTokens);
            double additionalTokensRequired = (double)tokens - accumulatedTokensToUse;
            microsToWait = additionalTokensRequired / this.tokensPerMicros;
            this.accumulatedTokens -= accumulatedTokensToUse;
            this.nextTokenAvailability = this.nextTokenAvailability.plus((long)microsToWait, ChronoUnit.MICROS);
        }
        this.sleeper.sleep((long)microsToWait);
    }

    private void accumulateTokens() {
        Instant now = Instant.now(this.clock);
        if (now.isAfter(this.nextTokenAvailability)) {
            long elapsedTimeMicros = RateLimiter.microsBetweenExact(this.nextTokenAvailability, now);
            double newTokens = this.tokensPerMicros * (double)elapsedTimeMicros;
            this.accumulatedTokens = Math.min(this.accumulatedTokensLimit, this.accumulatedTokens + newTokens);
            this.nextTokenAvailability = now;
        }
    }

    private static long microsBetweenExact(Instant start, Instant end) {
        try {
            return ChronoUnit.MICROS.between(start, end);
        }
        catch (ArithmeticException e) {
            if (end.isAfter(start)) {
                return Long.MAX_VALUE;
            }
            return 0L;
        }
    }

    Instant getNextTokenAvailability() {
        return this.nextTokenAvailability;
    }

    static final class TimeUnitSleeper
    implements Sleeper {
        TimeUnitSleeper() {
        }

        @Override
        public void sleep(long microsecondsToSleep) throws InterruptedException {
            TimeUnit.MICROSECONDS.sleep(microsecondsToSleep);
        }
    }

    public static interface Sleeper {
        public void sleep(long var1) throws InterruptedException;
    }
}

