/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.searchablesnapshots.store.input;

import java.io.Closeable;
import java.io.EOFException;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.elasticsearch.blobcache.BlobCacheUtils;
import org.elasticsearch.blobcache.common.BlobCacheBufferedIndexInput;
import org.elasticsearch.common.CheckedSupplier;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Streams;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.xpack.searchablesnapshots.store.IndexInputStats;
import org.elasticsearch.xpack.searchablesnapshots.store.input.MetadataCachingIndexInput;

public final class DirectBlobContainerIndexInput
extends BlobCacheBufferedIndexInput {
    private static final Logger logger = LogManager.getLogger(DirectBlobContainerIndexInput.class);
    private long position;
    @Nullable
    private StreamForSequentialReads streamForSequentialReads;
    private long sequentialReadSize;
    private static final long NO_SEQUENTIAL_READ_OPTIMIZATION = 0L;
    private final BlobContainer blobContainer;
    private final BlobStoreIndexShardSnapshot.FileInfo fileInfo;
    private final IndexInputStats stats;
    private final long offset;
    private final long length;
    private volatile boolean isClone;
    private AtomicBoolean closed;

    public DirectBlobContainerIndexInput(String name, BlobContainer blobContainer, BlobStoreIndexShardSnapshot.FileInfo fileInfo, IOContext context, IndexInputStats stats, long sequentialReadSize) {
        this(name, blobContainer, fileInfo, DirectBlobContainerIndexInput.bufferSize((IOContext)context), stats, 0L, 0L, fileInfo.length(), sequentialReadSize);
        stats.incrementOpenCount();
    }

    private DirectBlobContainerIndexInput(String name, BlobContainer blobContainer, BlobStoreIndexShardSnapshot.FileInfo fileInfo, int bufferSize, IndexInputStats stats, long position, long offset, long length, long sequentialReadSize) {
        super(name, bufferSize);
        this.position = position;
        assert (sequentialReadSize >= 0L);
        this.sequentialReadSize = sequentialReadSize;
        this.blobContainer = Objects.requireNonNull(blobContainer);
        this.fileInfo = Objects.requireNonNull(fileInfo);
        assert (!fileInfo.metadata().hashEqualsContents()) : "this method should only be used with blobs that are NOT stored in metadata's hash field (fileInfo: " + fileInfo + ")";
        this.stats = Objects.requireNonNull(stats);
        this.offset = offset;
        this.length = length;
        this.closed = new AtomicBoolean(false);
        this.isClone = false;
    }

    protected void readInternal(ByteBuffer b) throws IOException {
        assert (MetadataCachingIndexInput.assertCurrentThreadIsNotCacheFetchAsync());
        int bytesToRead = b.remaining();
        if (MetadataCachingIndexInput.maybeReadChecksumFromFileInfo(this.fileInfo, this.getFilePointer() + this.offset, this.isClone, b)) {
            logger.trace("read footer of file [{}], bypassing all caches", (Object)this.fileInfo.physicalName());
        } else {
            this.doReadInternal(b);
        }
        assert ((long)b.remaining() == 0L) : b.remaining();
        this.stats.addLuceneBytesRead(bytesToRead);
    }

    private void doReadInternal(ByteBuffer b) throws IOException {
        if (this.closed.get()) {
            throw new IOException(this + " is closed");
        }
        if (this.fileInfo.numberOfParts() == 1) {
            this.readInternalBytes(0, this.position, b, b.remaining());
        } else {
            while (b.hasRemaining()) {
                int currentPart = Math.toIntExact(this.position / this.fileInfo.partSize().getBytes());
                long remainingBytesInPart = currentPart < this.fileInfo.numberOfParts() - 1 ? (long)(currentPart + 1) * this.fileInfo.partSize().getBytes() - this.position : (long)BlobCacheUtils.toIntBytes((long)(this.fileInfo.length() - this.position));
                int read = BlobCacheUtils.toIntBytes((long)Math.min((long)b.remaining(), remainingBytesInPart));
                this.readInternalBytes(currentPart, this.position % this.fileInfo.partSize().getBytes(), b, read);
            }
        }
    }

    private void readInternalBytes(int part, long pos, ByteBuffer b, int length) throws IOException {
        int optimizedReadSize = this.readOptimized(part, pos, b, length);
        assert (optimizedReadSize <= length);
        this.position += (long)optimizedReadSize;
        if (optimizedReadSize < length) {
            long startTimeNanos = this.stats.currentTimeNanos();
            try (InputStream inputStream = this.openBlobStream(part, pos + (long)optimizedReadSize, length - optimizedReadSize);){
                int directReadSize = Streams.read((InputStream)inputStream, (ByteBuffer)b, (int)(length - optimizedReadSize));
                if (directReadSize < length - optimizedReadSize) {
                    throw new EOFException("Read past EOF at [" + this.position + "] with length [" + this.fileInfo.partBytes(part) + "]");
                }
                assert (optimizedReadSize + directReadSize == length) : optimizedReadSize + " and " + directReadSize + " vs " + length;
                this.position += (long)directReadSize;
                long endTimeNanos = this.stats.currentTimeNanos();
                this.stats.addDirectBytesRead(directReadSize, endTimeNanos - startTimeNanos);
            }
        }
    }

    private int readOptimized(int part, long pos, ByteBuffer b, int length) throws IOException {
        if (this.sequentialReadSize == 0L) {
            return 0;
        }
        int read = 0;
        if (this.streamForSequentialReads == null) {
            read = this.readFromNewSequentialStream(part, pos, b, length);
        } else if (this.streamForSequentialReads.canContinueSequentialRead(part, pos)) {
            read = this.streamForSequentialReads.read(b, length);
            if (this.streamForSequentialReads.isFullyRead()) {
                this.streamForSequentialReads.close();
                this.streamForSequentialReads = null;
            } else assert (read == length) : length + " remaining";
            if (read < length) {
                read += this.readFromNewSequentialStream(part, pos + (long)read, b, length - read);
            }
        } else {
            assert (!this.streamForSequentialReads.isFullyRead());
            this.sequentialReadSize = 0L;
            this.closeStreamForSequentialReads();
        }
        return read;
    }

    private void closeStreamForSequentialReads() throws IOException {
        try {
            IOUtils.close((Closeable)this.streamForSequentialReads);
        }
        finally {
            this.streamForSequentialReads = null;
        }
    }

    private int readFromNewSequentialStream(int part, long pos, ByteBuffer b, int length) throws IOException {
        assert (this.streamForSequentialReads == null) : "should only be called when a new stream is needed";
        assert (this.sequentialReadSize > 0L) : "should only be called if optimizing sequential reads";
        long streamLength = Math.min(this.sequentialReadSize, this.fileInfo.partBytes(part) - pos);
        if (streamLength <= (long)length) {
            return 0;
        }
        InputStream inputStream = this.openBlobStream(part, pos, streamLength);
        this.streamForSequentialReads = new StreamForSequentialReads(new FilterInputStream(inputStream){
            private final LongAdder bytesRead;
            private final LongAdder timeNanos;
            {
                this.bytesRead = new LongAdder();
                this.timeNanos = new LongAdder();
            }

            private int onOptimizedRead(CheckedSupplier<Integer, IOException> read) throws IOException {
                long startTimeNanos = DirectBlobContainerIndexInput.this.stats.currentTimeNanos();
                int result = (Integer)read.get();
                long endTimeNanos = DirectBlobContainerIndexInput.this.stats.currentTimeNanos();
                if (result != -1) {
                    this.bytesRead.add(result);
                    this.timeNanos.add(endTimeNanos - startTimeNanos);
                }
                return result;
            }

            @Override
            public int read() throws IOException {
                return this.onOptimizedRead((CheckedSupplier<Integer, IOException>)((CheckedSupplier)() -> super.read()));
            }

            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                return this.onOptimizedRead((CheckedSupplier<Integer, IOException>)((CheckedSupplier)() -> super.read(b, off, len)));
            }

            @Override
            public void close() throws IOException {
                super.close();
                DirectBlobContainerIndexInput.this.stats.addOptimizedBytesRead(Math.toIntExact(this.bytesRead.sumThenReset()), this.timeNanos.sumThenReset());
            }
        }, part, pos, streamLength);
        int read = this.streamForSequentialReads.read(b, length);
        assert (read == length) : read + " vs " + length;
        assert (!this.streamForSequentialReads.isFullyRead());
        return read;
    }

    protected void seekInternal(long pos) throws IOException {
        BlobCacheUtils.ensureSeek((long)pos, (IndexInput)this);
        if (this.position != this.offset + pos) {
            this.position = this.offset + pos;
            this.closeStreamForSequentialReads();
        }
    }

    public DirectBlobContainerIndexInput clone() {
        DirectBlobContainerIndexInput clone = (DirectBlobContainerIndexInput)super.clone();
        clone.sequentialReadSize = 0L;
        clone.closed = new AtomicBoolean(false);
        clone.isClone = true;
        return clone;
    }

    public IndexInput slice(String sliceName, long offset, long length) throws IOException {
        BlobCacheUtils.ensureSlice((String)sliceName, (long)offset, (long)length, (IndexInput)this);
        DirectBlobContainerIndexInput slice = new DirectBlobContainerIndexInput(sliceName, this.blobContainer, this.fileInfo, this.getBufferSize(), this.stats, this.position, this.offset + offset, length, 0L);
        slice.isClone = true;
        slice.seek(0L);
        return slice;
    }

    public void close() throws IOException {
        if (this.closed.compareAndSet(false, true)) {
            if (!this.isClone) {
                this.stats.incrementCloseCount();
            }
            this.closeStreamForSequentialReads();
        }
    }

    public long length() {
        return this.length;
    }

    public String toString() {
        return super.toString() + "[read seq=" + (this.streamForSequentialReads != null ? "yes" : "no") + "]";
    }

    private InputStream openBlobStream(int part, long pos, long length) throws IOException {
        assert (MetadataCachingIndexInput.assertCurrentThreadMayAccessBlobStore());
        this.stats.addBlobStoreBytesRequested(length);
        return this.blobContainer.readBlob(OperationPurpose.SNAPSHOT_DATA, this.fileInfo.partName(part), pos, length);
    }

    private static class StreamForSequentialReads
    implements Closeable {
        private final InputStream inputStream;
        private final int part;
        private long pos;
        private final long maxPos;

        StreamForSequentialReads(InputStream inputStream, int part, long pos, long streamLength) {
            this.inputStream = Objects.requireNonNull(inputStream);
            this.part = part;
            this.pos = pos;
            this.maxPos = pos + streamLength;
        }

        boolean canContinueSequentialRead(int part, long pos) {
            return this.part == part && this.pos == pos;
        }

        int read(ByteBuffer b, int length) throws IOException {
            int read;
            assert (this.pos < this.maxPos) : "should not try and read from a fully-read stream";
            int totalRead = Streams.read((InputStream)this.inputStream, (ByteBuffer)b, (int)length);
            int n = read = totalRead > 0 ? totalRead : -1;
            assert (read <= length) : read + " vs " + length;
            this.pos += (long)read;
            return read;
        }

        boolean isFullyRead() {
            assert (this.pos <= this.maxPos);
            return this.pos >= this.maxPos;
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
        }
    }
}

