/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.io.IOException;
import java.nio.file.Path;
import java.security.AccessControlException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedTrustManager;
import org.elasticsearch.common.ssl.KeyStoreUtil;
import org.elasticsearch.common.ssl.PemUtils;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.common.ssl.SslFileUtil;
import org.elasticsearch.common.ssl.SslTrustConfig;
import org.elasticsearch.common.ssl.StoredCertificate;

public final class PemTrustConfig
implements SslTrustConfig {
    private static final String CA_FILE_TYPE = "PEM certificate_authorities";
    private final List<String> certificateAuthorities;
    private final Path basePath;

    public PemTrustConfig(List<String> certificateAuthorities, Path basePath) {
        this.certificateAuthorities = Collections.unmodifiableList(certificateAuthorities);
        this.basePath = basePath;
    }

    @Override
    public Collection<Path> getDependentFiles() {
        return this.resolveFiles();
    }

    @Override
    public Collection<? extends StoredCertificate> getConfiguredCertificates() {
        ArrayList<StoredCertificate> info = new ArrayList<StoredCertificate>(this.certificateAuthorities.size());
        for (String caPath : this.certificateAuthorities) {
            for (Certificate cert : this.readCertificates(List.of(this.resolveFile(caPath)))) {
                if (!(cert instanceof X509Certificate)) continue;
                info.add(new StoredCertificate((X509Certificate)cert, caPath, "PEM", null, false));
            }
        }
        return info;
    }

    @Override
    public X509ExtendedTrustManager createTrustManager() {
        List<Path> paths = this.resolveFiles();
        try {
            List<Certificate> certificates = this.readCertificates(paths);
            KeyStore store = KeyStoreUtil.buildTrustStore(certificates);
            return KeyStoreUtil.createTrustManager(store, TrustManagerFactory.getDefaultAlgorithm());
        }
        catch (GeneralSecurityException e) {
            throw new SslConfigException("cannot create trust using PEM certificates [" + SslFileUtil.pathsToString(paths) + "]", e);
        }
    }

    private List<Path> resolveFiles() {
        return this.certificateAuthorities.stream().map(this::resolveFile).toList();
    }

    private Path resolveFile(String other) {
        return this.basePath.resolve(other);
    }

    private List<Certificate> readCertificates(List<Path> paths) {
        try {
            return PemUtils.readCertificates(paths);
        }
        catch (AccessControlException e) {
            throw SslFileUtil.accessControlFailure(CA_FILE_TYPE, paths, e, this.basePath);
        }
        catch (IOException e) {
            throw SslFileUtil.ioException(CA_FILE_TYPE, paths, e);
        }
        catch (GeneralSecurityException e) {
            throw SslFileUtil.securityException(CA_FILE_TYPE, paths, e);
        }
        catch (SslConfigException e) {
            throw SslFileUtil.configException(CA_FILE_TYPE, paths, e);
        }
    }

    public String toString() {
        return "PEM-trust{" + SslFileUtil.pathsToString(this.resolveFiles()) + "}";
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PemTrustConfig that = (PemTrustConfig)o;
        return Objects.equals(this.certificateAuthorities, that.certificateAuthorities);
    }

    public int hashCode() {
        return Objects.hash(this.certificateAuthorities);
    }
}

