/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ccr.repository;

import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.function.LongConsumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefIterator;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.metrics.CounterMetric;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.CombinedRateLimiter;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.KeyedLock;
import org.elasticsearch.core.AbstractRefCounted;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.shard.IndexEventListener;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardClosedException;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.ccr.CcrSettings;

public class CcrRestoreSourceService
extends AbstractLifecycleComponent
implements IndexEventListener {
    private static final Logger logger = LogManager.getLogger(CcrRestoreSourceService.class);
    private final Map<String, RestoreSession> onGoingRestores = ConcurrentCollections.newConcurrentMap();
    private final Map<IndexShard, HashSet<String>> sessionsForShard = new HashMap<IndexShard, HashSet<String>>();
    private final ThreadPool threadPool;
    private final CcrSettings ccrSettings;
    private final CounterMetric throttleTime = new CounterMetric();

    public CcrRestoreSourceService(ThreadPool threadPool, CcrSettings ccrSettings) {
        this.threadPool = threadPool;
        this.ccrSettings = ccrSettings;
    }

    public synchronized void afterIndexShardClosed(ShardId shardId, @Nullable IndexShard indexShard, Settings indexSettings) {
        HashSet<String> sessions;
        if (indexShard != null && (sessions = this.sessionsForShard.remove(indexShard)) != null) {
            for (String sessionUUID : sessions) {
                RestoreSession restore = this.onGoingRestores.remove(sessionUUID);
                assert (restore != null) : "Session UUID [" + sessionUUID + "] registered for shard but not found in ongoing restores";
                restore.decRef();
            }
        }
    }

    protected void doStart() {
    }

    protected void doStop() {
    }

    protected synchronized void doClose() {
        this.sessionsForShard.clear();
        this.onGoingRestores.values().forEach(AbstractRefCounted::decRef);
        this.onGoingRestores.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized Store.MetadataSnapshot openSession(String sessionUUID, IndexShard indexShard) throws IOException {
        boolean success = false;
        RestoreSession restore = null;
        try {
            if (this.onGoingRestores.containsKey(sessionUUID)) {
                logger.debug("not opening new session [{}] as it already exists", (Object)sessionUUID);
                restore = this.onGoingRestores.get(sessionUUID);
            } else {
                logger.debug("opening session [{}] for shard [{}]", (Object)sessionUUID, (Object)indexShard.shardId());
                if (indexShard.state() == IndexShardState.CLOSED) {
                    throw new IndexShardClosedException(indexShard.shardId(), "cannot open ccr restore session if shard closed");
                }
                Engine.IndexCommitRef commitRef = indexShard.acquireSafeIndexCommit();
                Set<String> fileNames = Set.copyOf(commitRef.getIndexCommit().getFileNames());
                restore = new RestoreSession(sessionUUID, indexShard, commitRef, fileNames, this.scheduleTimeout(sessionUUID));
                this.onGoingRestores.put(sessionUUID, restore);
                HashSet sessions = this.sessionsForShard.computeIfAbsent(indexShard, s -> new HashSet());
                sessions.add(sessionUUID);
            }
            Store.MetadataSnapshot metadata = restore.getMetadata();
            success = true;
            Store.MetadataSnapshot metadataSnapshot = metadata;
            return metadataSnapshot;
        }
        finally {
            if (!success) {
                this.onGoingRestores.remove(sessionUUID);
                if (restore != null) {
                    restore.decRef();
                }
            }
        }
    }

    public void ensureSessionShardIdConsistency(String sessionUUID, ShardId shardId) {
        RestoreSession restore = this.onGoingRestores.get(sessionUUID);
        if (restore == null) {
            logger.debug("could not get session [{}] because session not found", (Object)sessionUUID);
            throw new IllegalArgumentException("session [" + sessionUUID + "] not found");
        }
        ShardId sessionShardId = restore.indexShard.shardId();
        if (!sessionShardId.equals((Object)shardId)) {
            throw new IllegalArgumentException("session [" + sessionUUID + "] shardId [" + sessionShardId + "] does not match requested shardId [" + shardId + "]");
        }
    }

    public void ensureFileNameIsKnownToSession(String sessionUUID, String fileName) {
        RestoreSession restore = this.onGoingRestores.get(sessionUUID);
        if (restore == null) {
            logger.debug("could not get session [{}] because session not found", (Object)sessionUUID);
            throw new IllegalArgumentException("session [" + sessionUUID + "] not found");
        }
        if (!restore.fileNames.contains(fileName)) {
            throw new IllegalArgumentException("invalid file name [" + fileName + "]");
        }
    }

    public void closeSession(String sessionUUID) {
        this.internalCloseSession(sessionUUID, true);
    }

    public synchronized SessionReader getSessionReader(String sessionUUID) {
        RestoreSession restore = this.onGoingRestores.get(sessionUUID);
        if (restore == null) {
            logger.debug("could not get session [{}] because session not found", (Object)sessionUUID);
            throw new IllegalArgumentException("session [" + sessionUUID + "] not found");
        }
        restore.idle = false;
        return new SessionReader(restore, this.ccrSettings, arg_0 -> ((CounterMetric)this.throttleTime).inc(arg_0));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void internalCloseSession(String sessionUUID, boolean throwIfSessionMissing) {
        RestoreSession restore;
        CcrRestoreSourceService ccrRestoreSourceService = this;
        synchronized (ccrRestoreSourceService) {
            restore = this.onGoingRestores.remove(sessionUUID);
            if (restore == null) {
                if (throwIfSessionMissing) {
                    logger.debug("could not close session [{}] because session not found", (Object)sessionUUID);
                    throw new IllegalArgumentException("session [" + sessionUUID + "] not found");
                }
                return;
            }
            HashSet<String> sessions = this.sessionsForShard.get(restore.indexShard);
            assert (sessions != null) : "No session UUIDs for shard even though one [" + sessionUUID + "] is active in ongoing restores";
            if (sessions != null) {
                boolean removed = sessions.remove(sessionUUID);
                assert (removed) : "No session found for UUID [" + sessionUUID + "]";
                if (sessions.isEmpty()) {
                    this.sessionsForShard.remove(restore.indexShard);
                }
            }
        }
        restore.decRef();
    }

    private Scheduler.Cancellable scheduleTimeout(String sessionUUID) {
        TimeValue idleTimeout = this.ccrSettings.getRecoveryActivityTimeout();
        return this.threadPool.scheduleWithFixedDelay(() -> this.maybeTimeout(sessionUUID), idleTimeout, (Executor)this.threadPool.generic());
    }

    private void maybeTimeout(String sessionUUID) {
        RestoreSession restoreSession = this.onGoingRestores.get(sessionUUID);
        if (restoreSession != null) {
            if (restoreSession.idle) {
                this.internalCloseSession(sessionUUID, false);
            } else {
                restoreSession.idle = true;
            }
        }
    }

    public long getThrottleTime() {
        return this.throttleTime.count();
    }

    private static class RestoreSession
    extends AbstractRefCounted {
        private final String sessionUUID;
        private final IndexShard indexShard;
        private final Engine.IndexCommitRef commitRef;
        private final Set<String> fileNames;
        private final Scheduler.Cancellable timeoutTask;
        private final KeyedLock<String> keyedLock = new KeyedLock();
        private final Map<String, IndexInput> cachedInputs = new ConcurrentHashMap<String, IndexInput>();
        private volatile boolean idle = false;

        private RestoreSession(String sessionUUID, IndexShard indexShard, Engine.IndexCommitRef commitRef, Set<String> fileNames, Scheduler.Cancellable timeoutTask) {
            this.sessionUUID = sessionUUID;
            this.indexShard = indexShard;
            this.commitRef = commitRef;
            this.fileNames = fileNames;
            this.timeoutTask = timeoutTask;
        }

        private Store.MetadataSnapshot getMetadata() throws IOException {
            this.indexShard.store().incRef();
            try {
                Store.MetadataSnapshot metadataSnapshot = this.indexShard.store().getMetadata(this.commitRef.getIndexCommit());
                return metadataSnapshot;
            }
            finally {
                this.indexShard.store().decRef();
            }
        }

        private long readFileBytes(String fileName, BytesReference reference) throws IOException {
            try (Releasable ignored = this.keyedLock.acquire((Object)fileName);){
                BytesRef ref;
                IndexInput indexInput = this.cachedInputs.computeIfAbsent(fileName, f -> {
                    try {
                        return this.commitRef.getIndexCommit().getDirectory().openInput(fileName, IOContext.READONCE);
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(e);
                    }
                });
                BytesRefIterator refIterator = reference.iterator();
                while ((ref = refIterator.next()) != null) {
                    indexInput.readBytes(ref.bytes, ref.offset, ref.length);
                }
                long offsetAfterRead = indexInput.getFilePointer();
                if (offsetAfterRead == indexInput.length()) {
                    this.cachedInputs.remove(fileName);
                    IOUtils.close((Closeable)indexInput);
                }
                long l = offsetAfterRead;
                return l;
            }
        }

        protected void closeInternal() {
            logger.debug("closing session [{}] for shard [{}]", (Object)this.sessionUUID, (Object)this.indexShard.shardId());
            assert (!this.keyedLock.hasLockedKeys()) : "Should not hold any file locks when closing";
            this.timeoutTask.cancel();
            IOUtils.closeWhileHandlingException(this.cachedInputs.values());
            IOUtils.closeWhileHandlingException((Closeable)this.commitRef);
        }
    }

    public static class SessionReader
    implements Closeable {
        private final RestoreSession restoreSession;
        private final CcrSettings ccrSettings;
        private final LongConsumer throttleListener;

        private SessionReader(RestoreSession restoreSession, CcrSettings ccrSettings, LongConsumer throttleListener) {
            this.restoreSession = restoreSession;
            this.ccrSettings = ccrSettings;
            this.throttleListener = throttleListener;
            restoreSession.incRef();
        }

        public long readFileBytes(String fileName, BytesReference reference) throws IOException {
            CombinedRateLimiter rateLimiter = this.ccrSettings.getRateLimiter();
            long throttleTime = rateLimiter.maybePause(reference.length());
            this.throttleListener.accept(throttleTime);
            return this.restoreSession.readFileBytes(fileName, reference);
        }

        @Override
        public void close() {
            this.restoreSession.decRef();
        }
    }
}

