/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.threadpool;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.SizeValue;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionHandler;
import org.elasticsearch.common.util.concurrent.EsThreadPoolExecutor;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.ReportingService;
import org.elasticsearch.threadpool.ExecutorBuilder;
import org.elasticsearch.threadpool.FixedExecutorBuilder;
import org.elasticsearch.threadpool.ScalingExecutorBuilder;
import org.elasticsearch.threadpool.ScheduledCancellableAdapter;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPoolInfo;
import org.elasticsearch.threadpool.ThreadPoolStats;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentFragment;
import org.elasticsearch.xcontent.XContentBuilder;

public class ThreadPool
implements ReportingService<ThreadPoolInfo>,
Scheduler {
    private static final Logger logger = LogManager.getLogger(ThreadPool.class);
    public static final Map<String, ThreadPoolType> THREAD_POOL_TYPES = Map.ofEntries(Map.entry("same", ThreadPoolType.DIRECT), Map.entry("generic", ThreadPoolType.SCALING), Map.entry("get", ThreadPoolType.FIXED), Map.entry("analyze", ThreadPoolType.FIXED), Map.entry("write", ThreadPoolType.FIXED), Map.entry("search", ThreadPoolType.FIXED), Map.entry("search_worker", ThreadPoolType.FIXED), Map.entry("search_coordination", ThreadPoolType.FIXED), Map.entry("management", ThreadPoolType.SCALING), Map.entry("flush", ThreadPoolType.SCALING), Map.entry("refresh", ThreadPoolType.SCALING), Map.entry("warmer", ThreadPoolType.SCALING), Map.entry("snapshot", ThreadPoolType.SCALING), Map.entry("snapshot_meta", ThreadPoolType.SCALING), Map.entry("force_merge", ThreadPoolType.FIXED), Map.entry("fetch_shard_started", ThreadPoolType.SCALING), Map.entry("fetch_shard_store", ThreadPoolType.SCALING), Map.entry("search_throttled", ThreadPoolType.FIXED), Map.entry("system_read", ThreadPoolType.FIXED), Map.entry("system_write", ThreadPoolType.FIXED), Map.entry("system_critical_read", ThreadPoolType.FIXED), Map.entry("system_critical_write", ThreadPoolType.FIXED));
    private final Map<String, ExecutorHolder> executors;
    private final ThreadPoolInfo threadPoolInfo;
    private final CachedTimeThread cachedTimeThread;
    private final ThreadContext threadContext;
    private final Map<String, ExecutorBuilder> builders;
    private final ScheduledThreadPoolExecutor scheduler;
    private final long slowSchedulerWarnThresholdNanos;
    public static final Setting<TimeValue> ESTIMATED_TIME_INTERVAL_SETTING = Setting.timeSetting("thread_pool.estimated_time_interval", TimeValue.timeValueMillis((long)200L), TimeValue.ZERO, Setting.Property.NodeScope);
    public static final Setting<TimeValue> LATE_TIME_INTERVAL_WARN_THRESHOLD_SETTING = Setting.timeSetting("thread_pool.estimated_time_interval.warn_threshold", TimeValue.timeValueSeconds((long)5L), TimeValue.ZERO, Setting.Property.NodeScope);
    public static final Setting<TimeValue> SLOW_SCHEDULER_TASK_WARN_THRESHOLD_SETTING = Setting.timeSetting("thread_pool.scheduler.warn_threshold", TimeValue.timeValueSeconds((long)5L), TimeValue.ZERO, Setting.Property.NodeScope);

    public Collection<ExecutorBuilder> builders() {
        return Collections.unmodifiableCollection(this.builders.values());
    }

    public ThreadPool(Settings settings, ExecutorBuilder<?> ... customBuilders) {
        assert (Node.NODE_NAME_SETTING.exists(settings));
        HashMap builders = new HashMap();
        int allocatedProcessors = EsExecutors.allocatedProcessors(settings);
        int halfProc = ThreadPool.halfAllocatedProcessors(allocatedProcessors);
        int halfProcMaxAt5 = ThreadPool.halfAllocatedProcessorsMaxFive(allocatedProcessors);
        int halfProcMaxAt10 = ThreadPool.halfAllocatedProcessorsMaxTen(allocatedProcessors);
        int genericThreadPoolMax = ThreadPool.boundedBy(4 * allocatedProcessors, 128, 512);
        builders.put("generic", new ScalingExecutorBuilder("generic", 4, genericThreadPoolMax, TimeValue.timeValueSeconds((long)30L), false));
        builders.put("write", new FixedExecutorBuilder(settings, "write", allocatedProcessors, 10000, new EsExecutors.TaskTrackingConfig(true, 0.1)));
        int searchOrGetThreadPoolSize = ThreadPool.searchOrGetThreadPoolSize(allocatedProcessors);
        builders.put("get", new FixedExecutorBuilder(settings, "get", searchOrGetThreadPoolSize, 1000, EsExecutors.TaskTrackingConfig.DO_NOT_TRACK));
        builders.put("analyze", new FixedExecutorBuilder(settings, "analyze", 1, 16, EsExecutors.TaskTrackingConfig.DO_NOT_TRACK));
        builders.put("search", new FixedExecutorBuilder(settings, "search", searchOrGetThreadPoolSize, 1000, EsExecutors.TaskTrackingConfig.DEFAULT));
        builders.put("search_worker", new FixedExecutorBuilder(settings, "search_worker", searchOrGetThreadPoolSize, -1, EsExecutors.TaskTrackingConfig.DEFAULT));
        builders.put("search_coordination", new FixedExecutorBuilder(settings, "search_coordination", halfProc, 1000, EsExecutors.TaskTrackingConfig.DEFAULT));
        builders.put("auto_complete", new FixedExecutorBuilder(settings, "auto_complete", Math.max(allocatedProcessors / 4, 1), 100, EsExecutors.TaskTrackingConfig.DEFAULT));
        builders.put("search_throttled", new FixedExecutorBuilder(settings, "search_throttled", 1, 100, EsExecutors.TaskTrackingConfig.DEFAULT));
        builders.put("management", new ScalingExecutorBuilder("management", 1, ThreadPool.boundedBy(allocatedProcessors, 1, 5), TimeValue.timeValueMinutes((long)5L), false));
        builders.put("flush", new ScalingExecutorBuilder("flush", 1, halfProcMaxAt5, TimeValue.timeValueMinutes((long)5L), false));
        builders.put("refresh", new ScalingExecutorBuilder("refresh", 1, halfProcMaxAt10, TimeValue.timeValueMinutes((long)5L), false));
        builders.put("warmer", new ScalingExecutorBuilder("warmer", 1, halfProcMaxAt5, TimeValue.timeValueMinutes((long)5L), false));
        int maxSnapshotCores = ThreadPool.getMaxSnapshotThreadPoolSize(allocatedProcessors);
        builders.put("snapshot", new ScalingExecutorBuilder("snapshot", 1, maxSnapshotCores, TimeValue.timeValueMinutes((long)5L), false));
        builders.put("snapshot_meta", new ScalingExecutorBuilder("snapshot_meta", 1, Math.min(allocatedProcessors * 3, 50), TimeValue.timeValueSeconds((long)30L), false));
        builders.put("fetch_shard_started", new ScalingExecutorBuilder("fetch_shard_started", 1, 2 * allocatedProcessors, TimeValue.timeValueMinutes((long)5L), false));
        builders.put("force_merge", new FixedExecutorBuilder(settings, "force_merge", ThreadPool.oneEighthAllocatedProcessors(allocatedProcessors), -1, EsExecutors.TaskTrackingConfig.DO_NOT_TRACK));
        builders.put("cluster_coordination", new FixedExecutorBuilder(settings, "cluster_coordination", 1, -1, EsExecutors.TaskTrackingConfig.DO_NOT_TRACK));
        builders.put("fetch_shard_store", new ScalingExecutorBuilder("fetch_shard_store", 1, 2 * allocatedProcessors, TimeValue.timeValueMinutes((long)5L), false));
        builders.put("system_read", new FixedExecutorBuilder(settings, "system_read", halfProcMaxAt5, 2000, EsExecutors.TaskTrackingConfig.DO_NOT_TRACK));
        builders.put("system_write", new FixedExecutorBuilder(settings, "system_write", halfProcMaxAt5, 1000, new EsExecutors.TaskTrackingConfig(true, 0.1)));
        builders.put("system_critical_read", new FixedExecutorBuilder(settings, "system_critical_read", halfProcMaxAt5, 2000, EsExecutors.TaskTrackingConfig.DO_NOT_TRACK));
        builders.put("system_critical_write", new FixedExecutorBuilder(settings, "system_critical_write", halfProcMaxAt5, 1500, new EsExecutors.TaskTrackingConfig(true, 0.1)));
        for (ExecutorBuilder<?> builder : customBuilders) {
            if (builders.containsKey(builder.name())) {
                throw new IllegalArgumentException("builder with name [" + builder.name() + "] already exists");
            }
            builders.put(builder.name(), builder);
        }
        this.builders = Collections.unmodifiableMap(builders);
        this.threadContext = new ThreadContext(settings);
        HashMap<String, ExecutorHolder> executors = new HashMap<String, ExecutorHolder>();
        for (Map.Entry entry : builders.entrySet()) {
            Object executorSettings = ((ExecutorBuilder)entry.getValue()).getSettings(settings);
            ExecutorHolder executorHolder = ((ExecutorBuilder)entry.getValue()).build(executorSettings, this.threadContext);
            if (executors.containsKey(executorHolder.info.getName())) {
                throw new IllegalStateException("duplicate executors with name [" + executorHolder.info.getName() + "] registered");
            }
            logger.debug("created thread pool: {}", (Object)((ExecutorBuilder)entry.getValue()).formatInfo(executorHolder.info));
            executors.put((String)entry.getKey(), executorHolder);
        }
        executors.put("same", new ExecutorHolder(EsExecutors.DIRECT_EXECUTOR_SERVICE, new Info("same", ThreadPoolType.DIRECT)));
        this.executors = Map.copyOf(executors);
        List<Info> infos = executors.values().stream().filter(holder -> !holder.info.getName().equals("same")).map(holder -> holder.info).toList();
        this.threadPoolInfo = new ThreadPoolInfo(infos);
        this.scheduler = Scheduler.initScheduler(settings, "scheduler");
        this.slowSchedulerWarnThresholdNanos = SLOW_SCHEDULER_TASK_WARN_THRESHOLD_SETTING.get(settings).nanos();
        this.cachedTimeThread = new CachedTimeThread(EsExecutors.threadName(settings, "[timer]"), ESTIMATED_TIME_INTERVAL_SETTING.get(settings).millis(), LATE_TIME_INTERVAL_WARN_THRESHOLD_SETTING.get(settings).millis());
        this.cachedTimeThread.start();
    }

    protected ThreadPool() {
        this.builders = Map.of();
        this.executors = Map.of();
        this.cachedTimeThread = null;
        this.threadPoolInfo = new ThreadPoolInfo(List.of());
        this.slowSchedulerWarnThresholdNanos = 0L;
        this.threadContext = new ThreadContext(Settings.EMPTY);
        this.scheduler = null;
    }

    public long relativeTimeInMillis() {
        return TimeValue.nsecToMSec((long)this.relativeTimeInNanos());
    }

    public long relativeTimeInNanos() {
        return this.cachedTimeThread.relativeTimeInNanos();
    }

    public long rawRelativeTimeInMillis() {
        return TimeValue.nsecToMSec((long)System.nanoTime());
    }

    public long absoluteTimeInMillis() {
        return this.cachedTimeThread.absoluteTimeInMillis();
    }

    @Override
    public ThreadPoolInfo info() {
        return this.threadPoolInfo;
    }

    public Info info(String name) {
        ExecutorHolder holder = this.executors.get(name);
        if (holder == null) {
            return null;
        }
        return holder.info;
    }

    public ThreadPoolStats stats() {
        ArrayList<ThreadPoolStats.Stats> stats = new ArrayList<ThreadPoolStats.Stats>();
        for (ExecutorHolder holder : this.executors.values()) {
            String name = holder.info.getName();
            if ("same".equals(name)) continue;
            int threads = -1;
            int queue = -1;
            int active = -1;
            long rejected = -1L;
            int largest = -1;
            long completed = -1L;
            ExecutorService executorService = holder.executor();
            if (executorService instanceof ThreadPoolExecutor) {
                ThreadPoolExecutor threadPoolExecutor = (ThreadPoolExecutor)executorService;
                threads = threadPoolExecutor.getPoolSize();
                queue = threadPoolExecutor.getQueue().size();
                active = threadPoolExecutor.getActiveCount();
                largest = threadPoolExecutor.getLargestPoolSize();
                completed = threadPoolExecutor.getCompletedTaskCount();
                RejectedExecutionHandler rejectedExecutionHandler = threadPoolExecutor.getRejectedExecutionHandler();
                if (rejectedExecutionHandler instanceof EsRejectedExecutionHandler) {
                    EsRejectedExecutionHandler handler = (EsRejectedExecutionHandler)rejectedExecutionHandler;
                    rejected = handler.rejected();
                }
            }
            stats.add(new ThreadPoolStats.Stats(name, threads, queue, active, rejected, largest, completed));
        }
        return new ThreadPoolStats(stats);
    }

    public ExecutorService generic() {
        return this.executor("generic");
    }

    public ExecutorService executor(String name) {
        ExecutorHolder holder = this.executors.get(name);
        if (holder == null) {
            String message = "no executor service found for [" + name + "]";
            assert (false) : message;
            throw new IllegalArgumentException(message);
        }
        return holder.executor();
    }

    @Override
    public Scheduler.ScheduledCancellable schedule(Runnable command, TimeValue delay, Executor executor) {
        final Runnable contextPreservingRunnable = this.threadContext.preserveContext(command);
        Runnable toSchedule = executor != EsExecutors.DIRECT_EXECUTOR_SERVICE ? new ThreadedRunnable(contextPreservingRunnable, executor) : (this.slowSchedulerWarnThresholdNanos > 0L ? new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                long startTime = ThreadPool.this.relativeTimeInNanos();
                try {
                    contextPreservingRunnable.run();
                }
                finally {
                    long took = ThreadPool.this.relativeTimeInNanos() - startTime;
                    if (took > ThreadPool.this.slowSchedulerWarnThresholdNanos) {
                        logger.warn("execution of [{}] took [{}ms] which is above the warn threshold of [{}ms]", (Object)contextPreservingRunnable, (Object)TimeUnit.NANOSECONDS.toMillis(took), (Object)TimeUnit.NANOSECONDS.toMillis(ThreadPool.this.slowSchedulerWarnThresholdNanos));
                    }
                }
            }

            public String toString() {
                return contextPreservingRunnable.toString();
            }
        } : contextPreservingRunnable);
        return new ScheduledCancellableAdapter(this.scheduler.schedule(toSchedule, delay.millis(), TimeUnit.MILLISECONDS));
    }

    public void scheduleUnlessShuttingDown(TimeValue delay, Executor executor, Runnable command) {
        try {
            this.schedule(command, delay, executor);
        }
        catch (EsRejectedExecutionException e) {
            if (e.isExecutorShutdown()) {
                logger.debug(() -> Strings.format((String)"could not schedule execution of [%s] after [%s] on [%s] as executor is shut down", (Object[])new Object[]{command, delay, executor}), (Throwable)e);
            }
            throw e;
        }
    }

    @Override
    public Scheduler.Cancellable scheduleWithFixedDelay(Runnable command, TimeValue interval, Executor executor) {
        Scheduler.ReschedulingRunnable runnable = new Scheduler.ReschedulingRunnable(command, interval, executor, this, e -> {
            if (logger.isDebugEnabled()) {
                logger.debug(() -> Strings.format((String)"scheduled task [%s] was rejected on thread pool [%s]", (Object[])new Object[]{command, executor}), (Throwable)e);
            }
        }, e -> logger.warn(() -> Strings.format((String)"failed to run scheduled task [%s] on thread pool [%s]", (Object[])new Object[]{command, executor}), (Throwable)e));
        runnable.start();
        return runnable;
    }

    protected final void stopCachedTimeThread() {
        this.cachedTimeThread.running = false;
        this.cachedTimeThread.interrupt();
    }

    public void shutdown() {
        this.stopCachedTimeThread();
        this.scheduler.shutdown();
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            executor.executor().shutdown();
        }
    }

    public void shutdownNow() {
        this.stopCachedTimeThread();
        this.scheduler.shutdownNow();
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            executor.executor().shutdownNow();
        }
    }

    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        boolean result = this.scheduler.awaitTermination(timeout, unit);
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            result &= executor.executor().awaitTermination(timeout, unit);
        }
        this.cachedTimeThread.join(unit.toMillis(timeout));
        return result;
    }

    public ScheduledExecutorService scheduler() {
        return this.scheduler;
    }

    static int boundedBy(int value, int min, int max) {
        assert (min < max) : min + " vs " + max;
        return Math.min(max, Math.max(min, value));
    }

    static int halfAllocatedProcessors(int allocatedProcessors) {
        return (allocatedProcessors + 1) / 2;
    }

    static int halfAllocatedProcessorsMaxFive(int allocatedProcessors) {
        return ThreadPool.boundedBy(ThreadPool.halfAllocatedProcessors(allocatedProcessors), 1, 5);
    }

    static int halfAllocatedProcessorsMaxTen(int allocatedProcessors) {
        return ThreadPool.boundedBy(ThreadPool.halfAllocatedProcessors(allocatedProcessors), 1, 10);
    }

    static int twiceAllocatedProcessors(int allocatedProcessors) {
        return ThreadPool.boundedBy(2 * allocatedProcessors, 2, Integer.MAX_VALUE);
    }

    public static int oneEighthAllocatedProcessors(int allocatedProcessors) {
        return ThreadPool.boundedBy(allocatedProcessors / 8, 1, Integer.MAX_VALUE);
    }

    public static int searchOrGetThreadPoolSize(int allocatedProcessors) {
        return allocatedProcessors * 3 / 2 + 1;
    }

    static int getMaxSnapshotThreadPoolSize(int allocatedProcessors) {
        ByteSizeValue maxHeapSize = ByteSizeValue.ofBytes(Runtime.getRuntime().maxMemory());
        return ThreadPool.getMaxSnapshotThreadPoolSize(allocatedProcessors, maxHeapSize);
    }

    static int getMaxSnapshotThreadPoolSize(int allocatedProcessors, ByteSizeValue maxHeapSize) {
        if (maxHeapSize.compareTo(new ByteSizeValue(750L, ByteSizeUnit.MB)) < 0) {
            return ThreadPool.halfAllocatedProcessorsMaxFive(allocatedProcessors);
        }
        return 10;
    }

    public static boolean terminate(ExecutorService service, long timeout, TimeUnit timeUnit) {
        if (service != null) {
            service.shutdown();
            if (ThreadPool.awaitTermination(service, timeout, timeUnit)) {
                return true;
            }
            service.shutdownNow();
            return ThreadPool.awaitTermination(service, timeout, timeUnit);
        }
        return false;
    }

    private static boolean awaitTermination(ExecutorService service, long timeout, TimeUnit timeUnit) {
        try {
            if (service.awaitTermination(timeout, timeUnit)) {
                return true;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }

    public static boolean terminate(ThreadPool pool, long timeout, TimeUnit timeUnit) {
        if (pool != null) {
            pool.shutdown();
            if (ThreadPool.awaitTermination(pool, timeout, timeUnit)) {
                return true;
            }
            pool.shutdownNow();
            return ThreadPool.awaitTermination(pool, timeout, timeUnit);
        }
        return false;
    }

    private static boolean awaitTermination(ThreadPool threadPool, long timeout, TimeUnit timeUnit) {
        try {
            if (threadPool.awaitTermination(timeout, timeUnit)) {
                return true;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }

    public ThreadContext getThreadContext() {
        return this.threadContext;
    }

    public static boolean assertNotScheduleThread(String reason) {
        assert (!Thread.currentThread().getName().contains("scheduler")) : "Expected current thread [" + Thread.currentThread() + "] to not be the scheduler thread. Reason: [" + reason + "]";
        return true;
    }

    public static boolean assertCurrentThreadPool(String ... permittedThreadPoolNames) {
        String threadName = Thread.currentThread().getName();
        assert (threadName.startsWith("TEST-") || threadName.startsWith("LuceneTestCase") || Arrays.stream(permittedThreadPoolNames).anyMatch(n -> threadName.contains("[" + n + "]"))) : threadName + " not in " + Arrays.toString(permittedThreadPoolNames) + " nor a test thread";
        return true;
    }

    public static boolean assertInSystemContext(ThreadPool threadPool) {
        String threadName = Thread.currentThread().getName();
        assert (threadName.startsWith("TEST-") || threadName.startsWith("LuceneTestCase") || threadPool.getThreadContext().isSystemContext()) : threadName + " is not running in the system context nor a test thread";
        return true;
    }

    public static boolean assertCurrentMethodIsNotCalledRecursively() {
        StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
        assert (stackTraceElements.length >= 3) : stackTraceElements.length;
        assert (stackTraceElements[0].getMethodName().equals("getStackTrace")) : stackTraceElements[0];
        assert (stackTraceElements[1].getMethodName().equals("assertCurrentMethodIsNotCalledRecursively")) : stackTraceElements[1];
        StackTraceElement testingMethod = stackTraceElements[2];
        for (int i = 3; i < stackTraceElements.length; ++i) {
            assert (!stackTraceElements[i].getClassName().equals(testingMethod.getClassName()) || !stackTraceElements[i].getMethodName().equals(testingMethod.getMethodName())) : testingMethod.getClassName() + "#" + testingMethod.getMethodName() + " is called recursively";
        }
        return true;
    }

    public static class Names {
        public static final String SAME = "same";
        public static final String GENERIC = "generic";
        public static final String CLUSTER_COORDINATION = "cluster_coordination";
        public static final String GET = "get";
        public static final String ANALYZE = "analyze";
        public static final String WRITE = "write";
        public static final String SEARCH = "search";
        public static final String SEARCH_WORKER = "search_worker";
        public static final String SEARCH_COORDINATION = "search_coordination";
        public static final String AUTO_COMPLETE = "auto_complete";
        public static final String SEARCH_THROTTLED = "search_throttled";
        public static final String MANAGEMENT = "management";
        public static final String FLUSH = "flush";
        public static final String REFRESH = "refresh";
        public static final String WARMER = "warmer";
        public static final String SNAPSHOT = "snapshot";
        public static final String SNAPSHOT_META = "snapshot_meta";
        public static final String FORCE_MERGE = "force_merge";
        public static final String FETCH_SHARD_STARTED = "fetch_shard_started";
        public static final String FETCH_SHARD_STORE = "fetch_shard_store";
        public static final String SYSTEM_READ = "system_read";
        public static final String SYSTEM_WRITE = "system_write";
        public static final String SYSTEM_CRITICAL_READ = "system_critical_read";
        public static final String SYSTEM_CRITICAL_WRITE = "system_critical_write";
    }

    static class ExecutorHolder {
        private final ExecutorService executor;
        public final Info info;

        ExecutorHolder(ExecutorService executor, Info info) {
            assert (executor instanceof EsThreadPoolExecutor || executor == EsExecutors.DIRECT_EXECUTOR_SERVICE);
            this.executor = executor;
            this.info = info;
        }

        ExecutorService executor() {
            return this.executor;
        }
    }

    public static class Info
    implements Writeable,
    ToXContentFragment {
        private final String name;
        private final ThreadPoolType type;
        private final int min;
        private final int max;
        private final TimeValue keepAlive;
        private final SizeValue queueSize;

        public Info(String name, ThreadPoolType type) {
            this(name, type, -1);
        }

        public Info(String name, ThreadPoolType type, int size) {
            this(name, type, size, size, null, null);
        }

        public Info(String name, ThreadPoolType type, int min, int max, @Nullable TimeValue keepAlive, @Nullable SizeValue queueSize) {
            this.name = name;
            this.type = type;
            this.min = min;
            this.max = max;
            this.keepAlive = keepAlive;
            this.queueSize = queueSize;
        }

        public Info(StreamInput in) throws IOException {
            this.name = in.readString();
            this.type = ThreadPoolType.fromType(in.readString());
            this.min = in.readInt();
            this.max = in.readInt();
            this.keepAlive = in.readOptionalTimeValue();
            this.queueSize = in.readOptionalWriteable(SizeValue::new);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.name);
            out.writeString(this.type.getType());
            out.writeInt(this.min);
            out.writeInt(this.max);
            out.writeOptionalTimeValue(this.keepAlive);
            out.writeOptionalWriteable(this.queueSize);
        }

        public String getName() {
            return this.name;
        }

        public ThreadPoolType getThreadPoolType() {
            return this.type;
        }

        public int getMin() {
            return this.min;
        }

        public int getMax() {
            return this.max;
        }

        @Nullable
        public TimeValue getKeepAlive() {
            return this.keepAlive;
        }

        @Nullable
        public SizeValue getQueueSize() {
            return this.queueSize;
        }

        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject(this.name);
            builder.field("type", this.type.getType());
            if (this.type == ThreadPoolType.SCALING) {
                assert (this.min != -1);
                builder.field("core", this.min);
                assert (this.max != -1);
                builder.field("max", this.max);
            } else {
                assert (this.max != -1);
                builder.field("size", this.max);
            }
            if (this.keepAlive != null) {
                builder.field("keep_alive", this.keepAlive.toString());
            }
            if (this.queueSize == null) {
                builder.field("queue_size", -1);
            } else {
                builder.field("queue_size", this.queueSize.singles());
            }
            builder.endObject();
            return builder;
        }
    }

    public static enum ThreadPoolType {
        DIRECT("direct"),
        FIXED("fixed"),
        FIXED_AUTO_QUEUE_SIZE("fixed_auto_queue_size"),
        SCALING("scaling");

        private final String type;
        private static final Map<String, ThreadPoolType> TYPE_MAP;

        public String getType() {
            return this.type;
        }

        private ThreadPoolType(String type) {
            this.type = type;
        }

        public static ThreadPoolType fromType(String type) {
            ThreadPoolType threadPoolType = TYPE_MAP.get(type);
            if (threadPoolType == null) {
                throw new IllegalArgumentException("no ThreadPoolType for " + type);
            }
            return threadPoolType;
        }

        static {
            TYPE_MAP = Arrays.stream(ThreadPoolType.values()).collect(Collectors.toUnmodifiableMap(ThreadPoolType::getType, Function.identity()));
        }
    }

    static class CachedTimeThread
    extends Thread {
        final long interval;
        private final TimeChangeChecker timeChangeChecker;
        volatile boolean running = true;
        volatile long relativeNanos;
        volatile long absoluteMillis;

        CachedTimeThread(String name, long intervalMillis, long thresholdMillis) {
            super(name);
            this.interval = intervalMillis;
            this.relativeNanos = System.nanoTime();
            this.absoluteMillis = System.currentTimeMillis();
            this.timeChangeChecker = new TimeChangeChecker(thresholdMillis, this.absoluteMillis, this.relativeNanos);
            this.setDaemon(true);
        }

        long relativeTimeInNanos() {
            if (0L < this.interval) {
                return this.relativeNanos;
            }
            return System.nanoTime();
        }

        long absoluteTimeInMillis() {
            if (0L < this.interval) {
                return this.absoluteMillis;
            }
            return System.currentTimeMillis();
        }

        @Override
        public void run() {
            while (this.running && 0L < this.interval) {
                this.relativeNanos = System.nanoTime();
                this.absoluteMillis = System.currentTimeMillis();
                this.timeChangeChecker.check(this.absoluteMillis, this.relativeNanos);
                try {
                    Thread.sleep(this.interval);
                }
                catch (InterruptedException e) {
                    this.running = false;
                    return;
                }
            }
        }
    }

    static class ThreadedRunnable
    implements Runnable {
        private final Runnable runnable;
        private final Executor executor;

        ThreadedRunnable(Runnable runnable, Executor executor) {
            this.runnable = runnable;
            this.executor = executor;
        }

        @Override
        public void run() {
            try {
                this.executor.execute(this.runnable);
            }
            catch (EsRejectedExecutionException e) {
                if (e.isExecutorShutdown()) {
                    logger.debug(() -> Strings.format((String)"could not schedule execution of [%s] on [%s] as executor is shut down", (Object[])new Object[]{this.runnable, this.executor}), (Throwable)e);
                }
                throw e;
            }
        }

        public int hashCode() {
            return this.runnable.hashCode();
        }

        public boolean equals(Object obj) {
            return this.runnable.equals(obj);
        }

        public String toString() {
            return "[threaded] " + this.runnable.toString();
        }
    }

    static class TimeChangeChecker {
        private final long thresholdMillis;
        private final long thresholdNanos;
        private long absoluteMillis;
        private long relativeNanos;

        TimeChangeChecker(long thresholdMillis, long absoluteMillis, long relativeNanos) {
            this.thresholdMillis = thresholdMillis;
            this.thresholdNanos = TimeValue.timeValueMillis((long)thresholdMillis).nanos();
            this.absoluteMillis = absoluteMillis;
            this.relativeNanos = relativeNanos;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void check(long newAbsoluteMillis, long newRelativeNanos) {
            if (this.thresholdMillis <= 0L) {
                return;
            }
            try {
                long deltaMillis = newAbsoluteMillis - this.absoluteMillis;
                if (deltaMillis > this.thresholdMillis) {
                    delta = TimeValue.timeValueMillis((long)deltaMillis);
                    logger.warn("timer thread slept for [{}/{}ms] on absolute clock which is above the warn threshold of [{}ms]", (Object)delta, (Object)deltaMillis, (Object)this.thresholdMillis);
                } else if (deltaMillis < 0L) {
                    delta = TimeValue.timeValueMillis((long)(-deltaMillis));
                    logger.warn("absolute clock went backwards by [{}/{}ms] while timer thread was sleeping", (Object)delta, (Object)(-deltaMillis));
                }
                long deltaNanos = newRelativeNanos - this.relativeNanos;
                if (deltaNanos > this.thresholdNanos) {
                    TimeValue delta = TimeValue.timeValueNanos((long)deltaNanos);
                    logger.warn("timer thread slept for [{}/{}ns] on relative clock which is above the warn threshold of [{}ms]", (Object)delta, (Object)deltaNanos, (Object)this.thresholdMillis);
                } else if (deltaNanos < 0L) {
                    TimeValue delta = TimeValue.timeValueNanos((long)(-deltaNanos));
                    logger.error("relative clock went backwards by [{}/{}ns] while timer thread was sleeping", (Object)delta, (Object)(-deltaNanos));
                    assert (false) : "System::nanoTime time should be monotonic";
                }
            }
            finally {
                this.absoluteMillis = newAbsoluteMillis;
                this.relativeNanos = newRelativeNanos;
            }
        }
    }
}

