/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.cli;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.nio.CharBuffer;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.security.auth.x500.X500Principal;
import joptsimple.OptionException;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import joptsimple.OptionSpecBuilder;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMEncryptor;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.openssl.jcajce.JcePEMEncryptorBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.cli.MultiCommand;
import org.elasticsearch.cli.ProcessInfo;
import org.elasticsearch.cli.Terminal;
import org.elasticsearch.cli.UserException;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.cli.EnvironmentAwareCommand;
import org.elasticsearch.common.network.InetAddresses;
import org.elasticsearch.common.ssl.PemUtils;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.CheckedFunction;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.env.Environment;
import org.elasticsearch.xcontent.ConstructingObjectParser;
import org.elasticsearch.xcontent.ContextParser;
import org.elasticsearch.xcontent.ObjectParser;
import org.elasticsearch.xcontent.ParseField;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentParserConfiguration;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xpack.core.ssl.CertParsingUtils;
import org.elasticsearch.xpack.security.cli.CertGenUtils;
import org.elasticsearch.xpack.security.cli.HttpCertificateCommand;

class CertificateTool
extends MultiCommand {
    private static final String AUTO_GEN_CA_DN = "CN=Elastic Certificate Tool Autogenerated CA";
    private static final String DESCRIPTION = "Simplifies certificate creation for use with the Elastic Stack";
    private static final String DEFAULT_CSR_ZIP = "csr-bundle.zip";
    private static final String DEFAULT_CERT_ZIP = "certificate-bundle.zip";
    private static final String DEFAULT_CA_ZIP = "elastic-stack-ca.zip";
    private static final String DEFAULT_CA_P12 = "elastic-stack-ca.p12";
    private static final BouncyCastleProvider BC_PROV = new BouncyCastleProvider();
    static final String DEFAULT_CERT_NAME = "instance";
    private static final CharsetEncoder ASCII_ENCODER = StandardCharsets.US_ASCII.newEncoder();
    private static final int DEFAULT_DAYS = 1095;
    private static final int FILE_EXTENSION_LENGTH = 4;
    static final int MAX_FILENAME_LENGTH = 251;
    private static final Pattern ALLOWED_FILENAME_CHAR_PATTERN = Pattern.compile("[a-zA-Z0-9!@#$%^&{}\\[\\]()_+\\-=,.~'` ]{1,251}");
    private static final int DEFAULT_KEY_SIZE = 2048;
    static final String OLD_OPENSSL_VERSION = "1.1.0";
    static final int MAX_PASSWORD_OLD_OPENSSL = 50;
    static final String INTRO_TEXT = "This tool assists you in the generation of X.509 certificates and certificate\nsigning requests for use with SSL/TLS in the Elastic stack.";
    static final String INSTANCE_EXPLANATION = "* An instance is any piece of the Elastic Stack that requires an SSL certificate.\n  Depending on your configuration, Elasticsearch, Logstash, Kibana, and Beats\n  may all require a certificate and private key.\n* The minimum required value for each instance is a name. This can simply be the\n  hostname, which will be used as the Common Name of the certificate. A full\n  distinguished name may also be used.\n* A filename value may be required for each instance. This is necessary when the\n  name would result in an invalid file or directory name. The name provided here\n  is used as the directory name (within the zip) and the prefix for the key and\n  certificate files. The filename is required if you are prompted and the name\n  is not displayed in the prompt.\n* IP addresses and DNS names are optional. Multiple values can be specified as a\n  comma separated string. If no IP addresses or DNS names are provided, you may\n  disable hostname verification in your SSL configuration.".indent(4);
    static final String CA_EXPLANATION = "* All certificates generated by this tool will be signed by a certificate authority (CA)\n  unless the --self-signed command line option is specified.\n  The tool can automatically generate a new CA for you, or you can provide your own with\n  the --ca or --ca-cert command line options.".indent(4);

    CertificateTool() {
        super(DESCRIPTION);
        this.subcommands.put("csr", new SigningRequestCommand());
        this.subcommands.put("cert", new GenerateCertificateCommand());
        this.subcommands.put("ca", new CertificateAuthorityCommand());
        this.subcommands.put("http", new HttpCertificateCommand());
    }

    protected void execute(Terminal terminal, OptionSet options, ProcessInfo processInfo) throws Exception {
        try {
            super.execute(terminal, options, processInfo);
        }
        catch (OptionException e) {
            if (e.options().size() == 1 && e.options().contains("keep-ca-key")) {
                throw new UserException(64, "Generating certificates without providing a CA is no longer supported.\nPlease first generate a CA with the 'ca' sub-command and provide the ca file \nwith either --ca or --ca-cert/--ca-key to generate certificates.");
            }
            throw e;
        }
    }

    @SuppressForbidden(reason="resolve paths against CWD for a CLI tool")
    static Path resolvePath(String pathStr) {
        return PathUtils.get((String)pathStr, (String[])new String[0]).normalize();
    }

    static Collection<CertificateInformation> parseAndValidateFile(Terminal terminal, Path file) throws Exception {
        Collection<CertificateInformation> config = CertificateTool.parseFile(file);
        boolean hasError = false;
        for (CertificateInformation certInfo : config) {
            List<String> errors = certInfo.validate();
            if (errors.size() <= 0) continue;
            hasError = true;
            terminal.errorPrintln(Terminal.Verbosity.SILENT, "Configuration for instance " + certInfo.name.originalName + " has invalid details");
            for (String message : errors) {
                terminal.errorPrintln(Terminal.Verbosity.SILENT, " * " + message);
            }
            terminal.errorPrintln("");
        }
        if (hasError) {
            throw new UserException(78, "File " + file + " contains invalid configuration details (see messages above)");
        }
        return config;
    }

    static Collection<CertificateInformation> parseFile(Path file) throws Exception {
        try (BufferedReader reader = Files.newBufferedReader(file);){
            XContentParser xContentParser = XContentType.YAML.xContent().createParser(XContentParserConfiguration.EMPTY, (Reader)reader);
            Collection collection = (Collection)CertificateToolParser.PARSER.parse(xContentParser, new ArrayList(), null);
            return collection;
        }
    }

    static PEMEncryptor getEncrypter(char[] password) {
        return new JcePEMEncryptorBuilder("AES-128-CBC").setProvider((Provider)BC_PROV).build(password);
    }

    static boolean checkAndConfirmPasswordLengthForOpenSSLCompatibility(char[] password, Terminal terminal, boolean confirm) {
        if (password.length > 50) {
            terminal.println(Terminal.Verbosity.SILENT, (CharSequence)"Warning: Your password exceeds 50 characters. Versions of OpenSSL older than 1.1.0 may not be able to read this file.");
            if (confirm) {
                return terminal.promptYesNo("Do you want to continue?", true);
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static <T, E extends Exception> T withPassword(String description, char[] password, Terminal terminal, boolean checkLength, CheckedFunction<char[], T, E> body) throws E {
        if (password == null) {
            char[] promptedValue;
            do {
                promptedValue = terminal.readSecret("Enter password for " + description + " : ");
            } while (checkLength && !CertificateTool.checkAndConfirmPasswordLengthForOpenSSLCompatibility(promptedValue, terminal, true));
            try {
                Object object = body.apply((Object)promptedValue);
                return (T)object;
            }
            finally {
                Arrays.fill(promptedValue, '\u0000');
            }
        }
        if (checkLength) {
            CertificateTool.checkAndConfirmPasswordLengthForOpenSSLCompatibility(password, terminal, false);
        }
        return (T)body.apply((Object)password);
    }

    private static void fullyWriteZipFile(Path file, Writer writer) throws Exception {
        CertificateTool.fullyWriteFile(file, (CheckedConsumer<OutputStream, Exception>)((CheckedConsumer)outputStream -> {
            try (ZipOutputStream zipOutputStream = new ZipOutputStream((OutputStream)outputStream, StandardCharsets.UTF_8);
                 JcaPEMWriter pemWriter = new JcaPEMWriter((java.io.Writer)new OutputStreamWriter((OutputStream)zipOutputStream, StandardCharsets.UTF_8));){
                writer.write(zipOutputStream, pemWriter);
            }
        }));
    }

    private static void checkDirectory(Path path, Terminal terminal) throws UserException {
        Path parent = path.getParent();
        if (Files.isDirectory(parent, new LinkOption[0])) {
            return;
        }
        if (Files.exists(parent, new LinkOption[0])) {
            terminal.errorPrintln(Terminal.Verbosity.SILENT, "Path " + parent + " exists, but is not a directory. Cannot write to " + path);
            throw new UserException(73, "Cannot write to " + path);
        }
        if (terminal.promptYesNo("Directory " + parent + " does not exist. Do you want to create it?", true)) {
            try {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new UserException(73, "Cannot create directory " + parent, (Throwable)e);
            }
        } else {
            throw new UserException(73, "Directory " + parent + " does not exist");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void fullyWriteFile(Path file, CheckedConsumer<OutputStream, Exception> writer) throws Exception {
        assert (file != null);
        assert (writer != null);
        boolean success = false;
        if (Files.exists(file, new LinkOption[0])) {
            throw new UserException(74, "Output file '" + file + "' already exists");
        }
        try (OutputStream outputStream = Files.newOutputStream(file, StandardOpenOption.CREATE_NEW);){
            writer.accept((Object)outputStream);
            PosixFileAttributeView view = Files.getFileAttributeView(file, PosixFileAttributeView.class, new LinkOption[0]);
            if (view != null) {
                view.setPermissions(Sets.newHashSet((Object[])new PosixFilePermission[]{PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE}));
            }
            success = true;
        }
        finally {
            if (!success) {
                Files.deleteIfExists(file);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static PrivateKey readPrivateKey(Path path, char[] password, Terminal terminal) throws Exception {
        AtomicReference<char[]> passwordReference = new AtomicReference<char[]>(password);
        try {
            PrivateKey privateKey = PemUtils.readPrivateKey((Path)path, () -> {
                if (password != null) {
                    return password;
                }
                char[] promptedValue = terminal.readSecret("Enter password for CA private key (" + path.getFileName() + ") : ");
                passwordReference.set(promptedValue);
                return promptedValue;
            });
            return privateKey;
        }
        finally {
            if (passwordReference.get() != null) {
                Arrays.fill(passwordReference.get(), '\u0000');
            }
        }
    }

    static GeneralNames getSubjectAlternativeNamesValue(List<String> ipAddresses, List<String> dnsNames, List<String> commonNames) {
        HashSet<GeneralName> generalNameList = new HashSet<GeneralName>();
        for (String ip : ipAddresses) {
            generalNameList.add(new GeneralName(7, ip));
        }
        for (String dns : dnsNames) {
            generalNameList.add(new GeneralName(2, dns));
        }
        for (String cn : commonNames) {
            generalNameList.add(CertGenUtils.createCommonName(cn));
        }
        if (generalNameList.isEmpty()) {
            return null;
        }
        return new GeneralNames(generalNameList.toArray(new GeneralName[0]));
    }

    static boolean isAscii(char[] str) {
        return ASCII_ENCODER.canEncode((CharSequence)CharBuffer.wrap(str));
    }

    private static char[] getChars(String password) {
        return password == null ? null : password.toCharArray();
    }

    static class SigningRequestCommand
    extends CertificateCommand {
        SigningRequestCommand() {
            super("generate certificate signing requests");
            this.acceptInstanceDetails();
            this.acceptInputFile();
        }

        public void execute(Terminal terminal, OptionSet options, Environment env, ProcessInfo processInfo) throws Exception {
            terminal.println((CharSequence)CertificateTool.INTRO_TEXT);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"The 'csr' mode generates certificate signing requests that can be sent to");
            terminal.println((CharSequence)"a trusted certificate authority");
            terminal.println((CharSequence)"    * By default, this generates a single CSR for a single instance.");
            terminal.println((CharSequence)"    * You can use the '-multiple' option to generate CSRs for multiple");
            terminal.println((CharSequence)"       instances, each with their own private key.");
            terminal.println((CharSequence)"    * The '-in' option allows for the CSR generation to be automated");
            terminal.println((CharSequence)"       by describing the details of each instance in a YAML file");
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)INSTANCE_EXPLANATION);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"The 'csr' mode produces a single zip file which contains the certificate");
            terminal.println((CharSequence)"signing requests and private keys for each instance.");
            terminal.println((CharSequence)"    * Each certificate signing request is provided as a standard PEM encoding of a PKCS#10 CSR.");
            terminal.println((CharSequence)"    * Each key is provided as a PEM encoding of an RSA private key");
            terminal.println((CharSequence)"");
            Path output = this.resolveOutputPath(terminal, options, CertificateTool.DEFAULT_CSR_ZIP);
            int keySize = this.getKeySize(options);
            Collection<CertificateInformation> certificateInformations = this.getCertificateInformationList(terminal, options);
            this.generateAndWriteCsrs(output, keySize, certificateInformations);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)("Certificate signing requests have been written to " + output));
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"This file should be properly secured as it contains the private keys for all");
            terminal.println((CharSequence)"instances.");
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"After unzipping the file, there will be a directory for each instance containing");
            terminal.println((CharSequence)"the certificate signing request and the private key. Provide the certificate");
            terminal.println((CharSequence)"signing requests to your certificate authority. Once you have received the");
            terminal.println((CharSequence)"signed certificate, copy the signed certificate, key, and CA certificate to the");
            terminal.println((CharSequence)"configuration directory of the Elastic product that they will be used for and");
            terminal.println((CharSequence)"follow the SSL configuration instructions in the product guide.");
        }

        void generateAndWriteCsrs(Path output, int keySize, Collection<CertificateInformation> certInfo) throws Exception {
            CertificateTool.fullyWriteZipFile(output, (outputStream, pemWriter) -> {
                for (CertificateInformation certificateInformation : certInfo) {
                    KeyPair keyPair = CertGenUtils.generateKeyPair(keySize);
                    GeneralNames sanList = CertificateTool.getSubjectAlternativeNamesValue(certificateInformation.ipAddresses, certificateInformation.dnsNames, certificateInformation.commonNames);
                    PKCS10CertificationRequest csr = CertGenUtils.generateCSR(keyPair, certificateInformation.name.x500Principal, sanList);
                    String dirName = certificateInformation.name.filename + "/";
                    ZipEntry zipEntry = new ZipEntry(dirName);
                    assert (zipEntry.isDirectory());
                    outputStream.putNextEntry(zipEntry);
                    outputStream.putNextEntry(new ZipEntry(dirName + certificateInformation.name.filename + ".csr"));
                    pemWriter.writeObject((Object)csr);
                    pemWriter.flush();
                    outputStream.closeEntry();
                    outputStream.putNextEntry(new ZipEntry(dirName + certificateInformation.name.filename + ".key"));
                    pemWriter.writeObject((Object)keyPair.getPrivate());
                    pemWriter.flush();
                    outputStream.closeEntry();
                }
            });
        }
    }

    static class GenerateCertificateCommand
    extends CertificateCommand {
        OptionSpec<Void> selfSigned;

        GenerateCertificateCommand() {
            super("generate X.509 certificates and keys");
            this.acceptCertificateGenerationOptions();
            this.acceptInstanceDetails();
            this.acceptsCertificateAuthority();
            this.acceptInputFile();
            this.selfSigned = this.parser.accepts("self-signed", "generate self signed certificates").availableUnless(this.caPkcs12PathSpec, new OptionSpec[]{this.caCertPathSpec});
        }

        public void execute(Terminal terminal, OptionSet options, Environment env, ProcessInfo processInfo) throws Exception {
            String filesDescription;
            boolean writeZipFile;
            terminal.println((CharSequence)CertificateTool.INTRO_TEXT);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"The 'cert' mode generates X.509 certificate and private keys.");
            terminal.println((CharSequence)"    * By default, this generates a single certificate and key for use");
            terminal.println((CharSequence)"       on a single instance.");
            terminal.println((CharSequence)"    * The '-multiple' option will prompt you to enter details for multiple");
            terminal.println((CharSequence)"       instances and will generate a certificate and key for each one");
            terminal.println((CharSequence)"    * The '-in' option allows for the certificate generation to be automated by describing");
            terminal.println((CharSequence)"       the details of each instance in a YAML file");
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)INSTANCE_EXPLANATION);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)CA_EXPLANATION);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"By default the 'cert' mode produces a single PKCS#12 output file which holds:");
            terminal.println((CharSequence)"    * The instance certificate");
            terminal.println((CharSequence)"    * The private key for the instance certificate");
            terminal.println((CharSequence)"    * The CA certificate");
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"If you specify any of the following options:");
            terminal.println((CharSequence)"    * -pem (PEM formatted output)");
            terminal.println((CharSequence)"    * -multiple (generate multiple certificates)");
            terminal.println((CharSequence)"    * -in (generate certificates from an input file)");
            terminal.println((CharSequence)"then the output will be be a zip file containing individual certificate/key files");
            terminal.println((CharSequence)"");
            CAInfo caInfo = this.getCAInfo(terminal, options, env);
            Collection<CertificateInformation> certInfo = this.getCertificateInformationList(terminal, options);
            boolean usePemFormat = this.usePemFormat(options);
            boolean bl = writeZipFile = options.has(this.multipleNodesSpec) || options.has(this.inputFileSpec) || usePemFormat;
            Object outputName = writeZipFile ? CertificateTool.DEFAULT_CERT_ZIP : (options.has(this.nameSpec) ? (String)this.nameSpec.value(options) + ".p12" : "elastic-certificates.p12");
            Path output = this.resolveOutputPath(terminal, options, (String)outputName);
            this.generateAndWriteSignedCertificates(output, writeZipFile, options, certInfo, caInfo, terminal);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)("Certificates written to " + output));
            terminal.println((CharSequence)"");
            if (certInfo.size() > 1) {
                terminal.println(Terminal.Verbosity.NORMAL, (CharSequence)"This file should be properly secured as it contains the private keys for ");
                terminal.print(Terminal.Verbosity.NORMAL, "all instances");
            } else {
                terminal.println(Terminal.Verbosity.NORMAL, (CharSequence)"This file should be properly secured as it contains the private key for ");
                terminal.print(Terminal.Verbosity.NORMAL, "your instance.");
            }
            terminal.println((CharSequence)"");
            if (writeZipFile) {
                terminal.println((CharSequence)"After unzipping the file, there will be a directory for each instance.");
                if (usePemFormat) {
                    terminal.println((CharSequence)"Each instance has a certificate and private key.");
                    filesDescription = "the certificate, key, and CA certificate";
                } else {
                    terminal.println((CharSequence)"Each instance has a single PKCS#12 (.p12) file containing the instance");
                    terminal.println((CharSequence)"certificate, instance private key and the CA certificate");
                    filesDescription = "this '.p12' file";
                }
            } else {
                terminal.println((CharSequence)"This file is a self contained file and can be copied and used 'as is'");
                filesDescription = "this '.p12' file";
            }
            terminal.println((CharSequence)"For each Elastic product that you wish to configure, you should copy");
            terminal.println((CharSequence)(filesDescription + " to the relevant configuration directory"));
            terminal.println((CharSequence)"and then follow the SSL configuration instructions in the product guide.");
            terminal.println((CharSequence)"");
            if (usePemFormat || caInfo != null && !caInfo.generated) {
                terminal.println((CharSequence)"For client applications, you may only need to copy the CA certificate and");
                terminal.println((CharSequence)"configure the client to trust this certificate.");
            }
        }

        @Override
        CAInfo getCAInfo(Terminal terminal, OptionSet options, Environment env) throws Exception {
            if (!(options.has(this.selfSigned) || options.has(this.caPkcs12PathSpec) || options.has(this.caCertPathSpec))) {
                throw new UserException(64, "Must specify either --ca or --ca-cert/--ca-key or --self-signed");
            }
            return options.has(this.selfSigned) ? null : super.getCAInfo(terminal, options, env);
        }

        void generateAndWriteSignedCertificates(Path output, boolean writeZipFile, OptionSet options, Collection<CertificateInformation> certs, CAInfo caInfo, Terminal terminal) throws Exception {
            CertificateTool.checkDirectory(output, terminal);
            int keySize = this.getKeySize(options);
            int days = this.getDays(options);
            char[] outputPassword = super.getOutputPassword(options);
            if (writeZipFile) {
                boolean usePem = this.usePemFormat(options);
                boolean usePassword = super.useOutputPassword(options);
                CertificateTool.fullyWriteZipFile(output, (outputStream, pemWriter) -> {
                    for (CertificateInformation certificateInformation : certs) {
                        CertificateAndKey pair = GenerateCertificateCommand.generateCertificateAndKey(certificateInformation, caInfo, keySize, days);
                        String dirName = certificateInformation.name.filename + "/";
                        ZipEntry zipEntry = new ZipEntry(dirName);
                        assert (zipEntry.isDirectory());
                        outputStream.putNextEntry(zipEntry);
                        String entryBase = dirName + certificateInformation.name.filename;
                        if (usePem) {
                            outputStream.putNextEntry(new ZipEntry(entryBase + ".crt"));
                            pemWriter.writeObject((Object)pair.cert);
                            pemWriter.flush();
                            outputStream.closeEntry();
                            String keyFileName = entryBase + ".key";
                            outputStream.putNextEntry(new ZipEntry(keyFileName));
                            if (usePassword) {
                                CertificateTool.withPassword(keyFileName, outputPassword, terminal, true, password -> {
                                    pemWriter.writeObject((Object)pair.key, CertificateTool.getEncrypter(password));
                                    return null;
                                });
                            } else {
                                pemWriter.writeObject((Object)pair.key);
                            }
                            pemWriter.flush();
                            outputStream.closeEntry();
                            continue;
                        }
                        String fileName = entryBase + ".p12";
                        outputStream.putNextEntry(new ZipEntry(fileName));
                        GenerateCertificateCommand.writePkcs12(fileName, outputStream, certificateInformation.name.originalName, pair, caInfo == null ? null : caInfo.certAndKey.cert, outputPassword, terminal);
                        outputStream.closeEntry();
                    }
                });
            } else {
                assert (certs.size() == 1);
                CertificateInformation certificateInformation = certs.iterator().next();
                CertificateAndKey pair = GenerateCertificateCommand.generateCertificateAndKey(certificateInformation, caInfo, keySize, days);
                CertificateTool.fullyWriteFile(output, (CheckedConsumer<OutputStream, Exception>)((CheckedConsumer)stream -> GenerateCertificateCommand.writePkcs12(output.getFileName().toString(), stream, certificateInformation.name.originalName, pair, caInfo == null ? null : caInfo.certAndKey.cert, outputPassword, terminal)));
            }
        }

        private static CertificateAndKey generateCertificateAndKey(CertificateInformation certificateInformation, CAInfo caInfo, int keySize, int days) throws Exception {
            KeyPair keyPair = CertGenUtils.generateKeyPair(keySize);
            X509Certificate certificate = caInfo != null ? CertGenUtils.generateSignedCertificate(certificateInformation.name.x500Principal, CertificateTool.getSubjectAlternativeNamesValue(certificateInformation.ipAddresses, certificateInformation.dnsNames, certificateInformation.commonNames), keyPair, caInfo.certAndKey.cert, caInfo.certAndKey.key, days) : CertGenUtils.generateSignedCertificate(certificateInformation.name.x500Principal, CertificateTool.getSubjectAlternativeNamesValue(certificateInformation.ipAddresses, certificateInformation.dnsNames, certificateInformation.commonNames), keyPair, null, null, false, days, null);
            return new CertificateAndKey(certificate, keyPair.getPrivate());
        }
    }

    static class CertificateAuthorityCommand
    extends CertificateCommand {
        CertificateAuthorityCommand() {
            super("generate a new local certificate authority");
            this.acceptCertificateGenerationOptions();
            this.acceptsCertificateAuthorityName();
            this.caPasswordSpec = this.outputPasswordSpec;
        }

        public void execute(Terminal terminal, OptionSet options, Environment env, ProcessInfo processInfo) throws Exception {
            terminal.println((CharSequence)CertificateTool.INTRO_TEXT);
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"The 'ca' mode generates a new 'certificate authority'");
            terminal.println((CharSequence)"This will create a new X.509 certificate and private key that can be used");
            terminal.println((CharSequence)"to sign certificate when running in 'cert' mode.");
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"Use the 'ca-dn' option if you wish to configure the 'distinguished name'");
            terminal.println((CharSequence)"of the certificate authority");
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"By default the 'ca' mode produces a single PKCS#12 output file which holds:");
            terminal.println((CharSequence)"    * The CA certificate");
            terminal.println((CharSequence)"    * The CA's private key");
            terminal.println((CharSequence)"");
            terminal.println((CharSequence)"If you elect to generate PEM format certificates (the -pem option), then the output will");
            terminal.println((CharSequence)"be a zip file containing individual files for the CA certificate and private key");
            terminal.println((CharSequence)"");
            CAInfo caInfo = this.generateCA(terminal, options);
            boolean writeZipFile = this.usePemFormat(options);
            Path output = this.resolveOutputPath(terminal, options, writeZipFile ? CertificateTool.DEFAULT_CA_ZIP : CertificateTool.DEFAULT_CA_P12);
            this.writeCertificateAuthority(output, caInfo, writeZipFile, terminal);
        }

        private void writeCertificateAuthority(Path output, CAInfo caInfo, boolean writePemZip, Terminal terminal) throws Exception {
            CertificateTool.checkDirectory(output, terminal);
            if (writePemZip) {
                CertificateTool.fullyWriteZipFile(output, (outputStream, pemWriter) -> CertificateAuthorityCommand.writeCAInfo(outputStream, pemWriter, caInfo, true));
            } else {
                String fileName = output.getFileName().toString();
                CertificateTool.fullyWriteFile(output, (CheckedConsumer<OutputStream, Exception>)((CheckedConsumer)outputStream -> CertificateAuthorityCommand.writePkcs12(fileName, outputStream, "ca", caInfo.certAndKey, null, caInfo.password, terminal)));
            }
        }
    }

    static class CertificateInformation {
        final Name name;
        final List<String> ipAddresses;
        final List<String> dnsNames;
        final List<String> commonNames;

        CertificateInformation(String name, String filename, List<String> ipAddresses, List<String> dnsNames, List<String> commonNames) {
            this.name = Name.fromUserProvidedName(name, filename);
            this.ipAddresses = ipAddresses == null ? Collections.emptyList() : ipAddresses;
            this.dnsNames = dnsNames == null ? Collections.emptyList() : dnsNames;
            this.commonNames = commonNames == null ? Collections.emptyList() : commonNames;
        }

        List<String> validate() {
            ArrayList<String> errors = new ArrayList<String>();
            if (this.name.error != null) {
                errors.add(this.name.error);
            }
            for (String ip : this.ipAddresses) {
                if (InetAddresses.isInetAddress((String)ip)) continue;
                errors.add("[" + ip + "] is not a valid IP address");
            }
            for (String dnsName : this.dnsNames) {
                if (DERIA5String.isIA5String((String)dnsName)) continue;
                errors.add("[" + dnsName + "] is not a valid DNS name");
            }
            return errors;
        }
    }

    static class Name {
        final String originalName;
        final X500Principal x500Principal;
        final String filename;
        final String error;

        private Name(String name, X500Principal x500Principal, String filename, String error) {
            this.originalName = name;
            this.x500Principal = x500Principal;
            this.filename = filename;
            this.error = error;
        }

        static Name fromUserProvidedName(String name, String filename) {
            X500Principal principal;
            if ("ca".equals(name)) {
                return new Name(name, null, null, "[ca] may not be used as an instance name");
            }
            if (name == null) {
                return new Name("", null, null, "instance name may not be null");
            }
            try {
                principal = name.contains("=") ? new X500Principal(name) : new X500Principal("CN=" + name);
            }
            catch (IllegalArgumentException e) {
                String error = "[" + name + "] could not be converted to a valid DN\n" + e.getMessage() + "\n" + ExceptionsHelper.stackTrace((Throwable)e);
                return new Name(name, null, null, error);
            }
            boolean validFilename = Name.isValidFilename(filename);
            if (!validFilename) {
                return new Name(name, principal, null, "[" + filename + "] is not a valid filename");
            }
            return new Name(name, principal, CertificateTool.resolvePath(filename).toString(), null);
        }

        static boolean isValidFilename(String name) {
            return ALLOWED_FILENAME_CHAR_PATTERN.matcher(name).matches() && ALLOWED_FILENAME_CHAR_PATTERN.matcher(CertificateTool.resolvePath(name).toString()).matches() && !name.startsWith(".");
        }

        public String toString() {
            return this.getClass().getSimpleName() + "{original=[" + this.originalName + "] principal=[" + this.x500Principal + "] file=[" + this.filename + "] err=[" + this.error + "]}";
        }
    }

    private static class CertificateToolParser {
        private static final ObjectParser<List<CertificateInformation>, Void> PARSER = new ObjectParser("certgen");

        private CertificateToolParser() {
        }

        static {
            ConstructingObjectParser instanceParser = new ConstructingObjectParser("instances", a -> new CertificateInformation((String)a[0], (String)(a[1] == null ? a[0] : a[1]), (List)a[2], (List)a[3], (List)a[4]));
            instanceParser.declareString(ConstructingObjectParser.constructorArg(), new ParseField("name", new String[0]));
            instanceParser.declareString(ConstructingObjectParser.optionalConstructorArg(), new ParseField("filename", new String[0]));
            instanceParser.declareStringArray(ConstructingObjectParser.optionalConstructorArg(), new ParseField("ip", new String[0]));
            instanceParser.declareStringArray(ConstructingObjectParser.optionalConstructorArg(), new ParseField("dns", new String[0]));
            instanceParser.declareStringArray(ConstructingObjectParser.optionalConstructorArg(), new ParseField("cn", new String[0]));
            PARSER.declareObjectArray(List::addAll, (ContextParser)instanceParser, new ParseField("instances", new String[0]));
        }
    }

    private static interface Writer {
        public void write(ZipOutputStream var1, JcaPEMWriter var2) throws Exception;
    }

    static class CAInfo {
        final CertificateAndKey certAndKey;
        final boolean generated;
        final char[] password;

        CAInfo(X509Certificate caCert, PrivateKey privateKey) {
            this(caCert, privateKey, false, null);
        }

        CAInfo(X509Certificate caCert, PrivateKey privateKey, boolean generated, char[] password) {
            this.certAndKey = new CertificateAndKey(caCert, privateKey);
            this.generated = generated;
            this.password = password;
        }
    }

    static class CertificateAndKey {
        final X509Certificate cert;
        final PrivateKey key;

        CertificateAndKey(X509Certificate cert, PrivateKey key) {
            this.cert = cert;
            this.key = key;
        }
    }

    static abstract class CertificateCommand
    extends EnvironmentAwareCommand {
        final OptionSpec<String> outputPathSpec;
        final OptionSpec<String> outputPasswordSpec;
        final OptionSpec<Integer> keysizeSpec;
        OptionSpec<Void> pemFormatSpec;
        OptionSpec<Integer> daysSpec;
        OptionSpec<String> caPkcs12PathSpec;
        OptionSpec<String> caCertPathSpec;
        OptionSpec<String> caKeyPathSpec;
        OptionSpec<String> caPasswordSpec;
        OptionSpec<String> caDnSpec;
        OptionSpec<Void> multipleNodesSpec;
        OptionSpec<String> nameSpec;
        OptionSpec<String> dnsNamesSpec;
        OptionSpec<String> ipAddressesSpec;
        OptionSpec<String> inputFileSpec;

        CertificateCommand(String description) {
            super(description);
            this.outputPathSpec = this.parser.accepts("out", "path to the output file that should be produced").withRequiredArg();
            this.outputPasswordSpec = this.parser.accepts("pass", "password for generated private keys").withOptionalArg();
            this.keysizeSpec = this.parser.accepts("keysize", "size in bits of RSA keys").withRequiredArg().ofType(Integer.class);
        }

        final void acceptCertificateGenerationOptions() {
            this.pemFormatSpec = this.parser.accepts("pem", "output certificates and keys in PEM format instead of PKCS#12");
            this.daysSpec = this.parser.accepts("days", "number of days that the generated certificates are valid").withRequiredArg().ofType(Integer.class);
        }

        final void acceptsCertificateAuthority() {
            this.caPkcs12PathSpec = this.parser.accepts("ca", "path to an existing ca key pair (in PKCS#12 format)").withRequiredArg();
            this.caCertPathSpec = this.parser.accepts("ca-cert", "path to an existing ca certificate").availableUnless(this.caPkcs12PathSpec, new OptionSpec[0]).withRequiredArg();
            this.caKeyPathSpec = this.parser.accepts("ca-key", "path to an existing ca private key").availableIf(this.caCertPathSpec, new OptionSpec[0]).requiredIf(this.caCertPathSpec, new OptionSpec[0]).withRequiredArg();
            this.caPasswordSpec = this.parser.accepts("ca-pass", "password for an existing ca private key or the generated ca private key").withOptionalArg();
            this.acceptsCertificateAuthorityName();
        }

        void acceptsCertificateAuthorityName() {
            OptionSpecBuilder builder = this.parser.accepts("ca-dn", "distinguished name to use for the generated ca. defaults to CN=Elastic Certificate Tool Autogenerated CA");
            if (this.caPkcs12PathSpec != null) {
                builder = builder.availableUnless(this.caPkcs12PathSpec, new OptionSpec[0]);
            }
            if (this.caCertPathSpec != null) {
                builder = builder.availableUnless(this.caCertPathSpec, new OptionSpec[0]);
            }
            this.caDnSpec = builder.withRequiredArg();
        }

        final void acceptInstanceDetails() {
            this.multipleNodesSpec = this.parser.accepts("multiple", "generate files for multiple instances");
            this.nameSpec = this.parser.accepts("name", "name of the generated certificate").availableUnless(this.multipleNodesSpec, new OptionSpec[0]).withRequiredArg();
            this.dnsNamesSpec = this.parser.accepts("dns", "comma separated DNS names").availableUnless(this.multipleNodesSpec, new OptionSpec[0]).withRequiredArg();
            this.ipAddressesSpec = this.parser.accepts("ip", "comma separated IP addresses").availableUnless(this.multipleNodesSpec, new OptionSpec[0]).withRequiredArg();
        }

        final void acceptInputFile() {
            this.inputFileSpec = this.parser.accepts("in", "file containing details of the instances in yaml format").withRequiredArg();
        }

        OptionParser getParser() {
            return this.parser;
        }

        Path resolveOutputPath(Terminal terminal, OptionSet options, String defaultFilename) throws IOException {
            return CertificateCommand.resolveOutputPath(terminal, (String)this.outputPathSpec.value(options), defaultFilename);
        }

        static Path resolveOutputPath(Terminal terminal, String userOption, String defaultFilename) {
            Path file;
            if (userOption != null) {
                file = CertificateTool.resolvePath(userOption);
            } else {
                file = CertificateTool.resolvePath(defaultFilename);
                String input = terminal.readText("Please enter the desired output file [" + file + "]: ");
                if (!input.isEmpty()) {
                    file = CertificateTool.resolvePath(input);
                }
            }
            return file.toAbsolutePath();
        }

        final int getKeySize(OptionSet options) {
            if (options.has(this.keysizeSpec)) {
                return (Integer)this.keysizeSpec.value(options);
            }
            return 2048;
        }

        final int getDays(OptionSet options) {
            if (options.has(this.daysSpec)) {
                return (Integer)this.daysSpec.value(options);
            }
            return 1095;
        }

        boolean usePemFormat(OptionSet options) {
            return options.has(this.pemFormatSpec);
        }

        boolean useOutputPassword(OptionSet options) {
            return options.has(this.outputPasswordSpec);
        }

        char[] getOutputPassword(OptionSet options) {
            return CertificateTool.getChars((String)this.outputPasswordSpec.value(options));
        }

        protected Path resolvePath(OptionSet options, OptionSpec<String> spec) {
            String value = (String)spec.value(options);
            if (Strings.isNullOrEmpty((String)value)) {
                return null;
            }
            return CertificateTool.resolvePath(value);
        }

        CAInfo getCAInfo(Terminal terminal, OptionSet options, Environment env) throws Exception {
            if (options.has(this.caPkcs12PathSpec)) {
                return this.loadPkcs12CA(terminal, options, env);
            }
            if (options.has(this.caCertPathSpec)) {
                return this.loadPemCA(terminal, options, env);
            }
            terminal.println((CharSequence)"Note: Generating certificates without providing a CA certificate is deprecated.");
            terminal.println((CharSequence)"      A CA certificate will become mandatory in the next major release.");
            terminal.println((CharSequence)"");
            return this.generateCA(terminal, options);
        }

        private CAInfo loadPkcs12CA(Terminal terminal, OptionSet options, Environment env) throws Exception {
            Path path = this.resolvePath(options, this.caPkcs12PathSpec);
            char[] passwordOption = CertificateTool.getChars((String)this.caPasswordSpec.value(options));
            Map keys = (Map)CertificateTool.withPassword("CA (" + path + ")", passwordOption, terminal, false, password -> CertParsingUtils.readPkcs12KeyPairs((Path)path, (char[])password, a -> password));
            if (keys.size() != 1) {
                throw new IllegalArgumentException("expected a single key in file [" + path.toAbsolutePath() + "] but found [" + keys.size() + "]");
            }
            Map.Entry pair = keys.entrySet().iterator().next();
            return new CAInfo((X509Certificate)pair.getKey(), (PrivateKey)pair.getValue());
        }

        private CAInfo loadPemCA(Terminal terminal, OptionSet options, Environment env) throws Exception {
            if (!options.hasArgument(this.caKeyPathSpec)) {
                throw new UserException(64, "Option " + this.caCertPathSpec + " also requires " + this.caKeyPathSpec);
            }
            Path cert = this.resolvePath(options, this.caCertPathSpec);
            Path key = this.resolvePath(options, this.caKeyPathSpec);
            String password = (String)this.caPasswordSpec.value(options);
            X509Certificate caCert = CertParsingUtils.readX509Certificate((Path)cert);
            PrivateKey privateKey = CertificateTool.readPrivateKey(key, CertificateTool.getChars(password), terminal);
            return new CAInfo(caCert, privateKey);
        }

        CAInfo generateCA(Terminal terminal, OptionSet options) throws Exception {
            String dn = (String)this.caDnSpec.value(options);
            if (Strings.isNullOrEmpty((String)dn)) {
                dn = CertificateTool.AUTO_GEN_CA_DN;
            }
            X500Principal x500Principal = new X500Principal(dn);
            KeyPair keyPair = CertGenUtils.generateKeyPair(this.getKeySize(options));
            X509Certificate caCert = CertGenUtils.generateCACertificate(x500Principal, keyPair, this.getDays(options));
            if (options.hasArgument(this.caPasswordSpec)) {
                char[] password = CertificateTool.getChars((String)this.caPasswordSpec.value(options));
                CertificateTool.checkAndConfirmPasswordLengthForOpenSSLCompatibility(password, terminal, false);
                return new CAInfo(caCert, keyPair.getPrivate(), true, password);
            }
            if (options.has(this.caPasswordSpec)) {
                return (CAInfo)CertificateTool.withPassword("CA Private key", null, terminal, true, p -> new CAInfo(caCert, keyPair.getPrivate(), true, (char[])p.clone()));
            }
            return new CAInfo(caCert, keyPair.getPrivate(), true, null);
        }

        Collection<CertificateInformation> getCertificateInformationList(Terminal terminal, OptionSet options) throws Exception {
            String name;
            Path input = this.resolvePath(options, this.inputFileSpec);
            if (input != null) {
                return CertificateTool.parseAndValidateFile(terminal, input.toAbsolutePath());
            }
            if (options.has(this.multipleNodesSpec)) {
                return CertificateCommand.readMultipleCertificateInformation(terminal);
            }
            Function<String, Stream> splitByComma = v -> Arrays.stream(Strings.splitStringByCommaToArray((String)v));
            List<String> dns = this.dnsNamesSpec.values(options).stream().flatMap(splitByComma).collect(Collectors.toList());
            List<String> ip = this.ipAddressesSpec.values(options).stream().flatMap(splitByComma).collect(Collectors.toList());
            List<String> cn = null;
            String fileName = Name.isValidFilename(name = this.getCertificateName(options)) ? name : CertificateCommand.requestFileName(terminal, name);
            CertificateInformation information = new CertificateInformation(name, fileName, ip, dns, cn);
            List<String> validationErrors = information.validate();
            if (validationErrors.isEmpty()) {
                return Collections.singleton(information);
            }
            validationErrors.forEach(arg_0 -> ((Terminal)terminal).errorPrintln(arg_0));
            return Collections.emptyList();
        }

        protected String getCertificateName(OptionSet options) {
            return options.has(this.nameSpec) ? (String)this.nameSpec.value(options) : CertificateTool.DEFAULT_CERT_NAME;
        }

        static Collection<CertificateInformation> readMultipleCertificateInformation(Terminal terminal) {
            HashMap<String, CertificateInformation> map = new HashMap<String, CertificateInformation>();
            boolean done = false;
            while (!done) {
                String exit;
                String name = terminal.readText("Enter instance name: ");
                if (!name.isEmpty()) {
                    List<String> commonNames;
                    List<String> dnsList;
                    String filename = CertificateCommand.requestFileName(terminal, name);
                    String ipAddresses = terminal.readText("Enter IP Addresses for instance (comma-separated if more than one) []: ");
                    String dnsNames = terminal.readText("Enter DNS names for instance (comma-separated if more than one) []: ");
                    List<String> ipList = Arrays.asList(Strings.splitStringByCommaToArray((String)ipAddresses));
                    CertificateInformation information = new CertificateInformation(name, filename, ipList, dnsList = Arrays.asList(Strings.splitStringByCommaToArray((String)dnsNames)), commonNames = null);
                    List<String> validationErrors = information.validate();
                    if (validationErrors.isEmpty()) {
                        if (map.containsKey(name)) {
                            terminal.println((CharSequence)("Overwriting previously defined instance information [" + name + "]"));
                        }
                        map.put(name, information);
                    } else {
                        for (String validationError : validationErrors) {
                            terminal.println((CharSequence)validationError);
                        }
                        terminal.println((CharSequence)"Skipping entry as invalid values were found");
                    }
                } else {
                    terminal.println((CharSequence)"A name must be provided");
                }
                if ("y".equals(exit = terminal.readText("Would you like to specify another instance? Press 'y' to continue entering instance information: "))) continue;
                done = true;
            }
            return map.values();
        }

        private static String requestFileName(Terminal terminal, String certName) {
            boolean isNameValidFilename = Name.isValidFilename(certName);
            String filename;
            while (!(filename = terminal.readText("Enter name for directories and files of " + certName + (String)(isNameValidFilename ? " [" + certName + "]" : "") + ": ")).isEmpty() || !isNameValidFilename) {
                if (Name.isValidFilename(filename)) {
                    return filename;
                }
                terminal.errorPrintln(Terminal.Verbosity.SILENT, "'" + filename + "' is not a valid filename");
            }
            return certName;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        static void writeCAInfo(ZipOutputStream outputStream, JcaPEMWriter pemWriter, CAInfo info, boolean includeKey) throws Exception {
            String caDirName = CertificateCommand.createCaDirectory(outputStream);
            outputStream.putNextEntry(new ZipEntry(caDirName + "ca.crt"));
            pemWriter.writeObject((Object)info.certAndKey.cert);
            pemWriter.flush();
            outputStream.closeEntry();
            if (includeKey) {
                outputStream.putNextEntry(new ZipEntry(caDirName + "ca.key"));
                if (info.password != null && info.password.length > 0) {
                    try {
                        PEMEncryptor encryptor = CertificateTool.getEncrypter(info.password);
                        pemWriter.writeObject((Object)info.certAndKey.key, encryptor);
                    }
                    finally {
                        Arrays.fill(info.password, '\u0000');
                    }
                } else {
                    pemWriter.writeObject((Object)info.certAndKey.key);
                }
                pemWriter.flush();
                outputStream.closeEntry();
            }
        }

        private static String createCaDirectory(ZipOutputStream outputStream) throws IOException {
            String caDirName = "ca/";
            ZipEntry zipEntry = new ZipEntry("ca/");
            assert (zipEntry.isDirectory());
            outputStream.putNextEntry(zipEntry);
            return "ca/";
        }

        static void writePkcs12(String fileName, OutputStream output, String alias, CertificateAndKey pair, X509Certificate caCert, char[] password, Terminal terminal) throws Exception {
            KeyStore pkcs12 = KeyStore.getInstance("PKCS12");
            pkcs12.load(null);
            CertificateTool.withPassword(fileName, password, terminal, true, p12Password -> {
                if (CertificateTool.isAscii(p12Password)) {
                    pkcs12.setKeyEntry(alias, pair.key, (char[])p12Password, new Certificate[]{pair.cert});
                    if (caCert != null) {
                        pkcs12.setCertificateEntry("ca", caCert);
                    }
                    pkcs12.store(output, (char[])p12Password);
                    return null;
                }
                throw new UserException(78, "PKCS#12 passwords must be plain ASCII");
            });
        }
    }
}

