/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.ml.action.OpenJobAction;
import org.elasticsearch.xpack.core.ml.action.StartDataFrameAnalyticsAction;
import org.elasticsearch.xpack.core.ml.action.StartDatafeedAction;
import org.elasticsearch.xpack.core.ml.utils.MlTaskParams;
import org.elasticsearch.xpack.ml.notifications.AnomalyDetectionAuditor;
import org.elasticsearch.xpack.ml.notifications.DataFrameAnalyticsAuditor;

public class MlAssignmentNotifier
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(MlAssignmentNotifier.class);
    static final Duration MIN_CHECK_UNASSIGNED_INTERVAL = Duration.ofSeconds(30L);
    static final Duration LONG_TIME_UNASSIGNED_INTERVAL = Duration.ofMinutes(15L);
    static final Duration MIN_REPORT_INTERVAL = Duration.ofHours(6L);
    private final AnomalyDetectionAuditor anomalyDetectionAuditor;
    private final DataFrameAnalyticsAuditor dataFrameAnalyticsAuditor;
    private final ThreadPool threadPool;
    private final Clock clock;
    private Map<TaskNameAndId, UnassignedTimeAndReportTime> unassignedInfoByTask = Map.of();
    private volatile Instant lastLogCheck;

    MlAssignmentNotifier(AnomalyDetectionAuditor anomalyDetectionAuditor, DataFrameAnalyticsAuditor dataFrameAnalyticsAuditor, ThreadPool threadPool, ClusterService clusterService) {
        this(anomalyDetectionAuditor, dataFrameAnalyticsAuditor, threadPool, clusterService, Clock.systemUTC());
    }

    MlAssignmentNotifier(AnomalyDetectionAuditor anomalyDetectionAuditor, DataFrameAnalyticsAuditor dataFrameAnalyticsAuditor, ThreadPool threadPool, ClusterService clusterService, Clock clock) {
        this.anomalyDetectionAuditor = anomalyDetectionAuditor;
        this.dataFrameAnalyticsAuditor = dataFrameAnalyticsAuditor;
        this.threadPool = threadPool;
        this.clock = clock;
        this.lastLogCheck = clock.instant();
        clusterService.addListener((ClusterStateListener)this);
    }

    private static String executorName() {
        return "generic";
    }

    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.localNodeMaster()) {
            this.unassignedInfoByTask = Map.of();
            return;
        }
        Instant now = this.clock.instant();
        if (this.lastLogCheck.plus(MIN_CHECK_UNASSIGNED_INTERVAL).isBefore(now)) {
            this.lastLogCheck = now;
            this.threadPool.executor(MlAssignmentNotifier.executorName()).execute(() -> this.logLongTimeUnassigned(now, event.state()));
        }
        if (!event.metadataChanged()) {
            return;
        }
        this.threadPool.executor(MlAssignmentNotifier.executorName()).execute(() -> this.auditChangesToMlTasks(event));
    }

    private void auditChangesToMlTasks(ClusterChangedEvent event) {
        PersistentTasksCustomMetadata currentTasks;
        PersistentTasksCustomMetadata previousTasks = (PersistentTasksCustomMetadata)event.previousState().getMetadata().custom("persistent_tasks");
        if (Objects.equals(previousTasks, currentTasks = (PersistentTasksCustomMetadata)event.state().getMetadata().custom("persistent_tasks"))) {
            return;
        }
        this.auditMlTasks(event.previousState().nodes(), event.state().nodes(), previousTasks, currentTasks, false);
    }

    public void auditUnassignedMlTasks(DiscoveryNodes nodes, PersistentTasksCustomMetadata tasks) {
        this.auditMlTasks(nodes, nodes, tasks, tasks, true);
    }

    private void auditMlTasks(DiscoveryNodes previousNodes, DiscoveryNodes currentNodes, PersistentTasksCustomMetadata previousTasks, PersistentTasksCustomMetadata currentTasks, boolean alwaysAuditUnassigned) {
        if (currentTasks == null) {
            return;
        }
        for (PersistentTasksCustomMetadata.PersistentTask currentTask : currentTasks.tasks()) {
            String nodeName;
            String nodeName2;
            boolean wasTaskAssigned;
            boolean isTaskAssigned;
            PersistentTasksCustomMetadata.Assignment currentAssignment = currentTask.getAssignment();
            PersistentTasksCustomMetadata.PersistentTask previousTask = previousTasks != null ? previousTasks.getTask(currentTask.getId()) : null;
            PersistentTasksCustomMetadata.Assignment previousAssignment = previousTask != null ? previousTask.getAssignment() : null;
            boolean bl = isTaskAssigned = currentAssignment.getExecutorNode() != null;
            if (Objects.equals(currentAssignment, previousAssignment) && (isTaskAssigned || !alwaysAuditUnassigned)) continue;
            boolean bl2 = wasTaskAssigned = previousAssignment != null && previousAssignment.getExecutorNode() != null;
            if ("xpack/ml/job".equals(currentTask.getTaskName())) {
                String jobId = ((OpenJobAction.JobParams)currentTask.getParams()).getJobId();
                if (isTaskAssigned) {
                    Object msg = "Opening job";
                    if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                        nodeName2 = MlAssignmentNotifier.nodeName(currentNodes, currentAssignment.getExecutorNode());
                        msg = (String)msg + " on node [" + nodeName2 + "]";
                    }
                    this.anomalyDetectionAuditor.info(jobId, (String)msg);
                    continue;
                }
                if (alwaysAuditUnassigned) {
                    if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                        this.anomalyDetectionAuditor.warning(jobId, "No node found to open job. Reasons [" + currentAssignment.getExplanation() + "]");
                        continue;
                    }
                    this.anomalyDetectionAuditor.warning(jobId, "Awaiting capacity to open job.");
                    continue;
                }
                if (!wasTaskAssigned) continue;
                if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                    nodeName = MlAssignmentNotifier.nodeName(previousNodes, previousAssignment.getExecutorNode());
                    this.anomalyDetectionAuditor.info(jobId, "Job unassigned from node [" + nodeName + "]");
                    continue;
                }
                this.anomalyDetectionAuditor.info(jobId, "Job relocating.");
                continue;
            }
            if ("xpack/ml/datafeed".equals(currentTask.getTaskName())) {
                StartDatafeedAction.DatafeedParams datafeedParams = (StartDatafeedAction.DatafeedParams)currentTask.getParams();
                String jobId = datafeedParams.getJobId();
                if (jobId == null) continue;
                if (isTaskAssigned) {
                    String msg = "Starting datafeed [" + datafeedParams.getDatafeedId() + "]";
                    if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                        String nodeName3 = MlAssignmentNotifier.nodeName(currentNodes, currentAssignment.getExecutorNode());
                        msg = msg + "] on node [" + nodeName3 + "]";
                    }
                    this.anomalyDetectionAuditor.info(jobId, msg);
                    continue;
                }
                if (alwaysAuditUnassigned) {
                    if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                        this.anomalyDetectionAuditor.warning(jobId, "No node found to start datafeed [" + datafeedParams.getDatafeedId() + "]. Reasons [" + currentAssignment.getExplanation() + "]");
                        continue;
                    }
                    this.anomalyDetectionAuditor.warning(jobId, "Awaiting capacity to start datafeed [" + datafeedParams.getDatafeedId() + "].");
                    continue;
                }
                if (wasTaskAssigned) {
                    if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                        nodeName2 = MlAssignmentNotifier.nodeName(previousNodes, previousAssignment.getExecutorNode());
                        this.anomalyDetectionAuditor.info(jobId, "Datafeed [" + datafeedParams.getDatafeedId() + "] unassigned from node [" + nodeName2 + "]");
                        continue;
                    }
                    this.anomalyDetectionAuditor.info(jobId, "Datafeed [" + datafeedParams.getDatafeedId() + "] relocating.");
                    continue;
                }
                logger.warn("[{}] No node found to start datafeed [{}]. Reasons [{}]", (Object)jobId, (Object)datafeedParams.getDatafeedId(), (Object)currentAssignment.getExplanation());
                continue;
            }
            if (!"xpack/ml/data_frame/analytics".equals(currentTask.getTaskName())) continue;
            String id = ((StartDataFrameAnalyticsAction.TaskParams)currentTask.getParams()).getId();
            if (isTaskAssigned) {
                if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                    nodeName = MlAssignmentNotifier.nodeName(currentNodes, currentAssignment.getExecutorNode());
                    this.dataFrameAnalyticsAuditor.info(id, "Starting analytics on node [" + nodeName + "]");
                    continue;
                }
                this.dataFrameAnalyticsAuditor.info(id, "Starting analytics.");
                continue;
            }
            if (alwaysAuditUnassigned) {
                if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                    this.dataFrameAnalyticsAuditor.warning(id, "No node found to start analytics. Reasons [" + currentAssignment.getExplanation() + "]");
                    continue;
                }
                this.dataFrameAnalyticsAuditor.warning(id, "Awaiting capacity to start analytics.");
                continue;
            }
            if (!wasTaskAssigned) continue;
            if (this.anomalyDetectionAuditor.includeNodeInfo()) {
                nodeName = MlAssignmentNotifier.nodeName(previousNodes, previousAssignment.getExecutorNode());
                this.anomalyDetectionAuditor.info(id, "Analytics unassigned from node [" + nodeName + "]");
                continue;
            }
            this.anomalyDetectionAuditor.info(id, "Analytics relocating.");
        }
    }

    static String nodeName(DiscoveryNodes nodes, String nodeId) {
        DiscoveryNode node = nodes.get(nodeId);
        if (node != null && Strings.hasLength((String)node.getName())) {
            return node.getName();
        }
        return nodeId;
    }

    private void logLongTimeUnassigned(Instant now, ClusterState state) {
        PersistentTasksCustomMetadata tasks = (PersistentTasksCustomMetadata)state.getMetadata().custom("persistent_tasks");
        if (tasks == null) {
            return;
        }
        List<String> itemsToReport = this.findLongTimeUnassignedTasks(now, tasks);
        if (!itemsToReport.isEmpty()) {
            logger.warn("ML persistent tasks unassigned for a long time [{}]", (Object)String.join((CharSequence)"|", itemsToReport));
        }
    }

    synchronized List<String> findLongTimeUnassignedTasks(Instant now, PersistentTasksCustomMetadata tasks) {
        assert (tasks != null);
        ArrayList<String> itemsToReport = new ArrayList<String>();
        Map<TaskNameAndId, UnassignedTimeAndReportTime> oldUnassignedInfoByTask = this.unassignedInfoByTask;
        HashMap<TaskNameAndId, UnassignedTimeAndReportTime> newUnassignedInfoByTask = new HashMap<TaskNameAndId, UnassignedTimeAndReportTime>();
        for (PersistentTasksCustomMetadata.PersistentTask task : tasks.tasks()) {
            Instant lastReportedTime;
            Instant firstUnassignedTime;
            String taskName;
            if (task.getExecutorNode() != null || !"xpack/ml/job".equals(taskName = task.getTaskName()) && !"xpack/ml/data_frame/analytics".equals(taskName)) continue;
            String mlId = ((MlTaskParams)task.getParams()).getMlId();
            TaskNameAndId key = new TaskNameAndId(taskName, mlId);
            UnassignedTimeAndReportTime previousInfo = oldUnassignedInfoByTask.get(key);
            if (previousInfo != null) {
                firstUnassignedTime = previousInfo.unassignedTime();
                if (firstUnassignedTime.plus(LONG_TIME_UNASSIGNED_INTERVAL).isBefore(now) && (previousInfo.reportTime() == null || previousInfo.reportTime().plus(MIN_REPORT_INTERVAL).isBefore(now))) {
                    lastReportedTime = now;
                    itemsToReport.add(Strings.format((String)"[%s]/[%s] unassigned for [%d] seconds", (Object[])new Object[]{taskName, mlId, ChronoUnit.SECONDS.between(firstUnassignedTime, now)}));
                } else {
                    lastReportedTime = previousInfo.reportTime();
                }
            } else {
                firstUnassignedTime = now;
                lastReportedTime = null;
            }
            newUnassignedInfoByTask.put(key, new UnassignedTimeAndReportTime(firstUnassignedTime, lastReportedTime));
        }
        this.unassignedInfoByTask = newUnassignedInfoByTask;
        return itemsToReport;
    }

    private record TaskNameAndId(String taskName, String mlId) {
    }

    private record UnassignedTimeAndReportTime(Instant unassignedTime, Instant reportTime) {
    }
}

