/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.RefCountingRunnable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.env.Environment;
import org.elasticsearch.license.LicensedFeature;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.node.Node;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.security.authc.Authentication;
import org.elasticsearch.xpack.core.security.authc.DomainConfig;
import org.elasticsearch.xpack.core.security.authc.Realm;
import org.elasticsearch.xpack.core.security.authc.RealmConfig;
import org.elasticsearch.xpack.core.security.authc.RealmDomain;
import org.elasticsearch.xpack.core.security.authc.RealmSettings;
import org.elasticsearch.xpack.security.Security;
import org.elasticsearch.xpack.security.authc.InternalRealms;
import org.elasticsearch.xpack.security.authc.esnative.ReservedRealm;

public class Realms
extends AbstractLifecycleComponent
implements Iterable<Realm> {
    private static final Logger logger = LogManager.getLogger(Realms.class);
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger((String)logger.getName());
    private final Settings settings;
    private final Environment env;
    private final Map<String, Realm.Factory> factories;
    private final XPackLicenseState licenseState;
    private final ThreadContext threadContext;
    private final ReservedRealm reservedRealm;
    private final List<Realm> allConfiguredRealms;
    private final Map<String, DomainConfig> domainNameToConfig;
    private final Map<RealmConfig.RealmIdentifier, Authentication.RealmRef> realmRefs;
    private volatile List<Realm> activeRealms;

    public Realms(Settings settings, Environment env, Map<String, Realm.Factory> factories, XPackLicenseState licenseState, ThreadContext threadContext, ReservedRealm reservedRealm) throws Exception {
        this.settings = settings;
        this.env = env;
        this.factories = factories;
        this.licenseState = licenseState;
        this.threadContext = threadContext;
        this.reservedRealm = reservedRealm;
        assert (((Boolean)XPackSettings.SECURITY_ENABLED.get(settings)).booleanValue()) : "security must be enabled";
        assert (factories.get("reserved") == null);
        Map realmToDomainConfig = RealmSettings.computeRealmNameToDomainConfigAssociation((Settings)settings);
        this.domainNameToConfig = realmToDomainConfig.values().stream().distinct().collect(Collectors.toMap(DomainConfig::name, Function.identity()));
        List<RealmConfig> realmConfigs = this.buildRealmConfigs();
        List<Realm> initialRealms = this.initRealms(realmConfigs);
        this.realmRefs = this.calculateRealmRefs(realmConfigs, realmToDomainConfig);
        initialRealms.forEach(realm -> realm.initRealmRef(this.realmRefs));
        this.allConfiguredRealms = initialRealms;
        this.allConfiguredRealms.forEach(r -> r.initialize(this.allConfiguredRealms, licenseState));
        assert (this.allConfiguredRealms.get(0) == reservedRealm) : "the first realm must be reserved realm";
        this.activeRealms = this.calculateLicensedRealms(licenseState.copyCurrentLicenseState());
        licenseState.addListener(this::recomputeActiveRealms);
    }

    private Map<RealmConfig.RealmIdentifier, Authentication.RealmRef> calculateRealmRefs(Collection<RealmConfig> realmConfigs, Map<String, DomainConfig> domainForRealm) {
        String nodeName = (String)Node.NODE_NAME_SETTING.get(this.settings);
        HashMap<RealmConfig.RealmIdentifier, Authentication.RealmRef> realmRefs = new HashMap<RealmConfig.RealmIdentifier, Authentication.RealmRef>();
        assert (realmConfigs.stream().noneMatch(rc -> rc.name().equals(this.reservedRealm.name()) && rc.type().equals(this.reservedRealm.type()))) : "reserved realm cannot be configured";
        realmRefs.put(new RealmConfig.RealmIdentifier(this.reservedRealm.type(), this.reservedRealm.name()), new Authentication.RealmRef(this.reservedRealm.name(), this.reservedRealm.type(), nodeName, null));
        for (RealmConfig realmConfig : realmConfigs) {
            RealmDomain realmDomain;
            RealmConfig.RealmIdentifier realmIdentifier = new RealmConfig.RealmIdentifier(realmConfig.type(), realmConfig.name());
            DomainConfig domainConfig = domainForRealm.get(realmConfig.name());
            if (domainConfig != null) {
                String domainName = domainConfig.name();
                HashSet<RealmConfig.RealmIdentifier> domainIdentifiers = new HashSet<RealmConfig.RealmIdentifier>();
                for (RealmConfig otherRealmConfig : realmConfigs) {
                    DomainConfig otherDomainConfig = domainForRealm.get(otherRealmConfig.name());
                    if (otherDomainConfig == null || !domainName.equals(otherDomainConfig.name())) continue;
                    domainIdentifiers.add(otherRealmConfig.identifier());
                }
                realmDomain = new RealmDomain(domainName, domainIdentifiers);
            } else {
                realmDomain = null;
            }
            realmRefs.put(realmIdentifier, new Authentication.RealmRef(realmIdentifier.getName(), realmIdentifier.getType(), nodeName, realmDomain));
        }
        return Map.copyOf(realmRefs);
    }

    protected void recomputeActiveRealms() {
        XPackLicenseState licenseStateSnapshot = this.licenseState.copyCurrentLicenseState();
        List<Realm> licensedRealms = this.calculateLicensedRealms(licenseStateSnapshot);
        logger.info("license mode is [{}], currently licensed security realms are [{}]", (Object)licenseStateSnapshot.getOperationMode().description(), (Object)Strings.collectionToCommaDelimitedString(licensedRealms));
        if (this.activeRealms != null) {
            this.activeRealms.stream().filter(r -> !licensedRealms.contains(r)).forEach(realm -> this.handleDisabledRealmDueToLicenseChange((Realm)realm, licenseStateSnapshot));
        }
        this.activeRealms = licensedRealms;
    }

    protected void handleDisabledRealmDueToLicenseChange(Realm realm, XPackLicenseState licenseStateSnapshot) {
        LicensedFeature.Persistent feature = Realms.getLicensedFeatureForRealm(realm.type());
        assert (feature != null) : "Realm [" + realm + "] with no licensed feature became inactive due to change to license mode [" + licenseStateSnapshot.getOperationMode() + "]";
        feature.stopTracking(licenseStateSnapshot, realm.name());
        logger.warn("The [{}.{}] realm has been automatically disabled due to a change in license [{}]", (Object)realm.type(), (Object)realm.name(), (Object)licenseStateSnapshot.statusDescription());
    }

    @Override
    public Iterator<Realm> iterator() {
        return this.getActiveRealms().iterator();
    }

    public List<Realm> getUnlicensedRealms() {
        List<Realm> activeSnapshot = this.activeRealms;
        if (activeSnapshot.equals(this.allConfiguredRealms)) {
            return Collections.emptyList();
        }
        return this.allConfiguredRealms.stream().filter(r -> !activeSnapshot.contains(r)).toList();
    }

    public Stream<Realm> stream() {
        return StreamSupport.stream(this.spliterator(), false);
    }

    public List<Realm> getActiveRealms() {
        assert (this.activeRealms != null) : "Active realms not configured";
        return this.activeRealms;
    }

    public DomainConfig getDomainConfig(String domainName) {
        return this.domainNameToConfig.get(domainName);
    }

    protected List<Realm> calculateLicensedRealms(XPackLicenseState licenseStateSnapshot) {
        return this.allConfiguredRealms.stream().filter(r -> Realms.checkLicense(r, licenseStateSnapshot)).toList();
    }

    private static boolean checkLicense(Realm realm, XPackLicenseState licenseState) {
        LicensedFeature.Persistent feature = Realms.getLicensedFeatureForRealm(realm.type());
        if (feature == null) {
            return true;
        }
        return feature.checkAndStartTracking(licenseState, realm.name());
    }

    public static boolean isRealmTypeAvailable(XPackLicenseState licenseState, String type) {
        LicensedFeature.Persistent feature = Realms.getLicensedFeatureForRealm(type);
        if (feature == null) {
            return true;
        }
        return feature.checkWithoutTracking(licenseState);
    }

    @Nullable
    private static LicensedFeature.Persistent getLicensedFeatureForRealm(String realmType) {
        assert (Strings.hasText((String)realmType)) : "Realm type must be provided (received [" + realmType + "])";
        if (InternalRealms.isInternalRealm(realmType)) {
            return InternalRealms.getLicensedFeature(realmType);
        }
        return Security.CUSTOM_REALMS_FEATURE;
    }

    public Realm realm(String name) {
        for (Realm realm : this.activeRealms) {
            if (!name.equals(realm.name())) continue;
            return realm;
        }
        return null;
    }

    public Realm.Factory realmFactory(String type) {
        return this.factories.get(type);
    }

    protected List<Realm> initRealms(List<RealmConfig> realmConfigs) throws Exception {
        ArrayList<Realm> realms = new ArrayList<Realm>();
        HashMap<String, Set<String>> nameToRealmIdentifier = new HashMap<String, Set<String>>();
        HashMap<Integer, Set<String>> orderToRealmName = new HashMap<Integer, Set<String>>();
        ArrayList<RealmConfig.RealmIdentifier> reservedPrefixedRealmIdentifiers = new ArrayList<RealmConfig.RealmIdentifier>();
        for (RealmConfig config : realmConfigs) {
            Realm.Factory factory = this.factories.get(config.identifier().getType());
            assert (factory != null) : "unknown realm type [" + config.identifier().getType() + "]";
            if (config.identifier().getName().startsWith("_")) {
                reservedPrefixedRealmIdentifiers.add(config.identifier());
            }
            if (!config.enabled()) {
                if (!logger.isDebugEnabled()) continue;
                logger.debug("realm [{}] is disabled", (Object)config.identifier());
                continue;
            }
            Realm realm = factory.create(config);
            nameToRealmIdentifier.computeIfAbsent(realm.name(), k -> new HashSet()).add(RealmSettings.realmSettingPrefix((String)realm.type()) + realm.name());
            orderToRealmName.computeIfAbsent(realm.order(), k -> new HashSet()).add(realm.name());
            realms.add(realm);
        }
        Realms.checkUniqueOrders(orderToRealmName);
        Collections.sort(realms);
        Realms.ensureUniqueExplicitlyConfiguredRealmNames(nameToRealmIdentifier);
        this.maybeAddBasicRealms(realms, realmConfigs);
        this.addReservedRealm(realms);
        Realms.logDeprecationForReservedPrefixedRealmNames(reservedPrefixedRealmIdentifiers);
        return Collections.unmodifiableList(realms);
    }

    public void usageStats(ActionListener<Map<String, Object>> listener) {
        XPackLicenseState licenseStateSnapshot = this.licenseState.copyCurrentLicenseState();
        HashMap realmMap = new HashMap();
        AtomicBoolean failed = new AtomicBoolean(false);
        List<Realm> realmList = this.getActiveRealms().stream().filter(r -> !"reserved".equals(r.type())).toList();
        try (RefCountingRunnable refs = new RefCountingRunnable(() -> {
            if (!failed.get()) {
                for (String type : this.factories.keySet()) {
                    assert (!"reserved".equals(type));
                    realmMap.compute(type, (key, value) -> {
                        if (value == null) {
                            return Map.of("enabled", false, "available", Realms.isRealmTypeAvailable(licenseStateSnapshot, type));
                        }
                        assert (value instanceof Map);
                        Map realmTypeUsage = (Map)value;
                        realmTypeUsage.put("enabled", true);
                        realmTypeUsage.put("available", true);
                        return value;
                    });
                }
                listener.onResponse((Object)realmMap);
            }
        });){
            for (Realm realm : realmList) {
                Releasable ref = refs.acquire();
                realm.usageStats(ActionListener.wrap(stats -> {
                    if (!failed.get()) {
                        Map map = realmMap;
                        synchronized (map) {
                            realmMap.compute(realm.type(), (key, value) -> {
                                if (value == null) {
                                    Map<String, Object> realmTypeUsage = Realms.convertToMapOfLists(stats);
                                    return realmTypeUsage;
                                }
                                assert (value instanceof Map);
                                Realms.combineMaps((Map)value, stats);
                                return value;
                            });
                        }
                        ref.close();
                    }
                }, e -> {
                    if (failed.compareAndSet(false, true)) {
                        listener.onFailure(e);
                    }
                }));
            }
        }
    }

    public Map<String, Object> domainUsageStats() {
        if (this.domainNameToConfig.isEmpty()) {
            return Map.of();
        }
        return this.domainNameToConfig.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> Map.of("realms", ((DomainConfig)entry.getValue()).memberRealmNames())));
    }

    public Map<RealmConfig.RealmIdentifier, Authentication.RealmRef> getRealmRefs() {
        return this.realmRefs;
    }

    protected void doStart() {
    }

    protected void doStop() {
    }

    protected void doClose() throws IOException {
        IOUtils.close(this.allConfiguredRealms.stream().filter(r -> r instanceof Closeable).map(r -> (Closeable)r).toList());
    }

    private void maybeAddBasicRealms(List<Realm> realms, List<RealmConfig> realmConfigs) throws Exception {
        Set<String> disabledBasicRealmTypes = Realms.findDisabledBasicRealmTypes(realmConfigs);
        Set realmTypes = realms.stream().map(Realm::type).collect(Collectors.toUnmodifiableSet());
        if (!disabledBasicRealmTypes.contains("native") && !realmTypes.contains("native")) {
            boolean enabled = this.settings.getAsBoolean("xpack.security.authc.native_users.enabled", Boolean.valueOf(true));
            Realms.ensureRealmNameIsAvailable(realms, "default_native");
            RealmConfig.RealmIdentifier nativeRealmId = new RealmConfig.RealmIdentifier("native", "default_native");
            RealmConfig realmConfig = new RealmConfig(nativeRealmId, Realms.buildSettingsforDefaultRealm(this.settings, nativeRealmId, Integer.MIN_VALUE, enabled), this.env, this.threadContext);
            realmConfigs.add(realmConfig);
            if (enabled) {
                realms.add(0, this.factories.get("native").create(realmConfig));
            }
        }
        if (!disabledBasicRealmTypes.contains("file") && !realmTypes.contains("file")) {
            Realms.ensureRealmNameIsAvailable(realms, "default_file");
            RealmConfig.RealmIdentifier fileRealmId = new RealmConfig.RealmIdentifier("file", "default_file");
            RealmConfig realmConfig = new RealmConfig(fileRealmId, Realms.buildSettingsforDefaultRealm(this.settings, fileRealmId, Integer.MIN_VALUE, true), this.env, this.threadContext);
            realmConfigs.add(realmConfig);
            realms.add(0, this.factories.get("file").create(realmConfig));
        }
    }

    private void addReservedRealm(List<Realm> realms) {
        Realms.ensureRealmNameIsAvailable(realms, "reserved");
        realms.add(0, this.reservedRealm);
    }

    private static void ensureRealmNameIsAvailable(List<Realm> realms, String realmName) {
        assert (realms.size() == realms.stream().map(Realm::name).collect(Collectors.toUnmodifiableSet()).size()) : "existing realm names must be unique";
        Realm misNamedRealm = realms.stream().filter(realm -> realmName.equals(realm.name())).findFirst().orElse(null);
        if (misNamedRealm != null) {
            throw new IllegalArgumentException("Found realm configured with name clashing with the [" + realmName + "] realm: [" + RealmSettings.realmSettingPrefix((String)misNamedRealm.type()) + misNamedRealm.name() + "]");
        }
    }

    private static Settings buildSettingsforDefaultRealm(Settings settings, RealmConfig.RealmIdentifier realmIdentifier, int order, boolean enabled) {
        String prefix = RealmSettings.realmSettingPrefix((RealmConfig.RealmIdentifier)realmIdentifier);
        Settings.Builder builder = Settings.builder().put(settings).put(prefix + "order", order);
        if (!enabled) {
            builder.put(prefix + "enabled", false);
        }
        return builder.build();
    }

    private static void checkUniqueOrders(Map<Integer, Set<String>> orderToRealmName) {
        String duplicateOrders = orderToRealmName.entrySet().stream().filter(entry -> ((Set)entry.getValue()).size() > 1).map(entry -> entry.getKey() + ": " + entry.getValue()).collect(Collectors.joining("; "));
        if (Strings.hasText((String)duplicateOrders)) {
            throw new IllegalArgumentException("Found multiple realms configured with the same order: " + duplicateOrders);
        }
    }

    private static void ensureUniqueExplicitlyConfiguredRealmNames(Map<String, Set<String>> nameToRealmIdentifier) {
        String duplicateRealms = nameToRealmIdentifier.entrySet().stream().filter(entry -> ((Set)entry.getValue()).size() > 1).map(entry -> (String)entry.getKey() + ": " + entry.getValue()).collect(Collectors.joining("; "));
        if (Strings.hasText((String)duplicateRealms)) {
            throw new IllegalArgumentException("Found multiple realms configured with the same name: " + duplicateRealms);
        }
    }

    private List<RealmConfig> buildRealmConfigs() {
        Map realmsSettings = RealmSettings.getRealmSettings((Settings)this.settings);
        HashSet<String> internalTypes = new HashSet<String>();
        ArrayList<String> kerberosRealmNames = new ArrayList<String>();
        ArrayList<RealmConfig> realmConfigs = new ArrayList<RealmConfig>();
        for (RealmConfig.RealmIdentifier identifier : realmsSettings.keySet()) {
            Realm.Factory factory = this.factories.get(identifier.getType());
            if (factory == null) {
                throw new IllegalArgumentException("unknown realm type [" + identifier.getType() + "] for realm [" + identifier + "]");
            }
            RealmConfig config = new RealmConfig(identifier, this.settings, this.env, this.threadContext);
            if (InternalRealms.isBuiltinRealm(identifier.getType())) {
                if (internalTypes.contains(identifier.getType())) {
                    throw new IllegalArgumentException("multiple [" + identifier.getType() + "] realms are configured. [" + identifier.getType() + "] is an internal realm and therefore there can only be one such realm configured");
                }
                internalTypes.add(identifier.getType());
            }
            if ("kerberos".equals(identifier.getType())) {
                kerberosRealmNames.add(identifier.getName());
                if (kerberosRealmNames.size() > 1) {
                    throw new IllegalArgumentException("multiple realms " + ((Object)kerberosRealmNames).toString() + " configured of type [" + identifier.getType() + "], [" + identifier.getType() + "] can only have one such realm configured");
                }
            }
            realmConfigs.add(config);
        }
        return realmConfigs;
    }

    private static Set<String> findDisabledBasicRealmTypes(List<RealmConfig> realmConfigs) {
        return realmConfigs.stream().filter(rc -> InternalRealms.isBuiltinRealm(rc.type())).filter(rc -> false == rc.enabled()).map(RealmConfig::type).collect(Collectors.toUnmodifiableSet());
    }

    private static void logDeprecationForReservedPrefixedRealmNames(List<RealmConfig.RealmIdentifier> realmIdentifiers) {
        if (!realmIdentifiers.isEmpty()) {
            deprecationLogger.warn(DeprecationCategory.SECURITY, "realm_name_with_reserved_prefix", "Found realm " + (realmIdentifiers.size() == 1 ? "name" : "names") + " with reserved prefix [{}]: [{}]. In a future major release, node will fail to start if any realm names start with reserved prefix.", new Object[]{"_", realmIdentifiers.stream().map(rid -> "xpack.security.authc.realms." + rid.getType() + "." + rid.getName()).sorted().collect(Collectors.joining("; "))});
        }
    }

    private static void combineMaps(Map<String, Object> mapA, Map<String, Object> mapB) {
        for (Map.Entry<String, Object> entry : mapB.entrySet()) {
            mapA.compute(entry.getKey(), (key, value) -> {
                if (value == null) {
                    return new ArrayList(Collections.singletonList(entry.getValue()));
                }
                assert (value instanceof List);
                ((List)value).add(entry.getValue());
                return value;
            });
        }
    }

    private static Map<String, Object> convertToMapOfLists(Map<String, Object> map) {
        Map converted = Maps.newMapWithExpectedSize((int)map.size());
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            converted.put(entry.getKey(), new ArrayList<Object>(Collections.singletonList(entry.getValue())));
        }
        return converted;
    }
}

